/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "toolFileFormat.h"

#include <visu_tools.h>
#include <string.h>
#include <stdlib.h>

/**
 * SECTION:toolFileFormat
 * @short_description: Describes file format objects (name patterns,
 * description, properties...).
 *
 * <para>When dealing with files, it is convenient to class then by
 * formats, one format for JPEG files, one other for
 * postscript... Such file formats are characterized by their
 * description, explaining what they are, one or more name pattern
 * (e.g. "*.jpg") and some properties (e.g. compression level for JPEG
 * file).</para>
 * <para>This module describes objects that can store all these
 * informations and deals with them. To create a new file format, use
 * tool_file_format_new(). A #ToolFileFormat object can be used in a
 * GtkFileChooser object, using a GtkFileFilter. tool_file_format_getLabel()
 * will return directly the label to give to the GtkFileFilter and the
 * name patterns can be passed to it also.</para>
 * <para>A file format property is a #ToolOption value. There are some
 * convenient routines to add common boolean or integer
 * properties. For instance, use tool_file_format_addPropertyBoolean()
 * to add a boolean property to a given #ToolFileFormat object. Then the stored
 * properties can be iterated on using a #ToolFileFormatIter iterator.</para>
 */

GQuark tool_file_format_getQuark(void)
{
  return g_quark_from_static_string("ToolFileFormat");
}

struct _ToolFileFormat
{
  /* This is the list of file patterns, for example (*.jpg; *.jpeg; *.jpe). */
  GList *fileType;
  gboolean ignoreFileType;
  
  /* This is a short text to describe this file format.
     It should not be bigger than 30 characters. */
  gchar *name;

  /* This is a private field, it is a concatenation
     of "name ({patterns},)". */
  gchar *labelString;

  /* This is a private field. It enables to add some properties
     to a file format. See fileFormatAdd_property*() and tool_file_format_property_get*()
     to control them. */
  GList *properties;
};

static ToolFileFormat* _new(gchar* descr, gchar** patterns, gboolean uncompleteList)
{
  int nbCharacterMax = 45;
  ToolFileFormat *res;
  int lg, n;

  lg = strlen(descr);
  if (lg > nbCharacterMax)
    {
      g_warning("The descr parameter given to 'tool_file_format_new' is bigger than"
		" %d characters and it will be truncated.", nbCharacterMax);
      lg = nbCharacterMax;
    }

  res = g_malloc(sizeof(ToolFileFormat));
  DBG_fprintf(stderr, "Tool ToolFileFormat: new file format %p.\n", (gpointer)res);
  res->name           = g_strndup(descr, lg);
  res->fileType       = (GList*)0;
  res->labelString    = (gchar*)0;
  res->properties     = (GList*)0;
  res->ignoreFileType = uncompleteList;
  for (n = 0; patterns[n]; n++)
    res->fileType = g_list_append(res->fileType, g_strdup(patterns[n]));
  
  return res;
}
static void _buildLabel(ToolFileFormat *format)
{
  GList *lst;
  GString *tmpStr;
  
  tmpStr = g_string_new(format->name);
  g_string_append_printf(tmpStr, " (");
  for (lst = format->fileType; lst; lst = g_list_next(lst))
    {
      g_string_append_printf(tmpStr, "%s", (gchar*) lst->data);
      if (lst->next)
	g_string_append_printf(tmpStr, ", ");
    }
  if (!format->fileType)
    g_string_append_printf(tmpStr, "no filter");
  if (format->fileType && format->ignoreFileType)
    g_string_append_printf(tmpStr, ", ...");
  g_string_append_printf(tmpStr, ")");
  format->labelString = tmpStr->str;
  g_string_free(tmpStr, FALSE);
}

/**
 * tool_file_format_newRestricted:
 * @descr: a short string to label a new file format.
 * @patterns: a set of patterns to identify files of this format.
 *
 * Allocate a new #ToolFileFormat. The @patterns argument is copied in
 * the #ToolFileFormat object and can be freed safely after the call to this method.
 * The @patterns list is not all the patterns supported by the format.
 *
 * Returns: (transfer none): a newly allocated ToolFileFormat,
 * or NULL if something goes wrong.
 */
ToolFileFormat* tool_file_format_newRestricted(gchar* descr, gchar** patterns)
{
  ToolFileFormat *format;

  g_return_val_if_fail(descr && descr[0] && patterns, (ToolFileFormat*)0);

  format = _new(descr, patterns, TRUE);
  _buildLabel(format);
  return format;
}

/**
 * tool_file_format_new:
 * @descr: a short string to label a new file format.
 * @patterns: (array zero-terminated=1) (element-type utf8): a set of patterns to identify files of this format.
 *
 * Allocate a new #ToolFileFormat. The @patterns argument is copied in
 * the #ToolFileFormat object and can be freed safely after the call to this method.
 *
 * Returns: (transfer none): a newly allocated ToolFileFormat,
 * or NULL if something goes wrong.
 */
ToolFileFormat* tool_file_format_new(gchar* descr, gchar** patterns)
{
  ToolFileFormat *format;

  g_return_val_if_fail(descr && descr[0] && patterns, (ToolFileFormat*)0);

  format = _new(descr, patterns, FALSE);
  _buildLabel(format);
  return format;
}
/**
 * tool_file_format_free:
 * @format: a #ToolFileFormat to free.
 *
 * Free all the memory used by @format. If @format is NULL, the method
 * writes a warning on the eror output.
 */
void tool_file_format_free(ToolFileFormat *format)
{
  GList *tmpLst;

  if (!format)
    return;

  if (format->name)
    g_free(format->name);

  if (format->labelString)
    g_free(format->labelString);

  for (tmpLst = format->fileType; tmpLst; tmpLst = g_list_next(tmpLst))
    g_free(tmpLst->data);
  g_list_free(format->fileType);

  for (tmpLst = format->properties; tmpLst; tmpLst = g_list_next(tmpLst))
    tool_option_free(tmpLst->data);
  g_list_free(format->properties);

  g_free(format);
}
/**
 * tool_file_format_copy:
 * @from: a #ToolFileFormat object.
 *
 * Copy the given file format @from a create a new one.
 *
 * Since: 3.6
 *
 * Returns: (transfer none): a newly created #ToolFileFormat. Should
 * be freed with tool_file_format_free().
 */
ToolFileFormat* tool_file_format_copy(const ToolFileFormat *from)
{
  ToolFileFormat *to;
  GList *lst;

  g_return_val_if_fail(from, (ToolFileFormat*)0);

  to = g_malloc(sizeof(ToolFileFormat));
  DBG_fprintf(stderr, "Tool ToolFileFormat: copy to file format %p.\n", (gpointer)to);
  to->name           = g_strdup(from->name);
  to->ignoreFileType = from->ignoreFileType;
  to->fileType       = (GList*)0;
  for (lst = from->fileType; lst; lst = g_list_next(lst))
    to->fileType = g_list_append(to->fileType, g_strdup((gchar*)lst->data));
  to->labelString    = (gchar*)0;
  _buildLabel(to);
  to->properties     = (GList*)0;
  for (lst = from->properties; lst; lst = g_list_next(lst))
    to->properties = g_list_append(to->properties,
                                   tool_option_copy((ToolOption*)lst->data));
  
  return to;
}
/**
 * tool_file_format_getName:
 * @format: a #ToolFileFormat.
 *
 * This method gives the name describing the file format.
 *
 * Returns: (transfer none): a string with the name. This string
 * should not be freed.
 */
const gchar* tool_file_format_getName(ToolFileFormat *format)
{
  DBG_fprintf(stderr, "Tool ToolFileFormat: get the name of format '%p'.\n",
	      (gpointer)format);
  g_return_val_if_fail(format, (gchar*)0);
  return format->name;
}
/**
 * tool_file_format_getLabel:
 * @format: a #ToolFileFormat.
 *
 * This method gives a label describing the file format.
 *
 * Returns: (transfer none): a string made by the name and all
 * the paterns of the given format, given in parentethis. This string
 * should not be freed.
 */
const gchar* tool_file_format_getLabel(ToolFileFormat *format)
{
  DBG_fprintf(stderr, "Tool ToolFileFormat: get the label of format '%p'.\n",
	      (gpointer)format);
  g_return_val_if_fail(format, (gchar*)0);
  return format->labelString;
}
/**
 * tool_file_format_getFilePatterns:
 * @format: a #ToolFileFormat.
 *
 * This method gives a list with the file patterns.
 *
 * Returns: (transfer none) (element-type utf8): a list with the file
 * patterns. This list should not be freed.
 *
 * Since: 3.6
 */
const GList* tool_file_format_getFilePatterns(ToolFileFormat *format)
{
  DBG_fprintf(stderr, "Tool ToolFileFormat: get the file patterns of format '%p'.\n",
	      (gpointer)format);
  g_return_val_if_fail(format, (GList*)0);
  return format->fileType;
}

/**
 * tool_file_format_canMatch:
 * @format: a #ToolFileFormat.
 *
 * This method is used to know if the file pattern list can be used
 * to match a given filename with @tool_file_format_match().
 *
 * Returns: TRUE if a call to @tool_file_format_match() is safe.
 *
 * Since: 3.6
 */
gboolean tool_file_format_canMatch(ToolFileFormat* format)
{
  g_return_val_if_fail(format, FALSE);

  return !format->ignoreFileType;
}
/**
 * tool_file_format_match:
 * @format: a #ToolFileFormat ;
 * @filename: a string to match.
 *
 * This method try to match the given string to one of the patterns of
 * the #ToolFileFormat @format.
 *
 * Returns: the matching pattern, if any.
 */
const gchar* tool_file_format_match(ToolFileFormat *format, const gchar *filename)
{
  GList *tmpLst;

  g_return_val_if_fail(format, FALSE);

  DBG_fprintf(stderr, "Tool FileFormat: try to match '%s' with '%s' -> ",
              filename, format->name);
  tmpLst = format->fileType;
  while(tmpLst && !g_pattern_match_simple(tmpLst->data, filename))
    tmpLst = g_list_next(tmpLst);
  DBG_fprintf(stderr, "%d.\n", (tmpLst != (GList*)0));
  return (tmpLst)?(const gchar*)tmpLst->data:(const gchar*)0;
}

/**
 * tool_file_format_addPropertyBoolean:
 * @format: the #ToolFileFormat object.
 * @name: a name ;
 * @label: a description ;
 * @defaultVal: a default value.
 *
 * Add a new boolean property to the file format @format.
 *
 * Returns: (transfer none): a newly created #ToolOption, free with tool_option_free().
 */
ToolOption* tool_file_format_addPropertyBoolean(ToolFileFormat *format,
                                                const gchar *name, const gchar *label,
                                                gboolean defaultVal)
{
  ToolOption *opt;
  GValue *val;

  opt = tool_option_new(name, label, G_TYPE_BOOLEAN);
  val = tool_option_getValue(opt);
  g_value_set_boolean(val, defaultVal);
  format->properties = g_list_append(format->properties, (gpointer)opt);

  return opt;
}
/**
 * tool_file_format_addPropertyInt:
 * @format: the #ToolFileFormat object.
 * @name: a name ;
 * @label: a description ;
 * @defaultVal: a default value.
 *
 * Add a new integer property to the file format @format.
 *
 * Returns: (transfer none): a newly created #ToolOption, free with tool_option_free().
 */
ToolOption* tool_file_format_addPropertyInt(ToolFileFormat *format,
                                            const gchar *name, const gchar *label,
                                            gint defaultVal)
{
  ToolOption *opt;
  GValue *val;

  opt = tool_option_new(name, label, G_TYPE_INT);
  val = tool_option_getValue(opt);
  g_value_set_int(val, defaultVal);
  format->properties = g_list_append(format->properties, (gpointer)opt);

  return opt;
}

/**
 * tool_file_format_getPropertyList:
 * @format: a #ToolFileFormat.
 *
 * Use this method to get a list of stored properties for the given #ToolFileFormat.
 * The GList is composed of #ToolOption objects.
 * 
 * Returns: (transfer none) (element-type ToolOption*): a
 * GList* that is read-only.
 */
const GList* tool_file_format_getPropertyList(ToolFileFormat *format)
{
  g_return_val_if_fail(format, (GList*)0);
  return format->properties;
}

/**
 * tool_file_format_iterNextProperty:
 * @format: a #ToolFileFormat object.
 * @iter: an iterator.
 *
 * Run to the next property of the file format @format. The iterator
 * attributes are updated so it's convenient to access the property
 * values and details, see #ToolFileFormatIter.
 *
 * Since: 3.6
 *
 * Returns: TRUE if any.
 */
gboolean tool_file_format_iterNextProperty(ToolFileFormat *format,
                                           ToolFileFormatIter *iter)
{
  if (!iter->lst)
    iter->lst = format->properties;
  else
    iter->lst = g_list_next(iter->lst);

  if (!iter->lst)
    return FALSE;

  iter->name  = (gchar*)tool_option_getName ((ToolOption*)iter->lst->data);
  iter->label = (gchar*)tool_option_getLabel((ToolOption*)iter->lst->data);
  iter->val   = tool_option_getValue((ToolOption*)iter->lst->data);

  return TRUE;
}

/**
 * tool_file_format_getPropertyByName:
 * @format: a #ToolFileFormat.
 * @name: the name of a property.
 *
 * Retrieve the corresponding property to @name from the biven
 * @format.
 *
 * Since: 3.5
 *
 * Returns: (transfer none): the found #ToolOption or NULL.
 */
ToolOption* tool_file_format_getPropertyByName(ToolFileFormat *format,
                                               const gchar *name)
{
  GList *lst;

  g_return_val_if_fail(format && name, (ToolOption*)0);

  DBG_fprintf(stderr, "Tool FileFormat: grep property '%s' from file format %p.\n",
              name, (gpointer)format);
  for (lst = format->properties; lst; lst = g_list_next(lst))
    if (!strcmp(name, tool_option_getName((ToolOption*)lst->data)))
      return (ToolOption*)lst->data;
  return (ToolOption*)0;
}
