/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_extension.h"
#include "visu_configFile.h"
#include "coreTools/toolConfigFile.h"
#include "extensions/externalVisuExtensions.h"

#include <stdlib.h>
#include <string.h>
#include <GL/gl.h>
#include <GL/glu.h> 

#include "visu_tools.h"

/**
 * SECTION:visu_extension
 * @short_description: All objects drawn by V_Sim are defined in by a
 * #VisuExtension object
 *
 * <para>All objects that are drawn by V_Sim are handled by a
 * #VisuExtension object. Such an object has an OpenGL list. This
 * list is only COMPILED. When V_Sim receives the 'OpenGLAskForReDraw'
 * or the 'OpenGLForceReDraw' signals, each list of all known
 * #VisuExtension are excecuted. This excecution can be canceled if
 * the used flag of the #VisuExtension object is set to FALSE. The
 * order in which the lists are called depends on the priority of the
 * #VisuExtension object. This priority is set to
 * #VISU_EXTENSION_PRIORITY_NORMAL as default value, but it can be
 * tune by a call to visu_extension_setPriority(). This priority is
 * an integer, the lower it is, the sooner the list is
 * excecuted.</para>
 *
 * <para>The method registerVisuExtension() is used to declare to
 * V_Sim that there is a new #VisuExtension object available. This
 * allows to create extension when V_Sim is already
 * running. Nevertheless, an extension must be initialized in the
 * initialisation process, it is better to add an
 * #initVisuExtensionFunc method in the listInitExtensionFunc array
 * declared in extensions/externalVisuExtensions.h.</para>
 *
 * <para>Once again, the OpenGL list corresponding to an OpenGL
 * extension is COMPILE only. Then, OpenGL methods like glIsenabled()
 * are totally unusefull because it is called when the list is
 * compiled not when the list is called. If the extension needs to
 * alter some OpenGL state, such as desable GL_LIGHTING, it needs to
 * set a flag for the extension. With this flag, V_Sim will save the
 * OpenGL states and restore it when the list is called. Use
 * visu_extension_setSaveOpenGLState() to set this flag.</para>
 */

#define FLAG_PARAMETER_MODE "extension_render"
#define DESC_PARAMETER_MODE "Rules the way OpenGl draws extensions (see opengl_render); name (string) value (string)"
static gboolean readExtensionRendering(gchar **lines, int nbLines, int position,
				       VisuData *dataObj, GError **error);
static void exportParametersRendering(GString *data, VisuData *dataObj);

/* A GHashTable to store all the available OpenGL extensions
   in the system. The keys are the name of each extension. */
static GList *allExtensions;

/* This flag is TRUE when some priorities are new or have changed, and 
   the list should be reordered. */
static gboolean VisuExtension_reorderingNeeded;
/* Method used to compare the priority of two extensions. */
static gint compareExtensionPriority(gconstpointer a, gconstpointer b);

static void callList(GList *lst, VisuRenderingModeId *renderingMode,
		     VisuRenderingModeId globalRenderingMode);

/***************/
/* Public part */
/***************/


/**
 * visu_extension_new:
 * @name: (type filename): name of the extension, in ASCII, used as id in the config files,
 * @nameI18n: (type utf8): name in UTF8 that can be translated and shown to user,
 * @description: (allow-none): a brief description of the extension (can be null),
 * @objectListId: an int to identify an list of OpenGL objects (null
 * if this extension as no OpenGL object,
 * @rebuild: (allow-none) (scope call): handler to a method that is called every time V_Sim needs
 * to create again the OpenGL object list. If NULL, nothing is called.
 *
 * Create a new VisuExtension with the specified name, description
 * and OpenGL object list. The priority is set by default to
 * #VISU_EXTENSION_PRIORITY_NORMAL. The flag used to store
 * the OpenGL state is put to FALSE by default (see
 * visu_extension_setSaveOpenGLState() to chance it).
 *
 * Returns: (transfer none): the new VisuExtension or null if something wrong happens.
 */
VisuExtension* visu_extension_new(const gchar* name, const gchar *nameI18n,
				  const gchar* description,
				  int objectListId, VisuExtensionRebuildFunc rebuild)
{
  VisuExtension *extension;

  extension = g_malloc(sizeof(VisuExtension));
  extension->name = g_strdup(name);
  extension->nameI18n = g_strdup(nameI18n);
  if (description)
    extension->description = g_strdup(description);
  else
    extension->description = (char*)0;

  extension->objectListId = objectListId;
  extension->rebuild = rebuild;
  extension->priority = VISU_EXTENSION_PRIORITY_NORMAL;
  extension->saveState = FALSE;
  extension->isSensitiveToRenderingMode = FALSE;
  extension->preferedRenderingMode = followGeneralSetting;

  extension->used = 0;

  return extension;
}

/* Free all the allocated attributes of the specified extension. */
void visu_extension_free(VisuExtension* extension)
{
  if (!extension)
    return;
  if (extension->name)
    g_free(extension->name);
  if (extension->nameI18n)
    g_free(extension->nameI18n);
  if (extension->description)
    g_free(extension->description);
  g_free(extension);
}
/* Get if the extension is used or not. */
int visu_extension_getActive(VisuExtension* extension)
{
  if (extension)
    return extension->used;
  else
    return 0;
}
/* Set if an extension is actually used or not. */
void visu_extension_setActive(VisuExtension* extension, int value)
{
  if (!extension || extension->used == value)
    return;

  extension->used = value;
}
GList* visuExtensions_getList(void)
{
  if (VisuExtension_reorderingNeeded)
    {
      DBG_fprintf(stderr, "Visu Extension : sorting known extension depending on their priority.\n");
      allExtensions = g_list_sort(allExtensions, compareExtensionPriority);
      VisuExtension_reorderingNeeded = FALSE;
    }
  return allExtensions;
}
VisuExtension* VisuExtensionGet_fromName(const gchar* name)
{
  GList *pnt;
  VisuExtension *ext;

  DBG_fprintf(stderr, "Visu Extension: get '%s' from list.\n", name);

  pnt = allExtensions;
  while (pnt)
    {
      ext = (VisuExtension*)pnt->data;
      if (!strcmp(ext->name, name))
	return ext;
      pnt = g_list_next(pnt);
    }
  return (VisuExtension*)0;
}

void visu_extension_setPriority(VisuExtension* extension, guint priority)
{
  g_return_if_fail(extension);
  extension->priority = priority;
  VisuExtension_reorderingNeeded = TRUE;
}

void visu_extension_setSaveOpenGLState(VisuExtension *extension, gboolean saveState)
{
  g_return_if_fail(extension);
  extension->saveState = saveState;
}

void visu_extension_setSensitiveToRenderingMode(VisuExtension* extension,
						 gboolean status)
{
  g_return_if_fail(extension);
  extension->isSensitiveToRenderingMode = status;
}
gboolean visu_extension_setPreferedRenderingMode(VisuExtension* extension,
						  VisuRenderingModeId value)
{
  g_return_val_if_fail(extension, FALSE);
  g_return_val_if_fail(value < nb_renderingModes ||
		       value == followGeneralSetting, FALSE);

  if (extension->preferedRenderingMode == value)
    return FALSE;

  extension->preferedRenderingMode = value;

  return TRUE;
}



/* A method used by user to registered a new extension. */
void visuExtensions_add(VisuExtension *extension)
{
  DBG_fprintf(stderr, "Visu Extension : registering a new OpenGL extension ... ");
  g_return_if_fail(extension && extension->name && extension->name[0]);

  allExtensions = g_list_append(allExtensions, (gpointer)extension);

  VisuExtension_reorderingNeeded = TRUE;
  DBG_fprintf(stderr, "'%s' (%p).\n", extension->name, (gpointer)extension);
}

/* A method used by user to remove a previously registered extension. */
void visuExtensions_remove(VisuExtension *extension)
{
  DBG_fprintf(stderr, "Visu Extension : removing a registered OpenGL"
	      " extension (%p).\n", (gpointer)extension);
  g_return_if_fail(extension);

  allExtensions = g_list_remove(allExtensions, (gpointer)extension);
}

void visuExtensions_callList(const char *name, gboolean lastOnly)
{
  VisuExtension *ext;
  VisuRenderingModeId renderingMode, globalRenderingMode;
  GList lst;

  DBG_fprintf(stderr, "Visu Extension: call '%s' list.\n", name);

  globalRenderingMode = openGLGet_globalRenderingOption();
  renderingMode = globalRenderingMode;
  ext = VisuExtensionGet_fromName(name);
  g_return_if_fail(ext);

  if (ext->used && ((lastOnly && ext->priority == VISU_EXTENSION_PRIORITY_LAST) ||
		    (!lastOnly && ext->priority < VISU_EXTENSION_PRIORITY_LAST)) &&
      ext->objectListId > 1000)
    {
      lst.data = (gpointer)ext;
      lst.next = lst.prev = (GList*)0;
      callList(&lst, &renderingMode, globalRenderingMode);
      if (renderingMode != globalRenderingMode)
	/* Return the rendering mode to normal. */
	openGLApply_renderingMode(globalRenderingMode);
    }
}
/* For each extension that has a valid list (id > 1000) a glCallList is raised. */
void visuExtensions_callAllLists(void)
{
  GList *pnt, *lst;
  VisuRenderingModeId renderingMode, globalRenderingMode;

  if (VisuExtension_reorderingNeeded)
    {
      DBG_fprintf(stderr, "Visu Extension: sorting known extension depending on their priority.\n");
      allExtensions = g_list_sort(allExtensions, compareExtensionPriority);
      VisuExtension_reorderingNeeded = FALSE;
    }
  globalRenderingMode = openGLGet_globalRenderingOption();
  renderingMode = globalRenderingMode;
  lst = (GList*)0;
  for (pnt = allExtensions; pnt && ((VisuExtension*)pnt->data)->priority <
         VISU_EXTENSION_PRIORITY_LAST; pnt = g_list_next(pnt))
    if (((VisuExtension*)pnt->data)->used && ((VisuExtension*)pnt->data)->objectListId > 1000)
      lst = g_list_append(lst, pnt->data);
  callList(lst, &renderingMode, globalRenderingMode);
  g_list_free(lst);
  if (renderingMode != globalRenderingMode)
    /* Return the rendering mode to normal. */
    openGLApply_renderingMode(globalRenderingMode);
}
/**
 * visuExtensions_callAllLastLists:
 *
 * Call all lists whose priority is exactly
 * #VISU_EXTENSION_PRIORITY_LAST. To draw other priority, see
 * visuExtensions_callList().
 *
 * Since: 3.6
 */
void visuExtensions_callAllLastLists(void)
{
  GList *pnt, *lst;
  VisuRenderingModeId renderingMode, globalRenderingMode;

  if (VisuExtension_reorderingNeeded)
    {
      DBG_fprintf(stderr, "Visu Extension: sorting known extension depending on their priority.\n");
      allExtensions = g_list_sort(allExtensions, compareExtensionPriority);
      VisuExtension_reorderingNeeded = FALSE;
    }
  DBG_fprintf(stderr, "Visu Extension: draw PRIORITY_LAST only.\n");
  globalRenderingMode = openGLGet_globalRenderingOption();
  renderingMode = globalRenderingMode;
  for (pnt = allExtensions; pnt && ((VisuExtension*)pnt->data)->priority <
         VISU_EXTENSION_PRIORITY_LAST; pnt = g_list_next(pnt));
  lst = (GList*)0;
  for (; pnt; pnt = g_list_next(pnt))
    if (((VisuExtension*)pnt->data)->used && ((VisuExtension*)pnt->data)->objectListId > 1000)
      lst = g_list_append(lst, pnt->data);
  callList(lst, &renderingMode, globalRenderingMode);
  g_list_free(lst);
  if (renderingMode != globalRenderingMode)
    /* Return the rendering mode to normal. */
    openGLApply_renderingMode(globalRenderingMode);
}
void visuExtensions_rebuildAllLists(VisuData *dataObj)
{
  GList *pnt;

  DBG_fprintf(stderr, "Visu Extension: Rebuilding all lists...\n");
  if (VisuExtension_reorderingNeeded)
    {
      DBG_fprintf(stderr, "Visu Extension: sorting known extension depending on their priority.\n");
      allExtensions = g_list_sort(allExtensions, compareExtensionPriority);
      VisuExtension_reorderingNeeded = FALSE;
    }
  for (pnt = allExtensions; pnt; pnt = g_list_next(pnt))
    if (((VisuExtension*)pnt->data)->used &&
        ((VisuExtension*)pnt->data)->rebuild)
      {
        DBG_fprintf(stderr, "Visu Extension: rebuild extension %s (list %d).\n",
                    ((VisuExtension*)pnt->data)->name,
                    ((VisuExtension*)pnt->data)->objectListId);
        ((VisuExtension*)pnt->data)->rebuild(dataObj);
      }
}
void visuExtensions_rebuildList(VisuData *dataObj, const char *name)
{
  VisuExtension *ext;

  DBG_fprintf(stderr, "Visu Extension: rebuilding '%s' list.\n", name);
  ext = VisuExtensionGet_fromName(name);
  g_return_if_fail(ext);

  if (ext->used && ext->rebuild)
    ext->rebuild(dataObj);
}


/****************/
/* Private area */
/****************/

/* Initialise all the variable of this part. */
int initVisuExtensions(void)
{
  VisuConfigFileEntry *confEntry;

  allExtensions = (GList*)0;
  VisuExtension_reorderingNeeded = FALSE;

  confEntry = visu_configFile_addEntry(VISU_CONFIGFILE_PARAMETER,
				      FLAG_PARAMETER_MODE,
				      DESC_PARAMETER_MODE,
				      1, readExtensionRendering);
  visu_configFile_entry_setVersion(confEntry, 3.4f);
  visu_configFile_addExportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParametersRendering);
  return 1;
}
void loadExtensions(void)
{
  VisuExtension *extension;
  int i, res;

  res = 1;
  for (i = 0; listInitExtensionFunc[i]; i++)
    {
      extension = listInitExtensionFunc[i]();
      if (!extension)
	res = 0;
      visuExtensions_add(extension);
    }
  
  if (!res)
    g_warning("Some OpenGL extensions can't initialse.\n");
}

static void callList(GList *lst, VisuRenderingModeId *renderingMode,
		     VisuRenderingModeId globalRenderingMode)
{
  VisuExtension *ext;
  GTimer *timer;
  gulong fractionTimer;

#if DEBUG == 1
  timer = g_timer_new();
  g_timer_start(timer);
#endif

  for (; lst; lst = g_list_next(lst))
    {
      ext = (VisuExtension*)lst->data;
      /* The extension needs its own rendering mode. */
      if (ext->isSensitiveToRenderingMode &&
	  ext->preferedRenderingMode < nb_renderingModes)
	{
	  if (ext->preferedRenderingMode != *renderingMode)
	    {
	      openGLApply_renderingMode(ext->preferedRenderingMode);
	      *renderingMode = ext->preferedRenderingMode;
	    }
	}
      else
	{
	  if (*renderingMode != globalRenderingMode)
	    {
	      openGLApply_renderingMode(globalRenderingMode);
	      *renderingMode = globalRenderingMode;
	    }
	}
      /* Save OpenGL state if necessary. */
      if (ext->saveState)
	{
	  DBG_fprintf(stderr, "Visu Extension: save state.\n");
	  glPushAttrib(GL_ENABLE_BIT);
	}

      if (ext->isSensitiveToRenderingMode && *renderingMode == SmoothAndEdge)
        {
          glPushAttrib(GL_ENABLE_BIT);
          glEnable(GL_POLYGON_OFFSET_FILL);
          glPolygonOffset(1.0, 1.0);
        }

      /* Call the compiled list. */
      DBG_fprintf(stderr, "Visu Extension: call list %d (%s)",
                  ext->objectListId, ext->name);
      glCallList(ext->objectListId);
      DBG_fprintf(stderr, " at %g micro-s",
                  g_timer_elapsed(timer, &fractionTimer)*1e6);
      DBG_fprintf(stderr, ".\n");

      /* Add a wireframe draw if renderingMode is SmoothAndEdge. */
      if (ext->isSensitiveToRenderingMode && *renderingMode == SmoothAndEdge)
        {
          glDisable(GL_POLYGON_OFFSET_FILL);
          glDisable(GL_LIGHTING);
          glColor3f (0.0, 0.0, 0.0);
          glLineWidth(1);
          glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
          glCallList(ext->objectListId);
          glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
          glPopAttrib();
        }

      if (ext->saveState)
	{
	  DBG_fprintf(stderr, "Visu Extension: restore state.\n");
	  glPopAttrib();
	}
    }

#if DEBUG == 1
  g_timer_stop(timer);
  g_timer_destroy(timer);
#endif
}

static gint compareExtensionPriority(gconstpointer a, gconstpointer b)
{
  if (((VisuExtension*)a)->priority < ((VisuExtension*)b)->priority)
    return (gint)-1;
  else if (((VisuExtension*)a)->priority > ((VisuExtension*)b)->priority)
    return (gint)+1;
  else
    return (gint)0;
}

static gboolean readExtensionRendering(gchar **lines, int nbLines, int position,
				       VisuData *dataObj _U_, GError **error)
{
  gchar **val;
  VisuExtension *ext;
  VisuRenderingModeId id;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readString(lines[0], position, &val, 2, FALSE, error))
    return FALSE;
  ext = VisuExtensionGet_fromName(val[0]);
  if (!ext)
    {
      *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_VALUE,
			   _("Parse error at line %d: the extension"
			     " '%s' is unknown.\n"), position, val[0]);
      g_strfreev(val);
      return FALSE;
    }
  if (!openGLGet_renderingFromName(val[1], &id))
    {
      *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_VALUE,
			   _("Parse error at line %d: the rendering mode"
			     " '%s' is unknown.\n"), position, val[1]);
      g_strfreev(val);
      return FALSE;
    }
  g_strfreev(val);
  visu_extension_setPreferedRenderingMode(ext, id);

  return TRUE;
}
static void exportParametersRendering(GString *data, VisuData *dataObj _U_)
{
  GList *tmp;
  VisuExtension *ext;
  const char **names;

  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_MODE);

  names = openGLGet_allRenderingModes();
  tmp = allExtensions;
  while (tmp)
    {
      ext = (VisuExtension*)tmp->data;
      if (ext->isSensitiveToRenderingMode &&
	  ext->preferedRenderingMode < nb_renderingModes)
	g_string_append_printf(data, "%s: %s %s\n", FLAG_PARAMETER_MODE,
			       ext->name, names[ext->preferedRenderingMode]);
      tmp = g_list_next(tmp);
    }
  g_string_append_printf(data, "\n");
}
