#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include <gtk/gtk.h>

#include "v3dmh.h"
#include "v3dmp.h"
#include "v3dmodel.h"

#include "editor.h"
#include "editorp.h"
#include "editormodel.h"
#include "editorlist.h"
#include "editortexture.h"
#include "editorhf.h"
#include "editorselect.h"
#include "editorundo.h"

#include "vmapixmaps.h"
#include "vmaundo.h"

#include "config.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


/* Model header item handling. */
void EditorListHeaderSetHeightFieldBaseDir(
        ma_editor_struct *editor,
        const char *path
);
char *EditorListHeaderGetHeightFieldBaseDir(ma_editor_struct *editor);
void EditorListHeaderSetTextureBaseDir(
        ma_editor_struct *editor, const char *path
);
char *EditorListHeaderGetTextureBaseDir(ma_editor_struct *editor);

/* Model primitive handling. */
void EditorListPrimitivesSetComment(
	GtkWidget *clist, int row,
	void *p, gbool has_change
);
void EditorListPrimitivesSetTranslate(
	GtkWidget *clist, int row,
	void *p, gbool has_change
);
void EditorListPrimitivesSetDrawn(
	GtkWidget *clist, int row,
	void *p, gbool has_change
);
void EditorListPrimitivesSetColor(
	GtkWidget *clist, int row,
	void *p, gbool has_change
);
void EditorListPrimitivesSetTexture(
	GtkWidget *clist, int row,
	void *p, gbool has_change
);
void EditorListPrimitivesSetHeightField(
        GtkWidget *clist, int row,
        void *p, gbool has_change
);
void EditorListPrimitivesSet(
        GtkWidget *clist, int row,
        void *p, gbool has_change
);

/* Model listing handling. */
void EditorListModelsSet(
	GtkWidget *clist, int row,
	v3d_model_struct *model, gbool has_change
);


/* Massive list IO (RG means real to gui list). */
int EditorListAddModelsRG(ma_editor_struct *editor);
int EditorListAddPrimitivesRG(
	ma_editor_struct *editor, v3d_model_struct *model
);
int EditorListAddValuesRG(
	ma_editor_struct *editor, void *p
);

int EditorListValuePromptApply(
	ma_editor_struct *editor, GtkWidget *entry, gbool record_undo
);
int EditorListValuePromptFetch(
	ma_editor_struct *editor, void *p
);

void EditorListDeleteModelsG(ma_editor_struct *editor);
void EditorListDeletePrimitivesG(ma_editor_struct *editor);
void EditorListDeleteValuesG(ma_editor_struct *editor);
void EditorListDeleteAllLists(ma_editor_struct *editor);


#define VAL_NULL_STR	"(null)"

#define RADTODEG(r)	((r) * 180 / PI)
#define DEGTORAD(d)     ((d) * PI / 180)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))


#define PRIM_TYPE_NAME_LIMIT		256



/*
 *      Sets all model header items of type
 *      V3DMH_TYPE_HEIGHTFIELD_BASE_DIRECTORY to the specified value.
 */
void EditorListHeaderSetHeightFieldBaseDir(
        ma_editor_struct *editor,
        const char *path
)
{
	int i;
	void *p;


	if((editor == NULL) ||
           (path == NULL)
	)
	    return;

	for(i = 0; i < editor->total_mh_items; i++)
	{
	    p = editor->mh_item[i];
	    if(p == NULL)
		continue;

	    if((*(int *)p) == V3DMH_TYPE_HEIGHTFIELD_BASE_DIRECTORY)
	    {
		mh_heightfield_base_directory_struct *mh_hbd =
		    (mh_heightfield_base_directory_struct *)p;

		free(mh_hbd->path);
		mh_hbd->path = strdup(path);
	    }
	}

	return;
}

/*
 *	Returns the pointer to the path of the first model header item
 *	of type V3DMH_TYPE_HEIGHTFIELD_BASE_DIRECTORY.
 *
 *	Can return NULL on no match, return must not be free'ed.
 */
char *EditorListHeaderGetHeightFieldBaseDir(ma_editor_struct *editor)
{
        int i;
        void *p;


        if(editor == NULL)
            return(NULL);

        for(i = 0; i < editor->total_mh_items; i++)
        {
            p = editor->mh_item[i];
            if(p == NULL)
                continue;

            if((*(int *)p) == V3DMH_TYPE_HEIGHTFIELD_BASE_DIRECTORY)
            {
                mh_heightfield_base_directory_struct *mh_hbd =
                    (mh_heightfield_base_directory_struct *)p;
 
		return(mh_hbd->path);
            }
        }

	return(NULL);
}


/*
 *	Sets all model header items of type
 *	V3DMH_TYPE_TEXTURE_BASE_DIRECTORY to the specified value.
 *
 *	Will not create a new model header item if non exist.
 *
 *	Will not update any related resources like re-realizing
 *	primitives on the editor, reload textures on the editor, or update
 *	the texture browser.
 */
void EditorListHeaderSetTextureBaseDir(
        ma_editor_struct *editor, const char *path
)
{
        int i, htype;
        void *h;


        if((editor == NULL) || (path == NULL))
            return;

        for(i = 0; i < editor->total_mh_items; i++)
        {
            h = editor->mh_item[i];
            if(h == NULL)
                continue;
	    else
		htype = (*(int *)h);

            if(htype == V3DMH_TYPE_TEXTURE_BASE_DIRECTORY)
            {
                mh_texture_base_directory_struct *mh_texture_base =
                    (mh_texture_base_directory_struct *)h;

                free(mh_texture_base->path);
                mh_texture_base->path = strdup(path);
            }
        }

        return;
}

/*
 *      Returns the pointer to the path of the first model header item
 *      of type V3DMH_TYPE_TEXTURE_BASE_DIRECTORY.
 *
 *      Can return NULL on no match, return must not be free'ed.
 */
char *EditorListHeaderGetTextureBaseDir(ma_editor_struct *editor)
{
        int i;
        void *p;


        if(editor == NULL)
            return(NULL);

        for(i = 0; i < editor->total_mh_items; i++)
        {
            p = editor->mh_item[i];
            if(p == NULL)
                continue;

            if((*(int *)p) == V3DMH_TYPE_TEXTURE_BASE_DIRECTORY)
            {
                mh_texture_base_directory_struct *mh_tbd =
                    (mh_texture_base_directory_struct *)p;

                return(mh_tbd->path);
            }
        }
         
        return(NULL);
}





/*
 *	Adds a new clist item of type V3DMP_TYPE_COMMENT.
 *
 *	The given clist should be the editor's primitives_list.
 */
void EditorListPrimitivesSetComment(
	GtkWidget *clist, int row, void *p, gbool has_change
)
{
        gchar *val[1], *comment = NULL;
	mp_comment_struct *mp_comment;
	char *strptr;
	int icon_num = VMA_PIXMAP_MP_COMMENT_20x20;
	GdkPixmap *pixmap_gdk;
        GdkBitmap *mask;
        char text[256];


        if((clist == NULL) || (row < 0) || (p == NULL))
            return;

	(*text) = '\0';

	if((*(int *)p) == V3DMP_TYPE_COMMENT)
	    mp_comment = p;
	else
	    return;

	if(mp_comment->total_lines > 0)
	    comment = (gchar *)mp_comment->line[0];
	else
	    comment = NULL;

	if(comment == NULL)
	{
	    strcat(text, "comment ");
	    strcat(text, VAL_NULL_STR);
	}
	else
	{
	    strncat(text, comment, 80);
	}	    
	text[80 - 1] = '\0';

	/* Strip new lines if any. */
	strptr = strchr(text, '\n');
	if(strptr == NULL)
	    strptr = strchr(text, '\r');
	if(strptr != NULL)
	    *strptr = '\0';

        val[0] = text;
	VMAPixmapsListGetValues(
	    &vma_pixmaps_list, icon_num,
	    &pixmap_gdk, &mask, NULL
	);
        if(GTK_CLIST(clist)->rows <= row)
            gtk_clist_insert(GTK_CLIST(clist), row, val);
        gtk_clist_set_pixtext(
            GTK_CLIST(clist),
            row, 0,             /* Row, column. */
            text, VMA_LIST_PIXMAP_TEXT_SPACING,
            pixmap_gdk, mask
        );

	return;
}


/*
 *	Appends or sets a clist item for the given
 *	translate, untranslate, rotate, or unrotate primitive.
 */
void EditorListPrimitivesSetTranslate(
        GtkWidget *clist, int row,
        void *p, gbool has_change
)
{
	mp_translate_struct *translate;
	mp_rotate_struct *rotate;
        int icon_num = VMA_PIXMAP_MP_COMMENT_20x20;
        GdkPixmap *pixmap_gdk;
        GdkBitmap *mask;
        gchar *val[1];
        char text[256];

        if((clist == NULL) || (row < 0) || (p == NULL))
            return;

        *text = '\0';

        switch(*(int *)p)
        {
          case V3DMP_TYPE_TRANSLATE:
	    translate = p;
            sprintf(text, "translate(%.2f,%.2f,%.2f)",
		translate->x, translate->y, translate->z
	    );
	    icon_num = VMA_PIXMAP_MP_TRANSLATE_20x20;
            break;

          case V3DMP_TYPE_UNTRANSLATE:
            sprintf(text, "untranslate");
            icon_num = VMA_PIXMAP_MP_TRANSLATE_20x20;
            break;   

          case V3DMP_TYPE_ROTATE:
	    rotate = p;
            sprintf(text, "rotate(%.0f,%.0f,%.0f)",
		RADTODEG(rotate->heading),
		RADTODEG(rotate->pitch),
		RADTODEG(rotate->bank)
	    );
            icon_num = VMA_PIXMAP_MP_ROTATE_20x20;
            break;

          case V3DMP_TYPE_UNROTATE:
            sprintf(text, "unrotate");
            icon_num = VMA_PIXMAP_MP_ROTATE_20x20;
            break;

          default:
            strcpy(text, "not translate or rotate");
            break;
        }

        val[0] = text;
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, icon_num,
            &pixmap_gdk, &mask, NULL
        );
        if(GTK_CLIST(clist)->rows <= row)
            gtk_clist_insert(GTK_CLIST(clist), row, val);
        gtk_clist_set_pixtext(
            GTK_CLIST(clist), 
            row, 0,             /* Row, column. */
            text, VMA_LIST_PIXMAP_TEXT_SPACING,   
            pixmap_gdk, mask
        );

	return;
}

/*
 *	Appends or sets a clist item for the given primitive which
 *      must be drawable (ie V3DMP_TYPE_POINT, V3DMP_TYPE_POLYGON, etc).
 */
void EditorListPrimitivesSetDrawn(
        GtkWidget *clist, int row,
        void *p, gbool has_change
)
{
	mp_line_strip_struct *line_strip;
	mp_line_loop_struct *line_loop;
	mp_triangle_strip_struct *triangle_strip;
	mp_triangle_fan_struct *triangle_fan;
	mp_quad_strip_struct *quad_strip;
	mp_polygon_struct *polygon;

        int icon_num = VMA_PIXMAP_MP_COMMENT_20x20;
        GdkPixmap *pixmap_gdk;
        GdkBitmap *mask;
        gchar *val[1];
	char text[PRIM_TYPE_NAME_LIMIT + 256];

        if((clist == NULL) || (row < 0) || (p == NULL))
            return;

	switch(*(int *)p)
	{
	  case V3DMP_TYPE_POINT:
	    sprintf(text, "point");
	    icon_num = VMA_PIXMAP_MP_POINT_20x20;
	    break;

          case V3DMP_TYPE_LINE:
            sprintf(text, "line");
	    icon_num = VMA_PIXMAP_MP_LINE_20x20;
            break;

	  case V3DMP_TYPE_LINE_STRIP:
	    line_strip = p;
	    sprintf(text, "line_strip(%i)", line_strip->total);
	    icon_num = VMA_PIXMAP_MP_LINESTRIP_20x20;
	    break;

          case V3DMP_TYPE_LINE_LOOP:
            line_loop = p;
            sprintf(text, "line_loop(%i)", line_loop->total);
	    icon_num = VMA_PIXMAP_MP_LINELOOP_20x20;
            break;

          case V3DMP_TYPE_TRIANGLE:
            sprintf(text, "triangle");
	    icon_num = VMA_PIXMAP_MP_TRIANGLE_20x20;
            break;

          case V3DMP_TYPE_TRIANGLE_STRIP:
            triangle_strip = p;
            sprintf(text, "triangle_strip(%i)", triangle_strip->total);
            icon_num = VMA_PIXMAP_MP_TRIANGLESTRIP_20x20;
            break;

          case V3DMP_TYPE_TRIANGLE_FAN:
            triangle_fan = p;   
            sprintf(text, "triangle_fan(%i)", triangle_fan->total);
            icon_num = VMA_PIXMAP_MP_TRIANGLEFAN_20x20;
            break;

          case V3DMP_TYPE_QUAD:
            sprintf(text, "quad");
            icon_num = VMA_PIXMAP_MP_QUAD_20x20;
            break;

          case V3DMP_TYPE_QUAD_STRIP:
            quad_strip = p;   
            sprintf(text, "quad_strip(%i)", quad_strip->total);
            icon_num = VMA_PIXMAP_MP_QUADSTRIP_20x20;
            break;

          case V3DMP_TYPE_POLYGON:
            polygon = p;
            sprintf(text, "polygon(%i)", polygon->total);
	    icon_num = VMA_PIXMAP_MP_POLYGON_20x20;
            break;

	  default:
	    strcpy(text, "not drawable");
	    break;
	}

        val[0] = text;
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, icon_num,
            &pixmap_gdk, &mask, NULL    
        );
        if(GTK_CLIST(clist)->rows <= row)
            gtk_clist_insert(GTK_CLIST(clist), row, val);
        gtk_clist_set_pixtext(
            GTK_CLIST(clist), 
            row, 0,             /* Row, column. */
            text, VMA_LIST_PIXMAP_TEXT_SPACING,      
            pixmap_gdk, mask
        );

	return;
}

/*
 *	Appends or sets a clist item for the given primitive of type
 *	MP_TYPE_COLOR.
 */
void EditorListPrimitivesSetColor(
        GtkWidget *clist, int row,
	void *p, gbool has_change
)
{
	mp_color_struct *color;
        int icon_num = VMA_PIXMAP_MP_COMMENT_20x20;
        GdkPixmap *pixmap_gdk;
        GdkBitmap *mask;
        gchar *val[1];
        char text[PRIM_TYPE_NAME_LIMIT + 256];

        if((clist == NULL) || (row < 0) || (p == NULL))
            return;   

        switch(*(int *)p)
        {     
          case V3DMP_TYPE_COLOR:
	    color = p;
            sprintf(text, "color(%.2f,%.2f,%.2f,%.2f)",
		color->a,
		color->r,
		color->g,
		color->b
	    );
	    icon_num = VMA_PIXMAP_MP_COLOR_20x20;
	    break;

          default:
            strcpy(text, "not a set color");
            break;
        }

        val[0] = text;
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, icon_num,
            &pixmap_gdk, &mask, NULL
        );
        if(GTK_CLIST(clist)->rows <= row)
            gtk_clist_insert(GTK_CLIST(clist), row, val);
        gtk_clist_set_pixtext(
            GTK_CLIST(clist),  
            row, 0,             /* Row, column. */
            text, VMA_LIST_PIXMAP_TEXT_SPACING,
            pixmap_gdk, mask
        );

        return;
}

/*
 *      Appends or sets a clist item for the given primitive of type
 *      texture select or orient (ie V3DMP_TYPE_TEXTURE_SELECT,
 *	V3DMP_TYPE_TEXTURE_ORIENT_YZ, etc)
 */
void EditorListPrimitivesSetTexture(
        GtkWidget *clist, int row,
        void *p, gbool has_change
)
{
	mp_texture_select_struct *texture_select;
        int icon_num = VMA_PIXMAP_MP_COMMENT_20x20;
        GdkPixmap *pixmap_gdk;
        GdkBitmap *mask;

        gchar *val[1];
        char text[PRIM_TYPE_NAME_LIMIT + 256];


        if((clist == NULL) || (row < 0) || (p == NULL))
            return;

        switch(*(int *)p)
        {
          case V3DMP_TYPE_TEXTURE_SELECT:
            texture_select = p;
	    *text = '\0';
	    strcat(text, "texture_select (\"");
	    strncat(
		text,
		((texture_select->name == NULL) ?
		    VAL_NULL_STR : texture_select->name
		),
		80
	    );
            strcat(text, "\")");
	    text[80 - 1] = '\0';
	    icon_num = VMA_PIXMAP_MP_TEXTURE_SELECT_20x20;
            break;

	  case V3DMP_TYPE_TEXTURE_ORIENT_XY:
	    strcpy(text, "texture_orient XY");
	    icon_num = VMA_PIXMAP_MP_TEXTURE_ORIENT_20x20;
	    break;

          case V3DMP_TYPE_TEXTURE_ORIENT_YZ:
            strcpy(text, "texture_orient YZ");
            icon_num = VMA_PIXMAP_MP_TEXTURE_ORIENT_20x20;
            break;

          case V3DMP_TYPE_TEXTURE_ORIENT_XZ:
            strcpy(text, "texture_orient XZ");
            icon_num = VMA_PIXMAP_MP_TEXTURE_ORIENT_20x20;
            break;

          case V3DMP_TYPE_TEXTURE_OFF:
            strcpy(text, "texture_off");
            icon_num = VMA_PIXMAP_MP_TEXTURE_OFF_20x20;
            break;

          default:
            strcpy(text, "not a texture manipulation");
            break;
	}

        val[0] = text;
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, icon_num,
            &pixmap_gdk, &mask, NULL
        );
        if(GTK_CLIST(clist)->rows <= row)
            gtk_clist_insert(GTK_CLIST(clist), row, val);
        gtk_clist_set_pixtext(
            GTK_CLIST(clist),
            row, 0,             /* Row, column. */
            text, VMA_LIST_PIXMAP_TEXT_SPACING,
            pixmap_gdk, mask
        );

        return;
}

/*
 *	Appends or sets a clist item for the given primitive of type
 *      V3DMP_TYPE_HEIGHTFIELD_LOAD.
 */
void EditorListPrimitivesSetHeightField(
        GtkWidget *clist, int row,
        void *p, gbool has_change
)
{
        mp_heightfield_load_struct *hfl;
        int icon_num = VMA_PIXMAP_MP_COMMENT_20x20;
        GdkPixmap *pixmap_gdk;
        GdkBitmap *mask;

        gchar *val[1];
        char text[PRIM_TYPE_NAME_LIMIT + 256];


        if((clist == NULL) || (row < 0) || (p == NULL))   
            return;

        switch(*(int *)p)
        {
          case V3DMP_TYPE_HEIGHTFIELD_LOAD:
            hfl = p;
            *text = '\0';
            strcat(text, "heightfield_load (\"");
            strncat(
                text,
                ((hfl->path == NULL) ?
                    VAL_NULL_STR : hfl->path
                ),
                80
            );
            strcat(text, "\")");
            text[80 - 1] = '\0';
            icon_num = VMA_PIXMAP_MP_HEIGHTFIELD_LOAD_20x20;
            break;

          default:
            strcpy(text, "not a heightfield_load");
            break;
        }

        val[0] = text;
        VMAPixmapsListGetValues(
            &vma_pixmaps_list, icon_num,
            &pixmap_gdk, &mask, NULL
        );
        if(GTK_CLIST(clist)->rows <= row)
            gtk_clist_insert(GTK_CLIST(clist), row, val);
        gtk_clist_set_pixtext(
            GTK_CLIST(clist),
            row, 0,             /* Row, column. */
            text, VMA_LIST_PIXMAP_TEXT_SPACING,
            pixmap_gdk, mask
        );

        return;
}

/*
 *	Sets the gui primitives list item from the real primitive
 *	on the selected model on the editor.
 *
 *      The given clist should be the editor's primitives_list.
 */
void EditorListPrimitivesSet(           
        GtkWidget *clist, int row,
        void *p, gbool has_change
)
{
        int i = row;
        GtkWidget *w = clist;
        int icon_num = VMA_PIXMAP_MP_COMMENT_20x20;
        GdkPixmap *pixmap_gdk;
        GdkBitmap *mask;
	gchar text[256];
	gchar *val[1];


        if(w == NULL)
            return;

        if(p == NULL)
	{
	    val[0] = VAL_NULL_STR;
            VMAPixmapsListGetValues(
		&vma_pixmaps_list, icon_num,
		&pixmap_gdk, &mask, NULL
	    );
	    if(GTK_CLIST(w)->rows <= i)
		gtk_clist_insert(GTK_CLIST(w), i, val);
	    gtk_clist_set_pixtext(
		GTK_CLIST(w),
		i, 0,		/* Row, column. */
		VAL_NULL_STR, VMA_LIST_PIXMAP_TEXT_SPACING,
		pixmap_gdk, mask
	    );
	}
	else
	{
	    switch(*(int *)p)
	    {
                  case V3DMP_TYPE_COMMENT:
                    EditorListPrimitivesSetComment(
                        w, i, p, has_change
                    );
                    break;

                  case V3DMP_TYPE_TRANSLATE:
                  case V3DMP_TYPE_UNTRANSLATE:   
                  case V3DMP_TYPE_ROTATE:
                  case V3DMP_TYPE_UNROTATE:
                    EditorListPrimitivesSetTranslate(
                        w, i, p, has_change
                    );
                    break;

                  case V3DMP_TYPE_POINT:
                  case V3DMP_TYPE_LINE:
                  case V3DMP_TYPE_LINE_STRIP:
                  case V3DMP_TYPE_LINE_LOOP:
                  case V3DMP_TYPE_TRIANGLE:  
                  case V3DMP_TYPE_TRIANGLE_STRIP:
                  case V3DMP_TYPE_TRIANGLE_FAN:
                  case V3DMP_TYPE_QUAD:
                  case V3DMP_TYPE_QUAD_STRIP:
                  case V3DMP_TYPE_POLYGON:
                    EditorListPrimitivesSetDrawn(
                        w, i, p, has_change
                    );
                    break;

                 case V3DMP_TYPE_COLOR:
                    EditorListPrimitivesSetColor(
                        w, i, p, has_change
                    );
                    break;

                  case V3DMP_TYPE_TEXTURE_SELECT:
                  case V3DMP_TYPE_TEXTURE_ORIENT_XY:
                  case V3DMP_TYPE_TEXTURE_ORIENT_YZ:
                  case V3DMP_TYPE_TEXTURE_ORIENT_XZ:
                  case V3DMP_TYPE_TEXTURE_OFF:
                    EditorListPrimitivesSetTexture(
                        w, i, p, has_change
                    );
                    break;

                  case V3DMP_TYPE_HEIGHTFIELD_LOAD:
                    EditorListPrimitivesSetHeightField(
                        w, i, p, has_change
                    );
                    break;

                  default:
                    sprintf(text, "unknown(%i)",
                        (*(int *)p)
                    );
                    val[0] = text;
                    VMAPixmapsListGetValues(
                        &vma_pixmaps_list, icon_num,
                        &pixmap_gdk, &mask, NULL
                    );
                    if(GTK_CLIST(w)->rows <= i)
                        gtk_clist_insert(GTK_CLIST(w), i, val);
                    gtk_clist_set_pixtext(
                        GTK_CLIST(w),
                        i, 0,		/* Row, column. */
			text, VMA_LIST_PIXMAP_TEXT_SPACING,
			pixmap_gdk, mask
		    );
                    break;
	    }
	}

	return;
}

/*
 *      Sets the gui models list item value based on values on the
 *	given model.
 */
void EditorListModelsSet(
        GtkWidget *clist, int row,
        v3d_model_struct *model, gbool has_change
)
{
	int i = row;
        int icon_num = VMA_PIXMAP_MODEL_OTHER_20x20;
	GtkWidget *w = clist;
        GdkPixmap *pixmap_gdk;
        GdkBitmap *mask;
	gchar *val[1];


        if(w == NULL)
            return;

        if(model == NULL)
        {
            VMAPixmapsListGetValues(
                &vma_pixmaps_list, icon_num,
                &pixmap_gdk, &mask, NULL
            );
            if(GTK_CLIST(w)->rows <= i)
	    {
		val[0] = strdup(VAL_NULL_STR);
                gtk_clist_insert(GTK_CLIST(w), i, val);
		free(val[0]);
	    }
            gtk_clist_set_pixtext(
                GTK_CLIST(w),
                i, 0,		/* Row, column. */
                VAL_NULL_STR, VMA_LIST_PIXMAP_TEXT_SPACING,
                pixmap_gdk, mask
            );
        }
	else
	{
	    if(model->type == V3D_MODEL_TYPE_STANDARD)
	    {
		if(model->flags & V3D_MODEL_FLAG_HIDE)
		    icon_num = VMA_PIXMAP_MODEL_HIDDEN_20x20;
		else
		    icon_num = VMA_PIXMAP_MODEL_20x20;
	    }

            VMAPixmapsListGetValues(
                &vma_pixmaps_list, icon_num,
                &pixmap_gdk, &mask, NULL
            );
	    if(GTK_CLIST(w)->rows <= i)
	    {
		val[0] = strdup("");
		gtk_clist_insert(GTK_CLIST(w), i, val);
		free(val[0]);
	    }
	    gtk_clist_set_pixtext(
		GTK_CLIST(w),
		i, 0,		/* Row, column. */
		((model->name == NULL) ?
		    VAL_NULL_STR : model->name),
		VMA_LIST_PIXMAP_TEXT_SPACING,
		pixmap_gdk, mask
	    );
	}

	return;
}



/*
 *	Add Real models list to GUI list.
 *
 *	GUI list will not be cleared.
 */
int EditorListAddModelsRG(ma_editor_struct *editor)
{
	int i;
        GtkWidget *w;


        if(editor == NULL)
            return(-1);

        w = editor->models_list;
        if(w == NULL)
	    return(-1);

	gtk_clist_freeze(GTK_CLIST(w));

	for(i = 0; i < editor->total_models; i++)
	    EditorListModelsSet(
		w, i, editor->model[i], FALSE
	    );

        gtk_clist_thaw(GTK_CLIST(w)); 

        return(0);
}

/*
 *	Add primitives from the specified model list to the GUI
 *	primitives list on the editor.
 *
 *	GUI list will not be cleared.
 */
int EditorListAddPrimitivesRG(
	ma_editor_struct *editor, v3d_model_struct *model
)
{
        int i;
	void *p;
        GtkWidget *w;


	if((editor == NULL) ||
           (model == NULL)
	)
	    return(-1);

        w = editor->primitives_list;
        if(w == NULL)
            return(-1);

        gtk_clist_freeze(GTK_CLIST(w));

        for(i = 0; i < model->total_primitives; i++)
        {
            p = model->primitive[i];
	    EditorListPrimitivesSet(w, i, p, FALSE);
        }

        gtk_clist_thaw(GTK_CLIST(w)); 

        return(0);
}

/*
 *	Add values from the specified primitive to the GUI values list
 *	on the editor.
 *
 *	Does not clear the list.
 */
int EditorListAddValuesRG(
        ma_editor_struct *editor, void *p
)
{
        int i;
        GtkWidget *w;
        gchar *val[4], *heading[4];

	mp_vertex_struct *v, *n, *tc;

	mp_translate_struct *mp_translate;
	mp_rotate_struct *mp_rotate;
	mp_point_struct *mp_point;
	mp_line_struct *mp_line;
	mp_line_strip_struct *mp_line_strip;
	mp_line_loop_struct *mp_line_loop;
	mp_triangle_struct *mp_triangle;
	mp_triangle_strip_struct *mp_triangle_strip;
        mp_triangle_fan_struct *mp_triangle_fan;
	mp_quad_struct *mp_quad;
        mp_quad_strip_struct *mp_quad_strip;
	mp_polygon_struct *mp_polygon;
        mp_color_struct *mp_color;
	mp_texture_select_struct *mp_texture_select;
        mp_texture_orient_xy_struct *mp_texture_xy;
	mp_texture_orient_yz_struct *mp_texture_yz;
	mp_texture_orient_xz_struct *mp_texture_xz;
	mp_heightfield_load_struct *mp_heightfield_load;

        char text[256];
        char text2[256];
        char text3[256];


        if((editor == NULL) || (p == NULL))
            return(-1);

        w = editor->values_list;
        if(w == NULL)
            return(-1);

	/* Reset text strings. */
	(*text) = '\0';
	(*text2) = '\0';
	(*text3) = '\0';

	/* Reset header string pointers. */
	heading[0] = "";
	heading[1] = "";
	heading[2] = "";
	heading[3] = NULL;

        gtk_clist_freeze(GTK_CLIST(w));

#define NORMAL_HEADING_STR	"Normal"
#define TEXCOORD_HEADING_STR	"TexCoord"
#define DO_APPEND_TEXT	\
{ \
 val[0] = text; \
 val[1] = text2; \
 val[2] = text3; \
 val[3] = NULL; \
 gtk_clist_append(GTK_CLIST(w), val); \
}

	switch(*(int *)p)
	{
	  case V3DMP_TYPE_TRANSLATE:
	    mp_translate = p;
	    sprintf(text, "%.2f, %.2f, %.2f",
		mp_translate->x,
		mp_translate->y,
		mp_translate->z
	    );
	    DO_APPEND_TEXT
	    heading[0] = "Translation";
	    break;

          case V3DMP_TYPE_ROTATE:
	    mp_rotate = p;
            sprintf(text, "%.2f, %.2f, %.2f",
                RADTODEG(mp_rotate->heading),
                RADTODEG(mp_rotate->pitch),
                RADTODEG(mp_rotate->bank)
            );
            DO_APPEND_TEXT
            heading[0] = "Rotation";
            break;


	  case V3DMP_TYPE_POINT:
#define PTR_NAME	mp_point
	    PTR_NAME = (mp_point_struct *)p;
            for(i = 0; i < V3DMP_POINT_NVERTEX; i++)
	    {
                v = &PTR_NAME->v[i];
                sprintf(text, "%i: %.2f, %.2f, %.2f",
                    i, v->x, v->y, v->z
                );
                n = &PTR_NAME->n[i];
                sprintf(text2, "%.2f, %.2f, %.2f",
                    n->x, n->y, n->z
                );
                tc = &PTR_NAME->tc[i];
                sprintf(text3, "%.2f, %.2f, %.2f",
                    tc->x, tc->y, tc->z
                );
                DO_APPEND_TEXT
            }
            heading[0] = "Point Vertex";
            heading[1] = NORMAL_HEADING_STR;
            heading[2] = TEXCOORD_HEADING_STR;
            break;

	  case V3DMP_TYPE_LINE:
#undef PTR_NAME
#define PTR_NAME	mp_line
            PTR_NAME = (mp_line_struct *)p;
            for(i = 0; i < V3DMP_LINE_NVERTEX; i++)
            {
                v = &PTR_NAME->v[i];
                sprintf(text, "%i: %.2f, %.2f, %.2f",
                    i, v->x, v->y, v->z
                );
                n = &PTR_NAME->n[i];
                sprintf(text2, "%.2f, %.2f, %.2f",
                    n->x, n->y, n->z
                );
                tc = &PTR_NAME->tc[i];
                sprintf(text3, "%.2f, %.2f, %.2f",
                    tc->x, tc->y, tc->z
                );
                DO_APPEND_TEXT
            }
            heading[0] = "Line Vertex";
            heading[1] = NORMAL_HEADING_STR;
            heading[2] = TEXCOORD_HEADING_STR;
	    break;

          case V3DMP_TYPE_LINE_STRIP:
#undef PTR_NAME
#define PTR_NAME	mp_line_strip
            PTR_NAME = (mp_line_strip_struct *)p;
            for(i = 0; i < PTR_NAME->total; i++)
            {
                v = PTR_NAME->v[i];
		if(v != NULL)
                    sprintf(text, "%i: %.2f, %.2f, %.2f",
                        i, v->x, v->y, v->z
                    );
                n = PTR_NAME->n[i];
		if(n != NULL)
		    sprintf(text2, "%.2f, %.2f, %.2f",
                        n->x, n->y, n->z
                    );
                tc = PTR_NAME->tc[i];
		if(tc != NULL)
                    sprintf(text3, "%.2f, %.2f, %.2f",
                        tc->x, tc->y, tc->z
                    );
                DO_APPEND_TEXT
            }
            heading[0] = "Line Strip Vertex";
            heading[1] = NORMAL_HEADING_STR;
            heading[2] = TEXCOORD_HEADING_STR;
	    break;

	  case V3DMP_TYPE_LINE_LOOP:
#undef PTR_NAME
#define PTR_NAME	mp_line_loop
            PTR_NAME = (mp_line_loop_struct *)p;
            for(i = 0; i < PTR_NAME->total; i++)
            {
                v = PTR_NAME->v[i];
                if(v != NULL)
                    sprintf(text, "%i: %.2f, %.2f, %.2f",
                        i, v->x, v->y, v->z
                    );
                n = PTR_NAME->n[i];
                if(n != NULL)
                    sprintf(text2, "%.2f, %.2f, %.2f",
                        n->x, n->y, n->z
                    );
                tc = PTR_NAME->tc[i];
                if(tc != NULL)
                    sprintf(text3, "%.2f, %.2f, %.2f",
                        tc->x, tc->y, tc->z
                    );
                DO_APPEND_TEXT
            }
            heading[0] = "Line Loop Vertex";
            heading[1] = NORMAL_HEADING_STR;
            heading[2] = TEXCOORD_HEADING_STR;
	    break;

          case V3DMP_TYPE_TRIANGLE:
#undef PTR_NAME
#define PTR_NAME	mp_triangle
            PTR_NAME = (mp_triangle_struct *)p;
            for(i = 0; i < V3DMP_TRIANGLE_NVERTEX; i++)
            {
                v = &PTR_NAME->v[i];
                sprintf(text, "%i: %.2f, %.2f, %.2f",
                    i, v->x, v->y, v->z
                );
                n = &PTR_NAME->n[i];
                sprintf(text2, "%.2f, %.2f, %.2f",
                    n->x, n->y, n->z
                );
                tc = &PTR_NAME->tc[i];
                sprintf(text3, "%.2f, %.2f, %.2f",
                    tc->x, tc->y, tc->z
                );
                DO_APPEND_TEXT
            }
            heading[0] = "Triangle Vertex";
            heading[1] = NORMAL_HEADING_STR;
            heading[2] = TEXCOORD_HEADING_STR;
            break;

          case V3DMP_TYPE_TRIANGLE_STRIP:
#undef PTR_NAME
#define PTR_NAME	mp_triangle_strip
            PTR_NAME = (mp_triangle_strip_struct *)p;
            for(i = 0; i < PTR_NAME->total; i++)
            {
                v = PTR_NAME->v[i];   
                if(v != NULL)
                    sprintf(text, "%i: %.2f, %.2f, %.2f",
                        i, v->x, v->y, v->z
                    );
                n = PTR_NAME->n[i];
                if(n != NULL)
                    sprintf(text2, "%.2f, %.2f, %.2f",
                        n->x, n->y, n->z
                    );
                tc = PTR_NAME->tc[i];
                if(tc != NULL)
                    sprintf(text3, "%.2f, %.2f, %.2f",
                        tc->x, tc->y, tc->z
                    );
                DO_APPEND_TEXT
            }
            heading[0] = "Triangle Strip Vertex";
            heading[1] = NORMAL_HEADING_STR;
            heading[2] = TEXCOORD_HEADING_STR;
            break;

          case V3DMP_TYPE_TRIANGLE_FAN:
#undef PTR_NAME
#define PTR_NAME	mp_triangle_fan
            PTR_NAME = (mp_triangle_fan_struct *)p;
            for(i = 0; i < PTR_NAME->total; i++)
            {
                v = PTR_NAME->v[i];
                if(v != NULL)
                    sprintf(text, "%i: %.2f, %.2f, %.2f",
                        i, v->x, v->y, v->z
                    );
                n = PTR_NAME->n[i];
                if(n != NULL)
                    sprintf(text2, "%.2f, %.2f, %.2f",
                        n->x, n->y, n->z
                    );
                tc = PTR_NAME->tc[i];
                if(tc != NULL)
                    sprintf(text3, "%.2f, %.2f, %.2f",
                        tc->x, tc->y, tc->z
                    );
                DO_APPEND_TEXT
            }
            heading[0] = "Triangle Fan Vertex";
            heading[1] = NORMAL_HEADING_STR;
            heading[2] = TEXCOORD_HEADING_STR;
            break;

          case V3DMP_TYPE_QUAD:
#undef PTR_NAME
#define PTR_NAME	mp_quad
            PTR_NAME = (mp_quad_struct *)p;
            for(i = 0; i < V3DMP_QUAD_NVERTEX; i++)
            {
                v = &PTR_NAME->v[i];
                sprintf(text, "%i: %.2f, %.2f, %.2f",
                    i, v->x, v->y, v->z
                );
                n = &PTR_NAME->n[i];
                sprintf(text2, "%.2f, %.2f, %.2f",
                    n->x, n->y, n->z
                );
                tc = &PTR_NAME->tc[i];
                sprintf(text3, "%.2f, %.2f, %.2f",
                    tc->x, tc->y, tc->z
                );
                DO_APPEND_TEXT
            }
            heading[0] = "Quad Vertex";
            heading[1] = NORMAL_HEADING_STR;
            heading[2] = TEXCOORD_HEADING_STR;
            break;

          case V3DMP_TYPE_QUAD_STRIP:
#undef PTR_NAME
#define PTR_NAME	mp_quad_strip
            PTR_NAME = (mp_quad_strip_struct *)p;
            for(i = 0; i < PTR_NAME->total; i++)
            {
                v = PTR_NAME->v[i];
                if(v != NULL)  
                    sprintf(text, "%i: %.2f, %.2f, %.2f",
                        i, v->x, v->y, v->z
                    );
                n = PTR_NAME->n[i];
                if(n != NULL)
                    sprintf(text2, "%.2f, %.2f, %.2f",
                        n->x, n->y, n->z
                    );
                tc = PTR_NAME->tc[i];
                if(tc != NULL)
                    sprintf(text3, "%.2f, %.2f, %.2f",
                        tc->x, tc->y, tc->z
                    );
                DO_APPEND_TEXT
            }
            heading[0] = "Quad Strip Vertex";
            heading[1] = NORMAL_HEADING_STR;
            heading[2] = TEXCOORD_HEADING_STR;
            break;

         case V3DMP_TYPE_POLYGON:
#undef PTR_NAME
#define PTR_NAME	mp_polygon
            PTR_NAME = (mp_polygon_struct *)p;
            for(i = 0; i < PTR_NAME->total; i++)
            {
                v = PTR_NAME->v[i];
                if(v != NULL)
                    sprintf(text, "%i: %.2f, %.2f, %.2f",
                        i, v->x, v->y, v->z
                    );
                n = PTR_NAME->n[i];  
                if(n != NULL)
                    sprintf(text2, "%.2f, %.2f, %.2f",
                        n->x, n->y, n->z
                    );
                tc = PTR_NAME->tc[i];
                if(tc != NULL)
                    sprintf(text3, "%.2f, %.2f, %.2f",
                        tc->x, tc->y, tc->z
                    );
                DO_APPEND_TEXT
            }
            heading[0] = "Polygon Vertex";
            heading[1] = NORMAL_HEADING_STR;
            heading[2] = TEXCOORD_HEADING_STR;
            break;
#undef PTR_NAME

	  case V3DMP_TYPE_COLOR:
	    mp_color = p;
	    sprintf(text, "(a,r,g,b): %.2f, %.2f, %.2f, %.2f",
		mp_color->a,
		mp_color->r,
		mp_color->g,
		mp_color->b
	    );
	    DO_APPEND_TEXT
            sprintf(text, "ambient: %.2f",
                mp_color->ambient
            );
            DO_APPEND_TEXT
            sprintf(text, "diffuse: %.2f",
                mp_color->diffuse
            );
            DO_APPEND_TEXT
            sprintf(text, "specular: %.2f",
                mp_color->specular
            );
            DO_APPEND_TEXT
            sprintf(text, "shininess: %.2f",
                mp_color->shininess
            );
            DO_APPEND_TEXT
            sprintf(text, "emission: %.2f",
                mp_color->emission
            );
            DO_APPEND_TEXT
	    heading[0] = "Color";
	    break;

	  case V3DMP_TYPE_TEXTURE_SELECT:
	    mp_texture_select = p;
	    (*text) = '\0';
	    strncat(
		text,
		((mp_texture_select->name == NULL) ?
		    VAL_NULL_STR : mp_texture_select->name
		),
		80
	    );
	    DO_APPEND_TEXT
	    heading[0] = "Name";
            break;

	  case V3DMP_TYPE_TEXTURE_ORIENT_XY:
	    mp_texture_xy = p;
            sprintf(text, "(x,y,dx,dy): %.2f, %.2f, %.2f, %.2f",
                mp_texture_xy->x,
                mp_texture_xy->y,
                mp_texture_xy->dx,
                mp_texture_xy->dy
            );
            DO_APPEND_TEXT
	    heading[0] = "TexOrient XY";
	    break;

          case V3DMP_TYPE_TEXTURE_ORIENT_YZ:
            mp_texture_yz = p;
            sprintf(text, "(y,z,dy,dz): %.2f, %.2f, %.2f, %.2f",
                mp_texture_yz->y,
                mp_texture_yz->z,
                mp_texture_yz->dy,
                mp_texture_yz->dz
            );
            DO_APPEND_TEXT
            heading[0] = "TexOrient YZ"; 
            break;

          case V3DMP_TYPE_TEXTURE_ORIENT_XZ:
            mp_texture_xz = p;
            sprintf(text, "(x,z,dx,dz): %.2f, %.2f, %.2f, %.2f",
                mp_texture_xz->x,
                mp_texture_xz->z,
                mp_texture_xz->dx,
                mp_texture_xz->dz
            );
            DO_APPEND_TEXT
            heading[0] = "TexOrient XZ"; 
            break;

          case V3DMP_TYPE_HEIGHTFIELD_LOAD:
            mp_heightfield_load = p;
            (*text) = '\0';
            strncat(
                text,
                ((mp_heightfield_load->path == NULL) ?
                    VAL_NULL_STR : mp_heightfield_load->path
                ),
                80
            );
            DO_APPEND_TEXT
            sprintf(text, "(xl,yl,zl): %.2f, %.2f, %.2f",
                mp_heightfield_load->x_length,
                mp_heightfield_load->y_length,
                mp_heightfield_load->z_length
            );
            DO_APPEND_TEXT
            sprintf(text, "(x,y,z): %.2f, %.2f, %.2f",
                mp_heightfield_load->x,
                mp_heightfield_load->y,
                mp_heightfield_load->z
            );
            DO_APPEND_TEXT
            sprintf(text, "(h,p,b): %.2f, %.2f, %.2f",
                RADTODEG(mp_heightfield_load->heading),
                RADTODEG(mp_heightfield_load->bank),
                RADTODEG(mp_heightfield_load->pitch)
            );
            DO_APPEND_TEXT
            heading[0] = "HeightField";
            break;


	  default:
	    break;
	}


#undef DO_APPEND_TEXT
#undef NORMAL_HEADING_STR
#undef TEXCOORD_HEADING_STR

	/* Update column titles. */
	gtk_clist_set_column_title(
	    GTK_CLIST(w), 0, heading[0]
	);
        gtk_clist_set_column_title(
            GTK_CLIST(w), 1, heading[1]
        );
        gtk_clist_set_column_title(
            GTK_CLIST(w), 2, heading[2]
        );

        gtk_clist_thaw(GTK_CLIST(w)); 

	return(0);
}


/* Tempory texture orient structure type for
 * EditorListValuePromptApply().
 */
typedef struct {
	double i, j, di, dj;
} texorient_recval_struct;
typedef struct {
        double x_length, y_length, z_length;
	double x, y, z;
	double heading, pitch, bank;
} heightfield_values_recval_struct;  

/*
 *	Applies values from the values text prompts to the
 *	selected values list item on the values Real and GUI lists.
 *
 *	The widget entry should be the one that triggered the apply.
 *	If entry is NULL then it will be assumed to be a set vertex.
 */
int EditorListValuePromptApply(
	ma_editor_struct *editor, GtkWidget *entry, gbool record_undo
)
{
	gbool set_normal = FALSE;
        gbool set_texcoord = FALSE;
        gbool set_vertex = FALSE;
	gbool set_texorient = FALSE;
	gbool set_heightfield_load = FALSE;
	gbool set_heightfield_values = FALSE;

        GtkWidget *w, *clist, *primitives_clist;
        gchar *text_ptr;

	int model_num, value_num, pn;
	v3d_model_struct *model_ptr;
	void *p;
	mp_vertex_struct *v = NULL, *tc = NULL, *n = NULL, old_v, new_v;

	texorient_recval_struct old_texorient, new_texorient;
	heightfield_values_recval_struct old_heightfield_values;

        mp_translate_struct *mp_translate;
        mp_rotate_struct *mp_rotate;
	mp_point_struct *mp_point;
        mp_line_struct *mp_line;
	mp_line_strip_struct *mp_line_strip;
	mp_line_loop_struct *mp_line_loop;
        mp_triangle_struct *mp_triangle;
	mp_triangle_strip_struct *mp_triangle_strip;
        mp_triangle_fan_struct *mp_triangle_fan;
        mp_quad_struct *mp_quad;
        mp_quad_strip_struct *mp_quad_strip;
        mp_polygon_struct *mp_polygon;
        mp_color_struct *mp_color;
        mp_texture_select_struct *mp_texture_select;
        mp_texture_orient_xy_struct *mp_texture_xy;
        mp_texture_orient_yz_struct *mp_texture_yz;
        mp_texture_orient_xz_struct *mp_texture_xz;
	mp_heightfield_load_struct *mp_heightfield_load;

	char text[256];


        if(editor == NULL)
            return(-1);

	clist = editor->values_list;
	if(clist == NULL)
	    return(-1);

	primitives_clist = editor->primitives_list;
	if(primitives_clist == NULL)
	    return(-1);

	/* Make sure only one primitive is selected and get the
	 * value of that.
	 */
	if(editor->total_selected_primitives == 1)
	    pn = editor->selected_primitive[0];
	else
	    return(0);

	/* Get selected value item number. */
/* Itterate through all selected values? */
	value_num = EditorGetSelected(
	    editor->selected_value, editor->total_selected_values,
	    0
	);

	/* Get pointer to selected model. */
	model_num = EditorSelectedModelIndex(editor);
	model_ptr = V3DModelListGetPtr(
	    editor->model, editor->total_models, model_num
	);

	/* Don't bother applying values if no value, no primitive,
	 * or no model is selected.
	 */
	if((pn < 0) || (value_num < 0) || (model_ptr == NULL))
	    return(0);

	/* Get selected primitive. */
	p = V3DMPListGetPtr(
	    model_ptr->primitive, model_ptr->total_primitives, pn
	);
	if(p == NULL)
	    return(0);

#define DO_GET_TEXT_PTR		\
{ \
 if(w != NULL) \
  text_ptr = gtk_entry_get_text(GTK_ENTRY(w)); \
 else \
  text_ptr = NULL; \
}

#define DO_SET_VALUES_ITEM_VERTEX	\
{ \
 /* Check which entry this apply is for (if any). */ \
 if((entry != NULL) && (VMA_PRIMITIVES_MAX_VALUES >= 9)) \
 { \
  if((entry == editor->values_text[0]) || \
     (entry == editor->values_text[1]) || \
     (entry == editor->values_text[2]) \
  ) \
   set_vertex = TRUE; \
  else if((entry == editor->values_text[3]) || \
          (entry == editor->values_text[4]) || \
          (entry == editor->values_text[5]) \
  ) \
   set_normal = TRUE; \
  else if((entry == editor->values_text[6]) || \
          (entry == editor->values_text[7]) || \
          (entry == editor->values_text[8]) \
  ) \
   set_texcoord = TRUE; \
 } \
 else if(entry == NULL) \
 { \
  /* Assume set vertex if entry is NULL. */ \
  set_vertex = TRUE; \
 } \
 \
 /* Set vertex? */ \
 if(set_vertex && (v != NULL)) \
 { \
  /* Record old vertex values. */ \
  old_v.x = v->x; \
  old_v.y = v->y; \
  old_v.z = v->z; \
  /* Get new vertex values. */ \
  w = editor->values_text[0]; \
  DO_GET_TEXT_PTR \
  if(text_ptr != NULL) \
   new_v.x = atof(text_ptr); \
  w = editor->values_text[1]; \
  DO_GET_TEXT_PTR \
  if(text_ptr != NULL) \
   new_v.y = atof(text_ptr); \
  w = editor->values_text[2]; \
  DO_GET_TEXT_PTR \
  if(text_ptr != NULL) \
   new_v.z = atof(text_ptr); \
 \
  /* Set new vertex values. */ \
  v->x = new_v.x; \
  v->y = new_v.y; \
  v->z = new_v.z; \
 \
  /* Update selected values list item. */ \
  sprintf(text, "%i: %.2f, %.2f, %.2f", \
   vtx_num, v->x, v->y, v->z \
  ); \
  gtk_clist_set_text( \
   GTK_CLIST(clist), \
   value_num, 0, \
   text \
  ); \
 } \
 /* Set normal? */ \
 else if(set_normal && (n != NULL)) \
 { \
  /* Record old normal values. */ \
  old_v.x = n->x; \
  old_v.y = n->y; \
  old_v.z = n->z; \
  /* Get new vertex values. */ \
  w = editor->values_text[3]; \
  DO_GET_TEXT_PTR \
  if(text_ptr != NULL) \
   new_v.x = atof(text_ptr); \
  w = editor->values_text[4]; \
  DO_GET_TEXT_PTR \
  if(text_ptr != NULL) \
   new_v.y = atof(text_ptr); \
  w = editor->values_text[5]; \
  DO_GET_TEXT_PTR \
  if(text_ptr != NULL) \
   new_v.z = atof(text_ptr); \
 \
  /* Set new normal values. */ \
  n->x = new_v.x; \
  n->y = new_v.y; \
  n->z = new_v.z; \
 \
  /* Update selected values list item. */ \
  sprintf(text, "%.2f, %.2f, %.2f", \
   n->x, n->y, n->z \
  ); \
  gtk_clist_set_text( \
   GTK_CLIST(clist), \
   value_num, 1, \
   text \
  ); \
 } \
 /* Set texcoord? */ \
 else if(set_texcoord && (tc != NULL)) \
 { \
  /* Record old texcoord values. */ \
  old_v.x = tc->x; \
  old_v.y = tc->y; \
  old_v.z = tc->z; \
  /* Get new vertex values. */ \
  w = editor->values_text[6]; \
  DO_GET_TEXT_PTR \
  if(text_ptr != NULL) \
   new_v.x = atof(text_ptr); \
  w = editor->values_text[7]; \
  DO_GET_TEXT_PTR \
  if(text_ptr != NULL) \
   new_v.y = atof(text_ptr); \
  w = editor->values_text[8]; \
  DO_GET_TEXT_PTR \
  if(text_ptr != NULL) \
   new_v.z = atof(text_ptr); \
 \
  /* Set new texcoord values. */ \
  tc->x = new_v.x; \
  tc->y = new_v.y; \
  tc->z = new_v.z; \
 \
  /* Update selected values list item. */ \
  sprintf(text, "%.2f, %.2f, %.2f", \
   tc->x, tc->y, tc->z \
  ); \
  gtk_clist_set_text( \
   GTK_CLIST(clist), \
   value_num, 2, \
   text \
  ); \
 } \
}



	/* Begin setting values from values text prompts by the
	 * selected primitive type.
	 */
	switch(*(int *)p)
	{
	  case V3DMP_TYPE_TRANSLATE:
	    mp_translate = p;
            if((value_num == 0) &&
               (VMA_PRIMITIVES_MAX_VALUES >= 3)
            )
            {
		w = editor->values_text[0];
		DO_GET_TEXT_PTR
		if(text_ptr != NULL)
		    mp_translate->x = atof(text_ptr);
                w = editor->values_text[1];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_translate->y = atof(text_ptr);
                w = editor->values_text[2];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_translate->z = atof(text_ptr);

                sprintf(text, "%.2f, %.2f, %.2f",
                    mp_translate->x,
                    mp_translate->y,
		    mp_translate->z
                );
                gtk_clist_set_text(
                    GTK_CLIST(clist),
                    value_num, 0,
                    text
                );

		/* Update primitive item listing on primitives list. */
		EditorListPrimitivesSetTranslate(
		    primitives_clist, pn,
		    p, TRUE
		);
	    }
	    break;

          case V3DMP_TYPE_ROTATE:
            mp_rotate = p;
            if((value_num == 0) &&
               (VMA_PRIMITIVES_MAX_VALUES >= 3)
            )   
            {
                w = editor->values_text[0];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_rotate->heading = DEGTORAD(atof(text_ptr));
                w = editor->values_text[1];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_rotate->pitch = DEGTORAD(atof(text_ptr));
                w = editor->values_text[2];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_rotate->bank = DEGTORAD(atof(text_ptr));

                sprintf(text, "%.2f, %.2f, %.2f",
                    RADTODEG(mp_rotate->heading),
                    RADTODEG(mp_rotate->bank),   
                    RADTODEG(mp_rotate->pitch)
                );
		gtk_clist_set_text(
		    GTK_CLIST(clist),
		    value_num, 0,
		    text
		);

                /* Update primitive item listing on primitives list. */
                EditorListPrimitivesSetTranslate(   
                    primitives_clist, pn,
                    p, TRUE
                );
	    }
            break;


          case V3DMP_TYPE_POINT:
#define PTR_NAME	mp_point
            PTR_NAME = (mp_point_struct *)p;
            if(1)
            {
		int vtx_num = value_num;
		if((vtx_num >= 0) && (vtx_num < V3DMP_POINT_NVERTEX))
		{
		    n = &PTR_NAME->n[vtx_num];
                    tc = &PTR_NAME->tc[vtx_num];
		    v = &PTR_NAME->v[vtx_num];
                }
		DO_SET_VALUES_ITEM_VERTEX
	    }
	    break;

	  case V3DMP_TYPE_LINE:
#undef PTR_NAME
#define PTR_NAME	mp_line
            PTR_NAME = (mp_line_struct *)p;
            if(1)
            {
                int vtx_num = value_num;
                if((vtx_num >= 0) && (vtx_num < V3DMP_LINE_NVERTEX))
                {
                    n = &PTR_NAME->n[vtx_num];
                    tc = &PTR_NAME->tc[vtx_num];
                    v = &PTR_NAME->v[vtx_num];  
                }
                DO_SET_VALUES_ITEM_VERTEX
            }
	    break;

          case V3DMP_TYPE_LINE_STRIP:
#undef PTR_NAME
#define PTR_NAME	mp_line_strip
            PTR_NAME = (mp_line_strip_struct *)p;
            if(1)
            {
                int vtx_num = value_num;
                if((vtx_num >= 0) && (vtx_num < PTR_NAME->total))
                {
                    n = PTR_NAME->n[vtx_num];
                    tc = PTR_NAME->tc[vtx_num];
                    v = PTR_NAME->v[vtx_num];
                }
                DO_SET_VALUES_ITEM_VERTEX
            }
            break;

          case V3DMP_TYPE_LINE_LOOP:
#undef PTR_NAME
#define PTR_NAME	mp_line_loop
            PTR_NAME = (mp_line_loop_struct *)p;
            if(1)
            {
                int vtx_num = value_num;
                if((vtx_num >= 0) && (vtx_num < PTR_NAME->total))
                {
                    n = PTR_NAME->n[vtx_num];  
                    tc = PTR_NAME->tc[vtx_num];
                    v = PTR_NAME->v[vtx_num];
                }
                DO_SET_VALUES_ITEM_VERTEX
            }
            break;

          case V3DMP_TYPE_TRIANGLE:
#undef PTR_NAME
#define PTR_NAME	mp_triangle
            PTR_NAME = (mp_triangle_struct *)p;
            if(1)
            {
                int vtx_num = value_num;
                if((vtx_num >= 0) && (vtx_num < V3DMP_TRIANGLE_NVERTEX))
                {
                    n = &PTR_NAME->n[vtx_num];
                    tc = &PTR_NAME->tc[vtx_num];
                    v = &PTR_NAME->v[vtx_num];  
                }
                DO_SET_VALUES_ITEM_VERTEX
            }
	    break;

          case V3DMP_TYPE_TRIANGLE_STRIP:
#undef PTR_NAME
#define PTR_NAME	mp_triangle_strip
            PTR_NAME = (mp_triangle_strip_struct *)p;
            if(1)
            {
                int vtx_num = value_num;
                if((vtx_num >= 0) && (vtx_num < PTR_NAME->total))
                {
                    n = PTR_NAME->n[vtx_num];  
                    tc = PTR_NAME->tc[vtx_num];
                    v = PTR_NAME->v[vtx_num];
                }
                DO_SET_VALUES_ITEM_VERTEX
            }
            break;

          case V3DMP_TYPE_TRIANGLE_FAN:
#undef PTR_NAME
#define PTR_NAME	mp_triangle_fan
            PTR_NAME = (mp_triangle_fan_struct *)p;
            if(1)
            {
                int vtx_num = value_num;
                if((vtx_num >= 0) && (vtx_num < PTR_NAME->total))
                {
                    n = PTR_NAME->n[vtx_num];  
                    tc = PTR_NAME->tc[vtx_num];
                    v = PTR_NAME->v[vtx_num];
                }
                DO_SET_VALUES_ITEM_VERTEX
            }
            break;
 
          case V3DMP_TYPE_QUAD:
#undef PTR_NAME
#define PTR_NAME	mp_quad
            PTR_NAME = (mp_quad_struct *)p;
            if(1)
            {
                int vtx_num = value_num;
                if((vtx_num >= 0) && (vtx_num < V3DMP_QUAD_NVERTEX))
                {
                    n = &PTR_NAME->n[vtx_num];
                    tc = &PTR_NAME->tc[vtx_num];
                    v = &PTR_NAME->v[vtx_num];
                }
                DO_SET_VALUES_ITEM_VERTEX
            }
            break;

          case V3DMP_TYPE_QUAD_STRIP:
#undef PTR_NAME
#define PTR_NAME	mp_quad_strip
            PTR_NAME = (mp_quad_strip_struct *)p;
            if(1)
            {
                int vtx_num = value_num;
                if((vtx_num >= 0) && (vtx_num < PTR_NAME->total))
                {
                    n = PTR_NAME->n[vtx_num];  
                    tc = PTR_NAME->tc[vtx_num];
                    v = PTR_NAME->v[vtx_num];
                }
                DO_SET_VALUES_ITEM_VERTEX
            }
            break;

          case V3DMP_TYPE_POLYGON:
#undef PTR_NAME
#define PTR_NAME	mp_polygon
            PTR_NAME = (mp_polygon_struct *)p;
            if(1)
            {
                int vtx_num = value_num;
                if((vtx_num >= 0) && (vtx_num < PTR_NAME->total))
                {
                    n = PTR_NAME->n[vtx_num];
                    tc = PTR_NAME->tc[vtx_num];
                    v = PTR_NAME->v[vtx_num];
                }
                DO_SET_VALUES_ITEM_VERTEX
            }
            break;
#undef PTR_NAME

          case V3DMP_TYPE_COLOR:
            mp_color = p;
            if((value_num == 0) &&
               (VMA_PRIMITIVES_MAX_VALUES >= 4)
            )
            {
                w = editor->values_text[0];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_color->a = CLIP(atof(text_ptr), 0.0, 1.0);
                w = editor->values_text[1];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_color->r = CLIP(atof(text_ptr), 0.0, 1.0);
                w = editor->values_text[2];   
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_color->g = CLIP(atof(text_ptr), 0.0, 1.0);
                w = editor->values_text[3];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_color->b = CLIP(atof(text_ptr), 0.0, 1.0);

                sprintf(text, "(a,r,g,b): %.2f, %.2f, %.2f, %.2f",
                    mp_color->a,
                    mp_color->r,
                    mp_color->g,
                    mp_color->b
                );
                gtk_clist_set_text(
                    GTK_CLIST(clist),
                    value_num, 0,
                    text
                );

                /* Update primitive item listing on primitives list. */
                EditorListPrimitivesSetColor(
                    primitives_clist, pn,
                    p, TRUE
                );
            }
            else if((value_num == 1) &&
                    (VMA_PRIMITIVES_MAX_VALUES >= 1)
            )
	    {
		/* Ambient. */
                w = editor->values_text[0];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_color->ambient = CLIP(atof(text_ptr), 0.0, 1.0);

                sprintf(text, "ambient: %.2f",
                    mp_color->ambient
                );
                gtk_clist_set_text(
                    GTK_CLIST(clist),
                    value_num, 0,
                    text
                );

		/* No need to update row on primitives list. */
	    }
            else if((value_num == 2) &&
                    (VMA_PRIMITIVES_MAX_VALUES >= 1)
            )
            {
                /* Diffuse. */
                w = editor->values_text[0];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_color->diffuse = CLIP(atof(text_ptr), 0.0, 1.0);

                sprintf(text, "diffuse: %.2f",
                    mp_color->diffuse
                );
                gtk_clist_set_text(
                    GTK_CLIST(clist),
                    value_num, 0,
                    text
                );

                /* No need to update row on primitives list. */
            }     
            else if((value_num == 3) &&
                    (VMA_PRIMITIVES_MAX_VALUES >= 1)
            )
            {
                /* Specular. */
                w = editor->values_text[0];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_color->specular = CLIP(atof(text_ptr), 0.0, 1.0);

                sprintf(text, "specular: %.2f",
                    mp_color->specular
                );
                gtk_clist_set_text(
                    GTK_CLIST(clist),
                    value_num, 0,
                    text
                );

                /* No need to update row on primitives list. */
            }
            else if((value_num == 4) &&
                    (VMA_PRIMITIVES_MAX_VALUES >= 1)
            )
            {
                /* Shininess. */
                w = editor->values_text[0];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_color->shininess = CLIP(atof(text_ptr), 0.0, 1.0);

                sprintf(text, "shininess: %.2f",
                    mp_color->shininess
                );
                gtk_clist_set_text(
                    GTK_CLIST(clist),
                    value_num, 0,
                    text
                );
                /* No need to update row on primitives list. */
            }
            else if((value_num == 5) &&
                    (VMA_PRIMITIVES_MAX_VALUES >= 1)
            ) 
            {
                /* Emission. */
                w = editor->values_text[0];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_color->emission = CLIP(atof(text_ptr), 0.0, 1.0);

                sprintf(text, "emission: %.2f",
                    mp_color->emission
                );
                gtk_clist_set_text(
                    GTK_CLIST(clist),
                    value_num, 0,
                    text
                );
                /* No need to update row on primitives list. */
            }
            break;

          case V3DMP_TYPE_TEXTURE_SELECT:
            mp_texture_select = p;
            if((value_num == 0) &&
               (VMA_PRIMITIVES_MAX_VALUES >= 1)
            )
            {
                w = editor->values_text[0];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
		{
		    int tex_num;
		    v3d_texture_ref_struct *t;

		    /* Set new referance name on primitive. */
		    free(mp_texture_select->name);
		    mp_texture_select->name = strdup(text_ptr);

		    /* Match texture number on editor. */
		    t = EditorMatchTextureByName(
			editor, mp_texture_select->name,
			&tex_num
		    );
		    if((tex_num >= 0) && (t != NULL))
			mp_texture_select->client_data = (void *)tex_num;
		    else
			mp_texture_select->client_data = (void *)0;

                    /* Update item in values list. */
                    *text = '\0';
                    strncat(
                        text,
                        (mp_texture_select->name == NULL) ?
                            VAL_NULL_STR : mp_texture_select->name,
                        80
                    );
                    gtk_clist_set_text(
                        GTK_CLIST(clist),
                        value_num, 0,
                        text
                    );

                    /* Update primitive item listing on primitives list. */
                    EditorListPrimitivesSetTexture(
                        primitives_clist, pn,
                        p, TRUE
                    );

		    /* Realize texture select. */
		    EditorTexturePrimitiveRealize(
                        editor, mp_texture_select, TRUE
                    );
		}
	    }
	    break;

          case V3DMP_TYPE_TEXTURE_ORIENT_XY:
#define PRIM_PTR	mp_texture_xy
            PRIM_PTR = p;
            if((value_num == 0) &&
               (VMA_PRIMITIVES_MAX_VALUES >= 4)
            )
            {
		old_texorient.i = PRIM_PTR->x;
                old_texorient.j = PRIM_PTR->y;
                old_texorient.di = PRIM_PTR->dx;
                old_texorient.dj = PRIM_PTR->dy;

                w = editor->values_text[0];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    new_texorient.i = atof(text_ptr);
                w = editor->values_text[1];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)   
                    new_texorient.j = atof(text_ptr);
                w = editor->values_text[2];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    new_texorient.di = atof(text_ptr);
                w = editor->values_text[3];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    new_texorient.dj = atof(text_ptr);

                PRIM_PTR->x = new_texorient.i;
                PRIM_PTR->y = new_texorient.j;
                PRIM_PTR->dx = new_texorient.di;
                PRIM_PTR->dy = new_texorient.dj;
                sprintf(text, "(x,y,dx,dy): %.2f, %.2f, %.2f, %.2f",
                    PRIM_PTR->x,
                    PRIM_PTR->y,
                    PRIM_PTR->dx,
                    PRIM_PTR->dy
                );
                gtk_clist_set_text( 
                    GTK_CLIST(clist),
                    value_num, 0,
                    text
                );
		set_texorient = TRUE;
            }
#undef PRIM_PTR
            break;

          case V3DMP_TYPE_TEXTURE_ORIENT_YZ:
#define PRIM_PTR        mp_texture_yz
            PRIM_PTR = p;
            if((value_num == 0) &&
               (VMA_PRIMITIVES_MAX_VALUES >= 4)
            )
            {
                old_texorient.i = PRIM_PTR->y;
                old_texorient.j = PRIM_PTR->z;
                old_texorient.di = PRIM_PTR->dy;
                old_texorient.dj = PRIM_PTR->dz;

                w = editor->values_text[0];   
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    new_texorient.i = atof(text_ptr);
                w = editor->values_text[1];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)   
                    new_texorient.j = atof(text_ptr);
                w = editor->values_text[2];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    new_texorient.di = atof(text_ptr);
                w = editor->values_text[3];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    new_texorient.dj = atof(text_ptr);

                PRIM_PTR->y = new_texorient.i;
                PRIM_PTR->z = new_texorient.j;
                PRIM_PTR->dy = new_texorient.di;
                PRIM_PTR->dz = new_texorient.dj;
                sprintf(text, "(y,z,dy,dz): %.2f, %.2f, %.2f, %.2f",
                    PRIM_PTR->y,
                    PRIM_PTR->z,
                    PRIM_PTR->dy,
                    PRIM_PTR->dz
                );
                gtk_clist_set_text(
                    GTK_CLIST(clist),
                    value_num, 0,
                    text
                );
                set_texorient = TRUE;
            }
#undef PRIM_PTR
            break;

          case V3DMP_TYPE_TEXTURE_ORIENT_XZ:
#define PRIM_PTR	mp_texture_xz
            PRIM_PTR = p;
            if((value_num == 0) &&
               (VMA_PRIMITIVES_MAX_VALUES >= 4)
            )
            {
                old_texorient.i = PRIM_PTR->x;
                old_texorient.j = PRIM_PTR->z;
                old_texorient.di = PRIM_PTR->dx;
                old_texorient.dj = PRIM_PTR->dz;

                w = editor->values_text[0];   
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    new_texorient.i = atof(text_ptr);
                w = editor->values_text[1];
                DO_GET_TEXT_PTR 
                if(text_ptr != NULL)
                    new_texorient.j = atof(text_ptr);
                w = editor->values_text[2];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL) 
                    new_texorient.di = atof(text_ptr);
                w = editor->values_text[3];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL) 
                    new_texorient.dj = atof(text_ptr);

                PRIM_PTR->x = new_texorient.i;
                PRIM_PTR->z = new_texorient.j;
                PRIM_PTR->dx = new_texorient.di;
                PRIM_PTR->dz = new_texorient.dj;
                sprintf(text, "(x,z,dx,dz): %.2f, %.2f, %.2f, %.2f",
                    PRIM_PTR->x,
                    PRIM_PTR->z,
                    PRIM_PTR->dx,
                    PRIM_PTR->dz
                );
                gtk_clist_set_text(
                    GTK_CLIST(clist),
                    value_num, 0,
                    text
                );
                set_texorient = TRUE;
            }
#undef PRIM_PTR
            break;

	  case V3DMP_TYPE_HEIGHTFIELD_LOAD:
	    mp_heightfield_load = p;

	    old_heightfield_values.x_length = mp_heightfield_load->x_length;
            old_heightfield_values.y_length = mp_heightfield_load->y_length;
            old_heightfield_values.z_length = mp_heightfield_load->z_length;
            old_heightfield_values.x = mp_heightfield_load->x;
            old_heightfield_values.y = mp_heightfield_load->y;
            old_heightfield_values.z = mp_heightfield_load->z;
            old_heightfield_values.heading = mp_heightfield_load->heading;
            old_heightfield_values.pitch = mp_heightfield_load->pitch;
            old_heightfield_values.bank = mp_heightfield_load->bank;

            if((value_num == 0) &&
               (VMA_PRIMITIVES_MAX_VALUES >= 1)
	    )
            {
		/* Selected path value. */
                w = editor->values_text[0];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                {
                    /* Set new path. */
                    free(mp_heightfield_load->path);
                    mp_heightfield_load->path = strdup(text_ptr);

		    /* Rerealize heightfield data. */
		    EditorHFPrimitiveRealize(
			editor, mp_heightfield_load, TRUE
		    );

		    /* Update item in values list. */
		    *text = '\0';
		    strncat(
			text,
			(mp_heightfield_load->path == NULL) ?
			    VAL_NULL_STR : mp_heightfield_load->path,
			80
		    );
                    gtk_clist_set_text(
                        GTK_CLIST(clist),
                        value_num, 0,
                        text
                    );

		    /* Update primitive item listing on primitives list. */
                    EditorListPrimitivesSetHeightField(
                        primitives_clist, pn,
                        p, TRUE
                    );
                }

		set_heightfield_load = TRUE;
	    }
	    else if((value_num == 1) &&
                    (VMA_PRIMITIVES_MAX_VALUES >= 3)
	    )
            {
                /* Selected lengths value. */
                w = editor->values_text[0];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_heightfield_load->x_length = atof(text_ptr);
                w = editor->values_text[1];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_heightfield_load->y_length = atof(text_ptr);
                w = editor->values_text[2];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_heightfield_load->z_length = atof(text_ptr);

                /* Rerealize heightfield data. */
                EditorHFPrimitiveRealize(
                    editor, mp_heightfield_load, TRUE
                );

                /* Update item in values list. */
                sprintf(text, "(xl,yl,zl): %.2f, %.2f, %.2f",
                    mp_heightfield_load->x_length,
                    mp_heightfield_load->y_length,
                    mp_heightfield_load->z_length
                );
                gtk_clist_set_text(
                    GTK_CLIST(clist),
                    value_num, 0,
                    text
                );

                set_heightfield_values = TRUE;
            }
            else if((value_num == 2) &&
                    (VMA_PRIMITIVES_MAX_VALUES >= 3)
            )
            {
                /* Selected positions value. */
                w = editor->values_text[0];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_heightfield_load->x = atof(text_ptr);
                w = editor->values_text[1];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_heightfield_load->y = atof(text_ptr);
                w = editor->values_text[2];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_heightfield_load->z = atof(text_ptr);

                /* Update item in values list. */
                sprintf(text, "(x,y,z): %.2f, %.2f, %.2f",
                    mp_heightfield_load->x,
                    mp_heightfield_load->y,
                    mp_heightfield_load->z
                );
                gtk_clist_set_text(
                    GTK_CLIST(clist),
                    value_num, 0,
                    text
                );

                set_heightfield_values = TRUE;
            }
            else if((value_num == 3) &&
                    (VMA_PRIMITIVES_MAX_VALUES >= 3)
            )
            {
                /* Selected rotations value. */
                w = editor->values_text[0];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_heightfield_load->heading =
			DEGTORAD(atof(text_ptr));
                w = editor->values_text[1];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_heightfield_load->pitch =
			DEGTORAD(atof(text_ptr));
                w = editor->values_text[2];
                DO_GET_TEXT_PTR
                if(text_ptr != NULL)
                    mp_heightfield_load->bank =
			DEGTORAD(atof(text_ptr));

                /* Update item in values list. */
                sprintf(text, "(h,p,b): %.2f, %.2f, %.2f",
                    RADTODEG(mp_heightfield_load->heading),
                    RADTODEG(mp_heightfield_load->pitch),
                    RADTODEG(mp_heightfield_load->bank)
                );
                gtk_clist_set_text(
                    GTK_CLIST(clist),
                    value_num, 0,
                    text
                );    

                set_heightfield_values = TRUE;
            }
	    break;

	}

#undef DO_SET_VALUES_ITEM_VERTEX
#undef DO_GET_TEXT_PTR

        /* Record undo? */
        if(record_undo)
        {
            if(set_normal)
            {
                vma_undo_set_normal_struct *u = (vma_undo_set_normal_struct *)VMAUndoNew(
                    VMA_UNDO_TYPE_SET_NORMAL, "Set Normal"
                );  
		if(u != NULL)
		{
                    u->editor = editor;
                    u->model_num = model_num;
                    u->primitive_num = pn;
		    u->vertex_num = value_num;
		    u->n.x = old_v.x;
                    u->n.y = old_v.y;
                    u->n.z = old_v.z;
                    EditorUndoRecord(editor, u);
		}
            }
            else if(set_texcoord)
            {
                vma_undo_set_texcoord_struct *u = (vma_undo_set_texcoord_struct *)VMAUndoNew(
                    VMA_UNDO_TYPE_SET_TEXCOORD, "Set TexCoord"
                );
                if(u != NULL)
                {
		    u->editor = editor;
                    u->model_num = model_num;
                    u->primitive_num = pn;
		    u->vertex_num = value_num;
                    u->tc.x = old_v.x;
                    u->tc.y = old_v.y;
                    u->tc.z = old_v.z;
                    EditorUndoRecord(editor, u);
		}
            }
            else if(set_vertex)
            {
                vma_undo_set_vertex_struct *u = (vma_undo_set_vertex_struct *)VMAUndoNew(
                    VMA_UNDO_TYPE_SET_VERTEX, "Set Vertex"
                );
                if(u != NULL)
                {
                    u->editor = editor;
                    u->model_num = model_num;   
                    u->primitive_num = pn;
                    u->vertex_num = value_num;
                    u->v.x = old_v.x;
                    u->v.y = old_v.y;
                    u->v.z = old_v.z;
                    EditorUndoRecord(editor, u);
		}
            }
	    else if(set_texorient)
            {
                vma_undo_set_texorient_struct *u = (vma_undo_set_texorient_struct *)VMAUndoNew(
                    VMA_UNDO_TYPE_SET_TEXORIENT, "Set TexOrient"
                );
                if(u != NULL)
                {
                    u->editor = editor;
                    u->model_num = model_num;
                    u->primitive_num = pn;
		    u->i = old_texorient.i;
                    u->j = old_texorient.j;
                    u->di = old_texorient.di;
                    u->dj = old_texorient.dj;
                    EditorUndoRecord(editor, u);
                }
            }
	    else if(set_heightfield_values)
	    {
                vma_undo_set_heightfield_struct *u = (vma_undo_set_heightfield_struct *)VMAUndoNew(
                    VMA_UNDO_TYPE_SET_HEIGHTFIELD, "Set HeightField"
                );  
                if(u != NULL)
                {
                    u->editor = editor;
                    u->model_num = model_num;   
                    u->primitive_num = pn;
		    u->x_length = old_heightfield_values.x_length;
                    u->y_length = old_heightfield_values.y_length;
                    u->z_length = old_heightfield_values.z_length;
                    u->x = old_heightfield_values.x;
                    u->y = old_heightfield_values.y;
                    u->z = old_heightfield_values.z;
                    u->heading = old_heightfield_values.heading;
                    u->pitch = old_heightfield_values.pitch;
                    u->bank = old_heightfield_values.bank;
                    EditorUndoRecord(editor, u);
                }
	    }
/* Add more undo support here for different types of primitves. */




        }

        return(0);
}

/*
 *	Fetches selected value list item that of the selected
 *	primitive to the values text prompts.
 *
 *	If the editor's selected value item is -1 then the value text
 *	prompts will be unmapped.
 */
int EditorListValuePromptFetch(
        ma_editor_struct *editor, void *p
)
{
	int i, selected_value;
	GtkWidget *w;

	mp_vertex_struct *v, *n, *tc;

	mp_translate_struct *mp_translate;
	mp_rotate_struct *mp_rotate;
	mp_point_struct *mp_point;
	mp_line_struct *mp_line;
	mp_line_strip_struct *mp_line_strip;
	mp_line_loop_struct *mp_line_loop;
	mp_triangle_struct *mp_triangle;
	mp_triangle_strip_struct *mp_triangle_strip;
	mp_triangle_fan_struct *mp_triangle_fan;
	mp_quad_struct *mp_quad;
	mp_quad_strip_struct *mp_quad_strip;
        mp_polygon_struct *mp_polygon;
	mp_color_struct *mp_color;
	mp_texture_select_struct *mp_texture_select;
	mp_texture_orient_xy_struct *mp_texture_xy;
	mp_texture_orient_yz_struct *mp_texture_yz;
	mp_texture_orient_xz_struct *mp_texture_xz;
	mp_heightfield_load_struct *mp_heightfield_load;

	char *label_text_ptr;
	char text[256];
	char fmt_str[256];


	if((editor == NULL) || (p == NULL))
	    return(-1);

/* Unmaps the rest of the value prompts on the editor starting at i. */
#define DO_UNMAP_REST_VALUE_PROMPTS	\
{ \
 while(i < VMA_PRIMITIVES_MAX_VALUES) \
 { \
  w = editor->values_label[i]; \
  if(w != NULL) \
   gtk_widget_hide(w); \
  w = editor->values_text[i]; \
  if(w != NULL) \
   gtk_widget_hide(w); \
  w = editor->values_browse_btn[i]; \
  if(w != NULL) \
   gtk_widget_hide(w); \
  i++; \
 } \
}

/* Unmaps all value prompts on the editor. */
#define DO_UNMAP_ALL_VALUE_PROMPTS	\
{ \
 for(i = 0; i < VMA_PRIMITIVES_MAX_VALUES; i++) \
 { \
  w = editor->values_label[i]; \
  if(w != NULL) \
   gtk_widget_hide(w); \
  w = editor->values_text[i]; \
  if(w != NULL) \
   gtk_widget_hide(w); \
  w = editor->values_browse_btn[i]; \
  if(w != NULL) \
   gtk_widget_hide(w); \
 } \
}

/* Sets label widget w to the label value of label_text_ptr. */
#define DO_SET_LABEL_AND_MAP	\
{ \
 if((w != NULL) && \
    (label_text_ptr != NULL) \
 ) \
 { \
  gtk_label_set_text(GTK_LABEL(w), label_text_ptr); \
  gtk_widget_show(w); \
 } \
}

/* Sets entry widget w to the value of text text. */
#define DO_SET_TEXT_AND_MAP    \
{ \
 if(w != NULL) \
 { \
  gtk_entry_set_text(GTK_ENTRY(w), text); \
  gtk_widget_show(w); \
 } \
}

/* Sets up the value prompts on the editor for the given vertex structure
 * v if it is not NULL. Then unmaps the remaining value prompts if any.
 */
#define DO_SET_VALUE_PROMPTS_VERTEX	\
{ \
 /* Vertex pointer valid? */ \
 if((v != NULL) && (VMA_PRIMITIVES_MAX_VALUES >= 3)) \
 { \
  w = editor->values_label[0]; \
  label_text_ptr = "X:"; \
  DO_SET_LABEL_AND_MAP \
  w = editor->values_label[1]; \
  label_text_ptr = "Y:"; \
  DO_SET_LABEL_AND_MAP \
  w = editor->values_label[2]; \
  label_text_ptr = "Z:"; \
  DO_SET_LABEL_AND_MAP \
 \
  sprintf(fmt_str, "%%.%if", editor->vertex_decimals); \
 \
  w = editor->values_text[0]; \
  sprintf(text, fmt_str, v->x); \
  DO_SET_TEXT_AND_MAP \
  w = editor->values_text[1]; \
  sprintf(text, fmt_str, v->y); \
  DO_SET_TEXT_AND_MAP \
  w = editor->values_text[2]; \
  sprintf(text, fmt_str, v->z); \
  DO_SET_TEXT_AND_MAP \
 \
  /* Normal pointer valid? */ \
  if((n != NULL) && (VMA_PRIMITIVES_MAX_VALUES >= 6)) \
  { \
   w = editor->values_label[3]; \
   label_text_ptr = "NX:"; \
   DO_SET_LABEL_AND_MAP \
   w = editor->values_label[4]; \
   label_text_ptr = "NY:"; \
   DO_SET_LABEL_AND_MAP \
   w = editor->values_label[5]; \
   label_text_ptr = "NZ:"; \
   DO_SET_LABEL_AND_MAP \
 \
   /* Use vertex decimal precision. */ \
   sprintf(fmt_str, "%%.%if", editor->vertex_decimals); \
 \
   w = editor->values_text[3]; \
   sprintf(text, fmt_str, n->x); \
   DO_SET_TEXT_AND_MAP \
   w = editor->values_text[4]; \
   sprintf(text, fmt_str, n->y); \
   DO_SET_TEXT_AND_MAP \
   w = editor->values_text[5]; \
   sprintf(text, fmt_str, n->z); \
   DO_SET_TEXT_AND_MAP \
  } \
 \
  /* Texcoord pointer valid? */ \
  if((tc != NULL) && (VMA_PRIMITIVES_MAX_VALUES >= 9)) \
  { \
   w = editor->values_label[6]; \
   label_text_ptr = "TS:"; \
   DO_SET_LABEL_AND_MAP \
   w = editor->values_label[7]; \
   label_text_ptr = "TT:"; \
   DO_SET_LABEL_AND_MAP \
   w = editor->values_label[8]; \
   label_text_ptr = "TU:"; \
   DO_SET_LABEL_AND_MAP \
 \
   /* Use vertex decimal precision. */ \
   sprintf(fmt_str, "%%.%if", editor->vertex_decimals); \
 \
   w = editor->values_text[6]; \
   sprintf(text, fmt_str, tc->x); \
   DO_SET_TEXT_AND_MAP \
   w = editor->values_text[7]; \
   sprintf(text, fmt_str, tc->y); \
   DO_SET_TEXT_AND_MAP \
   w = editor->values_text[8]; \
   sprintf(text, fmt_str, tc->z); \
   DO_SET_TEXT_AND_MAP \
  } \
 \
  i = 9; \
  DO_UNMAP_REST_VALUE_PROMPTS \
 } \
 else \
 { \
  DO_UNMAP_ALL_VALUE_PROMPTS \
 } \
}

	/* Get editor's selected value item. */
	selected_value = EditorGetSelected(
            editor->selected_value, editor->total_selected_values,
            0
        );
	if(selected_value < 0)
	{
	    DO_UNMAP_ALL_VALUE_PROMPTS
	}
	else
	{
	    switch(*(int *)p)
	    {
	      case V3DMP_TYPE_TRANSLATE:
		mp_translate = p;
		if((selected_value == 0) &&
                   (VMA_PRIMITIVES_MAX_VALUES >= 3)
		)
		{
                    w = editor->values_label[0];
		    label_text_ptr = "X:";
		    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[1];
                    label_text_ptr = "Y:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[2];
                    label_text_ptr = "Z:";
                    DO_SET_LABEL_AND_MAP

		    sprintf(fmt_str, "%%.%if", editor->vertex_decimals);

                    w = editor->values_text[0];
		    sprintf(text, fmt_str, mp_translate->x);
		    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[1];
                    sprintf(text, fmt_str, mp_translate->y);
		    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[2];
                    sprintf(text, fmt_str, mp_translate->z);
		    DO_SET_TEXT_AND_MAP

		    i = 3;
		    DO_UNMAP_REST_VALUE_PROMPTS
		}
		else
		{
		    DO_UNMAP_ALL_VALUE_PROMPTS
		}
		break;

              case V3DMP_TYPE_ROTATE:
                mp_rotate = p;
                if((selected_value == 0) &&
                   (VMA_PRIMITIVES_MAX_VALUES >= 3)
                )
                {
                    w = editor->values_label[0];
                    label_text_ptr = "H:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[1];
                    label_text_ptr = "P:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[2];
                    label_text_ptr = "B:";
                    DO_SET_LABEL_AND_MAP

                    sprintf(fmt_str, "%%.%if", editor->vertex_angle_decimals);

                    w = editor->values_text[0];
                    sprintf(text, fmt_str, RADTODEG(mp_rotate->heading));
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[1];
                    sprintf(text, fmt_str, RADTODEG(mp_rotate->pitch));
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[2];
                    sprintf(text, fmt_str, RADTODEG(mp_rotate->bank));
                    DO_SET_TEXT_AND_MAP

                    i = 3;
                    DO_UNMAP_REST_VALUE_PROMPTS
		}
                else
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
		break;

             case V3DMP_TYPE_POINT:
#define PTR_NAME	mp_point
                PTR_NAME = (mp_point_struct *)p;
		i = selected_value;
		if((i >= 0) && (i < V3DMP_POINT_NVERTEX))
		{
		    v = &PTR_NAME->v[i];
		    n = &PTR_NAME->n[i];
		    tc = &PTR_NAME->tc[i];
                    DO_SET_VALUE_PROMPTS_VERTEX
                }
                else        
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
                break;

              case V3DMP_TYPE_LINE:
#undef PTR_NAME
#define PTR_NAME	mp_line
                PTR_NAME = (mp_line_struct *)p;
                i = selected_value;
                if((i >= 0) && (i < V3DMP_LINE_NVERTEX))
                {
                    v = &PTR_NAME->v[i];
                    n = &PTR_NAME->n[i];
                    tc = &PTR_NAME->tc[i];
                    DO_SET_VALUE_PROMPTS_VERTEX
                }
                else
                {   
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
                break;

              case V3DMP_TYPE_LINE_STRIP:
#undef PTR_NAME
#define PTR_NAME        mp_line_strip
                PTR_NAME = (mp_line_strip_struct *)p;
                i = selected_value;
                if((i >= 0) && (i < PTR_NAME->total))
                {
                    v = PTR_NAME->v[i];
                    n = PTR_NAME->n[i];  
                    tc = PTR_NAME->tc[i];
                    DO_SET_VALUE_PROMPTS_VERTEX
                }   
                else
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
                break;

              case V3DMP_TYPE_LINE_LOOP:
#undef PTR_NAME
#define PTR_NAME	mp_line_loop
                PTR_NAME = (mp_line_loop_struct *)p;
                i = selected_value;
                if((i >= 0) && (i < PTR_NAME->total))
                {
                    v = PTR_NAME->v[i];
                    n = PTR_NAME->n[i];
                    tc = PTR_NAME->tc[i];
                    DO_SET_VALUE_PROMPTS_VERTEX
                }
                else
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
                break;

             case V3DMP_TYPE_TRIANGLE:
#undef PTR_NAME
#define PTR_NAME	mp_triangle
                PTR_NAME = (mp_triangle_struct *)p;
                i = selected_value;
                if((i >= 0) && (i < V3DMP_TRIANGLE_NVERTEX))
                {
                    v = &PTR_NAME->v[i];
                    n = &PTR_NAME->n[i];
                    tc = &PTR_NAME->tc[i];
                    DO_SET_VALUE_PROMPTS_VERTEX
                }
                else
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
                break;

             case V3DMP_TYPE_TRIANGLE_STRIP:
#undef PTR_NAME
#define PTR_NAME	mp_triangle_strip
                PTR_NAME = (mp_triangle_strip_struct *)p;
                i = selected_value;
                if((i >= 0) && (i < PTR_NAME->total))
                {
                    v = PTR_NAME->v[i];   
                    n = PTR_NAME->n[i];
                    tc = PTR_NAME->tc[i];
                    DO_SET_VALUE_PROMPTS_VERTEX   
                }
                else
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
                break;

             case V3DMP_TYPE_TRIANGLE_FAN:
#undef PTR_NAME
#define PTR_NAME	mp_triangle_fan
                PTR_NAME = (mp_triangle_fan_struct *)p;
                i = selected_value;
                if((i >= 0) && (i < PTR_NAME->total))
                {
                    v = PTR_NAME->v[i];
                    n = PTR_NAME->n[i];
                    tc = PTR_NAME->tc[i];
                    DO_SET_VALUE_PROMPTS_VERTEX
                }
                else
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
		break;

             case V3DMP_TYPE_QUAD:
#undef PTR_NAME
#define PTR_NAME	mp_quad
                PTR_NAME = (mp_quad_struct *)p;
                i = selected_value;
                if((i >= 0) && (i < V3DMP_QUAD_NVERTEX))
                {
                    v = &PTR_NAME->v[i];   
                    n = &PTR_NAME->n[i];
                    tc = &PTR_NAME->tc[i];
                    DO_SET_VALUE_PROMPTS_VERTEX
                }
                else    
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
                break;

             case V3DMP_TYPE_QUAD_STRIP:
#undef PTR_NAME
#define PTR_NAME	mp_quad_strip
                PTR_NAME = (mp_quad_strip_struct *)p;
                i = selected_value;
                if((i >= 0) && (i < PTR_NAME->total))
                {
                    v = PTR_NAME->v[i];
                    n = PTR_NAME->n[i];
                    tc = PTR_NAME->tc[i];
                    DO_SET_VALUE_PROMPTS_VERTEX
                }
                else
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
		break;

             case V3DMP_TYPE_POLYGON:
#undef PTR_NAME
#define PTR_NAME	mp_polygon
                PTR_NAME = (mp_polygon_struct *)p;
                i = selected_value;
                if((i >= 0) && (i < PTR_NAME->total))
                {
                    v = PTR_NAME->v[i];
                    n = PTR_NAME->n[i];
                    tc = PTR_NAME->tc[i];
                    DO_SET_VALUE_PROMPTS_VERTEX
                }
                else
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
                break;
#undef PTR_NAME

             case V3DMP_TYPE_COLOR:
                mp_color = p;
                if((selected_value == 0) &&
                   (VMA_PRIMITIVES_MAX_VALUES >= 4)
                )
                {
                    w = editor->values_label[0];
                    label_text_ptr = "A:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[1];
                    label_text_ptr = "R:";
                    DO_SET_LABEL_AND_MAP  
                    w = editor->values_label[2];
                    label_text_ptr = "G:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[3];
                    label_text_ptr = "B:";
                    DO_SET_LABEL_AND_MAP

                    sprintf(fmt_str, "%%.2f");

                    w = editor->values_text[0];
                    sprintf(text, fmt_str, mp_color->a);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[1];
                    sprintf(text, fmt_str, mp_color->r);
                    DO_SET_TEXT_AND_MAP
		    w = editor->values_text[2];
                    sprintf(text, fmt_str, mp_color->g);
                    DO_SET_TEXT_AND_MAP
		    w = editor->values_text[3];
                    sprintf(text, fmt_str, mp_color->b);
                    DO_SET_TEXT_AND_MAP

                    w = editor->values_browse_btn[0];
                    if(w != NULL)
                        gtk_widget_show(w);

                    i = 4;
                    DO_UNMAP_REST_VALUE_PROMPTS
                }
                else if((selected_value == 1) &&
                        (VMA_PRIMITIVES_MAX_VALUES >= 1)
                )
                {
		    /* Ambient. */
                    w = editor->values_label[0];
                    label_text_ptr = "Ambient:";
                    DO_SET_LABEL_AND_MAP

                    sprintf(fmt_str, "%%.2f");

                    w = editor->values_text[0];
                    sprintf(text, fmt_str, mp_color->ambient);
                    DO_SET_TEXT_AND_MAP

                    w = editor->values_browse_btn[0];
                    if(w != NULL)
                        gtk_widget_show(w);

                    i = 1;
                    DO_UNMAP_REST_VALUE_PROMPTS
		}
                else if((selected_value == 2) &&
                        (VMA_PRIMITIVES_MAX_VALUES >= 1)
                )
                {
                    /* Diffuse. */
                    w = editor->values_label[0];
                    label_text_ptr = "Diffuse:";
                    DO_SET_LABEL_AND_MAP

                    sprintf(fmt_str, "%%.2f");

                    w = editor->values_text[0];
                    sprintf(text, fmt_str, mp_color->diffuse);
                    DO_SET_TEXT_AND_MAP

                    w = editor->values_browse_btn[0];
                    if(w != NULL)
                        gtk_widget_show(w);

                    i = 1;
                    DO_UNMAP_REST_VALUE_PROMPTS
                }
                else if((selected_value == 3) &&
                        (VMA_PRIMITIVES_MAX_VALUES >= 1)
                )
                {
                    /* Specular. */
                    w = editor->values_label[0];
                    label_text_ptr = "Specular:";
                    DO_SET_LABEL_AND_MAP

                    sprintf(fmt_str, "%%.2f");

                    w = editor->values_text[0];
                    sprintf(text, fmt_str, mp_color->specular);
                    DO_SET_TEXT_AND_MAP

                    w = editor->values_browse_btn[0];
                    if(w != NULL)
                        gtk_widget_show(w);

                    i = 1;
                    DO_UNMAP_REST_VALUE_PROMPTS
                }
                else if((selected_value == 4) &&
                        (VMA_PRIMITIVES_MAX_VALUES >= 1)
                )
                {
                    /* Shininess. */
                    w = editor->values_label[0];
                    label_text_ptr = "Shininess:";
                    DO_SET_LABEL_AND_MAP

                    sprintf(fmt_str, "%%.2f");

                    w = editor->values_text[0];
                    sprintf(text, fmt_str, mp_color->shininess);
                    DO_SET_TEXT_AND_MAP

                    w = editor->values_browse_btn[0];
                    if(w != NULL)
                        gtk_widget_show(w);

                    i = 1;
                    DO_UNMAP_REST_VALUE_PROMPTS
                }
                else if((selected_value == 5) &&
                        (VMA_PRIMITIVES_MAX_VALUES >= 1)
                )
                {
                    /* Emission. */
                    w = editor->values_label[0];
                    label_text_ptr = "Emission:";
                    DO_SET_LABEL_AND_MAP

                    sprintf(fmt_str, "%%.2f");

                    w = editor->values_text[0];
                    sprintf(text, fmt_str, mp_color->emission);
                    DO_SET_TEXT_AND_MAP

                    w = editor->values_browse_btn[0];
                    if(w != NULL)
                        gtk_widget_show(w);

                    i = 1;
                    DO_UNMAP_REST_VALUE_PROMPTS
                }
		else
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
		break;

            case V3DMP_TYPE_TEXTURE_SELECT:
                mp_texture_select = p;
                if((selected_value == 0) &&   
                   (VMA_PRIMITIVES_MAX_VALUES >= 1)
                )
                {
                    w = editor->values_label[0];
                    label_text_ptr = "Name:";
                    DO_SET_LABEL_AND_MAP

                    w = editor->values_text[0];
		    if(w != NULL)
		    {
			gtk_entry_set_text(
			    GTK_ENTRY(w),
			    ((mp_texture_select->name == NULL) ?
				VAL_NULL_STR : mp_texture_select->name
			    )
			);
			gtk_widget_show(w);
		    }

		    w = editor->values_browse_btn[0];
		    if(w != NULL)
			gtk_widget_show(w);

                    i = 1;
                    DO_UNMAP_REST_VALUE_PROMPTS 
                }
                else
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
                break;

              case V3DMP_TYPE_TEXTURE_ORIENT_XY:
                mp_texture_xy = p;
                if((selected_value == 0) &&
                   (VMA_PRIMITIVES_MAX_VALUES >= 4)
                )
                {
                    w = editor->values_label[0];
                    label_text_ptr = "X:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[1];
                    label_text_ptr = "Y:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[2];
                    label_text_ptr = "DX:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[3];
                    label_text_ptr = "DY:";
                    DO_SET_LABEL_AND_MAP

                    sprintf(fmt_str, "%%.%if", editor->vertex_decimals);

                    w = editor->values_text[0];
                    sprintf(text, fmt_str, mp_texture_xy->x);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[1];
                    sprintf(text, fmt_str, mp_texture_xy->y);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[2];
                    sprintf(text, fmt_str, mp_texture_xy->dx);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[3];
                    sprintf(text, fmt_str, mp_texture_xy->dy);
                    DO_SET_TEXT_AND_MAP

                    i = 4;
                    DO_UNMAP_REST_VALUE_PROMPTS
                }
                else
                {   
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
		break;

             case V3DMP_TYPE_TEXTURE_ORIENT_YZ:
                mp_texture_yz = p;
                if((selected_value == 0) &&
                   (VMA_PRIMITIVES_MAX_VALUES >= 4)
                )
                {
                    w = editor->values_label[0];
                    label_text_ptr = "Y:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[1];
                    label_text_ptr = "Z:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[2];
                    label_text_ptr = "DY:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[3];
                    label_text_ptr = "DZ:";
                    DO_SET_LABEL_AND_MAP

                    sprintf(fmt_str, "%%.%if", editor->vertex_decimals);

                    w = editor->values_text[0];
                    sprintf(text, fmt_str, mp_texture_yz->y);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[1];
                    sprintf(text, fmt_str, mp_texture_yz->z);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[2];
                    sprintf(text, fmt_str, mp_texture_yz->dy);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[3];
                    sprintf(text, fmt_str, mp_texture_yz->dz);
                    DO_SET_TEXT_AND_MAP

                    i = 4;
                    DO_UNMAP_REST_VALUE_PROMPTS
                }
                else
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
                break;

             case V3DMP_TYPE_TEXTURE_ORIENT_XZ:
                mp_texture_xz = p;
                if((selected_value == 0) &&
                   (VMA_PRIMITIVES_MAX_VALUES >= 4)
                )
                {
                    w = editor->values_label[0];
                    label_text_ptr = "Y:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[1];
                    label_text_ptr = "Z:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[2];
                    label_text_ptr = "DY:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[3];
                    label_text_ptr = "DZ:";
                    DO_SET_LABEL_AND_MAP

                    sprintf(fmt_str, "%%.%if", editor->vertex_decimals);

                    w = editor->values_text[0];
                    sprintf(text, fmt_str, mp_texture_xz->x);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[1];
                    sprintf(text, fmt_str, mp_texture_xz->z);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[2];
                    sprintf(text, fmt_str, mp_texture_xz->dx);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[3];
                    sprintf(text, fmt_str, mp_texture_xz->dz);
                    DO_SET_TEXT_AND_MAP

                    i = 4;
                    DO_UNMAP_REST_VALUE_PROMPTS
                }
                else
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
                }
                break;

            case V3DMP_TYPE_HEIGHTFIELD_LOAD:
                mp_heightfield_load = p;
                if((selected_value == 0) &&
                   (VMA_PRIMITIVES_MAX_VALUES >= 1)
                )
                {
		    /* Path. */
                    w = editor->values_label[0];
                    label_text_ptr = "Path:";
                    DO_SET_LABEL_AND_MAP

                    w = editor->values_text[0];
                    if(w != NULL)
                    {
                        gtk_entry_set_text(
                            GTK_ENTRY(w),
                            ((mp_heightfield_load->path == NULL) ?
                                VAL_NULL_STR : mp_heightfield_load->path
                            )
                        );
                        gtk_widget_show(w);
                    }

                    w = editor->values_browse_btn[0];
                    if(w != NULL)
                        gtk_widget_show(w);

                    i = 1;
                    DO_UNMAP_REST_VALUE_PROMPTS
                }
		else if((selected_value == 1) &&
                        (VMA_PRIMITIVES_MAX_VALUES >= 3)
                )
		{
		    /* Lengths. */
                    w = editor->values_label[0];
                    label_text_ptr = "X Length:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[1];
                    label_text_ptr = "Y Length:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[2];
                    label_text_ptr = "Z Length:";
                    DO_SET_LABEL_AND_MAP

                    sprintf(fmt_str, "%%.%if", editor->vertex_decimals);

                    w = editor->values_text[0];
                    sprintf(text, fmt_str, mp_heightfield_load->x_length);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[1];
                    sprintf(text, fmt_str, mp_heightfield_load->y_length);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[2];
                    sprintf(text, fmt_str, mp_heightfield_load->z_length);
                    DO_SET_TEXT_AND_MAP

                    w = editor->values_browse_btn[0];
                    if(w != NULL)
                        gtk_widget_show(w);

                    i = 3;
                    DO_UNMAP_REST_VALUE_PROMPTS
		}
                else if((selected_value == 2) &&
                        (VMA_PRIMITIVES_MAX_VALUES >= 1)
                )
                {           
                    /* Position. */
                    w = editor->values_label[0];
                    label_text_ptr = "X:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[1];
                    label_text_ptr = "Y:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[2];
                    label_text_ptr = "Z:";
                    DO_SET_LABEL_AND_MAP

                    sprintf(fmt_str, "%%.%if", editor->vertex_decimals);

                    w = editor->values_text[0];
                    sprintf(text, fmt_str, mp_heightfield_load->x);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[1];
                    sprintf(text, fmt_str, mp_heightfield_load->y);
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[2];
                    sprintf(text, fmt_str, mp_heightfield_load->z);
                    DO_SET_TEXT_AND_MAP

                    w = editor->values_browse_btn[0];
                    if(w != NULL)
                        gtk_widget_show(w);

                    i = 3;
                    DO_UNMAP_REST_VALUE_PROMPTS
                }
                else if((selected_value == 3) &&
                        (VMA_PRIMITIVES_MAX_VALUES >= 1)
                )
                {
                    /* Rotations. */
                    w = editor->values_label[0];
                    label_text_ptr = "H:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[1];
                    label_text_ptr = "P:";
                    DO_SET_LABEL_AND_MAP
                    w = editor->values_label[2];
                    label_text_ptr = "B:";
                    DO_SET_LABEL_AND_MAP

                    sprintf(fmt_str, "%%.%if", editor->vertex_angle_decimals);

                    w = editor->values_text[0];
                    sprintf(text, fmt_str, 
			RADTODEG(mp_heightfield_load->heading));
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[1];
                    sprintf(text, fmt_str,
			RADTODEG(mp_heightfield_load->pitch));
                    DO_SET_TEXT_AND_MAP
                    w = editor->values_text[2];
                    sprintf(text, fmt_str,
			RADTODEG(mp_heightfield_load->bank));
                    DO_SET_TEXT_AND_MAP

                    w = editor->values_browse_btn[0];
                    if(w != NULL)
                        gtk_widget_show(w);

                    i = 3;
                    DO_UNMAP_REST_VALUE_PROMPTS
                }
                else
                {
                    DO_UNMAP_ALL_VALUE_PROMPTS
		}
		break;

	      default:
		DO_UNMAP_ALL_VALUE_PROMPTS
		break;
	    }

	}

#undef DO_SET_VALUE_PROMPTS_VERTEX
#undef DO_UNMAP_ALL_VALUE_PROMPTS
#undef DO_UNMAP_REST_VALUE_PROMPTS
#undef DO_SET_LABEL_AND_MAP
#undef DO_SET_TEXT_AND_MAP

        return(0);
}

/*
 *	Deletes all items from the models GUI list.
 */
void EditorListDeleteModelsG(ma_editor_struct *editor)
{
        GtkWidget *w;

        if(editor == NULL)
            return;

        w = editor->models_list;
        if(w != NULL)
            gtk_clist_clear(GTK_CLIST(w));

        editor->selected_model_item = -1;

        return;
}

/*
 *	Deletes all items from the primitives GUI list.
 */
void EditorListDeletePrimitivesG(ma_editor_struct *editor)
{
        GtkWidget *w;

        if(editor == NULL)
            return;

        w = editor->primitives_list;
        if(w != NULL)
        {
            gtk_clist_clear(GTK_CLIST(w));
        }

	EditorUnselectAll(
	    &editor->selected_primitive,
	    &editor->total_selected_primitives
	);

        return;
}

/*
 *	Deletes all items from the values GUI list.
 */
void EditorListDeleteValuesG(ma_editor_struct *editor)
{
	gint i;
        GtkWidget *w;  
  
        if(editor == NULL)
            return;
 
	/* Clear values clist. */
        w = editor->values_list;
	if(w != NULL)
	{
	    gtk_clist_freeze(GTK_CLIST(w));

	    gtk_clist_clear(GTK_CLIST(w));

            gtk_clist_set_column_title(
                GTK_CLIST(w), 0, ""
            );
            gtk_clist_set_column_title(
                GTK_CLIST(w), 1, ""
            );
            gtk_clist_set_column_title(
                GTK_CLIST(w), 2, ""
            );

	    gtk_clist_thaw(GTK_CLIST(w));
	}

	/* Hide all value prompts and related widgets. */
	for(i = 0; i < VMA_PRIMITIVES_MAX_VALUES; i++)
	{
	    w = editor->values_label[i];
	    if(w != NULL)
		gtk_widget_hide(w);
	    w = editor->values_text[i];
	    if(w != NULL)
		gtk_widget_hide(w);
            w = editor->values_browse_btn[i];   
            if(w != NULL)
                gtk_widget_hide(w);
	}

	/* Mark as not having any selected value items. */
	EditorUnselectAll(
	    &editor->selected_value, &editor->total_selected_values
	);

	return;
}

/*
 *	Deletes all items on all the GUI and Real lists on
 *	the editor (except for undo and redo lists).
 */
void EditorListDeleteAllLists(ma_editor_struct *editor)
{
	if(editor == NULL)
	    return;

	/* Delete all models on the editor, this will unrealize any
	 * primitives on the models and deallocate those primitives as
	 * well.
	 */
	EditorModelDeleteAll(editor);

	/* Delete all model header items on the editor. */
	V3DMHListDeleteAll(&editor->mh_item, &editor->total_mh_items);

	/* GUI lists. */
	EditorListDeleteModelsG(editor);
        EditorListDeletePrimitivesG(editor);
        EditorListDeleteValuesG(editor);

	/* Textures. */
	EditorTextureListDeleteAll(editor); 

	return;
}
