#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <math.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "cdialog.h"
#include "csd.h"

#include "editor.h"
#include "editorcb.h"
#include "editoridialog.h"
#include "editorlightcb.h"
#include "editorviewcb.h"
#include "editorlist.h"

#include "msglist.h"
#include "messages.h"
#include "vmapixmaps.h"
#include "vma.h"
#include "vmautils.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


static void EditorLightPropertiesDrawingAreaDestroyCB(
	GtkObject *object, gpointer data
);
static void EditorLightPropertiesComboTextInsertCB(
	GtkEditable *editable,
	const gchar *text, gint length, gint *position,
	gpointer data
);
static void EditorLightPropertiesSelectColorCB(
	GtkWidget *widget, gpointer data
);
static void EditorLightPropertiesEnabledToggledCB(
	GtkWidget *widget, gpointer data
);
static void EditorLightPropertiesOmnidirectionalToggledCB(
	GtkWidget *widget, gpointer data
);
static gint EditorLightPropertiesGetLightNumber(
	ma_editor_idialog_struct *d, const gchar *entry_text
);
static void EditorLightPropertiesGetValues(
	ma_editor_idialog_struct *d, int light_num
);
static void EditorLightPropertiesMoveToCursorCB(
	GtkWidget *widget, gpointer data
);

void EditorLightToggleCB(GtkWidget *widget, gpointer data);
void EditorLightMoveToCursorCB(GtkWidget *widget, gpointer data);

void EditorLightPropertiesCB(GtkWidget *widget, gpointer data);
static void EditorLightPropertiesCloseCB(void *idialog, void *data);
static void EditorLightPropertiesUpdateCB(void *idialog, void *data);
static void EditorLightPropertiesApply(
	ma_editor_idialog_struct *d, gint light_num
);

#define LIGHT_NAME_MAX	256


#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))

#define RADTODEG(r)     ((r) * 180 / PI)
#define DEGTORAD(d)     ((d) * PI / 180.0)



/*
 *      Drawing area destroy callback.
 *
 *      This is just to deallocate its associated csd_color_struct.
 */
static void EditorLightPropertiesDrawingAreaDestroyCB(
	GtkObject *object, gpointer data
)
{
	csd_color_struct *color_ptr = (csd_color_struct *)data;
	if(color_ptr == NULL)
	    return;   

	/* Deallocate csd_color_struct. */
	free(color_ptr);

	return;
}

/*
 *      Light properties dialog combo entry insert_text callback.
 *
 *	This is to detect when a new item has been selected on the combo's
 *	list.
 */
static void EditorLightPropertiesComboTextInsertCB(
	GtkEditable *editable,
	const gchar *text, gint length, gint *position,
	gpointer data
)
{
	GtkWidget *w;
	gchar *name = NULL, *prev_name = NULL;
	ma_editor_idialog_struct *d = (ma_editor_idialog_struct *)data;
	if(d == NULL)
	    return;
	      
	/* If given text is valid, then make a copy of it (since the
	 * given text is not null terminated.
	 */
	if((text != NULL) && (length >= 0))
	{
	    name = (gchar *)malloc((length + 1) * sizeof(gchar));
	    memcpy(name, text, length * sizeof(gchar));
	    name[length] = '\0';
	}
	if(name == NULL)
	    return;

	/* Get last light name label widget. */
	w = EditorIDialogGetWidget(d, 18);
	if((w == NULL) ? 0 : GTK_IS_LABEL(w))
	{
	    gchar *strptr;

	    gtk_label_get(GTK_LABEL(w), &strptr);
	    if(strptr != NULL)
		prev_name = g_strdup(strptr);
	}

	/* Handle this callback only if dialog is mapped, because
	 * text changing may occure during setup of dialog while
	 * it is not mapped and we don't want to handle those callbacks.
	 */
	if(d->map_state)
	{
	    gint light_num, prev_light_num = -1;

	    /* Get previous selected light number on input dialog. */
	    if(prev_name != NULL)
		prev_light_num = EditorLightPropertiesGetLightNumber(
		    d, prev_name
		);

	    /* Get new light number to input dialog. */
	    light_num = EditorLightPropertiesGetLightNumber(d, name);

	    /* If previous light number is valid, apply values to
	     * that light.
	     */
	    if(prev_light_num > -1)
	    {
		EditorLightPropertiesApply(d, prev_light_num);
	    }
	    /* New light valid? */
	    if(light_num > -1)
	    {
		EditorLightPropertiesGetValues(d, light_num);
	    }
	}

	/* Update last light name label widget. */
	w = EditorIDialogGetWidget(d, 18);
	if((w == NULL) ? 0 : GTK_IS_LABEL(w))
	{
	    if(name != NULL)
		gtk_label_set_text(GTK_LABEL(w), name);
	}

	/* Deallocate our locally coppied name. */
	g_free(name);
	name = NULL;
	g_free(prev_name);
	prev_name = NULL;

	return;
}


/*
 *      Select color callback.
 *
 *      When a select color GtkButton is pressed, the color selection
 *      dialog will be mapped to select a color.
 *
 *      If a color is selected the color values will be updated on
 *      the GtkDrawingArea widget.
 *
 *      Input for data is the GtkDrawingArea.
 */
static void EditorLightPropertiesSelectColorCB(
	GtkWidget *widget, gpointer data
)
{
	static gbool reenterant = FALSE;
	GtkWidget *da = (GtkWidget *)data;
	csd_color_struct *start_color, *color_rtn;
	if(da == NULL)
	    return;

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	start_color = gtk_object_get_user_data(GTK_OBJECT(da));
	if(start_color == NULL)
	{
	    reenterant = FALSE;
	    return;
	}

	CSDSetTransientFor(gtk_widget_get_toplevel(da));
	if(CSDGetResponse(
	    "Select Color",
	    "Select", "Cancel",
	    start_color, &color_rtn,
	    NULL, NULL
	))
	{
	    if((color_rtn != NULL) && !GTK_WIDGET_NO_WINDOW(da))
	    {
		GdkColormap *colormap = gdk_window_get_colormap(da->window);
		GdkColor gdk_color;

		gdk_color.red = (guint16)(color_rtn->r * (guint16)(-1));
		gdk_color.green = (guint16)(color_rtn->g * (guint16)(-1));
		gdk_color.blue = (guint16)(color_rtn->b * (guint16)(-1));

		if(colormap != NULL)
		{
		    gdk_colormap_alloc_color(colormap, &gdk_color, TRUE, TRUE);
		    gdk_window_set_background(da->window, &gdk_color);
		    gdk_window_clear(da->window);
		    gdk_colormap_free_colors(colormap, &gdk_color, 1);
		}

		/* Update csd_color_struct on GtkDrawingArea widget. */
		if(start_color != color_rtn)
		    memcpy(start_color, color_rtn, sizeof(csd_color_struct));
	    }
	}
	CSDSetTransientFor(NULL);

	reenterant = FALSE;
}

/*
 *	Light enabled toggle callback.
 */
static void EditorLightPropertiesEnabledToggledCB(
	GtkWidget *widget, gpointer data
)
{
	static gbool reenterant = FALSE;
	gbool sensitive = FALSE;
	GtkWidget *w;
	ma_editor_idialog_struct *d = (ma_editor_idialog_struct *)data;
	if((widget == NULL) || (d == NULL))
	    return;

	if(reenterant)
	    return;
	else 
	    reenterant = TRUE;

	/* Get light enabled toggle widget. */
	w = EditorIDialogGetWidget(d, 1);
	if((w == NULL) ? 0 : GTK_IS_TOGGLE_BUTTON(w))
	{
	    sensitive = GTK_TOGGLE_BUTTON(w)->active;
	}

#define GTK_WIDGET_SENSITIVITY	\
{ \
 if(w != NULL) \
  gtk_widget_set_sensitive(w, sensitive); \
}

	w = EditorIDialogGetWidget(d, 2);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 3);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 4);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 5);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 6);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 7);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 8);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 9);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 10);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 11);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 12);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 13);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 14);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 15);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 16);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 17);
	GTK_WIDGET_SENSITIVITY


	/* Need to update omnidirectional related widgets. */
	if(sensitive)
	{
	    /* Get omnidirectional toggle widget. */
	    w = EditorIDialogGetWidget(d, 6);
	    if((w == NULL) ? 0 : GTK_IS_TOGGLE_BUTTON(w))
		EditorLightPropertiesOmnidirectionalToggledCB(w, d);
	}

#undef GTK_WIDGET_SENSITIVITY

	reenterant = FALSE;
	return;
}

/*
 *	Aprature omni directional toggle callback.
 */
static void EditorLightPropertiesOmnidirectionalToggledCB(
	GtkWidget *widget, gpointer data
)
{
	static gbool reenterant = FALSE;
	gbool sensitive = FALSE;
	GtkWidget *w;
	ma_editor_idialog_struct *d = (ma_editor_idialog_struct *)data;
	if((widget == NULL) || (d == NULL))
	    return;

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;

	/* Get light enabled toggle widget. */
	w = EditorIDialogGetWidget(d, 1);
	if((w == NULL) ? 0 : GTK_IS_TOGGLE_BUTTON(w))
	{
	    sensitive = GTK_TOGGLE_BUTTON(w)->active;
	}

	/* Get omnidirectional toggle widget. */
	w = EditorIDialogGetWidget(d, 6);
	if((w == NULL) ? 0 : GTK_IS_TOGGLE_BUTTON(w))
	{
	    if(sensitive)
		sensitive = !(GTK_TOGGLE_BUTTON(w)->active);
	}

#define GTK_WIDGET_SENSITIVITY	\
{ \
 if(w != NULL) \
  gtk_widget_set_sensitive(w, sensitive); \
}

	w = EditorIDialogGetWidget(d, 7);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 9);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 10);
	GTK_WIDGET_SENSITIVITY
	w = EditorIDialogGetWidget(d, 11);
	GTK_WIDGET_SENSITIVITY

#undef GTK_WIDGET_SENSITIVITY

	reenterant = FALSE;
	return;
}


/*
 *	Returns the light number on the input dialog's light number
 *	widget.
 *
 *	Can return -1 on error.
 */
static gint EditorLightPropertiesGetLightNumber(
	ma_editor_idialog_struct *d, const gchar *entry_text
)
{
	gint i;
	GList *glist;
	GtkWidget *w;
	GtkCombo *combo;

	if(d == NULL)
	    return(-1);

	/* Get the light number combo (widget 0). */
	w = EditorIDialogGetWidget(d, 0);
	if((w == NULL) ? 0 : GTK_IS_COMBO(w))
	    combo = GTK_COMBO(w);
	else
	    combo = NULL; 

	if(combo == NULL)
	    return(-1);

	if(entry_text == NULL)
	    entry_text = (const gchar *)gtk_entry_get_text(GTK_ENTRY(combo->entry));
	if(entry_text == NULL)
	    return(-1);

	i = 0;
	glist = GUIComboGetList(GTK_WIDGET(combo));
	if(glist != NULL)
	{
	    const gchar *glist_name;

	    while(glist != NULL)
	    {
		glist_name = (const gchar *)glist->data;
		if(glist_name != NULL)
		{
		    if(!strcasecmp(glist_name, entry_text))
			return(i);
		}

		i++;
		glist = glist->next;
	    }
	}

	return(-1);
}


/*
 *	Fetches values of the specified light number on the given
 *	input dialog's editor.
 */
static void EditorLightPropertiesGetValues(
	ma_editor_idialog_struct *d, int light_num
)
{
	GtkWidget *w;
	gbool omnidirectional_enabled = FALSE;
	guint decimals_position, decimals_angle;
	vma_light_struct *light_ptr;
	ma_editor_struct *editor;
	char fmt_str[80];
	char num_str[256];

	if(d == NULL)
	    return;

	editor = (ma_editor_struct *)d->editor_ptr;
	if(editor == NULL)
	    return;

	decimals_position = EditorVertexPositionDecimals();
	decimals_angle = EditorVertexAngleDecimals();

	if((light_num >= 0) && (light_num < VMA_LIGHTS_MAX))
	    light_ptr = &editor->light[light_num];
	else
	    light_ptr = NULL;
	if(light_ptr == NULL)
	    return;

	/* Get light enabled/disabled. */
	w = EditorIDialogGetWidget(d, 1);
	if((w == NULL) ? 0 : GTK_IS_TOGGLE_BUTTON(w))
	{
	    gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(w),
		(light_ptr->flags & VMA_LIGHT_FLAG_ENABLED) ? TRUE : FALSE
	    );
	    gtk_signal_emit_by_name(
		GTK_OBJECT(w), "toggled", (gpointer)d
	    );
	}

	/* Set position format string. */
	sprintf(fmt_str, "%%.%if", decimals_position);
	/* Get position. */
	w = EditorIDialogGetWidget(d, 2);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    sprintf(num_str, fmt_str, light_ptr->x);
	    gtk_entry_set_text(GTK_ENTRY(w), num_str);
	}
	w = EditorIDialogGetWidget(d, 3);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    sprintf(num_str, fmt_str, light_ptr->y);
	    gtk_entry_set_text(GTK_ENTRY(w), num_str);
	}
	w = EditorIDialogGetWidget(d, 4);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    sprintf(num_str, fmt_str, light_ptr->z);
	    gtk_entry_set_text(GTK_ENTRY(w), num_str);
	}

	/* Omnidirectional check. */
	w = EditorIDialogGetWidget(d, 6);
	if((w == NULL) ? 0 : GTK_IS_TOGGLE_BUTTON(w))
	{
	    gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(w),
		(light_ptr->spot_cutoff > (0.5 * PI)) ? TRUE : FALSE
	    );
	    gtk_signal_emit_by_name(
		GTK_OBJECT(w), "toggled", (gpointer)d
	    );
	    omnidirectional_enabled = GTK_TOGGLE_BUTTON(w)->active;
	}
	/* Spot cutoff. */
	w = EditorIDialogGetWidget(d, 7);
	if((w == NULL) ? 0 : GTK_IS_SPIN_BUTTON(w))
	{
	    gfloat theta = RADTODEG(light_ptr->spot_cutoff);
	    if(omnidirectional_enabled)
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), 0.0);
	    else
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), theta);
	}
	/* Spot exponent. */
	w = EditorIDialogGetWidget(d, 8);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    sprintf(num_str, "%f", light_ptr->spot_exponent);
	    gtk_entry_set_text(GTK_ENTRY(w), num_str);
	}

	/* Get direction. */
	w = EditorIDialogGetWidget(d, 9);
	if((w == NULL) ? 0 : GTK_IS_SPIN_BUTTON(w))
	{
	    gfloat theta = RADTODEG(light_ptr->heading);
	    gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), theta);
	}
	w = EditorIDialogGetWidget(d, 10);
	if((w == NULL) ? 0 : GTK_IS_SPIN_BUTTON(w))
	{
	    gfloat theta = RADTODEG(light_ptr->pitch);
	    gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), theta);
	}
	w = EditorIDialogGetWidget(d, 11);
	if((w == NULL) ? 0 : GTK_IS_SPIN_BUTTON(w))
	{
	    gfloat theta = RADTODEG(light_ptr->bank);
	    gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), theta);
	}

	/* Ambient. */
	w = EditorIDialogGetWidget(d, 12);
	if((w == NULL) ? 0 : GTK_IS_DRAWING_AREA(w))
	{
	    csd_color_struct *csd_color_ptr = (csd_color_struct *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	    if(csd_color_ptr != NULL)
	    {
		GdkColormap *colormap = gdk_window_get_colormap(w->window);
		GdkColor gdk_color;

		csd_color_ptr->a = 1.0;		/* Alpha always 1.0. */
		csd_color_ptr->r = light_ptr->ambient.r;
		csd_color_ptr->g = light_ptr->ambient.g;
		csd_color_ptr->b = light_ptr->ambient.b;

		gdk_color.red = (guint16)(csd_color_ptr->r * (guint16)(-1));
		gdk_color.green = (guint16)(csd_color_ptr->g * (guint16)(-1));
		gdk_color.blue = (guint16)(csd_color_ptr->b * (guint16)(-1));

		if(colormap != NULL)
		{
		    gdk_colormap_alloc_color(colormap, &gdk_color, TRUE, TRUE);
		    gdk_window_set_background(w->window, &gdk_color);
		    gdk_window_clear(w->window);
		    gdk_colormap_free_colors(colormap, &gdk_color, 1);
		}
	    }
	}
	/* Diffuse. */
	w = EditorIDialogGetWidget(d, 13);
	if((w == NULL) ? 0 : GTK_IS_DRAWING_AREA(w))  
	{
	    csd_color_struct *csd_color_ptr = (csd_color_struct *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	    if(csd_color_ptr != NULL)
	    {
		GdkColormap *colormap = gdk_window_get_colormap(w->window);
		GdkColor gdk_color;

		csd_color_ptr->a = 1.0;         /* Alpha always 1.0. */
		csd_color_ptr->r = light_ptr->diffuse.r;
		csd_color_ptr->g = light_ptr->diffuse.g;
		csd_color_ptr->b = light_ptr->diffuse.b;
   
		gdk_color.red = (guint16)(csd_color_ptr->r * (guint16)(-1));
		gdk_color.green = (guint16)(csd_color_ptr->g * (guint16)(-1));
		gdk_color.blue = (guint16)(csd_color_ptr->b * (guint16)(-1));

		if(colormap != NULL)
		{
		    gdk_colormap_alloc_color(colormap, &gdk_color, TRUE, TRUE);
		    gdk_window_set_background(w->window, &gdk_color);
		    gdk_window_clear(w->window);
		    gdk_colormap_free_colors(colormap, &gdk_color, 1);
		}
	    }
	}
	/* Specular. */
	w = EditorIDialogGetWidget(d, 14);
	if((w == NULL) ? 0 : GTK_IS_DRAWING_AREA(w))  
	{
	    csd_color_struct *csd_color_ptr = (csd_color_struct *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	    if(csd_color_ptr != NULL)
	    {
		GdkColormap *colormap = gdk_window_get_colormap(w->window);
		GdkColor gdk_color;

		csd_color_ptr->a = 1.0;         /* Alpha always 1.0. */
		csd_color_ptr->r = light_ptr->specular.r;
		csd_color_ptr->g = light_ptr->specular.g;
		csd_color_ptr->b = light_ptr->specular.b;
   
		gdk_color.red = (guint16)(csd_color_ptr->r * (guint16)(-1));
		gdk_color.green = (guint16)(csd_color_ptr->g * (guint16)(-1));
		gdk_color.blue = (guint16)(csd_color_ptr->b * (guint16)(-1));
 
		if(colormap != NULL)
		{
		    gdk_colormap_alloc_color(colormap, &gdk_color, TRUE, TRUE);
		    gdk_window_set_background(w->window, &gdk_color);
		    gdk_window_clear(w->window);
		    gdk_colormap_free_colors(colormap, &gdk_color, 1);
		}
	    }
	}

	/* Attenuation constant. */
	w = EditorIDialogGetWidget(d, 15);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    sprintf(num_str, "%f", light_ptr->attenuation_constant);
	    gtk_entry_set_text(GTK_ENTRY(w), num_str);
	}
	/* Attenuation linear. */
	w = EditorIDialogGetWidget(d, 16);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    sprintf(num_str, "%f", light_ptr->attenuation_linear);
	    gtk_entry_set_text(GTK_ENTRY(w), num_str);
	}
	/* Attenuation quadratic. */
	w = EditorIDialogGetWidget(d, 17);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    sprintf(num_str, "%f", light_ptr->attenuation_quadratic);
	    gtk_entry_set_text(GTK_ENTRY(w), num_str);
	}

	return;
}

/*
 *	Move to cursor button callback.
 */
static void EditorLightPropertiesMoveToCursorCB(
	GtkWidget *widget, gpointer data
)
{
	GtkWidget *w;
	char fmt_str[80];
	guint decimals_position;
	ma_editor_struct *editor;
	ma_editor_idialog_struct *d = (ma_editor_idialog_struct *)data;
	if(d == NULL)
	    return;

	editor = (ma_editor_struct *)d->editor_ptr;
	if(editor == NULL)
	    return;

	decimals_position = EditorVertexPositionDecimals();

	/* Set position format string. */
	sprintf(fmt_str, "%%.%if", decimals_position);

	/* Get X position widget. */
	w = EditorIDialogGetWidget(d, 2);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    char num_str[256];

	    sprintf(num_str, fmt_str, editor->vcursor_x);
	    gtk_entry_set_text(GTK_ENTRY(w), num_str);
	}   

	/* Get Y position widget. */
	w = EditorIDialogGetWidget(d, 3);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    char num_str[256];

	    sprintf(num_str, fmt_str, editor->vcursor_y);
	    gtk_entry_set_text(GTK_ENTRY(w), num_str);
	}

	/* Get Z position widget. */
	w = EditorIDialogGetWidget(d, 4);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    char num_str[256];

	    sprintf(num_str, fmt_str, editor->vcursor_z);
	    gtk_entry_set_text(GTK_ENTRY(w), num_str);
	}

	return;
}

/*
 *	Light enable/disable toggle callback.
 *
 *	Designed to be called from the editor's light button menu.
 */
void EditorLightToggleCB(GtkWidget *widget, gpointer data)
{
	gint light_num;
	ma_editor_struct *editor = (ma_editor_struct *)data;
	if(editor == NULL)
	    return;

	if(!editor->initialized || editor->processing)
	    return;

	light_num = EditorSelectedLightNumber(editor);
	if((light_num >= 0) && (light_num < VMA_LIGHTS_MAX))
	{
	    vma_light_struct *light_ptr = &editor->light[light_num];

	    if(light_ptr->flags & VMA_LIGHT_FLAG_ENABLED)
		light_ptr->flags &= ~VMA_LIGHT_FLAG_ENABLED;
	    else
		light_ptr->flags |= VMA_LIGHT_FLAG_ENABLED;

	    /* Update menus and redraw views. */
	    EditorUpdateMenus(editor);
	    EditorUpdateAllViewMenus(editor);
	    EditorRedrawAllViews(editor);
	}

	return;
}

/*
 *	Light move to cursor callback.
 */
void EditorLightMoveToCursorCB(GtkWidget *widget, gpointer data)
{
	gint light_num;
	ma_editor_struct *editor = (ma_editor_struct *)data;
	if(editor == NULL)
	    return;

	if(!editor->initialized || editor->processing)
	    return;

	light_num = EditorSelectedLightNumber(editor);
	if((light_num >= 0) && (light_num < VMA_LIGHTS_MAX))
	{
	    vma_light_struct *light_ptr = &editor->light[light_num];

	    light_ptr->x = editor->vcursor_x;
	    light_ptr->y = editor->vcursor_y;
	    light_ptr->z = editor->vcursor_z;

	    /* Update menus and redraw views. */
	    EditorUpdateMenus(editor);
	    EditorUpdateAllViewMenus(editor);
	    EditorRedrawAllViews(editor);
	}

	return;
}

/*
 *	Creates a color selector button as the next widget on the
 *	given input dialog d.
 */
static void EditorLightPropertiesCreateColorButton(
	ma_editor_idialog_struct *d,
	GtkWidget *parent,	/* An hbox. */
	const char *label,
	int label_width,
	const char *tooltip
)
{
	GdkColormap *colormap;
	GtkWidget *w, *btn, *da;
	csd_color_struct *color_ptr;
	GdkColor gdk_color;


	w = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent = w;

	w = gtk_label_new(label);
	gtk_widget_set_usize(w, label_width, -1);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	btn = w = gtk_button_new();
/*      gtk_widget_set_usize(w, 25, 25); */
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent = w;

	w = gtk_frame_new(NULL);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_container_border_width(GTK_CONTAINER(w), 1);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_OUT);
	gtk_widget_show(w);
	parent = w;

	/* Create drawing area. */
	da = w = gtk_drawing_area_new();
	gtk_drawing_area_size(GTK_DRAWING_AREA(w), 15, 15);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_realize(w);  
	gtk_widget_show(w);

	/* Allocate a new csd_color_struct for the drawing area. */
	color_ptr = (csd_color_struct *)calloc(
	    1, sizeof(csd_color_struct)
	);
	if(color_ptr != NULL)
	{
	    color_ptr->a = 1.0;
	    color_ptr->r = 0.0;  
	    color_ptr->g = 0.0;
	    color_ptr->b = 0.0;

	    gdk_color.red = (guint16)(color_ptr->r * (guint16)(-1));
	    gdk_color.green = (guint16)(color_ptr->g * (guint16)(-1));
	    gdk_color.blue = (guint16)(color_ptr->b * (guint16)(-1));

	    gtk_object_set_user_data(GTK_OBJECT(da), (gpointer)color_ptr);
	    gtk_signal_connect(
		GTK_OBJECT(da), "destroy",
		GTK_SIGNAL_FUNC(EditorLightPropertiesDrawingAreaDestroyCB),
		(gpointer)color_ptr
	    );
	}

	/* Set background color of GtkDrawingArea widget. */
	colormap = gdk_window_get_colormap(da->window);
	if(colormap != NULL)
	{
	    gdk_colormap_alloc_color(colormap, &gdk_color, TRUE, TRUE);
	    gdk_window_set_background(da->window, &gdk_color);
	    gdk_colormap_free_colors(colormap, &gdk_color, 1);
	}

	/* Set tooltip for the button. */
	GUISetWidgetTip(btn, tooltip);

	/* Record this widget on the dialog. */
	EditorIDialogRecordWidget(d, da);

	/* Attach the "clicked" signal to the GtkButton widget and pass
	 * the GtkDrawingArea widget as the data pointer (not the
	 * input dialog d).
	 */
	gtk_signal_connect(
	    GTK_OBJECT(btn), "clicked",
	    GTK_SIGNAL_FUNC(EditorLightPropertiesSelectColorCB),
	    (gpointer)da
	);

	return;
}


/*
 *	Light properties input dialog map callback.
 */
void EditorLightPropertiesCB(GtkWidget *widget, gpointer data)
{
	const char *msglist[] = VMA_MSGLIST_LTPROP_TOOLTIPS;
	GtkWidget *parent;
	ma_editor_idialog_struct *d;
	ma_editor_struct *editor = (ma_editor_struct *)data;
	if(editor == NULL)
	    return;

	if(!editor->initialized || editor->processing)
	    return;

/*
	if(VMAWriteProtectCheck(editor))
	    return;
 */

	d = &editor->idialog;		/* Editor's input dialog. */

	/* Reset input dialog. */
	EditorIDialogReset(editor, d, FALSE);

	/* Get input dialog's client vbox. */
	parent = EditorIDialogClientParent(d);
	if(parent != NULL) 
	{
	    /* Begin creating our widgets on input dialog, order;
	     *
	     *	0	Light number combo
	     *	1	Enabled/disabled check
	     *
	     *	2	Position X entry
	     *  3       Position Y entry
	     *  4       Position Z entry
	     *  5	Position set from cursor button
	     *
	     *  6       Omnidirectional check
	     *  7       Spot cutoff
	     *  8       Spot exponent
	     *
	     *  9       Direction heading spin
	     *  10      Direction pitch spin
	     *  11      Position bank spin
	     *
	     *	12	Ambient color button
	     *  13      Diffuse color button
	     *  14      Specular color button
	     *
	     *	15	Attenuation constant entry
	     *	16	Attenuation linear entry
	     *	17	Attenuation quadratic entry
	     *
	     *	18	Label to hold previous light name
	     */
	    gint	border_major = 5,
			border_minor = 2;
	    gint	bw = (100 + (2 * 3)),
			bh = (30 + (2 * 3));
	    guint	decimals_angle = EditorVertexAngleDecimals();
	    gint i, light_num;
	    gchar light_name[LIGHT_NAME_MAX];
	    GList *glist;
	    GtkAdjustment *adj;
	    GtkWidget *w, *parent2, *parent3, *parent4;
	    GtkCombo *combo;
	    void *w_rtn1, *w_rtn2;


	    /* Main vbox. */
	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Hbox for light number combo and enabled check. */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;


	    /* Create light names. */
	    glist = NULL;
	    for(i = 0; i < VMA_LIGHTS_MAX; i++)
	    {
		char tmp_text[256];

		sprintf(tmp_text, "Light %i", i);
		glist = g_list_append(glist, g_strdup(tmp_text));
	    }

	    /* Light number combo. */
	    w = (GtkWidget *)GUIComboCreate(
		NULL,		/* No label. */
		"Light 0",	/* Initial value. */
		glist,		/* Initial glist. */
		VMA_LIGHTS_MAX,	/* Max items. */
		&w_rtn1,
		(void *)d,	/* Client data. */
		NULL,		/* Don't need callbacks. */
		NULL
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    combo = (GtkCombo *)w_rtn1;
	    gtk_entry_set_editable(GTK_ENTRY(combo->entry), FALSE);
	    gtk_combo_set_case_sensitive(combo, TRUE);
	    gtk_signal_connect(
		GTK_OBJECT((GtkEditable *)combo->entry), "insert_text",
		GTK_SIGNAL_FUNC(EditorLightPropertiesComboTextInsertCB),
		(gpointer)d
	    );
	    gtk_widget_show(w);
	    /* Record as widget 0. */
	    EditorIDialogRecordWidget(d, (GtkWidget *)combo);

	    /* Free light names list. */
	    if(glist != NULL)
	    {
		g_list_foreach(glist, (GFunc)g_free, NULL);
		g_list_free(glist);
		glist = NULL;
	    }

	    /* Get light number on editor's light number spin. */
	    (*light_name) = '\0';
	    light_num = EditorSelectedLightNumber(editor);
	    if(light_num > -1)
	    {
		sprintf(light_name, "Light %i", light_num);
		gtk_entry_set_text(GTK_ENTRY(combo->entry), light_name);
	    }


	    /* Enabled/disabled check. */
	    w = gtk_check_button_new_with_label("Enabled");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_signal_connect( 
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(EditorLightPropertiesEnabledToggledCB),
		(gpointer)d
	    );
	    gtk_widget_show(w);
	    /* Record as widget 1. */
	    EditorIDialogRecordWidget(d, w);


	    /* Separator. */
	    w = gtk_hseparator_new();
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);


	    /* Position frame. */
	    w = gtk_frame_new("Position");
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Hbox for position entries and move to cursor button. */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_container_add(GTK_CONTAINER(parent3), w);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Position entries. */
	    w = GUIPromptBar(
		NULL, "X:",
		&w_rtn1, &w_rtn2
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize((GtkWidget *)w_rtn2, 100, -1);
	    GUISetWidgetTip(
		(GtkWidget *)w_rtn2,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_POSITION
		)  
	    );
	    gtk_widget_show(w);
	    /* Record as widget 2. */
	    EditorIDialogRecordWidget(d, w_rtn2);

	    w = GUIPromptBar(  
		NULL, "Y:",
		&w_rtn1, &w_rtn2
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize((GtkWidget *)w_rtn2, 100, -1);
	    GUISetWidgetTip(
		(GtkWidget *)w_rtn2,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_POSITION
		)
	    );
	    gtk_widget_show(w);
	    /* Record as widget 3. */
	    EditorIDialogRecordWidget(d, w_rtn2);

	    w = GUIPromptBar(
		NULL, "Z:",
		&w_rtn1, &w_rtn2
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize((GtkWidget *)w_rtn2, 100, -1);
	    GUISetWidgetTip(
		(GtkWidget *)w_rtn2,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_POSITION
		)
	    );
	    gtk_widget_show(w);
	    /* Record as widget 4. */
	    EditorIDialogRecordWidget(d, w_rtn2);

	    /* Set position from cursor button. */
	    w = gtk_button_new_with_label("From Cursor");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, bw, bh);
	    GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EditorLightPropertiesMoveToCursorCB),
		(gpointer)d
	    );
	    GUISetWidgetTip(
		w,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_POSITION_FROM_CURSOR
		)
	    );
	    gtk_widget_show(w);
	    /* Record as widget 5. */
	    EditorIDialogRecordWidget(d, w);


	    /* Aprature frame. */
	    w = gtk_frame_new("Aprature");
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Hbox for aprature entries. */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_container_add(GTK_CONTAINER(parent3), w);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Omnidirectional check. */
	    w = gtk_check_button_new_with_label("Omnidirectional");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(EditorLightPropertiesOmnidirectionalToggledCB),
		(gpointer)d
	    );
	    GUISetWidgetTip(
		w,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_OMNIDIRECTIONAL
		)
	    );
	    gtk_widget_show(w);
	    /* Record as widget 6. */
	    EditorIDialogRecordWidget(d, w);

	    /* Cutoff spin button. */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    w = gtk_label_new("Cutoff:");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0,            /* Initial. */
		0.0,            /* Minimum. */
		90.0,           /* Maximum. */
		1.0,            /* Step inc. */
		10.0,           /* Page inc. */
		10.0            /* Page size. */
	    );
	    w = gtk_spin_button_new(
		adj,
		1.0,            /* Climb rate (0.0 to 1.0). */
		decimals_angle  /* Digits. */
	    );
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    GUISetWidgetTip(   
		w,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_SPOT_CUTOFF
		)
	    );
	    gtk_widget_set_usize(w, 70, -1);
	    gtk_widget_show(w);
	    /* Record as widget 7. */
	    EditorIDialogRecordWidget(d, w);

	    /* Exponent entry. */
	    w = GUIPromptBar(
		NULL, "Exponent:",
		&w_rtn1, &w_rtn2
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize((GtkWidget *)w_rtn2, 100, -1);
	    GUISetWidgetTip(
		(GtkWidget *)w_rtn2,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_SPOT_EXPONENT
		)
	    );
	    gtk_widget_show(w);
	    /* Record as widget 8. */
	    EditorIDialogRecordWidget(d, w_rtn2);


	    /* Direction frame. */
	    w = gtk_frame_new("Direction");
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Hbox for direction spin button. */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_container_add(GTK_CONTAINER(parent3), w);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Direction spin buttons. */
	    /* Heading. */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    w = gtk_label_new("Heading:");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0,		/* Initial. */
		0.0,		/* Minimum. */
		360.0,		/* Maximum. */
		1.0,		/* Step inc. */
		10.0,		/* Page inc. */
		10.0		/* Page size. */
	    );
	    w = gtk_spin_button_new(
		adj,
		1.0,		/* Climb rate (0.0 to 1.0). */
		decimals_angle	/* Digits. */
	    );
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, 70, -1);
	    GUISetWidgetTip(
		w,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_DIRECTION
		)
	    );
	    gtk_widget_show(w);
	    /* Record as widget 9. */
	    EditorIDialogRecordWidget(d, w);

	    /* Pitch. */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    w = gtk_label_new("Pitch:");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0,            /* Initial. */   
		0.0,            /* Minimum. */
		360.0,          /* Maximum. */
		1.0,            /* Step inc. */
		10.0,           /* Page inc. */
		10.0            /* Page size. */
	    );
	    w = gtk_spin_button_new(
		adj,
		1.0,            /* Climb rate (0.0 to 1.0). */
		decimals_angle  /* Digits. */
	    );
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, 70, -1);
	    GUISetWidgetTip(
		w,                  
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_DIRECTION   
		)
	    );
	    gtk_widget_show(w);
	    /* Record as widget 10. */
	    EditorIDialogRecordWidget(d, w);

	    /* Bank. */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w); 
	    parent4 = w;
	    w = gtk_label_new("Bank:");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0,            /* Initial. */   
		0.0,            /* Minimum. */
		360.0,          /* Maximum. */
		1.0,            /* Step inc. */
		10.0,           /* Page inc. */
		10.0            /* Page size. */
	    );
	    w = gtk_spin_button_new(
		adj,
		1.0,            /* Climb rate (0.0 to 1.0). */
		decimals_angle  /* Digits. */
	    );
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, 70, -1);
	    GUISetWidgetTip(
		w,                  
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_DIRECTION   
		)
	    );
	    gtk_widget_show(w);
	    /* Record as widget 11. */
	    EditorIDialogRecordWidget(d, w);


	    /* Color frame. */
	    w = gtk_frame_new("Color");   
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Hbox for color buttons. */ 
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_container_add(GTK_CONTAINER(parent3), w);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Ambient (widget 12). */
	    EditorLightPropertiesCreateColorButton(
		d, parent3,
		"Ambient", 80,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_AMBIENT
		)
	    );
	    /* Diffuse (widget 13). */
	    EditorLightPropertiesCreateColorButton(
		d, parent3,
		"Diffuse", 80,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_DIFFUSE
		)
	    );
	    /* Ambient (widget 14). */
	    EditorLightPropertiesCreateColorButton(
		d, parent3,
		"Specular", 80,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_SPECULAR
		)
	    );


	    /* Attenuation frame. */
	    w = gtk_frame_new("Attenuation");
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Hbox for attenuation entries. */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_container_add(GTK_CONTAINER(parent3), w);
	    gtk_container_border_width(GTK_CONTAINER(w), border_major);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Constant attenuation factor entry. */
	    w = GUIPromptBar(
		NULL, "Constant:",
		&w_rtn1, &w_rtn2
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize((GtkWidget *)w_rtn2, 100, -1);
	    GUISetWidgetTip(
		(GtkWidget *)w_rtn2,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_ATTENUATION_CONSTANT
		)
	    );
	    gtk_widget_show(w);
	    /* Record as widget 15. */
	    EditorIDialogRecordWidget(d, w_rtn2);

	    /* Linear attenuation factor entry. */   
	    w = GUIPromptBar(
		NULL, "Linear:",
		&w_rtn1, &w_rtn2
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize((GtkWidget *)w_rtn2, 100, -1);
	    GUISetWidgetTip(
		(GtkWidget *)w_rtn2,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_ATTENUATION_LINEAR
		)
	    );
	    gtk_widget_show(w);
	    /* Record as widget 16. */
	    EditorIDialogRecordWidget(d, w_rtn2);

	    /* Quadratic attenuation factor entry. */
	    w = GUIPromptBar(
		NULL, "Quadratic:",
		&w_rtn1, &w_rtn2
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize((GtkWidget *)w_rtn2, 100, -1);
	    GUISetWidgetTip(
		(GtkWidget *)w_rtn2,
		MsgListMatchCaseMessage(
		    msglist,
		    VMA_MSGNAME_LTPROP_ATTENUATION_QUADRATIC
		)
	    );
	    gtk_widget_show(w);
	    /* Record as widget 17. */
	    EditorIDialogRecordWidget(d, w_rtn2);




	    /* Label to hold previous light name, this label is never
	     * shown.
	     */
	    w = gtk_label_new(light_name);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    /* Record as widget 18. */
	    EditorIDialogRecordWidget(d, w);
 

	    /* Set initial widget states. */
	    EditorLightPropertiesGetValues(d, light_num);
	}

	EditorIDialogMap(
	    editor, d,
	    "Light Properties",
	    NULL, "Update", "Close",
	    (void *)editor,
	    NULL,
	    EditorLightPropertiesUpdateCB,
	    EditorLightPropertiesCloseCB
	);

	return;
}

/*
 *	Light properties input dialog close callback.
 */
static void EditorLightPropertiesCloseCB(void *idialog, void *data)
{
	ma_editor_struct *editor;
	ma_editor_idialog_struct *d = (ma_editor_idialog_struct *)idialog;
	if(d == NULL)
	    return;

	editor = (ma_editor_struct *)d->editor_ptr;
	if(editor == NULL)
	    return;

	/* Update selected light on input dialog. */
	EditorLightPropertiesUpdateCB(idialog, data);

	/* Reset input dialog and unmap it. */
	EditorIDialogReset(editor, d, TRUE);

	return;
}

/*
 *      Light properties input dialog update callback.
 */
static void EditorLightPropertiesUpdateCB(void *idialog, void *data)
{
	gint light_num;
	ma_editor_idialog_struct *d = (ma_editor_idialog_struct *)idialog;
	if(d == NULL)
	    return;

	/* Get current light number on input dialog. */
	light_num = EditorLightPropertiesGetLightNumber(
	    d, NULL
	);
	/* Light valid to apply values to? */
	if(light_num > -1)
	   EditorLightPropertiesApply(d, light_num);

	return;
}

/*
 *	Light properties input dialog apply callback.
 */
static void EditorLightPropertiesApply(
	ma_editor_idialog_struct *d, gint light_num
)
{
	vma_light_struct *light_ptr;
	const gchar *cstrptr;
	GtkWidget *w;
	ma_editor_struct *editor;
	if(d == NULL)
	    return;

	editor = (ma_editor_struct *)d->editor_ptr;
	if(editor == NULL)
	    return;

	if(editor->processing)
	    return;

	/* Get pointer to light on editor. */
	if((light_num >= 0) && (light_num < VMA_LIGHTS_MAX))
	    light_ptr = &editor->light[light_num];
	else
	    light_ptr = NULL;
	if(light_ptr == NULL)
	    return;


	/* Begin fetching values from widgets. */
	EditorSetBusy(editor);

	/* Enabled state (widget 1). */
	w = EditorIDialogGetWidget(d, 1);
	if((w == NULL) ? 0 : GTK_IS_TOGGLE_BUTTON(w))
	{
	    if(GTK_TOGGLE_BUTTON(w)->active)
		light_ptr->flags |= VMA_LIGHT_FLAG_ENABLED;
	    else
		light_ptr->flags &= ~VMA_LIGHT_FLAG_ENABLED;
	}

	/* Position (widgets 2 to 4). */
	w = EditorIDialogGetWidget(d, 2);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    cstrptr = (const gchar *)gtk_entry_get_text(GTK_ENTRY(w));
	    if(cstrptr != NULL)
		light_ptr->x = atof(cstrptr);
	}
	w = EditorIDialogGetWidget(d, 3);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    cstrptr = (const gchar *)gtk_entry_get_text(GTK_ENTRY(w));
	    if(cstrptr != NULL)
		light_ptr->y = atof(cstrptr);
	}
	w = EditorIDialogGetWidget(d, 4);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    cstrptr = (const gchar *)gtk_entry_get_text(GTK_ENTRY(w));
	    if(cstrptr != NULL)
		light_ptr->z = atof(cstrptr);
	}

	/* Omnidirectional state (widget 6). */
	w = EditorIDialogGetWidget(d, 6);
	if((w == NULL) ? 0 : GTK_IS_TOGGLE_BUTTON(w))
	{
	    if(GTK_TOGGLE_BUTTON(w)->active) 
	    {
		light_ptr->spot_cutoff = (1.0 * PI);
	    }
	    else
	    {
		/* Spot cutoff (widget 7). */
		w = EditorIDialogGetWidget(d, 7);
		if((w == NULL) ? 0 : GTK_IS_SPIN_BUTTON(w))
		{
		    light_ptr->spot_cutoff = DEGTORAD(
			gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(w))
		    );
		}
	    }
	}
	/* Spot expnoent (widget 8). */
	w = EditorIDialogGetWidget(d, 8);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    cstrptr = (const gchar *)gtk_entry_get_text(GTK_ENTRY(w));
	    if(cstrptr != NULL)
		light_ptr->spot_exponent = atof(cstrptr);
	}

	/* Direction (widgets 9 to 11). */
	w = EditorIDialogGetWidget(d, 9);
	if((w == NULL) ? 0 : GTK_IS_SPIN_BUTTON(w))
	{
	    light_ptr->heading = DEGTORAD(
		gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(w))
	    );
	}
	w = EditorIDialogGetWidget(d, 10);
	if((w == NULL) ? 0 : GTK_IS_SPIN_BUTTON(w))
	{
	    light_ptr->pitch = DEGTORAD(
		gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(w))
	    );
	}
	w = EditorIDialogGetWidget(d, 11);
	if((w == NULL) ? 0 : GTK_IS_SPIN_BUTTON(w))
	{
	    light_ptr->bank = DEGTORAD(
		gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(w))
	    );
	}

	/* Ambient. */
	w = EditorIDialogGetWidget(d, 12);
	if((w == NULL) ? 0 : GTK_IS_DRAWING_AREA(w))
	{
	    csd_color_struct *csd_color_ptr = (csd_color_struct *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	    if(csd_color_ptr != NULL)
	    {
		light_ptr->ambient.a = 1.0;
		light_ptr->ambient.r = csd_color_ptr->r;
		light_ptr->ambient.g = csd_color_ptr->g;
		light_ptr->ambient.b = csd_color_ptr->b;
	    }
	}
	/* Diffuse. */
	w = EditorIDialogGetWidget(d, 13);
	if((w == NULL) ? 0 : GTK_IS_DRAWING_AREA(w))
	{
	    csd_color_struct *csd_color_ptr = (csd_color_struct *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	    if(csd_color_ptr != NULL)
	    {
		light_ptr->diffuse.a = 1.0;
		light_ptr->diffuse.r = csd_color_ptr->r;  
		light_ptr->diffuse.g = csd_color_ptr->g;
		light_ptr->diffuse.b = csd_color_ptr->b;
	    }
	}
	/* Specular. */
	w = EditorIDialogGetWidget(d, 14);
	if((w == NULL) ? 0 : GTK_IS_DRAWING_AREA(w))
	{
	    csd_color_struct *csd_color_ptr = (csd_color_struct *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	    if(csd_color_ptr != NULL)
	    {
		light_ptr->specular.a = 1.0;
		light_ptr->specular.r = csd_color_ptr->r;
		light_ptr->specular.g = csd_color_ptr->g;
		light_ptr->specular.b = csd_color_ptr->b;
	    }
	}

	/* Attenuation constant (widget 15). */
	w = EditorIDialogGetWidget(d, 15);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    cstrptr = (const gchar *)gtk_entry_get_text(GTK_ENTRY(w));
	    if(cstrptr != NULL)
		light_ptr->attenuation_constant = atof(cstrptr);
	}
	/* Attenuation linear (widget 16). */
	w = EditorIDialogGetWidget(d, 16);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    cstrptr = (const gchar *)gtk_entry_get_text(GTK_ENTRY(w));
	    if(cstrptr != NULL)
		light_ptr->attenuation_linear = atof(cstrptr);
	}
	/* Attenuation quadratic (widget 17). */
	w = EditorIDialogGetWidget(d, 17);
	if((w == NULL) ? 0 : GTK_IS_ENTRY(w))
	{
	    cstrptr = (const gchar *)gtk_entry_get_text(GTK_ENTRY(w));
	    if(cstrptr != NULL)
		light_ptr->attenuation_quadratic = atof(cstrptr);
	}

	/* Update menus and redraw views. */
	EditorUpdateMenus(editor);
	EditorUpdateAllViewMenus(editor);
	EditorRedrawAllViews(editor); 

	EditorSetReady(editor);

	return;
}
