#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include "../include/string.h"

#include "view.h"
#include "vmacfg.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


vma_cfg_item_struct *VMACFGItemNew(
	int type, const char *parameter
);
void *VMACFGItemGetValue(vma_cfg_item_struct *ci, int *type_rtn);
void VMACFGItemSetValue(vma_cfg_item_struct *ci, const void *value);
void VMACFGItemResetValue(vma_cfg_item_struct *ci);
void VMACFGItemReset(vma_cfg_item_struct *ci);
void VMACFGItemDelete(vma_cfg_item_struct *ci);

int VMACFGItemListMatchParameter(
	vma_cfg_item_struct *list, const char *parameter
);
void *VMACFGItemListMatchGetValue(
	vma_cfg_item_struct *list, const char *parameter, int *type_rtn
);
int VMACFGItemListGetValueI(
	vma_cfg_item_struct *list, const char *parameter
);
long VMACFGItemListGetValueL(
	vma_cfg_item_struct *list, const char *parameter
);
float VMACFGItemListGetValueF(
	vma_cfg_item_struct *list, const char *parameter 
);
double VMACFGItemListGetValueD(
	vma_cfg_item_struct *list, const char *parameter
);
char *VMACFGItemListGetValueS(
	vma_cfg_item_struct *list, const char *parameter
);

vma_cfg_item_struct *VMACFGItemListMatchSetValue(
	vma_cfg_item_struct *list, const char *parameter,
	const void *value, int create_as_needed
);


/*
 *	Allocates a new configuration item structure.
 *
 *	Parameter will be coppied to the parameter member of the
 *	structure however member value will be left NULL.
 */
vma_cfg_item_struct *VMACFGItemNew(
	int type, const char *parameter
)
{
	vma_cfg_item_struct *ci = (vma_cfg_item_struct *)calloc(
	    1, sizeof(vma_cfg_item_struct)
	);
	if(ci == NULL)
	    return(NULL);

	ci->type = type;
	ci->parameter = ((parameter != NULL) ? strdup(parameter) : NULL);
	ci->value = NULL;

	return(ci);
}

/*
 *	Returns the given configuration item ci's value and type.
 */
void *VMACFGItemGetValue(vma_cfg_item_struct *ci, int *type_rtn)
{
	if(type_rtn != NULL)
	    (*type_rtn) = -1;

	if(ci == NULL)
	    return(NULL);

	if(type_rtn != NULL)
	    (*type_rtn) = ci->type;

	return(ci->value);
}

/*
 *	Sets the given value to the given configuration item ci
 *	depending on ci's type.
 *
 *	The type of data that value points to must match the same
 *	type of data of the ci's type.
 *
 *	Note that ci->value may be reallocated during this call.
 */
void VMACFGItemSetValue(vma_cfg_item_struct *ci, const void *value)
{
	size_t size;
	int8_t *ptr_8;
	u_int8_t *ptr_u_8;
	int16_t *ptr_16;
	u_int16_t *ptr_u_16;
	int32_t *ptr_32;
	u_int32_t *ptr_u_32;
	int64_t *ptr_64;
	u_int64_t *ptr_u_64;
	float *ptr_float;
	double *ptr_double;
	vma_color_struct *ptr_color;


	if(ci == NULL)
	    return;

	/* Reset value on the cfg item, deallocating it. */
	VMACFGItemResetValue(ci);

	/* No given value? */
	if(value == NULL)
	    return;

	/* Set by cfg item value data type. */
	switch(ci->type)
	{
	  case VMA_CFG_ITEM_TYPE_NONE:
	    break;

	  case VMA_CFG_ITEM_TYPE_INT_8:
	    size = sizeof(int8_t);
	    ptr_8 = (int8_t *)malloc(size);
	    if(ptr_8 != NULL)
		memcpy(ptr_8, value, size);
	    ci->value = ptr_8;
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_8:
	    size = sizeof(u_int8_t);
	    ptr_u_8 = (u_int8_t *)malloc(size);
	    if(ptr_u_8 != NULL)
		memcpy(ptr_u_8, value, size);
	    ci->value = ptr_u_8;
	    break;

	  case VMA_CFG_ITEM_TYPE_INT_16:
	    size = sizeof(int16_t);
	    ptr_16 = (int16_t *)malloc(size);
	    if(ptr_16 != NULL)
		memcpy(ptr_16, value, size);
	    ci->value = ptr_16;
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_16:
	    size = sizeof(u_int16_t);
	    ptr_u_16 = (u_int16_t *)malloc(size);
	    if(ptr_u_16 != NULL)
		memcpy(ptr_u_16, value, size);
	    ci->value = ptr_u_16;
	    break;

	  case VMA_CFG_ITEM_TYPE_INT_32:
	    size = sizeof(int32_t);
	    ptr_32 = (int32_t *)malloc(size);
	    if(ptr_32 != NULL)
		memcpy(ptr_32, value, size);
	    ci->value = ptr_32;
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_32:
	    size = sizeof(u_int32_t);
	    ptr_u_32 = (u_int32_t *)malloc(size);
	    if(ptr_u_32 != NULL)
		memcpy(ptr_u_32, value, size);
	    ci->value = ptr_u_32;
	    break;

	  case VMA_CFG_ITEM_TYPE_INT_64:
	    size = sizeof(int64_t);
	    ptr_64 = (int64_t *)malloc(size);
	    if(ptr_64 != NULL)
		memcpy(ptr_64, value, size);
	    ci->value = ptr_64;
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_64:
	    size = sizeof(u_int64_t);
	    ptr_u_64 = (u_int64_t *)malloc(size);
	    if(ptr_u_64 != NULL)
		memcpy(ptr_u_64, value, size);
	    ci->value = ptr_u_64;
	    break;

	  case VMA_CFG_ITEM_TYPE_FLOAT:
	    size = sizeof(float);
	    ptr_float = (float *)malloc(size);
	    if(ptr_float != NULL)
		memcpy(ptr_float, value, size);
	    ci->value = ptr_float;
	    break;
	  case VMA_CFG_ITEM_TYPE_DOUBLE:
	    size = sizeof(double);
	    ptr_double = (double *)malloc(size);
	    if(ptr_double != NULL)
		memcpy(ptr_double, value, size);
	    ci->value = ptr_double;
	    break;

	  case VMA_CFG_ITEM_TYPE_STRING:
	    ci->value = strdup(
		(char *)value
	    );
	    break;

	  case VMA_CFG_ITEM_TYPE_COLOR:
	    size = sizeof(vma_color_struct);
	    ptr_color = (vma_color_struct *)malloc(size);
	    if(ptr_color != NULL)
		memcpy(ptr_color, value, size);
	    ci->value = ptr_color;
	    break;


	}
}

/*
 *	Deallocates the given configuration item ci's member value
 *	(as needed) and resets it to NULL.
 */
void VMACFGItemResetValue(vma_cfg_item_struct *ci)
{
	if(ci != NULL)
	{
	    free(ci->value);
	    ci->value = NULL;
	}
}

/*
 *	Deallocates all substructures on the given configuration item
 *	ci and sets the type to VMA_CFG_ITEM_TYPE_NONE.
 *
 *	The structure ci itself will not be deallocated.
 */
void VMACFGItemReset(vma_cfg_item_struct *ci)
{
	if(ci != NULL)
	{
	    /* Deallocate value. */
	    VMACFGItemResetValue(ci);

	    /* Deallocate parameter name. */
	    free(ci->parameter);
	    ci->parameter = NULL;

	    /* Reset type. */
	    ci->type = VMA_CFG_ITEM_TYPE_NONE;
	}
}

/*
 *	Deallocates all structures on the ci and the structure itself.
 */
void VMACFGItemDelete(vma_cfg_item_struct *ci)
{
	if(ci != NULL)
	{
	    /* Deallocate all of the given configuration item's members. */
	    VMACFGItemReset(ci);

	    /* Deallocate the configuration item structure itself. */
	    free(ci);
	}
}

/*
 *	Scans through the sequential list of configuration items and
 *	returns the index number of the one who's parameter matches the
 *	given parameter (case insensitive).
 *
 *	The last item in the sequential list needs to be all 0.
 *
 *	Returns -1 on failed match or error, otherwise returns
 *	the index of the valid item.
 */
int VMACFGItemListMatchParameter(
	vma_cfg_item_struct *list, const char *parameter
)
{
	int i = 0;
	vma_cfg_item_struct *ci;


	if((list == NULL) || (parameter == NULL))
	    return(-1);

	do
	{
	    ci = &list[i];

	    if(ci->parameter != NULL)
	    {
		if(!strcasecmp(ci->parameter, parameter))
		    return(i);
	    }

	    i++;

	} while((ci->type != 0) && (ci->parameter != NULL));

	return(-1);
}

/*
 *	Returns the pointer to the value and sets the type to the
 *	type of value the matched list item in the sequential list is.
 *
 *      The last item in the sequential list needs to be all 0.
 *
 *      Returns NULL on failed match or error, otherwise returns
 *      the pointer to the value and sets the type code.
 *
 *	The returned pointer should never be deallocated by the calling
 *	function.
 */
void *VMACFGItemListMatchGetValue(
	vma_cfg_item_struct *list, const char *parameter, int *type_rtn
)
{
	int i;


	if(type_rtn != NULL)
	    (*type_rtn) = -1;

	/* Check if given parameter matches the parameter of a 
	 * configuration item in the given list.
	 */
	i = VMACFGItemListMatchParameter(list, parameter);
	if(i > -1)
	    return(VMACFGItemGetValue(&list[i], type_rtn));
	else
	    return(NULL);
}

/*
 *	Returns value as an int, the parameter must match to an item
 *	who's type matches any of the following:
 *
 *		VMA_CFG_ITEM_TYPE_INT_8
 *		VMA_CFG_ITEM_TYPE_U_INT_8
 *		VMA_CFG_ITEM_TYPE_INT_16
 *		VMA_CFG_ITEM_TYPE_U_INT_16
 *		VMA_CFG_ITEM_TYPE_INT_32
 *		VMA_CFG_ITEM_TYPE_U_INT_32
 *
 *	Does not distinguish error.
 */
int VMACFGItemListGetValueI(
	vma_cfg_item_struct *list, const char *parameter
)
{
#define RTN_VAL_T	int
	RTN_VAL_T rtn_val = 0;
	int ci_num;
	vma_cfg_item_struct *ci_ptr;

	int8_t *ptr_8;
	u_int8_t *ptr_u8;
	int16_t *ptr_16;
	u_int16_t *ptr_u16;
	int32_t *ptr_32;
	u_int32_t *ptr_u32;


	/* Match by parameter. */
	ci_num = VMACFGItemListMatchParameter(list, parameter);
	if(ci_num < 0)
	    return(rtn_val);
	else
	    ci_ptr = &list[ci_num];

	/* No value? */
	if(ci_ptr->value == NULL)
	    return(rtn_val);

	/* Get return value by cfg item type. */
	switch(ci_ptr->type)
	{
	  case VMA_CFG_ITEM_TYPE_INT_8: 
	    ptr_8 = (int8_t *)ci_ptr->value;  
	    rtn_val = (RTN_VAL_T)(*ptr_8); 
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_8: 
	    ptr_u8 = (u_int8_t *)ci_ptr->value;  
	    rtn_val = (RTN_VAL_T)(*ptr_u8); 
	    break;

	  case VMA_CFG_ITEM_TYPE_INT_16:
	    ptr_16 = (int16_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_16);
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_16:
	    ptr_u16 = (u_int16_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u16);
	    break;

	  case VMA_CFG_ITEM_TYPE_INT_32:
	    ptr_32 = (int32_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_32);
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_32:
	    ptr_u32 = (u_int32_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u32);
	    break;
	}

	return(rtn_val);
#undef RTN_VAL_T
}

/*
 *      Returns value as an int, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              VMA_CFG_ITEM_TYPE_INT_8
 *              VMA_CFG_ITEM_TYPE_U_INT_8
 *              VMA_CFG_ITEM_TYPE_INT_16
 *              VMA_CFG_ITEM_TYPE_U_INT_16
 *              VMA_CFG_ITEM_TYPE_INT_32
 *              VMA_CFG_ITEM_TYPE_U_INT_32
 *		VMA_CFG_ITEM_TYPE_INT_64
 *		VMA_CFG_ITEM_TYPE_U_INT_64
 *
 *      Does not distinguish error.
 */
long VMACFGItemListGetValueL(
	vma_cfg_item_struct *list, const char *parameter
)
{
#define RTN_VAL_T	long
	RTN_VAL_T rtn_val = (long)0;
	int ci_num;
	vma_cfg_item_struct *ci_ptr;

	int8_t *ptr_8;
	u_int8_t *ptr_u8;
	int16_t *ptr_16;
	u_int16_t *ptr_u16;
	int32_t *ptr_32;
	u_int32_t *ptr_u32;
	int64_t *ptr_64;
	u_int64_t *ptr_u64;


	/* Match by parameter. */
	ci_num = VMACFGItemListMatchParameter(list, parameter);
	if(ci_num < 0)
	    return(rtn_val);
	else
	    ci_ptr = &list[ci_num];

	/* No value? */
	if(ci_ptr->value == NULL)
	    return(rtn_val);

	/* Get return value by cfg item type. */
	switch(ci_ptr->type)  
	{
	  case VMA_CFG_ITEM_TYPE_INT_8:
	    ptr_8 = (int8_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_8);
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_8:
	    ptr_u8 = (u_int8_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u8);
	    break;
 
	  case VMA_CFG_ITEM_TYPE_INT_16:
	    ptr_16 = (int16_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_16);
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_16:
	    ptr_u16 = (u_int16_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u16);
	    break;

	  case VMA_CFG_ITEM_TYPE_INT_32:
	    ptr_32 = (int32_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_32);
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_32:
	    ptr_u32 = (u_int32_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u32); 
	    break;

	  case VMA_CFG_ITEM_TYPE_INT_64:
	    ptr_64 = (int64_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_64);
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_64:
	    ptr_u64 = (u_int64_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u64);
	    break;
	}
	    
	return(rtn_val);
#undef RTN_VAL_T  
}           

/*
 *      Returns value as an int, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              VMA_CFG_ITEM_TYPE_INT_8
 *              VMA_CFG_ITEM_TYPE_U_INT_8
 *              VMA_CFG_ITEM_TYPE_INT_16
 *              VMA_CFG_ITEM_TYPE_U_INT_16
 *              VMA_CFG_ITEM_TYPE_INT_32
 *              VMA_CFG_ITEM_TYPE_U_INT_32
 *              VMA_CFG_ITEM_TYPE_INT_64
 *              VMA_CFG_ITEM_TYPE_U_INT_64
 *		VMA_CFG_ITEM_TYPE_FLOAT
 *
 *      Does not distinguish error.
 */
float VMACFGItemListGetValueF(
	vma_cfg_item_struct *list, const char *parameter
)
{
#define RTN_VAL_T	float
	RTN_VAL_T rtn_val = (float)0.0;
	int ci_num;
	vma_cfg_item_struct *ci_ptr;

	int8_t *ptr_8;
	u_int8_t *ptr_u8;
	int16_t *ptr_16;
	u_int16_t *ptr_u16; 
	int32_t *ptr_32;
	u_int32_t *ptr_u32;
	int64_t *ptr_64;
	u_int64_t *ptr_u64;
	float *ptr_f;


	/* Match by parameter. */
	ci_num = VMACFGItemListMatchParameter(list, parameter);
	if(ci_num < 0)  
	    return(rtn_val);
	else
	    ci_ptr = &list[ci_num];

	/* No value? */
	if(ci_ptr->value == NULL)
	    return(rtn_val);

	/* Get return value by cfg item type. */
	switch(ci_ptr->type)
	{
	  case VMA_CFG_ITEM_TYPE_INT_8: 
	    ptr_8 = (int8_t *)ci_ptr->value;  
	    rtn_val = (RTN_VAL_T)(*ptr_8); 
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_8: 
	    ptr_u8 = (u_int8_t *)ci_ptr->value;  
	    rtn_val = (RTN_VAL_T)(*ptr_u8); 
	    break;
	
	  case VMA_CFG_ITEM_TYPE_INT_16:
	    ptr_16 = (int16_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_16);
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_16:
	    ptr_u16 = (u_int16_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u16);
	    break;

	  case VMA_CFG_ITEM_TYPE_INT_32:
	    ptr_32 = (int32_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_32);
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_32:
	    ptr_u32 = (u_int32_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u32);
	    break;

	  case VMA_CFG_ITEM_TYPE_INT_64:
	    ptr_64 = (int64_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_64);
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_64:
	    ptr_u64 = (u_int64_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u64);
	    break;

	  case VMA_CFG_ITEM_TYPE_FLOAT:
	    ptr_f = (float *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_f);
	    break;
	}

	return(rtn_val);
#undef RTN_VAL_T
}

/*
 *      Returns value as an int, the parameter must match to an item
 *      who's type matches any of the following:
 *
 *              VMA_CFG_ITEM_TYPE_INT_8
 *              VMA_CFG_ITEM_TYPE_U_INT_8 
 *              VMA_CFG_ITEM_TYPE_INT_16
 *              VMA_CFG_ITEM_TYPE_U_INT_16  
 *              VMA_CFG_ITEM_TYPE_INT_32
 *              VMA_CFG_ITEM_TYPE_U_INT_32
 *              VMA_CFG_ITEM_TYPE_INT_64
 *              VMA_CFG_ITEM_TYPE_U_INT_64
 *              VMA_CFG_ITEM_TYPE_FLOAT
 *		VMA_CFG_ITEM_TYPE_DOUBLE
 *
 *      Does not distinguish error.
 */
double VMACFGItemListGetValueD(
	vma_cfg_item_struct *list, const char *parameter
)
{
#define RTN_VAL_T	double
	RTN_VAL_T rtn_val = 0.0;
	int ci_num;
	vma_cfg_item_struct *ci_ptr;

	int8_t *ptr_8;
	u_int8_t *ptr_u8;
	int16_t *ptr_16;
	u_int16_t *ptr_u16;
	int32_t *ptr_32;
	u_int32_t *ptr_u32;
	int64_t *ptr_64;
	u_int64_t *ptr_u64;
	float *ptr_f;
	double *ptr_d;


	/* Match by parameter. */
	ci_num = VMACFGItemListMatchParameter(list, parameter);
	if(ci_num < 0)
	    return(rtn_val);
	else
	    ci_ptr = &list[ci_num];

	/* No value? */
	if(ci_ptr->value == NULL)
	    return(rtn_val);

	/* Get return value by cfg item type. */
	switch(ci_ptr->type)
	{
	  case VMA_CFG_ITEM_TYPE_INT_8:
	    ptr_8 = (int8_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_8);
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_8:  
	    ptr_u8 = (u_int8_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u8);
	    break;

	  case VMA_CFG_ITEM_TYPE_INT_16:
	    ptr_16 = (int16_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_16);
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_16:
	    ptr_u16 = (u_int16_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u16);
	    break;

	  case VMA_CFG_ITEM_TYPE_INT_32:
	    ptr_32 = (int32_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_32);
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_32:
	    ptr_u32 = (u_int32_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u32);
	    break;

	  case VMA_CFG_ITEM_TYPE_INT_64:
	    ptr_64 = (int64_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_64);
	    break;
	  case VMA_CFG_ITEM_TYPE_U_INT_64:
	    ptr_u64 = (u_int64_t *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_u64);
	    break;

	  case VMA_CFG_ITEM_TYPE_FLOAT:
	    ptr_f = (float *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_f);
	    break;

	  case VMA_CFG_ITEM_TYPE_DOUBLE:
	    ptr_d = (double *)ci_ptr->value;
	    rtn_val = (RTN_VAL_T)(*ptr_d);
	    break;
	}

	return(rtn_val);
#undef RTN_VAL_T
}

/*
 *	Returns a pointer to the value of the matched parameter, the
 *	returned string may not be deallocated as it points directly
 *	to the matched cfg item's value. Can return NULL on error.
 *
 *	Only cfg items are of type VMA_CFG_ITEM_TYPE_STRING will be
 *	matched.
 */
char *VMACFGItemListGetValueS(
	vma_cfg_item_struct *list, const char *parameter
)
{
	int ci_num;
	vma_cfg_item_struct *ci_ptr;


	/* Match by parameter. */
	ci_num = VMACFGItemListMatchParameter(list, parameter);
	if(ci_num < 0)
	    return(NULL);
	else
	    ci_ptr = &list[ci_num];

	/* Not a string or not value? */
	if((ci_ptr->type != VMA_CFG_ITEM_TYPE_STRING) ||
	   (ci_ptr->value == NULL)
	)
	    return(NULL);
	else
	    return((char *)ci_ptr->value);
}


/*
 *      Sets the given value to the sequential list item found match
 *	the given parameter.
 *
 *      The last item in the sequential list needs to be all 0.
 *
 *	Returns the pointer to the new sequential list, if
 *	create_as_needed is false then pointer will be exactly the same
 *	as the list pointer value. If create_as_needed is true then
 *	the returned pointer will be a new pointer to the new
 *	sequential list and the old list will be deallocated.
 *
 *	Can return NULL on error if create_as_needed is true.
 */
vma_cfg_item_struct *VMACFGItemListMatchSetValue(
	vma_cfg_item_struct *list, const char *parameter,
	const void *value, int create_as_needed
)
{
	int i;


	if(list == NULL)
	    return(list);

	/* Are we allowed to create a new configuration item? */
	if(create_as_needed)
	{
	    /* Yes we can create a new configuration item, but first
	     * check if the given parameter matches the parameter of
	     * an existing configuration item in the given list.
	     */
	    i = VMACFGItemListMatchParameter(list, parameter);
	    if(i > -1)
	    {
		/* Got existing configuration item, so set it and
		 * return the pointer to it.
		 */
		VMACFGItemSetValue(&list[i], value);
		return(&list[i]);
	    }
	    else
	    {
/* Need to add code to allocate a new configuration item into the
 * given list.
 */
		return(NULL);
	    }
	}
	else
	{
	    /* Not allowed to create new configuration item, so just
	     * check if the given parameter matches the parameter of
	     * an existing configuration item in the given list.
	     */
	    i = VMACFGItemListMatchParameter(list, parameter);
	    if(i > -1)
	    {
		/* Got existing configuration item, so set it and
		 * return the pointer to it.
		 */
		VMACFGItemSetValue(&list[i], value);
		return(&list[i]);
	    }
	    else
	    {
		/* No match and not allowed to create new configuration 
		 * item so just return NULL.
		 */
		return(NULL);
	    }
  	}
}
