/*
 * Copyright (C) 2003  Stefan Kleine Stegemann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "DocumentWindow.h"
#include "DocumentWindowController.h"
#include "ViewPDF.h"

#include <Foundation/NSString.h>
#include <Foundation/NSException.h>
#include <Foundation/NSArray.h>
#include <Foundation/NSEnumerator.h>
#include <Foundation/NSUserDefaults.h>
#include <AppKit/NSButton.h>
#include <AppKit/NSEvent.h>
#include <AppKit/NSImage.h>
#include <AppKit/NSTextField.h>
#include <AppKit/NSFont.h>
#include <AppKit/NSProgressIndicator.h>
#include <AppKit/NSWindow+Toolbar.h>
#include <AppKit/NSToolbarItem.h>
#include <AppKit/NSText.h>

/* Toolbar Item identifiers */
static NSString* TbItemNextPage       = @"NextPage";
static NSString* TbItemPrevPage       = @"PrevPage";
static NSString* TbItemZoomIn         = @"ZoomIn";
static NSString* TbItemZoomOut        = @"ZoomOut";
static NSString* TbItemZoomToRealSize = @"SizeToRealSize";
static NSString* TbItemSizeToFit      = @"SizeToFit";
static NSString* TbItemSizeToFitWidth = @"SizeToFitWidth";


/*
 * Non-Public methods of the DocumentView class.
 */
@interface DocumentWindow(Private)
- (void) _createContentView;
- (void) _createDocumentView;
- (void) _createToolbar;
@end

/*
 * A window that displays a ViewableDocument and provides
 * some controls to control the view (eg. zoom controls).
 * A DocumentWindow is controlled by an instance of
 * DocumentWindowController. 
 */
@implementation DocumentWindow

/*
 * Designated initializer.
 */
- (id) init
{
   NSRect  contentRect;

   contentRect = NSMakeRect(100, 100, 400, 200);

   if ((self = [super initWithContentRect: contentRect
                     styleMask: (NSTitledWindowMask
                                 | NSClosableWindowMask
                                 | NSMiniaturizableWindowMask
                                 | NSResizableWindowMask)
                     backing: NSBackingStoreBuffered
                      defer: NO]))
   {
      myController = nil;
      toolbar = nil;
      findView = nil;
      isInFindMode = NO;
      fieldEditor = nil;

      [self setTitle: @"ViewPDF Window"];
      [self setMinSize: NSMakeSize(100, 100)];
      
      [self _createContentView];
      [self _createDocumentView];

      [[[self documentView] scrollView] setDisplayPageControls: YES];
   }

   return self;
}


/*
 * Set the windows controller.
 */
- (void) setController: (DocumentWindowController*)controller
{
   myController = controller;
   // myController needs not to be retained because this
   // is done by the document-based application model

   [[[documentView scrollView] firstPageButton] setAction: @selector(firstPage:)];
   [[[documentView scrollView] firstPageButton] setTarget: myController];
   [[[documentView scrollView] prevPageButton] setAction: @selector(previousPage:)];
   [[[documentView scrollView] prevPageButton] setTarget: myController];
   [[[documentView scrollView] nextPageButton] setAction: @selector(nextPage:)];
   [[[documentView scrollView] nextPageButton] setTarget: myController];   
   [[[documentView scrollView] lastPageButton] setAction: @selector(lastPage:)];
   [[[documentView scrollView] lastPageButton] setTarget: myController];
   [[[documentView scrollView] currentPageField] setDelegate: myController];
}


/*
 * Invoked by the controller when the controller has been
 * initialized with this window.
 */
- (void) windowInitializedWithController
{
   /* we initialize the toolbar here because the window
      controller set's the window size and calls this
      method afterwards.  */
   [self _createToolbar];
}



- (void) dealloc
{
   NSLog(@"dealloc DocumentWindow, retain count is %d", [self retainCount]);
   RELEASE(findView);
   RELEASE(fieldEditor);

   [super dealloc];
}


/*
 * Catch control-keys.
 */
- (void) keyDown: (NSEvent*)theEvent
{
   NSString* chars = [theEvent characters];
   unichar   firstChar = 0;
   BOOL      shiftKey = [theEvent modifierFlags] & NSShiftKeyMask;
   
   if ([chars length] > 0)
   {
      firstChar = [chars characterAtIndex: 0];
      switch (firstChar)
      {
         case NSLeftArrowFunctionKey:
            if (!shiftKey)
            {
               [myController scrollHorizontal: self direction: SCROLL_LEFT];
            }
            else
            {
               [myController previousPage: self];
            }
            break;

         case NSRightArrowFunctionKey:
            if (!shiftKey)
            {
               [myController scrollHorizontal: self direction: SCROLL_RIGHT];
            }
            else
            {
               [myController nextPage: self];
            }
            break;

         case NSUpArrowFunctionKey:
            [myController scrollVertical: self direction: SCROLL_UP];
            break;

         case NSDownArrowFunctionKey:
            [myController scrollVertical: self direction: SCROLL_DOWN];
            break;

         case NSPageUpFunctionKey:
            [myController scrollPage: self direction: SCROLL_UP];
            break;

         case NSPageDownFunctionKey:
            [myController scrollPage: self direction: SCROLL_DOWN];
            break;

         case 0x20:
            if (shiftKey)
            {
               [myController scrollPage: self direction: SCROLL_UP];
            }
            else
            {
               [myController scrollPage: self direction: SCROLL_DOWN];
            }
            break;

         case 27: // ESC
            if ([self isInFindMode])
            {
               [self exitFindMode];
            }
            break;
      }
   }
}


/*
 * Get a field editor that is able to catch the ESC key.
 */
- (NSText*) fieldEditor: (BOOL)createFlag forObject: (id)anObject
{
   if (!fieldEditor && createFlag)
   {
      fieldEditor = [[ExtendedNSTextView alloc] init];
      [fieldEditor setFieldEditor: YES];
      [fieldEditor setWindow: self];
   }

   return fieldEditor;
}


- (PDFContentView*) documentView
{
   return documentView;
}


- (void) toggleToolbar
{
   [super toggleToolbarShown: nil];

   [[NSUserDefaults standardUserDefaults] setBool: [toolbar isVisible]
                                           forKey: PrefsToolbarVisible];
}


- (void) enterFindMode
{
   NSRect contentFrame;
   NSRect fvFrame;
   NSRect docFrame;

   if ([self isInFindMode])
   {
      return;
   }

   contentFrame = [[self contentView] frame];

   fvFrame = NSMakeRect(0, 0, NSWidth(contentFrame), 40);

   if (findView == nil)
   {
      findView = [[FindView alloc] initWithFrame: fvFrame];
   }
   else
   {
      [findView setFrame: fvFrame];
   }

   docFrame = [[self documentView] frame];
   [[self documentView] setFrame:
      NSMakeRect(NSMinX(docFrame),
                 NSHeight(fvFrame),
                 NSWidth(docFrame),
                 NSHeight(docFrame) - NSHeight(fvFrame))];

   [[self contentView] addSubview: findView];
   [[self documentView] setNeedsDisplay: YES];

   [findView activate];

   isInFindMode = YES;
}


- (void) exitFindMode
{
   NSRect contentFrame;

   NSAssert([self isInFindMode], @"not in find mode");

   isInFindMode = NO;

   [findView deactivate];

   contentFrame = [[self contentView] frame];

   [findView removeFromSuperview];
   [[self documentView] setFrame:
        NSMakeRect(0, 0, NSWidth(contentFrame), NSHeight(contentFrame))];

   [[self documentView] setNeedsDisplay: YES];
}


- (BOOL) isInFindMode
{
   return isInFindMode;
}

@end


/* ----------------------------------------------------- */
/*  Category NSToolbarDelegate                           */
/* ----------------------------------------------------- */

@implementation DocumentWindow (NSToolbarDelegate)

- (NSToolbarItem*)toolbar: (NSToolbar*)toolbar
    itemForItemIdentifier: (NSString*)itemIdentifier
willBeInsertedIntoToolbar: (BOOL)flag
{
   NSToolbarItem* theItem;

   NSAssert(myController, @"cannot create toolbar item with no window controller set");

   theItem = [[NSToolbarItem alloc] initWithItemIdentifier: itemIdentifier];
   AUTORELEASE(theItem);

   if ([itemIdentifier isEqualToString: TbItemPrevPage])
   {
      [theItem setLabel: @"Previous Page"];
      [theItem setImage: [NSImage imageNamed: @"tb_prev_page.tiff"]];
      [theItem setTarget: myController];
      [theItem setAction: @selector(previousPage:)];
   }
   else if ([itemIdentifier isEqualToString: TbItemNextPage])
   {
      [theItem setLabel: @"Next Page"];
      [theItem setImage: [NSImage imageNamed: @"tb_next_page.tiff"]];
      [theItem setTarget: myController];
      [theItem setAction: @selector(nextPage:)];
   }
   else if ([itemIdentifier isEqualToString: TbItemZoomIn])
   {
      [theItem setLabel: @"Zoom In"];
      [theItem setImage: [NSImage imageNamed: @"zoom_in.tiff"]];
      [theItem setTarget: myController];
      [theItem setAction: @selector(zoomIn:)];
   }
   else if ([itemIdentifier isEqualToString: TbItemZoomOut])
   {
      [theItem setLabel: @"Zoom Out"];
      [theItem setImage: [NSImage imageNamed: @"zoom_out.tiff"]];
      [theItem setTarget: myController];
      [theItem setAction: @selector(zoomOut:)];
   }
   else if ([itemIdentifier isEqualToString: TbItemZoomToRealSize])
   {
      [theItem setLabel: @"1:1"];
      [theItem setImage: [NSImage imageNamed: @"zoom_11.tiff"]];
      [theItem setTarget: myController];
      [theItem setAction: @selector(zoomToRealSize:)];
   }
   else if ([itemIdentifier isEqualToString: TbItemSizeToFit])
   {
      [theItem setLabel: @"Fit Window"];
      [theItem setImage: [NSImage imageNamed: @"fit_page.tiff"]];
      [theItem setTarget: myController];
      [theItem setAction: @selector(sizePageToFit:)];
   }
   else if ([itemIdentifier isEqualToString: TbItemSizeToFitWidth])
   {
      [theItem setLabel: @"Fit Width"];
      [theItem setImage: [NSImage imageNamed: @"fit_width.tiff"]];
      [theItem setTarget: myController];
      [theItem setAction: @selector(sizePageToFitWidth:)];
   }

   return theItem;
}


- (NSArray*) toolbarAllowedItemIdentifiers: (NSToolbar*)aToolbar
{
   NSMutableArray* items = [[NSMutableArray alloc] init];
   AUTORELEASE(items);

   [items addObject: TbItemPrevPage];
   [items addObject: TbItemNextPage];
   [items addObject: TbItemZoomIn];
   [items addObject: TbItemZoomOut];
   [items addObject: TbItemZoomToRealSize];
   [items addObject: TbItemSizeToFit];
   [items addObject: TbItemSizeToFitWidth];

   return items;
}


- (NSArray*) toolbarDefaultItemIdentifiers: (NSToolbar*)aToolbar
{
   return [self toolbarAllowedItemIdentifiers: aToolbar];
}


- (NSArray *) toolbarSelectableItemIdentifiers: (NSToolbar *)toolbar
{
   return [NSArray array];
}

@end


/* ----------------------------------------------------- */
/*  Category Private                                     */
/* ----------------------------------------------------- */

@implementation DocumentWindow (Private)

/*
 * Create the window's content view.
 */
- (void) _createContentView
{
   NSView* contentView;
   NSRect contentRect;
   NSRect contentViewFrame;

   contentRect = [[self contentView] frame];
   contentViewFrame = NSMakeRect(0, 0, 
                                 contentRect.size.width,
                                 contentRect.size.height);
   contentView = [[NSView alloc] initWithFrame: contentViewFrame];
   [contentView setAutoresizesSubviews: YES];
   [contentView setAutoresizingMask: (NSViewWidthSizable |
                                      NSViewHeightSizable)];
   
   [self setContentView: contentView];

   RELEASE(contentView);
}


/*
 * Create the image view and add it to the window.
 */
- (void) _createDocumentView
{
   NSRect frame;

   frame = NSMakeRect(0,
                      0,
                      NSWidth([[self contentView] frame]),
                      NSHeight([[self contentView] frame]));

   documentView = [[PDFContentView alloc] initWithFrame: frame];
   [[self contentView] addSubview: documentView];
   
   RELEASE(documentView);
}


/*
 * Create the window's toolbar.
 */
- (void) _createToolbar
{
   BOOL isVisible;

   toolbar = [[NSToolbar alloc] initWithIdentifier: @"ViewPDFToolbar"];
   [toolbar setDelegate: self];
   [toolbar setAllowsUserCustomization: NO];
   [self setToolbar: toolbar];
   RELEASE(toolbar);

   isVisible = [[NSUserDefaults standardUserDefaults] boolForKey: PrefsToolbarVisible];
   [toolbar setVisible: isVisible];
}

@end


/* ----------------------------------------------------- */
/*  ExtendedNSTextView implementation                    */
/* ----------------------------------------------------- */

@implementation ExtendedNSTextView

- (id) initWithFrame: (NSRect)aFrame
       textContainer: (NSTextContainer*)aTextContainer
{
   if ((self = [super initWithFrame: aFrame textContainer: aTextContainer]))
   {
      window = nil;
   }

   return self;
}


- (void) dealloc
{
   [super dealloc];
}


- (void) setWindow: (DocumentWindow*)aWindow
{
   window = aWindow;
}


- (void) keyDown: (NSEvent*)theEvent
{
   NSString* chars = [theEvent characters];
   unichar   firstChar = 0;

   if ([chars length] > 0)
   {
      firstChar = [chars characterAtIndex: 0];
      if ((firstChar == 27) && [window isInFindMode])
      {
         [window exitFindMode];
         return;
      }
   }

   [super keyDown: theEvent];
}

@end

