/*
    vlad - an LDAP visualisation tool
    Copyright (C) 1999-2001 Robert Norris <rob@nauseum.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <lber.h>
#include <ldap.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "vlad.h"

static LDAP *ld = NULL;

static char *dn_attrs[] = { "dn", NULL };

static int vlad_get_lderrno(LDAP *ld) {
#ifdef USE_LDAP_OPENLDAP
#if LDAP_API_VERSION > 2000
	int ld_errno;

	ldap_get_option(ld, LDAP_OPT_ERROR_NUMBER, &ld_errno);
	return ld_errno;
#else
	return ld->ld_errno;
#endif
#endif
#ifdef USE_LDAP_NETSCAPE
	return ldap_get_lderrno(ld, NULL, NULL);
#endif
	}
	
static LDAPMessage *do_search(char *dn, char *filter, int scope, int verbose) {
	LDAPMessage *result;
	char *realfilter, **attrs;
	int realscope;

	if(filter)
		realfilter = filter;
	else
		realfilter = "objectclass=*";
	
	if(scope == 0) {
		attrs = NULL;
		realscope = LDAP_SCOPE_BASE; 
		}
	else {
		attrs = dn_attrs;
		realscope = LDAP_SCOPE_ONELEVEL;
		}
	
	if(!(ld = ldap_open(opt.host, opt.port))) {
		if(verbose) set_status("Couldn't connect to %s:%d", opt.host, opt.port);
		return NULL;
		}
	
	if(ldap_simple_bind_s(ld, opt.binddn, opt.bindpw)) {
		if(verbose) set_status("Couldn't bind to server: %s", ldap_err2string(vlad_get_lderrno(ld)));
		return NULL;
		}
	
	if(ldap_search_s(ld, dn, realscope, realfilter, attrs, 0, &result)) {
		if(verbose) set_status("Search failed: %s", ldap_err2string(vlad_get_lderrno(ld)));
		ldap_unbind_s(ld);
		return NULL;
		}
	
	return result;
	}

int get_attrs(struct entry *entry, int verbose) {
	LDAPMessage *result, *ldentry;
	BerElement *ber;
	char *attr, **values, *value;
	int i = 0, err = 0, count;

	if(!(result = do_search(entry->dn, NULL, 0, 1))) {
		err = 1; goto attrs_fail;
		}

	if(!(ldentry = ldap_first_entry(ld, result))) {
		ldap_msgfree(result);
		if(verbose) set_status("No attributes");
		err = 1; goto attrs_fail_unbind;
		}
	
	if(entry->attrs)
		free_attr_set(entry->attrs);

	if(!(entry->attrs = (struct attr **) malloc(sizeof(struct attr *)))) {
		if(verbose) set_status("Couldn't allocate memory");
		err = 1; goto attrs_fail_free_entry;
		}
	entry->attrs[0] = NULL;
	
	for(attr = ldap_first_attribute(ld, ldentry, &ber); attr; attr = ldap_next_attribute(ld, ldentry, ber)) {
		if(!(values = ldap_get_values(ld, ldentry, attr))) {
			if(verbose) set_status("Couldn't get attributes: %s", ldap_err2string(vlad_get_lderrno(ld)));
			err = 1; goto attrs_fail_free_ber;
			}

		for(count = 0, value = values[0]; value; count++, value = values[count]) {
			i++;
			if(!(entry->attrs = (struct attr **) realloc((void *) entry->attrs, sizeof(struct attr *) * (i + 1)))) {
				if(verbose) set_status("Couldn't allocate memory");
				err = 1; goto attrs_fail_free_attrs;
				}

			if(!(entry->attrs[i - 1] = create_attr_struct())) {
				if(verbose) set_status("Couldn't allocate memory");
				ldap_value_free(values);
				err = 1; goto attrs_fail_free_attrs;
				}

			entry->attrs[i - 1]->name = strdup(attr);
			entry->attrs[i - 1]->value = strdup(value);

			entry->attrs[i] = NULL;
			}

		ldap_value_free(values);
		}
	
	entry->attrs_timestamp = time(NULL);

	ldap_msgfree(ldentry);
	
	ldap_unbind_s(ld);
	ld = NULL;
	
	return 0;

attrs_fail_free_attrs:
	free_attr_set(entry->attrs);

attrs_fail_free_ber:
	ber_free(ber, 0);

attrs_fail_free_entry:
	ldap_msgfree(ldentry);

attrs_fail_unbind:
	ldap_unbind_s(ld);

attrs_fail:
	ld = NULL;
	return err;
	}

int get_subtree(struct entry *tree, int verbose) {
	LDAPMessage *result, *ldentry;
	struct entry *here;
	char *value, **explode;
	int err = 0;

	if(!(result = do_search(tree->dn, NULL, 1, verbose))) goto tree_fail;

	if(tree->subentries_head) {
		free_entry_list(tree->subentries_head);
		tree->subentries_head = tree->subentries_tail = NULL;
		}
	
	if((ldap_count_entries(ld, result)) <= 0) {
		tree->subentries_timestamp = time(NULL);
		err = 0; goto tree_fail_free_result;
		}

	for(ldentry = ldap_first_entry(ld, result); ldentry; ldentry = ldap_next_entry(ld, ldentry)) {
		if(!(value = ldap_get_dn(ld, ldentry))) {
			if(verbose) set_status("Couldn't get DN: %s", ldap_err2string(vlad_get_lderrno(ld)));
			err = 1; goto tree_fail_free_tree;
			}

		if(!(here = create_entry_struct())) {
			if(verbose) set_status("Couldn't allocate memory");
			err = 1; goto tree_fail_free_tree;
			}

		if(tree->subentries_tail) {
			tree->subentries_tail->next = here;
			here->prev = tree->subentries_tail;
			tree->subentries_tail = here;
			}
		else
			tree->subentries_head = tree->subentries_tail = here;
			
		here->parent = tree;
		here->dn = strdup(value);

		explode = ldap_explode_dn(here->dn, 0);
		here ->shortdn = strdup(explode[0]);
		ldap_value_free(explode);

		if((value = strchr(here->shortdn, '\n')))
			*value = '\0';

		here->depth = tree->depth + 1;
		}
	
	tree->subentries_timestamp = time(NULL);
	
	ldap_msgfree(ldentry);
	ldap_msgfree(result);
	
	ldap_unbind_s(ld);
	ld = NULL;
	
	return 0;

tree_fail_free_tree:
	free_entry_struct(tree);
	tree->subentries_timestamp = 0;

	ldap_msgfree(ldentry);

tree_fail_free_result:
	ldap_msgfree(result);

	ldap_unbind_s(ld);

tree_fail:
	ld = NULL;
	return err;
	}

int get_tree(struct entry *tree, int verbose) {
	struct entry *scan;

	if(get_subtree(tree, verbose)) return 1;
	
	if(!tree->subentries_head) return 0;

	for(scan = tree->subentries_head; scan; scan = scan->next)
		if(get_tree(scan, verbose))
			return 1;
	
	return 0;
	}
