(**
    Demo for VisualOberon. Shows a list of all available fonts.
    Note, that the fontlist may differ from the OS fontlist, because
    VisualOberon may do some filtering.
**)

MODULE FontList;

(*
    Demo for VisualOberon. Shows list of all available fonts.
    Copyright (C) 1998  Tim Teulings (rael@edge.ping.de)

    This file is part of VisualOberon.

    VisualOberon is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    VisualOberon is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with VisualOberon. If not, write to the Free Software
    Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VODisplay,
       G   := VOGUIObject,
       K   := VOKeyHandler,
       P   := VOPanel,
       T   := VOTable,
       TM  := VOTableModel,
              VOUsePrefs,
       W   := VOWindow,

       co  := IntStr,
              Err;

VAR
  d          : D.Display;
  w          : W.Window;
  keyHandler : K.KeyHandler;

  panel      : P.Panel;
  table      : T.Table;
  model      : TM.ASTableModel;


  msg2Exit   : D.Msg2Exit;

  PROCEDURE GetFontList(model : TM.ASTableModel);

 VAR
   fontList,
   font          : D.Font;
   fontCount,
   featureCount,
   x,pos         : LONGINT;
   features      : SET;
   colPos        : ARRAY MAX(SET) OF LONGINT;
   buffer        : ARRAY 20 OF CHAR;

  BEGIN
    fontList:=d.GetFontList();

    fontCount:=0;
    features:={};
    font:=fontList;
    WHILE font#NIL DO
      INC(fontCount);
      features:=features+font.features;
      font:=font.next;
    END;

    featureCount:=0;
    FOR x:=0 TO MAX(SET)-1 DO
      IF x IN features THEN
        INC(featureCount);
      END;
    END;

    model.SetSize(featureCount,fontCount);

    pos:=1;
    FOR x:=0 TO MAX(SET)-1 DO
      IF x IN features THEN
        CASE x OF
          D.fontName:        model.SetColumnString(pos,"Name"); model.SetColumnWidth(pos,30);
        | D.fontHeight:      model.SetColumnString(pos,"Height");
        | D.fontAscent:      model.SetColumnString(pos,"Ascent");
        | D.fontDescent:     model.SetColumnString(pos,"Descent");
        | D.fontPointHeight: model.SetColumnString(pos,"Height [points]");
        | D.fontHorizRes:    model.SetColumnString(pos,"Horiz. resolution");
        | D.fontVertRes:     model.SetColumnString(pos,"Vert. resolution");
        | D.fontAvWidth:     model.SetColumnString(pos,"Average width");
        | D.fontSetWidth:    model.SetColumnString(pos,"Width type");
        | D.fontSpacing:     model.SetColumnString(pos,"Spacing");
        | D.fontCharSet:     model.SetColumnString(pos,"Charset");
        ELSE
          model.SetColumnString(pos,"???");
        END;
        colPos[x]:=pos;
        INC(pos);
      END;
    END;

    pos:=1;
    font:=fontList;
    WHILE font#NIL DO
      FOR x:=0 TO MAX(SET)-1 DO
        IF x IN features THEN
          CASE x OF
            D.fontName:        model.SetString(colPos[x],pos,font.name^);
          | D.fontHeight:      co.IntToStr(font.height,buffer);
                               model.SetString(colPos[x],pos,buffer);
          | D.fontAscent:      co.IntToStr(font.ascent,buffer);
                               model.SetString(colPos[x],pos,buffer);
          | D.fontDescent:     co.IntToStr(font.descent,buffer);
                               model.SetString(colPos[x],pos,buffer);
          | D.fontPointHeight: co.IntToStr(font.pointHeight,buffer);
                               model.SetString(colPos[x],pos,buffer);
          | D.fontHorizRes:    co.IntToStr(font.horizRes,buffer);
                               model.SetString(colPos[x],pos,buffer);
          | D.fontVertRes:     co.IntToStr(font.vertRes,buffer);
                               model.SetString(colPos[x],pos,buffer);
          | D.fontAvWidth:     co.IntToStr(font.avWidth,buffer);
                               model.SetString(colPos[x],pos,buffer);
          | D.fontSetWidth:    CASE font.setWidth OF
                                 D.fontNormal:    model.SetString(colPos[x],pos,"normal");
                               | D.fontCondensed: model.SetString(colPos[x],pos,"condensed");
                               | D.fontNarrow:    model.SetString(colPos[x],pos,"narrow");
                               | D.fontDouble:    model.SetString(colPos[x],pos,"double");
                               ELSE
                                 model.SetString(colPos[x],pos,"???");
                               END;
          | D.fontSpacing:     CASE font.spacing OF
                                 D.fontFixed:         model.SetString(colPos[x],pos,"fixed");
                               |  D.fontProportional: model.SetString(colPos[x],pos,"proportional");
                               ELSE
                                 model.SetString(colPos[x],pos,"???");
                               END;
          | D.fontCharSet:     model.SetString(colPos[x],pos,font.charSet^);
          ELSE
          END;
        END;
      END;
      INC(pos);
      font:=font.next;
    END;

  END GetFontList;

BEGIN
  NEW(d);
  IF ~d.InitDisplay("FontList") THEN
    Err.String("Cannot open to display!"); Err.Ln;
    HALT(1);
  END;

  NEW(keyHandler);
  keyHandler.Init;

  NEW(panel);
  panel.Init;
  panel.Set(P.horizontal);
  panel.SetFlags({G.horizontalFlex,G.verticalFlex});

    NEW(model);
    model.Init;

    GetFontList(model);
(*    model2.SetSize(2,2);
    model2.SetColumnString(1,"\ebTestcolumn\en\n\esone");
    model2.SetColumnString(2,"\ebTestcolumn\en\n\estwo");
    model2.SetString(1,1,"(1,1)");
    model2.SetString(2,1,"(2,1)");
    model2.SetString(1,2,"(1,2)");
    model2.SetString(2,2,"(2,2)");
*)
    NEW(table);
    table.Init;
    table.SetFlags({G.horizontalFlex,G.verticalFlex});
    table.SetWidth(G.sizeFontRel,100);
    table.SetHeight(G.sizeFontRel,50);
    table.SetModel(model);
    table.SetSelectionType(T.singleLineSelect);
    keyHandler.AddFocusObject(table);
  panel.Add(table);

  NEW(w);
  w.Init;
  w.SetDisplay(d);
  w.SetTop(panel);
  w.SetTitle("FontList");
  w.AddKeyHandler(keyHandler);
  NEW(msg2Exit);
  msg2Exit.destination:=d;
  w.AddHandler(msg2Exit,W.closeMsg);
  w.Open;

  d.Handler;

  d.Deinit;
END FontList.