(**
   A simple boolean button with a selected and an unselected state.
**)

MODULE VOBool;

(*
    Implements a bool gadget.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VODisplay,
       E  := VOEvent,
       G  := VOGUIObject,
       O  := VOObject,
       P  := VOPrefs,
       V  := VOVecImage,
       VM := VOValue;

CONST
  clickedMsg * = 0;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (P.PrefsDesc)
                 image* : LONGINT; (* the frame to use for the button *)
               END;

  Bool*     = POINTER TO BoolDesc;

 (**
   Implementation of a bool gadget.
 **)

  BoolDesc* = RECORD (G.GadgetDesc)
                  prefs*   : Prefs;
                  image    : V.VecImage;
                  model-   : VM.ValueModel;
                  tmpState : BOOLEAN;
                END;

  (* messages *)

  ClickedMsg*     = POINTER TO ClickedMsgDesc;

  (**
    The PressedMsg generated everytime the button get clicked.
  **)

  ClickedMsgDesc* = RECORD (O.MessageDesc)
                      value* : BOOLEAN;
                    END;


VAR
  prefs* : Prefs;

  PROCEDURE (b : Bool) Init*;

  BEGIN
    b.Init^;

    b.prefs:=prefs;

    INCL(b.flags,G.canFocus);

    b.model:=NIL;
    b.label:=NIL;

    NEW(b.image);
    b.image.Init;
    b.image.SetFlags({G.horizontalFlex,G.verticalFlex});
  END Init;

  PROCEDURE (b : Bool) IsIn(x,y : LONGINT):BOOLEAN;

  BEGIN
    RETURN b.PointIsIn(x,y) OR ((b.label#NIL) & (b.label.PointIsIn(x,y)));
  END IsIn;


  (**
    Assign a bool model to the bool-gadget. The bool gadget
    will up to now always display the value of the bool model.
  **)

  PROCEDURE (b : Bool) SetModel*(model : VM.ValueModel);

  BEGIN
    IF b.model#NIL THEN
      b.UnattachModel(b.model);
    END;
    b.model:=model;
    b.AttachModel(model);
  END SetModel;

  PROCEDURE (b : Bool) CalcSize*(display : D.Display);

  BEGIN
    (*
      If the label has a custom way for displaying a focus frame
      we delegate the focus displaying to the label.
    *)
    IF (b.label#NIL) & ~b.label.StdFocus() & b.MayFocus() THEN
      EXCL(b.flags,G.stdFocus);
      INCL(b.label.flags,G.mayFocus);
    END;

    b.image.Set(b.prefs.image);
    b.image.CalcSize(display);
    b.width:=b.image.width;
    b.height:=b.image.height;
    b.minWidth:=b.image.minWidth;
    b.minHeight:=b.image.minHeight;

    b.CalcSize^(display);
  END CalcSize;

  PROCEDURE (b : Bool) GetFocus*(event : E.Event):G.Object;

  BEGIN
    IF ~b.visible OR b.disabled OR (b.model=NIL) THEN
      RETURN NIL;
    END;

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseDown)
      & b.IsIn(event.x,event.y)
      & (event.qualifier={}) & (event.button=E.button1) THEN
        b.tmpState:=b.model.GetBoolean();
        b.model.ToggleBoolean;
        RETURN b;
      END;
    ELSE
    END;
    RETURN NIL;
  END GetFocus;

  PROCEDURE (b : Bool) HandleEvent*(event : E.Event):BOOLEAN;

  VAR
    clicked : ClickedMsg;

  BEGIN
    WITH
      event : E.MouseEvent DO
      IF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        IF b.IsIn(event.x,event.y) THEN
          b.model.SetBoolean(~b.tmpState);
          (* Action: State changed *)
          NEW(clicked);
          clicked.value:=b.model.GetBoolean();
          b.Send(clicked,clickedMsg);
        END;
        RETURN TRUE;
      END;
    | event : E.MotionEvent DO
      IF b.IsIn(event.x,event.y) THEN
        b.model.SetBoolean(~b.tmpState);
      ELSE
        b.model.SetBoolean(b.tmpState);
      END;
    ELSE
    END;

    RETURN FALSE;
  END HandleEvent;

  PROCEDURE (b : Bool) HandleFocusEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    keysym : LONGINT;

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      IF keysym=E.space THEN
        b.model.ToggleBoolean;
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleFocusEvent;


  PROCEDURE (b : Bool) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    b.Draw^(x,y,draw);

    b.image.Resize(b.width,b.height);

    IF (b.model#NIL) & b.model.GetBoolean() THEN
      draw.mode:={D.selected};
    ELSE
      draw.mode:={};
    END;

    IF b.disabled & (G.canDisable IN b.image.flags) THEN
      INCL(b.draw.mode,D.disabled);
    END;

    b.image.Draw(b.x,b.y,draw);

    draw.mode:={};

    IF b.disabled & ~(G.canDisable IN b.image.flags) THEN
      b.DrawDisabled;
    END;
  END Draw;

  (**
    Draw the keyboard focus.
  **)

  PROCEDURE (b : Bool) DrawFocus*;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (b.label#NIL) & ~b.label.StdFocus() THEN
      b.label.DrawFocus;
    ELSE
      (* Delegate drawing to the baseclass *)
      b.DrawFocus^;
    END;
  END DrawFocus;

  (**
    Hide the keyboard focus.
  **)

  PROCEDURE (b : Bool) HideFocus*;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (b.label#NIL) & ~b.label.StdFocus() THEN
      b.label.HideFocus;
    ELSE
      (* Delegate drawing to the baseclass *)
      b.HideFocus^;
    END;
  END HideFocus;

  PROCEDURE (b : Bool) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF b.visible THEN
      b.Redraw;
    END;
  END Resync;

  PROCEDURE (b : Bool) Receive*(message : O.Message);

  BEGIN
    WITH message : G.LabelClickedMsg DO
      IF b.model#NIL THEN
        b.model.ToggleBoolean;
      END;
    ELSE
    END;
  END Receive;

  PROCEDURE (b : Bool) Hide*;

  BEGIN
    IF b.visible THEN
      b.image.Hide;
      b.Hide^;
    END;
  END Hide;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.image:=V.checkmark;
  END Init;


BEGIN
  NEW(prefs);
  prefs.Init;

END VOBool.


