(**
  Baseclass for all displayable objects.

  TODO
  * Implement Image.AddaptWidthToHeight and Image.AddaptHeightToWidth

  * Implement Object.HandleKeyShortcut (similar to Object.GetPosObject)

  * Multicolumn listview (should be rather easy).

  * A DirectoryModel (inherits from ListModel) for directory browsing.

  * A TextModel and corresponding gadget. Can load data from a file.

  * A InfoModel and corresponding gadgets (inherits from TextModel?).

  * A Presenter gadget, that displays pages of text, images and
    a number of blend-over effects.
**)

MODULE VOGUIObject;

(*
    Basesclass for all graphical objects.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT D := VODisplay,
       E := VOEvent,
       O := VOObject,
       P := VOPrefs;

CONST

  (* values for Object.flags *)

  horizontalFlex * =  0; (* Object can change its size in horizontal direction              *)
  verticalFlex   * =  1; (* Object can change its size in vertical direction                *)
  noHighlight    * =  2; (* Object shouldn't highlight when selected                        *)
  stdFocus       * =  3; (* Object uses the std way of displ. focus, ie. Object does it all *)
  hasFocus       * =  4; (* The objects currently has the focus                             *)
  canFocus       * =  5; (* We can handle keyboard focus events                             *)
  mayFocus       * =  6; (* It is possible that the object must draw its focus frame, it should therfor
                           interpet this flag to reserve some display space etc... *)
  showFocus      * =  7; (* We can't handle focus events, but should display one. Likely to happend,
                           when focus displaying has been delegated from super object      *)
  handleSC       * =  8; (* Object can handle shortcuts                                     *)
  scAlways       * =  9; (* Can handle sc even when not visible (menus f.e.)                *)
  canDisable     * = 10; (* Object has special handling for drawing disabled *)
  inited         * = 11; (* CalcSize has been called at least one *)

  (* values for minXXX and maxXXX *)

  minSize = 0;
  maxSize = MAX(LONGINT)-4;   (* Because of hardcoded focusborderwidth of 2 than might be added *)

  (* Constants for Object.GetPosObject *)

  menuGadget * = 0; (* Each Object can have its private context-sensitive menu       *)
  helpGadget * = 1; (* Each Object can have its private context-sensitive helpobject *)
  dragGadget * = 2; (* An object you can drag data from *)

  (* Constants for the different modes for Object.SetWidth and Object.setHeight *)

  sizeGiven     * = 0; (* Use the size given by the object itself *)
  sizeFontRel   * = 1; (* The size is given in fontrelativ units (Display.spaceXXX) *)
  sizePixel     * = 2; (* The size is given in pixel, you should avoid this for all costs *)
  sizeScreenRel * = 3; (* The size is given in procent of screensize (1..100) *)
  sizeFontPRel  * = 4; (* The size is given in fontrelativ percents (Display.spaceXXX) sizeFontPRel(100)=sizeFontRel(1) *)

  (* Constants for different kind of alignments an object may support *)

  alignLeft      * = 0;
  alignCenter    * = 1;
  alignRight     * = 2;
  alignBound     * = 3;
  alignTop       * = 4;
  alignBottom    * = 5;

  alignmentCount * = 6;

  (* shortcut events states *)

  pressed  * = 0;
  released * = 1;
  canceled * = 2;

TYPE
  AlignmentName = ARRAY 15 OF CHAR;

  Background* = POINTER TO BackgroundDesc;

  Object*     = POINTER TO ObjectDesc;

  (**
    An object can have a background color or an background object that is responsible
    for drawing the given region in the background style. Background object could draw
    the background simply using the background color (though you can achive this b
    y simply setting the background color without using an background object),
    a regular pattern, a titled image, an image stretched to fit the whole object or
    something even more complicated.

    The apropiate method of filling the background will be handled the DrawBackground
    method. It will fill the given area in the backgrojund area or will delegate
    drawing to the background object - if available.

    source stores the orign of this Background. E.g., yo assign a background object
    to a button. The button will delegate the background to it label. However in some
    cases the background must not be rendered relative to the size of the label but
    still relative to the button.
  **)

  BackgroundDesc* = RECORD
                      source* : Object; (* The object, the background is background of.*)
                    END;

  (**
    The baseclass for all GUI objects. All GUI-objects do have messaging so the class derives
    itself from VOObject.MsgObject
  **)


  ObjectDesc* = RECORD (D.ObjectDesc)
                  next*,                             (* All objects a singlelinked *)
                  parent*              : Object;

                  (* Sizing stuff *)

                  x*,y*,                             (* The current bounds of the object *)
                  width*,height*,

                  minWidth*,minHeight*,              (* Its minimal and maximal bounds *)
                  maxWidth*,maxHeight* : LONGINT;

                  oX-,oY-,
                  oWidth-,oHeight-    : LONGINT;     (* The outer bounds of the object
                                                        Inner bounds + possible focus frame *)

                  oMinWidth*,
                  oMinHeight*,                       (* Its minimal and maximal bounds *)
                  oMaxWidth*,
                  oMaxHeight*          : LONGINT;

                  setWidth*,
                  setHeight*,
                  setWidthMode*,
                  setHeightMode*       : LONGINT;    (* Stored values of SetWidth/Height untill CalcSize gets called *)

                  setMWidth*,
                  setMHeight*,
                  setMWidthMode*,
                  setMHeightMode*      : LONGINT;    (* Stored values of SetMinWidth/Height untill CalcSize gets called *)

                  (* generalFlags *)

                  flags*               : SET;        (* Some special flags *)

                  (* visibility stuff *)

                  background-          : LONGINT;
                  backgroundObject-    : Background;
                  visible*,                          (* The object is currently visible, i.e. not hidden *)
                  disabled*            : BOOLEAN;    (* Every object can be disabled *)

                  (* objects *)

                  display*             : D.Display;  (* An object has a pointer to its display.
                                                        Valid after calling Object.CalcSize *)
                  draw*                : D.DrawInfo; (* A pointer to the handle for all drawing operations.
                                                        Valid after the first call to Object.Draw *)

                  menu                 : D.Window;   (* pointer the context sensitive menuobject *)
                  help                 : D.Window;   (* pointer the context sensitive helpobject *)

                  focus*               : Object;
                  label*               : Object;     (* An object can have a label *)
                END;


  Image*      = POINTER TO ImageDesc;  (* An image is a special kind of Object *)

  (**
    The baseclass for all images. Images do not have any event-handling.
  **)

  ImageDesc*  = RECORD (ObjectDesc)
                END;


  Gadget*     = POINTER TO GadgetDesc;

  (**
    The baseclass for all gadgets. Gadgets have methods for event handling, unlike images for example.
  **)

  GadgetDesc* = RECORD (ObjectDesc)
                END;


  Group*      = POINTER TO GroupDesc;

  (**
    An group is a special kind of Object. Since it is a collection of
    objects, it may have to behave like an gadget, so we inherit from
    Gadget.
    It collects a number of object and presents them as one object to
    the rest of the world.
  **)

  GroupDesc*  = RECORD (GadgetDesc)
                  list*,               (* A linked list of all members *)
                  last*  : Object;
                  count* : LONGINT;    (* The number of members *)
                END;

  Prefs*      = POINTER TO PrefsDesc;

  (**
    Special Preferences object for graphical objects.
  **)

  PrefsDesc*  = RECORD (P.PrefsDesc)
                  background*  : Background;
                END;


 (* -- messages -- *)

  LabelClickedMsg*     = POINTER TO LabelClickedMsgDesc;

  (**
    The LabelClickedMsg maybe generated when someone clicks on your label.
  **)

  LabelClickedMsgDesc* = RECORD (O.MessageDesc)
                         END;


VAR
  alignments- : ARRAY alignmentCount OF AlignmentName;


  (* Some tool-functions *)

  (**
     Returns the maximum of the two numbers given.
  **)

  PROCEDURE MaxLong*(a,b : LONGINT):LONGINT;

  BEGIN
    IF a>=b THEN
      RETURN a;
    ELSE
      RETURN b;
    END;
  END MaxLong;

  (**
     Returns the minumum of the two numbers given.
  **)

  PROCEDURE MinLong*(a,b : LONGINT):LONGINT;

  BEGIN
    IF a<=b THEN
      RETURN a;
    ELSE
      RETURN b;
    END;
  END MinLong;

  (**
     Modifies @var{x} so that i is in the range
     between @var{a} and @var{b} including.
  **)

  PROCEDURE RoundRange*(x,a,b : LONGINT):LONGINT;

  BEGIN
    IF x<a THEN
      RETURN a;
    ELSIF x>b THEN
      RETURN b;
    ELSE
      RETURN x;
    END;
  END RoundRange;

  (* -------------------------------------------- *)

  (**
    Initilalize the object with the display.
  **)

  PROCEDURE (b : Background) CalcSize*(display : D.Display);

  BEGIN
  END CalcSize;

  (**
    Draw the background. The baseclass will simply fill the given area
    with the defined object background color.
  **)

  PROCEDURE (b : Background) Draw*(object : Object; x,y,w,h : LONGINT);

  BEGIN
    object.draw.PushForeground(object.background);
    object.draw.FillRectangle(x,y,w,h);
    object.draw.PopForeground;
  END Draw;

  (**
    Return a copy of the current background object.
  **)

  PROCEDURE (b : Background) Copy*():Background;

  VAR
    copy : Background;

  BEGIN
    NEW(copy);
    copy^:=b^;

    RETURN copy;
  END Copy;

  (**
    Free internal data.
  **)

  PROCEDURE (b : Background) Free*;

  BEGIN
  END Free;

  (* -------------------------------------------- *)

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.background:=NIL;
  END Init;

  PROCEDURE (p : Prefs) Free*;

  BEGIN
    IF p.background#NIL THEN
      p.background.Free;
      p.background:=NIL;
    END;

    p.Free^;
  END Free;

  (* -------------------------------------------- *)

  (**
    Every object needs an initialiser.

    If you provide an Init-method for an object you must first cann the
    Init-method of the baseclass.

    Note, that this does not hinder you ro provide function-like initializer, too.
    They just have to call this method.
  **)

  PROCEDURE (o : Object) Init*;

  BEGIN
    o.x:=0;
    o.y:=0;

    o.oX:=0;
    o.oY:=0;

    o.width:=0;
    o.height:=0;

    o.oWidth:=0;
    o.oHeight:=0;

    o.next:=NIL;

    o.parent:=NIL;

    o.minWidth:=minSize;
    o.minHeight:=minSize;
    o.maxWidth:=maxSize;
    o.maxHeight:=maxSize;

    o.oMinWidth:=minSize;
    o.oMinHeight:=minSize;
    o.oMaxWidth:=maxSize;
    o.oMaxHeight:=maxSize;


    o.setWidthMode:=sizeGiven;
    o.setHeightMode:=sizeGiven;
    o.setMWidthMode:=sizeGiven;
    o.setMHeightMode:=sizeGiven;

    o.flags:={stdFocus};
    o.visible:=FALSE;

    o.focus:=NIL;
    o.label:=NIL;

    o.background:=D.backgroundColor;
    o.backgroundObject:=NIL;
  END Init;

  (**
    Via this functions you can add the given flags to the
    flags already set.
  **)

  PROCEDURE (o : Object) SetFlags*(flags : SET);

  BEGIN
    o.flags:=o.flags+flags;
  END SetFlags;

  (**
    Via this functions you can remove the given flags from the
    flags already set.
  **)

  PROCEDURE (o : Object) RemoveFlags*(flags : SET);

  BEGIN
    o.flags:=o.flags-flags;
  END RemoveFlags;

  (**
    Via this functions you can clear all flags.

    NOTE
    This is a dangerous operations. There are many internal flags used by
    VO and it is likely that some things will not operate the way you
    want them to. Use removeFlags instead whenever possible.
  **)

  PROCEDURE (o : Object) ClearFlags*;

  BEGIN
    o.flags:={};
  END ClearFlags;

  (**
    Set the width of the object.
    "Mode" defines the interpretation of the given value.
  **)

  PROCEDURE (o : Object) SetWidth*(mode, value : LONGINT);

  BEGIN
    o.setWidth:=value;
    o.setWidthMode:=mode;
  END SetWidth;

  (**
    Set the height of the object.
    "Mode" defines the interpretation of the given value.
  **)

  PROCEDURE (o : Object) SetHeight*(mode, value : LONGINT);

  BEGIN
    o.setHeight:=value;
    o.setHeightMode:=mode;
  END SetHeight;

  (**
    Set the minWidth of the object.
    "Mode" defines the interpretation of the given value.
  **)

  PROCEDURE (o : Object) SetMinWidth*(mode, value : LONGINT);

  BEGIN
    o.setMWidth:=value;
    o.setMWidthMode:=mode;
  END SetMinWidth;

  (**
    Set the minHeight of the object.
    "Mode" defines the interpretation of the given value.
  **)

  PROCEDURE (o : Object) SetMinHeight*(mode, value : LONGINT);

  BEGIN
    o.setMHeight:=value;
    o.setMHeightMode:=mode;
  END SetMinHeight;

  (**
    Set the background color for the object. Behaviour for complex/group
    objects is undefined.
  **)

  PROCEDURE (o : Object) SetBackground*(backgroundColor : LONGINT);

  BEGIN
    o.background:=backgroundColor;
    o.backgroundObject:=NIL;
  END SetBackground;

  (**
    Set the background object for the object. Behaviour for complex/group
    objects is undefined.
  **)

  PROCEDURE (o : Object) SetBackgroundObject*(background : Background);

  BEGIN
    o.backgroundObject:=background;
  END SetBackgroundObject;

  PROCEDURE (o : Object) CopyBackground*(destination : Object);

  BEGIN
    ASSERT(destination#NIL);

    destination.backgroundObject:=o.backgroundObject;
    destination.background:=o.background;
  END CopyBackground;

  (**
    Assign a label to the object. Some object may support some
    tricky operations with their label. A checkbox f.e. may delegate its
    focus frame to its label.
  **)

  PROCEDURE (o : Object) SetLabelObject*(label : Object);

  BEGIN
    o.label:=label;
  END SetLabelObject;

  (**
    Assign an window that will displayed as tooltip. The baseclass will store
    the object within a private variable and will return it when GetHelpObject gets
    called.

    NOTE
    Special objects may overload this method this implement different behaviour.
    They must then also overload GetPosObject.
  **)

  PROCEDURE(o : Object) SetHelpObject*(helpObject : D.Window);

  BEGIN
    o.help:=helpObject;
  END SetHelpObject;

  (**
    Returns the help window for this object.

    NOTE
    Special objects may overload this method to f.e. implement dynamic tooltips.
    They must then also overload GetPosObject.
  **)

  PROCEDURE(o : Object) GetHelpObject*():D.Window;

  BEGIN
    RETURN o.help;
  END GetHelpObject;

  (**
    Assign a window that will displayed as menu. The baseclass will store
    the object within a private variable and will return it when GetMenuObject gets
    called.

    NOTE
    Special objects may overload this method this implement different behaviour.
    They must then also overload GetPosObject.
  **)

  PROCEDURE(o : Object) SetMenuObject*(menuObject : D.Window);

  BEGIN
    o.menu:=menuObject;
  END SetMenuObject;

  (**
    Returns the menu window for this object.

    NOTE
    Special objects may overload this method to f.e. implement dynamic menues.
    They must then also overload GetPosObject.
  **)

  PROCEDURE(o : Object) GetMenuObject*():D.Window;

  BEGIN
    RETURN o.menu;
  END GetMenuObject;

  (**
    Returns TRUE, when the object displays itself using the standard functionality
    of displaying focus frameing, e.g. stdFocus
    is in Object.flags.
  **)

  PROCEDURE (o : Object) StdFocus*():BOOLEAN;

  BEGIN
    RETURN stdFocus IN o.flags;
  END StdFocus;

  (**
    Returns TRUE, when the object has the focus, e.g. hasFocus
    is in Object.flags.
  **)

  PROCEDURE (o : Object) HasFocus*():BOOLEAN;

  BEGIN
    RETURN hasFocus IN o.flags;
  END HasFocus;

  (**
    Returns TRUE, when the object can handle keyboard focus events,a e.g. canFocus
    is in Object.flags.
  **)

  PROCEDURE (o : Object) CanFocus*():BOOLEAN;

  BEGIN
    RETURN canFocus IN o.flags;
  END CanFocus;

  (**
    Returns TRUE, when the object may show its focusframe,a e.g. mayFocus
    is in Object.flags.
  **)

  PROCEDURE (o : Object) MayFocus*():BOOLEAN;

  BEGIN
    RETURN mayFocus IN o.flags;
  END MayFocus;

  (**
    Returns TRUE, when the object should show its focusframe,a e.g. showFocus
    is in Object.flags.
  **)

  PROCEDURE (o : Object) ShowFocus*():BOOLEAN;

  BEGIN
    RETURN showFocus IN o.flags;
  END ShowFocus;

  (**
    Returns TRUE, when the object should display its focusframe,a e.g. showFocus
    or hasFocus is in Object.flags.
  **)

  PROCEDURE (o : Object) DisplayFocus*():BOOLEAN;

  BEGIN
    RETURN (hasFocus IN o.flags) OR (showFocus IN o.flags);
  END DisplayFocus;

  (**
    This methods returns a size calculated from the given value and type in respect with
    the given min and maximum ranges. If type=sizeGiven or unknown, the dfault value is returned.
  **)

  PROCEDURE (o : Object) GetSize*(value, type, min, max, default : LONGINT; horiz : BOOLEAN; display : D.Display):LONGINT;

  BEGIN
    CASE type OF
      sizeFontRel:
        IF horiz THEN
          RETURN RoundRange(value*display.spaceWidth,min,max);
        ELSE
          RETURN RoundRange(value*display.spaceHeight,min,max);
        END;
    | sizePixel:
          RETURN RoundRange(value,min,max);
    | sizeScreenRel:
        IF horiz THEN
          RETURN RoundRange((value*display.scrWidth) DIV 100,min,max);
        ELSE
          RETURN RoundRange((value*display.scrHeight) DIV 100,min,max);
        END;
    | sizeFontPRel:
        IF horiz THEN
          RETURN RoundRange((value*display.spaceWidth) DIV 100,min,max);
        ELSE
          RETURN RoundRange((value*display.spaceHeight) DIV 100,min,max);
        END;
    ELSE
      RETURN default;
    END;
  END GetSize;

  (**
    Tell the object it should calculate its size depending on its current settings

    This method normally gets calleed once before the first call of ObjectDraw.
    The window calls this method for its top object before it opens itself
    and draws its content. The top object call this method automatically for all
    its children.

    If this method get called for you object you normally have to simly
    propagate it to all your children and set the values for minWidth,
    minHeight, width and height. The setting of the max-values is optional,
    the defaultvalue for them is MAX(LONGINT).

    After this your *must* call this method of your baseclass.
  **)

  PROCEDURE (o : Object) CalcSize*(display : D.Display);

  BEGIN

    (* TODO: clean up *)

    o.display:=display;

    o.minWidth:=o.GetSize(o.setMWidth,o.setMWidthMode,o.minWidth,o.maxWidth,o.minWidth,TRUE,display);
    o.minHeight:=o.GetSize(o.setMHeight,o.setMHeightMode,o.minHeight,o.maxHeight,o.minHeight,FALSE,display);

    o.oMinWidth:=o.minWidth;
    o.oMinHeight:=o.minHeight;

    o.oMaxWidth:=o.maxWidth;
    o.oMaxHeight:=o.maxHeight;

    o.width:=RoundRange(o.width,o.minWidth,o.maxWidth);
    o.height:=RoundRange(o.height,o.minHeight,o.maxHeight);

    o.width:=o.GetSize(o.setWidth,o.setWidthMode,o.minWidth,o.maxWidth,o.width,TRUE,display);
    o.height:=o.GetSize(o.setHeight,o.setHeightMode,o.minHeight,o.maxHeight,o.height,FALSE,display);

    o.oWidth:=o.width;
    o.oHeight:=o.height;
    IF o.MayFocus() & o.StdFocus() THEN
      INC(o.oWidth,2*display.focusBorder);
      INC(o.oHeight,2*display.focusBorder);

      INC(o.oMinWidth,2*display.focusBorder);
      INC(o.oMinHeight,2*display.focusBorder);
      INC(o.oMaxWidth,2*display.focusBorder);
      INC(o.oMaxHeight,2*display.focusBorder);
    END;

    IF ~(horizontalFlex IN o.flags) THEN
      o.oMinWidth:=o.oWidth;
      o.minWidth:=o.width;
      o.oMaxWidth:=o.oWidth;
      o.maxWidth:=o.width;
    END;
    IF ~(verticalFlex IN o.flags) THEN
      o.oMinHeight:=o.oHeight;
      o.minHeight:=o.height;
      o.oMaxHeight:=o.oHeight;
      o.maxHeight:=o.height;
    END;

    INCL(o.flags,inited);

    IF o.backgroundObject#NIL THEN
      o.backgroundObject.CalcSize(display);
    END;
  END CalcSize;

  (**
    Returns true, if the object can grow in the stated direction

    An object is resizable, if

    * It hase the xxxFlex-flag set for that direction

    * And if its current size is smaller than it maximal size

  **)

  PROCEDURE (o : Object) CanGrow*(horiz : BOOLEAN):BOOLEAN;

  BEGIN
    IF horiz THEN
      IF horizontalFlex IN o.flags THEN
        RETURN o.width<o.maxWidth;
      ELSE
        RETURN FALSE;
      END;
    ELSE
      IF verticalFlex IN o.flags THEN
        RETURN o.height<o.maxHeight;
      ELSE
        RETURN FALSE;
      END;
    END;
  END CanGrow;

  (**
    Same as Object.CanGrow only for shrinking
  **)

  PROCEDURE (o : Object) CanShrink*(horiz : BOOLEAN):BOOLEAN;

  BEGIN
    IF horiz THEN
      IF horizontalFlex IN o.flags THEN
        RETURN o.width>o.minWidth;
      ELSE
        RETURN FALSE;
      END;
    ELSE
      IF verticalFlex IN o.flags THEN
        RETURN o.height>o.minHeight;
      ELSE
        RETURN FALSE;
      END;
    END;
  END CanShrink;

  (**
    This calls Object.CanGrow or Object.Can.Shrink depending on its options
  **)

  PROCEDURE (o : Object) CanResize*(grow, horiz : BOOLEAN):BOOLEAN;

  BEGIN
    IF grow THEN
      RETURN o.CanGrow(horiz);
    ELSE
      RETURN o.CanShrink(horiz);
    END;
  END CanResize;

  (**
    Returns TRUE if you can drag data from the object.
  **)

  PROCEDURE (o : Object) CanDrag*():BOOLEAN;

  BEGIN
    RETURN FALSE;
  END CanDrag;

  (**
    This function tries to resize the object to the given measurement

    Note, that the object may not have the giving size after calling since
    it have the follow the settings for o.flags and it maximal size

    Normally the features of this implementation are enought.
    But you can overload it (At the moment none of the classes does this).

  **)

  PROCEDURE (o : Object) Resize*(width,height : LONGINT);

  BEGIN
    IF o.MayFocus() & o.StdFocus() THEN
      IF (horizontalFlex IN o.flags) & (width>=0) THEN
        o.oWidth:=MaxLong(width,o.oMinWidth);
        o.width:=o.oWidth-2*o.display.focusBorder;
      END;
      IF (verticalFlex IN o.flags) & (height>=0) THEN
        o.oHeight:=MaxLong(height,o.oMinHeight);
        o.height:=o.oHeight-2*o.display.focusBorder;
      END;
    ELSE
      IF (horizontalFlex IN o.flags) & (width>=0) THEN
        o.width:=MaxLong(width,o.minWidth);
        o.oWidth:=o.width;
      END;
      IF (verticalFlex IN o.flags) & (height>=0) THEN
        o.height:=MaxLong(height,o.minHeight);
        o.oHeight:=o.height;
      END;
    END;
  END Resize;

  (**
    Tells the object to draw itself at the given position and with the mode
    set in draw.mode.

    You can igonore modes you do not support and simply assume mode=normal.

    You *must* call the Draw-method of your superclass first!
  **)

  PROCEDURE (o : Object) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    o.visible:=TRUE;

    o.x:=x;
    o.y:=y;

    o.oX:=o.x;
    o.oY:=o.y;

    IF o.MayFocus() & o.StdFocus() THEN
      INC(o.x,o.display.focusBorder);
      INC(o.y,o.display.focusBorder);
      IF o.DisplayFocus() & (o.focus#NIL) THEN
        o.focus.Resize(o.oWidth,o.oHeight);
        o.focus.Draw(o.oX,o.oY,o.draw);
      END;
    END;
    o.draw:=draw;
  END Draw;

  (**
    Tells the object to redraw itself
    You normally do not need to overload this method.
  **)

  PROCEDURE (o : Object) Redraw*;

  BEGIN
    IF o.visible THEN
      o.Draw(o.oX,o.oY,o.draw);
    END;
  END Redraw;

  (**
    Returns TRUE if the two areas intersect, else FALSE.
  **)

  PROCEDURE (o : Object) Intersect*(x,y,w,h : LONGINT):BOOLEAN;

  BEGIN
    RETURN ~(
       (y+h-1<o.y) (* above *)
    OR (y>o.y+o.height-1) (* under *)
    OR (x+w-1<o.x) (* left *)
    OR (x>o.x+o.width-1)); (* right *)
  END Intersect;

  PROCEDURE (o : Object) RestrictToBounds*(VAR x,y,w,h : LONGINT);

  BEGIN
    IF x<o.x THEN
      DEC(w,o.x-x);
      x:=o.x;
    END;
    IF x>=o.x+o.width THEN
      x:=o.x+o.width-1;
      w:=0;
    END;

    IF y<o.y THEN
      DEC(h,o.y-y);
      y:=o.y;
    END;
    IF y>=o.y+o.height THEN
      y:=o.y+o.height-1;
      h:=0;
    END;

    IF x+w>o.x+o.width-1 THEN
      w:=o.width-(x-o.x);
    END;

    IF y+h>o.y+o.height-1 THEN
      h:=o.height-(y-o.y);
    END;
  END RestrictToBounds;

  (**
    This method gets called by the display (or by the super object, which
    in turn got called by the display), when it has to get refreshed
    because of exposure. The exposed region is given as parameter and
    the object should try to refresh only that parts within the given
    area.

    The baseclass simply calls Object.Redraw when Object.IsIn return TRUE
    and the object is visible.
  **)

  PROCEDURE (o : Object) Refresh*(x,y,w,h : LONGINT);

  BEGIN
    IF o.visible & o.Intersect(x,y,w,h) THEN
      o.Redraw;
    END;
  END Refresh;

  (**
    This is just a convinience function. Call this if your are a gadget and want to
    show yourself disabled. It draws a pattern over the bound of the gadget.

    Not that this is just a proposal. You can of cause use other methods to show yourself
    disabled, but keep some style guide in mind.
  **)

  PROCEDURE (o : Object) DrawDisabled*;

  BEGIN
    o.draw.PushForeground(D.disabledColor);
    o.draw.PushPattern(D.disablePattern,D.disableWidth,D.disableHeight,D.fgPattern);
    o.draw.FillRectangle(o.x,o.y,o.width,o.height);
    o.draw.PopPattern;
    o.draw.PopForeground;
  END DrawDisabled;

  (**
    This is just a convinience function. Draws the given rectangle in the
    background color of the object.
  **)

  PROCEDURE (o : Object) DrawBackground*(x,y,w,h : LONGINT);

  BEGIN
    IF o.backgroundObject#NIL THEN
      o.backgroundObject.Draw(o,x,y,w,h);
    ELSE
      o.draw.PushForeground(o.background);
      o.draw.FillRectangle(x,y,w,h);
      o.draw.PopForeground;
    END;
  END DrawBackground;

  (**
    This is just a convinience function. Call this if your are an object and want to
    hide yourself. It just draw a rectangle in the backgroundcolor of the bounds of the
    gadget.

    Not that this is just a proposal. You can of cause use other methods to hide yourself.
  **)

  PROCEDURE (o : Object) DrawHide*;

  BEGIN
    o.draw.PushForeground(D.backgroundColor);
    o.draw.FillRectangle(o.x,o.y,o.width,o.height);
    o.draw.PopForeground;
  END DrawHide;


  (**
    Disable the object.
    The object should show it's disabled state via it's
    visual apperance.
  **)

  PROCEDURE (o : Object) Disable*(disable : BOOLEAN);

  BEGIN
    IF o.disabled#disable THEN
      o.disabled:=disable;
      o.Redraw;
    END;

    (* TODO: Loose the focus *)

  END Disable;

  (**
    Tell the object to hide itself.
    The space of the object should have the backgroundcolor after hiding.

    The method of the baseclass simply sets Object.visible to FALSE. If you
    implement your own object, you must overload this methods and hide your
    object (you may use Object.DrawHide for this). After that call
    Object.Hide of your baseclass. Since Object.Hide may have other purposes
    in the future, too.

    NOTE
    If you use other objects for your drawing you have to overload this
    method and call Hide for each of them to get Object.visible correct.
  **)

  PROCEDURE (o : Object) Hide*;

  BEGIN
    o.visible:=FALSE;

    (* Tell the windw, that the focus object hides *)
    IF o.HasFocus() THEN
      o.draw.vWindow.FocusObjectHides;
    END;
  END Hide;

  (**
    This method gets called when the window thinks, that the
    object should show somekind of keyboardfocus. This will
    happen, when the object gets the focus, or when the
    focus has to be refreshed because of a window refresh.

    NOTE
    You can use the supplied focusFrame but you need not to do.
    The baseclass does nothing.
  **)

  PROCEDURE (o : Object) DrawFocus*;

  BEGIN
    IF o.focus#NIL THEN
      o.focus.Resize(o.oWidth,o.oHeight);
      o.focus.Draw(o.oX,o.oY,o.draw);
    END;
  END DrawFocus;

  (**
    This method gets called when the window thinks, that the
    object should hide the keyboardfocus. This happens when
    the pbject loses the focus.
  **)

  PROCEDURE (o : Object) HideFocus*;

  BEGIN
    IF o.focus#NIL THEN
      o.focus.Hide;
    END;
  END HideFocus;

  (**
    Ask the window to change the focus to the next entry.
    This is usefull f.e. for stringgadgets. If you enter return, the
    gadget should get deactivated and the focus should change to the
    next element.

    NOTE
    The object must have the focus, otherwise nothing will happen.
  **)

  PROCEDURE (o : Object) LeaveFocus*;

  BEGIN
    IF o.HasFocus() THEN
      o.draw.vWindow.FocusNext;
    END;
  END LeaveFocus;

  (**
    Ask the object, if the given point it in its bounds.
    We also check if the pointer is within the bounds of the
    label of the object, if the object has one.
  **)

  PROCEDURE (o : Object) PointIsIn*(x,y : LONGINT):BOOLEAN;

  BEGIN
    IF ~o.visible THEN
      RETURN FALSE;
    END;

    RETURN (x>=o.x) & (y>=o.y) & (x<=o.x+o.oWidth-1) & (y<=o.y+o.oHeight-1);
  END PointIsIn;

  (**
    Tells if the mouse pointer is currently ober the object.
    We also check if the mouse pointer is within the bounds of the
    label of the object, if the object has one.
  **)

  PROCEDURE (o : Object) MouseIsIn*():BOOLEAN;

  VAR
    rx,ry,
    wx,wy  : LONGINT;

  BEGIN
    IF ~o.visible OR o.disabled OR (o.draw=NIL) THEN
      RETURN FALSE;
    ELSE
      o.draw.vWindow.GetMousePos(rx,ry,wx,wy);
      RETURN (wx>=o.x) & (wx<=o.x+o.width-1) & (wy>=o.y) & (wy<=o.y+o.height-1);
    END;
  END MouseIsIn;

  (**
    Ask the object, if it wants the focus after inverstigating the given event

    GetFocus returns the object, that whants to have the focus or NIL.
  **)

  PROCEDURE (g : Object) GetFocus*(event : E.Event):Object;

  BEGIN
    RETURN NIL;
  END GetFocus;

  (**
    After getting the focus with GetFocus you will get all
    events via this function.

    return TRUE if you are finished with eventhandling.
  **)

  PROCEDURE ( g : Object) HandleEvent*(event : E.Event):BOOLEAN;

  BEGIN
    RETURN TRUE; (* Return 'TRUE' if you want to loose focus *)
  END HandleEvent;

  (**
    Get a pointer to the object, which is in the given bounds and has
    an menu or help-object attached. If there is no object, return NIL.

    Return also true, if the object is currently not visible.
  **)

  PROCEDURE (o : Object) GetPosObject*(x,y : LONGINT; type : LONGINT):Object;

  BEGIN
    IF o.visible THEN
      IF o.PointIsIn(x,y) THEN
        CASE type OF
          menuGadget:
            IF o.menu#NIL THEN
              RETURN o;
            END;
        | helpGadget:
            IF o.help#NIL THEN
              RETURN o;
            END;
        | dragGadget:
            IF o.CanDrag() THEN
              RETURN o;
            END;
        END;
      END;
    END;
    RETURN NIL;
  END GetPosObject;

  (**
    Returns the object that coveres the given point and that supports
    drag and drop of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.
  **)

  PROCEDURE (o : Object) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):Object;

  BEGIN
    RETURN NIL;
  END GetDnDObject;

  (**
    Before freeing the object call this function.

    This method is not supported by the current objects.
  **)

  PROCEDURE (o : Object) Free*;
  BEGIN
  END Free;

  (* ----------------------------------- *)

  (**
    Called, when you got the keyboard focus.
  **)

  PROCEDURE (g : Gadget) CatchedFocus*;

  BEGIN
    INCL(g.flags,hasFocus);
    g.DrawFocus;
  END CatchedFocus;

  (**
    Call, when the keyboard focus has been taken away from you.
  **)

  PROCEDURE (g : Gadget) LostFocus*;

  BEGIN
    EXCL(g.flags,hasFocus);
    g.HideFocus;
  END LostFocus;

  (**
    This gets called, when you have the current keyboard focus and
    the users presses keys. You can expect to get only keyboard events.

    RESULT
    Return TRUE if you have handled the event, else FALSE.

    NOTE
    If you activtly have grabed the focus using Object.GetFocus and
    Object.handleEvent you will get the keyboardevents there. This
    function gets only called when you don't grabed the focus.
  **)

  PROCEDURE (g : Gadget) HandleFocusEvent*(event : E.KeyEvent):BOOLEAN;

  BEGIN
    RETURN FALSE;
  END HandleFocusEvent;

  (**
    Via this method you get informed about shortcut events you've
    registered via the keyboard handler of the window.

    PARAMETER
    id - the id you've registered your shortcut with.
    state - the state of the shortcut event. This may either be
    pressed, released or canceled.
  **)

  PROCEDURE (g : Gadget) HandleShortcutEvent*(id,state : LONGINT);

  BEGIN
  END HandleShortcutEvent;

  (* ----------------------------------- *)

  (**
    A group object is derived from Object. Its purpure is, to  collect a number
    of object and represent them to the ouer space as one object.
    It has some more methods and attributs.
  **)

  PROCEDURE (g : Group) Init*;

  BEGIN
    g.Init^;

    g.count:=0;
    g.list:=NIL;
  END Init;

  (**
    Disables all elements of the group object.
    Note, that inherited group objects may interpret beeing disabled
    different. VOTab f.e. disables itself.
  **)

  PROCEDURE (g : Group) Disable*(disabled : BOOLEAN);

  VAR
    help : Object;

  BEGIN
    help:=g.list;
    WHILE help#NIL DO
      help.Disable(disabled);
      help:=help.next;
    END;
  END Disable;

  (**
    Add a new Object to the group
    Removing objects is currently not supported

    Note that some group-objects have a more special functions for adding
    members. However Add should always be supported
  **)

  PROCEDURE (g : Group) Add*(object : Object);

  BEGIN
    IF g.list=NIL THEN
      g.list:=object;
    ELSE
      g.last.next:=object;
    END;
    g.last:=object;
    object.parent:=g;
    INC(g.count);
  END Add;

  (**
    The defaulthandler ask all members of the Group for the focus
  **)

  PROCEDURE (g : Group) GetFocus*(event : E.Event):Object;

  VAR
    object,
    help    : Object;

  BEGIN
    object:=g.list;
    WHILE object#NIL DO
      help:=object.GetFocus(event);
      IF help#NIL THEN
        RETURN help;
      END;
      object:=object.next;
    END;
    RETURN NIL;
  END GetFocus;

  (**
    The defaulthandler ask all members of the Group

    You have to overload this method if you do not use
    Group.Add and Group.list.
  **)

  PROCEDURE (g : Group) GetPosObject*(x,y : LONGINT; type : LONGINT):Object;

  VAR
    object,
    return  : Object;

  BEGIN
    object:=g.list;
    WHILE object#NIL DO
      return:=object.GetPosObject(x,y,type);
      IF return#NIL THEN
        RETURN return;
      END;
      object:=object.next;
    END;
    RETURN g.GetPosObject^(x,y,type);
  END GetPosObject;

  (**
    Returns the object that coveres the given point and that supports
    drag and drop of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.
  **)

  PROCEDURE (g : Group) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):Object;

  VAR
    object,
    return  : Object;

  BEGIN
    object:=g.list;
    WHILE object#NIL DO
      return:=object.GetDnDObject(x,y,drag);
      IF return#NIL THEN
        RETURN return;
      END;
      object:=object.next;
    END;
    RETURN g.GetDnDObject^(x,y,drag);
  END GetDnDObject;

  (**
    Refreshes the group object by refreshing all objects in the object list.
    The refresh retangle is preserved and will be handled to all subobjects.
  **)

  PROCEDURE (g : Group) Refresh*(x,y,w,h : LONGINT);

  VAR
    help : Object;

  BEGIN
    IF g.visible & g.Intersect(x,y,w,h) THEN
      g.RestrictToBounds(x,y,w,h);
      g.DrawBackground(x,y,w,h);

      help:=g.list;
      WHILE help#NIL DO
        help.Refresh(x,y,w,h);
        help:=help.next;
      END;
    END;
  END Refresh;

  (**
    Returns the alignment value corresponding to the the given string value.
    Returns -1, if the string contains no valid alignment name.
  **)

  PROCEDURE  GetAlignment*(name : ARRAY OF CHAR):LONGINT;

  VAR
    x : LONGINT;

  BEGIN
    FOR x:=0 TO alignmentCount-1 DO
      IF alignments[x]=name THEN
        RETURN x;
      END;
    END;
    RETURN -1;
  END GetAlignment;

BEGIN
  alignments[alignLeft]:="alignLeft";
  alignments[alignCenter]:="alignCenter";
  alignments[alignRight]:="alignRight";
  alignments[alignBound]:="alignBound";
  alignments[alignTop]:="alignTop";
  alignments[alignBottom]:="alignBottom";
END VOGUIObject.