(**
  This module implements loading and displaying of images.
  This implementation use the Imlib to handle loading and drawing.
**)

MODULE VOImage;

(*
    A class for loading and displaying images.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

<* IF LIB_HAVE_LIBIMLIB=TRUE THEN *>

IMPORT D   := VODisplay,
       G   := VOGUIObject,
       U   := VOUtil,

              Err,
       str := Strings,

       I   := Imlib,
              X11;

<*ELSE *>

IMPORT D   := VODisplay,
       G   := VOGUIObject,
       U   := VOUtil,

       str := Strings;

<* END *>

CONST
  plainMode  * = 0;
  buttonMode * = 1;

TYPE
  Image*     = POINTER TO ImageDesc;
  ImageDesc* = RECORD (G.ImageDesc)
                 filename- : U.Text;
<* IF LIB_HAVE_LIBIMLIB THEN *>
                 iImage    : I.ImagePtr;
                 image,
                 shape     : X11.Pixmap;
<* END *>
                 hAlign,
                 vAlign    : LONGINT;

                 mode      : LONGINT;
               END;

  (**
    Initialized the object. Must be called.
  **)

  PROCEDURE (i : Image) Init*;

  BEGIN
    i.Init^;

    i.filename:=NIL;

<* IF LIB_HAVE_LIBIMLIB THEN *>
    i.iImage:=NIL;
    i.image:=0;
    i.shape:=0;
<* END *>
    i.mode:=plainMode;
    i.hAlign:=G.alignCenter;
    i.vAlign:=G.alignCenter;
  END Init;

  (**
    Set the alignment of the image within the image area.
    See VOGUIObject for alignments supported.
  **)

  PROCEDURE (i : Image) SetAlignment*(horiz, vert : LONGINT);

  BEGIN
    i.hAlign:=horiz;
    i.vAlign:=vert;
  END SetAlignment;

  (**
    Set the filename of the image. Must be set before use.
  **)

  PROCEDURE (i : Image) SetFilename*(name : ARRAY OF CHAR);

  BEGIN
    NEW(i.filename,str.Length(name)+1);
    COPY(name,i.filename^);
  END SetFilename;

  (**
    Set the mode of the object.

    * plainMode
      Uses the fillColor when selecting.

    * buttonMode
      draw the image a little bit more left and down on select, so one gets
      a 3D feal when selecting-
  **)

  PROCEDURE (i : Image) SetMode*(mode : LONGINT);

  BEGIN
    i.mode:=mode;
  END SetMode;

<* IF LIB_HAVE_LIBIMLIB=TRUE THEN *>

  PROCEDURE (i : Image) FreePixmaps;

  BEGIN
    IF i.image#0 THEN
      I.Imlib_free_pixmap(i.display.imlib,i.image);
      i.image:=0;
      i.shape:=0;
    END;
  END FreePixmaps;

  PROCEDURE (i : Image) MakePixmap;

  BEGIN
    IF i.image=0 THEN
      IF I.Imlib_render(i.display.imlib,i.iImage,i.width,i.height)=0 THEN END;
      i.image:=I.Imlib_move_image(i.display.imlib,i.iImage);
      i.shape:=I.Imlib_move_mask(i.display.imlib,i.iImage);
    END;
  END MakePixmap;

  PROCEDURE (i : Image) Resize*(width,height : LONGINT);

  VAR
    oW,oH : LONGINT;

  BEGIN
    oW:=i.width;
    oH:=i.height;

    i.Resize^(width,height);

    IF (i.width#oW) OR (i.height#oH) THEN
      i.FreePixmaps;
    END;
  END Resize;

  PROCEDURE (i : Image) CalcSize*(display : D.Display);

  VAR
    window  : D.Window;

  BEGIN
    (*
      CalcSize may be called more than one but must not open the image twice,
      so we check f we already have been inited.
    *)

    IF ~(G.inited IN i.flags) THEN
      i.image:=0;
      i.shape:=0;

      i.iImage:=I.Imlib_load_image(display.imlib,i.filename^);

      IF i.iImage#NIL THEN
        i.width:=i.iImage.rgb_width;
        i.height:=i.iImage.rgb_height;

        IF i.mode=buttonMode THEN
          INC(i.width);
          INC(i.height);
        END;

      ELSE
        Err.String("Error loading "); Err.String(i.filename^); Err.Ln;
        i.width:=2*display.spaceWidth;
        i.height:=2*display.spaceHeight;
      END;

      i.minWidth:=0;
      i.minHeight:=0;

      (*
        We register ourself to our parent window.
        This asures that XpmImage.Free gets called when the window
        get destroyed. So we can free our ressources.
      *)

      window:=display.GetNewWindow();
      IF window#NIL THEN
        window.AddFreeList(i);
      END;
    END;

    i.CalcSize^(display);
  END CalcSize;

  PROCEDURE (i : Image) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    a,b : LONGINT;

  BEGIN
    i.Draw^(x,y,draw);

(*    IF (D.selected IN draw.mode) & (i.mode=plainMode) & ~(G.noHighlight IN i.flags) THEN
      draw.PushForeground(D.fillColor);
      draw.FillRectangle(i.x,i.y,i.width,i.height);
    ELSE
      draw.PushForeground(D.backgroundColor);
      draw.FillRectangle(i.x,i.y,i.width,i.height);
    END;
    draw.PopForeground;*)

    IF i.iImage#NIL THEN
      i.MakePixmap;

      CASE i.hAlign OF
        G.alignLeft:
          a:=i.x;
      | G.alignCenter:
          a:=i.x+(i.width-i.iImage.width) DIV 2;
      | G.alignRight:
          a:=i.x+i.width-i.iImage.width;
      END;

      CASE i.vAlign OF
        G.alignTop:
          b:=i.y;
      | G.alignCenter:
          b:=i.y+(i.height-i.iImage.height) DIV 2;
      | G.alignRight:
          b:=i.y+i.height-i.iImage.height;
      END;

      IF (i.mode=buttonMode) & (D.selected IN draw.mode) THEN
        INC(a);
        INC(b);
      END;

      IF i.shape#0 THEN
        X11.XSetClipMask(draw.display.display,draw.gc,i.shape);
        X11.XSetClipOrigin(draw.display.display,draw.gc,a,b);
      END;

      X11.XCopyArea(draw.display.display,i.image,draw.window,draw.gc,
                    0,0,i.width,i.height,a,b);
      IF i.shape#0 THEN
        draw.ReinstallClip;
      END;
    END;
  END Draw;

  PROCEDURE (i : Image) DrawPart*(x,y,w,h,dx,dy : LONGINT; draw : D.DrawInfo);

  BEGIN
    IF i.iImage#NIL THEN
      i.MakePixmap;
      X11.XCopyArea(draw.display.display,i.image,draw.window,draw.gc,
                    x,y,w,h,dx,dy);
    END;
  END DrawPart;

  PROCEDURE (i : Image) DrawTiled*(x,y,w,h,dx,dy : LONGINT; draw : D.DrawInfo);

  BEGIN
    IF i.iImage#NIL THEN
      i.MakePixmap;
      X11.XSetTile(draw.display.display,draw.gc,i.image);
      X11.XSetTSOrigin(draw.display.display,draw.gc,dx,dy);
      X11.XSetFillStyle(draw.display.display,draw.gc,X11.FillTiled);
      X11.XFillRectangle(draw.display.display,draw.window,draw.gc,x,y,w,h);
      (* TODO: Restore correct fill mode *)
      X11.XSetFillStyle(draw.display.display,draw.gc,X11.FillSolid);
    END;
  END DrawTiled;

  PROCEDURE (i : Image) CopyToBitmap(x,y,w,h,dx,dy : LONGINT; bitmap : D.Bitmap);

  BEGIN
    IF i.iImage#NIL THEN
      i.MakePixmap;
      X11.XCopyArea(bitmap.draw.display.display,i.image,bitmap.pixmap,bitmap.draw.gc,
                    x,y,w,h,dx,dy);
    END;
  END CopyToBitmap;

  PROCEDURE (i : Image) Hide*;

  BEGIN
    IF i.visible THEN
      i.DrawHide;
      i.Hide^;
    END;
  END Hide;

  (**
    Must be called before image gets garbage collected
    or you possibly get memory leaks.
  **)

  PROCEDURE (i : Image) Free*;

  BEGIN
    i.FreePixmaps;
    IF i.iImage#NIL THEN
      I.Imlib_destroy_image(i.display.imlib,i.iImage);
      i.iImage:=NIL;
    END;
  END Free;

<* ELSE *>

  PROCEDURE (i : Image) CalcSize*(display : D.Display);

  BEGIN
    i.width:=2*display.spaceWidth;
    i.height:=2*display.spaceHeight;

    i.CalcSize^(display);
  END CalcSize;

  PROCEDURE (i : Image) DrawPart*(x,y,w,h,dx,dy : LONGINT; draw : D.DrawInfo);

  BEGIN
  END DrawPart;

  PROCEDURE (i : Image) DrawTiled*(x,y,w,h,dx,dy : LONGINT; draw : D.DrawInfo);

  BEGIN
  END DrawTiled;

  PROCEDURE (i : Image) CopyToBitmap(x,y,w,h,dx,dy : LONGINT; bitmap : D.Bitmap);

  BEGIN
  END CopyToBitmap;

  PROCEDURE (i : Image) Free*;

  BEGIN
  END Free;

<* END *>

END VOImage.