(**
   Baseclass for all objects. Does also define the derived class
   @code{MsgObject} which implements message communication between
   objects. @code{Model}, the baseclass for all models, is also
   implemented here.
**)

MODULE VOObject;

(*
    Baseclass for all objects in VisualOberon.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT BinaryRider,
       Kernel,
       Types;

(*
  This module is the basemodule for all VO-stuff.

  It defines some abstract baseclasses used for inter-object communication
  and baseclasses for the model-viewer paradigm.
*)

CONST
  (* persistent object identifiers *)
  obEOL     = -1;          (* end of linked list marker *)
  obHANDLER = obEOL-1;     (* handler descriptor marker *)
  obMODEL   = obHANDLER-1; (* model descriptor marker   *)

  broadcastMsg* = -1;
  everyMsg*     = -2;
  actionMsg*    = -3;

  customMsg*    = -4;

TYPE
  Object*           = POINTER TO ObjectDesc;
  MsgObject*        = POINTER TO MsgObjectDesc;
  Handler*          = POINTER TO HandlerDesc;
  Message*          = POINTER TO MessageDesc;
  ResyncMsg*        = POINTER TO ResyncMsgDesc;
  Notify*           = POINTER TO NotifyDesc;
  Action*           = POINTER TO ActionDesc;
  HandlerEntry      = POINTER TO HandlerEntryDesc;
  Model*            = POINTER TO ModelDesc;
  ModelEntry        = POINTER TO ModelEntryDesc;

  (**
     The abstract baseclass for all objects in VO
  **)

  ObjectDesc*       = RECORD
                        ref- : LONGINT; (* used to identify this persistent object *)
                      END;

  (**
     This is a second abstract class derived from Object.

     MsgObject can send and recieve messages.
  **)

  MsgObjectDesc*    = RECORD (ObjectDesc)
                        handlerList : HandlerEntry; (* A list of handlers, each message is send to *)
                        id-         : LONGINT;      (* All Objects have an id *)
                      END;

  (**
     All handlers for an object are listed with this class as node.
  **)

  HandlerEntryDesc  = RECORD
                        next    : HandlerEntry;
                        type    : LONGINT;      (* The type of message the handler listens to *)
                        handler : Handler;      (* the handler itself *)
                      END;
  (**
    All message sending between objects goes through handlers.
    Using a handler between the sender and reciever of an message allows
    us to change the messagetyp on the fly.

    This is a simple way to get objects freely communicate with each other
    without knowing the messages the reciever knows.

    You simply insert a handler between sender reciever that converts the
    sender message to the reciever messsage.
  **)

  HandlerDesc*      = RECORD
                        destination* : MsgObject; (* The destination object all messages by this handler
                                                     should be send to. You have to initialize it
                                                     before you do MsgObject.AddHandler. *)
                        sourceType*  : LONGINT;
                      END;

  (**
    The (abstract) baseclass for all messages.
  **)

  MessageDesc*      = RECORD
                        source* : MsgObject; (* The object the message was send from *)
                      END;

  (**
    A special message for prameterless actions (like for example a event that
    gets triggered when a button is pressed). You store the type of the
    action in the action memebr variable. Using action instead of a self-derived
    class while reduce code bloat and class count.
  **)

  ActionDesc*       = RECORD (MessageDesc)
                        action* : LONGINT;
                      END;

  (**
    This message is the abstract baseclass for all resync-messages.
    a model can send to its viewers when the contents of the
    model have changed.
  **)

  ResyncMsgDesc*    = RECORD
                      END;

  NotifyDesc*       = RECORD (ResyncMsgDesc)
                        notify* : LONGINT;
                      END;

  (**
    The list of objects (viewers) an model has uses this as node.
  **)

  ModelEntryDesc    = RECORD
                        next   : ModelEntry;
                        object : MsgObject;  (* We could als use a pointer to Object, but this makes more sense for me *)
                      END;

  (**
    The abstract baseclass for all models. A model is a container
    for an in any way designed datacollection.

    A Object (viewer) does not hold its values itself but has a pointer to
    a model. The user changes the value of the model and not the value
    of the object. The model then sends an ResyncMsg to all its viewers
    to make them update the representation of the datas the model holds.
  **)

  ModelDesc* = RECORD (MsgObjectDesc)    (* This offers interesting features *)
                 objectList : ModelEntry; (* The list of objects *)
                 lastObject : ModelEntry;
               END;

  RefList = POINTER TO RefListDesc;



  RefListDesc = RECORD
                  obj  : Object;
                  next : RefList
                END;

VAR
  refCnt : LONGINT;
  refList: RefList;  (* list of all objects *)

  (**
    This routine extracts previously-defined objects.
  **)

  PROCEDURE FindRef * (id: LONGINT) : Object;

  VAR
    ref: RefList;

  BEGIN
    ref:=refList;
    WHILE (ref#NIL) & (ABS(ref.obj.ref)#id) DO
      ref:=ref.next
    END;
    IF ref#NIL THEN
      RETURN ref.obj
    ELSE RETURN NIL
    END
  END FindRef;

  (**
    Each object has an initialisation routine.

    If an object initializes itself, it must call the init-method of the baseclass.

    Note, that this method does not prevent a class to implement initialize-functions,
    but they must then call this method.
  **)

  PROCEDURE (o : Object) Init*;

  BEGIN
    INC(refCnt);
    o.ref:=refCnt
  END Init;

  (**
    Each object has a load routine.

    The persistent data is read from the passed reader.  If a derived class overrides
    this method, it must call the super-method before restoring its internal state.
  **)

  PROCEDURE (o : Object) Load* (r: BinaryRider.Reader);

  BEGIN
    (* nothing to do for the base object *)
  END Load;

  (**
    Each object has a store routine.

    The persistent data is written to the passed writer.  If a derived class overrides
    this method, it must call the super-method before storing its internal state.
  **)

  PROCEDURE (o : Object) Store* (w: BinaryRider.Writer) : BOOLEAN;

  VAR
    type: Types.Type;

  BEGIN
    (* we always write at least the object reference id *)
    IF o.ref<0 THEN
      w.WriteNum(-o.ref);    (* reference id used to identify this object *)
      RETURN FALSE           (* no need to store the rest of the object *)
    ELSE
      w.WriteNum(o.ref);
      o.ref:=-o.ref;         (* negate ref id to indicate written object *)

      (* output the type & module names *)
      type:=Types.TypeOf(o);
      w.WriteString(type.module.name^);
      w.WriteString(type.name^);

      RETURN TRUE            (* rest of object needs to be stored *)
    END
  END Store;

  (**
    All object have some kind of finalizer method. Such a method is needed
    for OS-specific ressources like filehandles or GUI stuff that does not get
    freed automatically by garbage collection.

    Later versions of oo2c/VisualOberon may have an direct interface to
    the GC to realize finalisation, since then me must use a not that nice
    method: objects can register themself a a window. The window will call
    Free for all registered objects on deletion.
  **)

  PROCEDURE (o : Object) Free*;

  BEGIN
  END Free;

  (**
    This generic object creation routine will read type object type
    information from the stream, create the object, and initialize
    it's fields from the stream -- using the associated Load method.

    If the object already exists in memory, a pointer to that object
    will be returned with no initialization necessary.
  **)

  PROCEDURE Create* (r: BinaryRider.Reader) : Object;
  VAR obj: Object;
    ref: RefList;
    name: ARRAY 256 OF CHAR;
    mod: Kernel.Module;
    type: Types.Type;
    id: LONGINT;
  BEGIN
    r.ReadNum(id);  (* reference id for this object *)
    IF id=0 THEN RETURN NIL END;  (* this is a NIL object *)
    obj:=FindRef(id);
    IF obj=NIL THEN
      (* determine the module for the object *)
      r.ReadString(name);   (* module name *)
      mod:=Kernel.modules;
      WHILE (mod#NIL) & (mod.name^#name) DO mod:=mod.next END;
      IF mod=NIL THEN RETURN NIL END;

      (* determine the type of the object *)
      r.ReadString(name);   (* type name *)
      type:=Types.This(mod, name);
      IF type=NIL THEN RETURN NIL END;
      Types.NewObj(obj, type);  (* create an object of this type *)

      (* add object to refList *)
      IF refList=NIL THEN NEW(refList); refList.next:=NIL
      ELSE NEW(ref); ref.next:=refList; refList:=ref
      END;
      refList.obj:=obj; obj.ref:=id;
      obj.Load(r);   (* initialize object fields *)
      RETURN obj
    ELSE RETURN obj
    END
  END Create;


  (**
    MsgObject inherits the Init-function from Object.
  **)

  PROCEDURE (m : MsgObject) Init*;

  BEGIN
    m.Init^;
    m.handlerList:=NIL;
  END Init;

  (**
    We must have the ability to set the id of an object
  **)

  PROCEDURE (m : MsgObject) SetId*(id : LONGINT);

  BEGIN
    m.id:=id;
  END SetId;

  (**
    This method of an object gets called when someone sends a method
    to it.
  **)

  PROCEDURE (m : MsgObject) Receive*(message : Message);

  BEGIN
    (* nothing to do *)
  END Receive;

  (**
    You can add a handler to an object.

    The handler gets then called after the sender sends the object
    and before the receiver recieves it.

    type defines the type of message the handler wants to get called for.
    The objects have to suply constants for that.
  **)

  PROCEDURE (m : MsgObject) AddHandler*(handler : Handler; type : LONGINT);

  VAR
    hdlEntry : HandlerEntry;

  BEGIN
    NEW(hdlEntry);
    hdlEntry.type:=type;
    hdlEntry.handler:=handler;
    hdlEntry.next:=m.handlerList;
    m.handlerList:=hdlEntry;
  END AddHandler;

  (**
    You also can remove a handler anytime.
  **)

  PROCEDURE (m : MsgObject) RemoveHandler*(handler : Handler);

  VAR
    help,
    last : HandlerEntry;

  BEGIN
    IF (m.handlerList#NIL) & (m.handlerList.handler=handler) THEN
      m.handlerList:=m.handlerList.next;
      RETURN;
    END;

    help:=m.handlerList.next;
    last:=m.handlerList;
    WHILE (help#NIL) & (help.handler#handler) DO
      last:=help;
      help:=help.next;
    END;
    IF help#NIL THEN
      last.next:=help.next;
    END;
  END RemoveHandler;

  (**
    You can also tell the object to send msgs of a specific type
    simply to another object.

    This is done by simply adding a default handler.
  **)

  PROCEDURE (m : MsgObject) Forward*(type : LONGINT; destination : MsgObject);

  VAR
    handler : Handler;

  BEGIN
    NEW(handler);
    handler.destination:=destination;
    m.AddHandler(handler,type);
 END Forward;

  (**
    Load the MsgObject object state information.
  **)

  PROCEDURE (m : MsgObject) Load* (r: BinaryRider.Reader);
  VAR s: LONGINT; type: LONGINT; h: Handler; o: Object;
  BEGIN
    r.ReadNum(s);
    WHILE s=obHANDLER DO
      NEW(h);                       (* read the list of handlers *)
      r.ReadNum(type);
      r.ReadNum(h.sourceType);
      o:=Create(r); h.destination:=o(MsgObject);    (* create and initialize the object *)
      m.AddHandler(h, type);
      r.ReadNum(s)
    END;
    r.ReadNum(m.id)
  END Load;

  (**
    Store the MsgObject object state information.
  **)

  PROCEDURE (m : MsgObject) Store* (w: BinaryRider.Writer) : BOOLEAN;
  VAR h: HandlerEntry; ok: BOOLEAN;
  BEGIN
    IF m.Store^(w) THEN    (* we need to store this object as well *)
      h:=m.handlerList;
      WHILE h#NIL DO
        (* store the linked list of HandlerEntry *)
        w.WriteNum(obHANDLER);
        w.WriteNum(h.type);
        w.WriteNum(h.handler.sourceType);
        IF h.handler.destination#NIL THEN
          ok:=h.handler.destination.Store(w)
        ELSE w.WriteNum(0)  (* NIL *)
        END;
        h:=h.next
      END;
      w.WriteNum(obEOL);
      w.WriteNum(m.id);
      RETURN TRUE
    END;
    RETURN FALSE
  END Store;

  (**
    This function of the handler gets called after the sender sends the object
    and before the receiver recieves it. This gives you the possibility to
    change the type of the message. Just create the new message and return it.

    You don't have copy the destination field, that will be done by Handler.Send
    automatically.
  **)

  PROCEDURE (h : Handler) Convert*(message : Message):Message;

  BEGIN
    RETURN message;
  END Convert;

  (**
    This method gets called by an object for each handler which is interested
    in the message.
  **)

  PROCEDURE (h : Handler) Send*(message : Message);

  VAR
    newMsg : Message;

  BEGIN
    newMsg:=h.Convert(message);
    IF newMsg#NIL THEN
      newMsg.source:=message.source;
      IF h.destination#NIL THEN
        h.destination.Receive(newMsg);
      END;
    END;
  END Send;

  (**
    Call this method if you want to send a message with a given type.
  **)

  PROCEDURE (m : MsgObject) Send*(message : Message; type : LONGINT);

  VAR
    hdlEntry : HandlerEntry;

  BEGIN
    message.source:=m;
    hdlEntry:=m.handlerList;
    WHILE hdlEntry#NIL DO
      IF (type=broadcastMsg)
      OR (hdlEntry.type=type)
      OR (hdlEntry.type=everyMsg) THEN
        hdlEntry.handler.Send(message);
      END;
      hdlEntry:=hdlEntry.next;
    END;
  END Send;

  (**
    This method gets called when a model wants you to resync yourself with
    its contents.
  **)

  PROCEDURE (m : MsgObject) Resync*(model : Model; msg : ResyncMsg);

  BEGIN
  END Resync;

  (**
    A model must be inited.
  **)

  PROCEDURE (m : Model) Init*;

  BEGIN
    m.Init^;
    m.objectList:=NIL;
    m.lastObject:=NIL;
  END Init;

  (**
    Add an object (viewer) to a model
  **)

  PROCEDURE (m : Model) AddObject*(object : MsgObject);

  VAR
    entry : ModelEntry;

  BEGIN
    NEW(entry);
    entry.object:=object;
    entry.next:=NIL;
    IF m.objectList=NIL THEN
      m.objectList:=entry;
    ELSE
      m.lastObject.next:=entry;
    END;
    m.lastObject:=entry;
  END AddObject;

  (**
    Load the model state information.
  **)

  PROCEDURE (m : Model) Load* (r: BinaryRider.Reader);
  VAR s: LONGINT; msg: MsgObject; o: Object;
  BEGIN
    r.ReadNum(s);
    WHILE s=obMODEL DO
      o:=Create(r); msg:=o(MsgObject);
      IF msg#NIL THEN m.AddObject(msg) END;
      r.ReadNum(s)
    END
  END Load;

  (**
    Store the model state information.
  **)

  PROCEDURE (m : Model) Store* (w: BinaryRider.Writer) : BOOLEAN;
  VAR o: ModelEntry; ok: BOOLEAN;
  BEGIN
    IF m.Store^(w) THEN        (* we need to store this object as well *)
      o:=m.objectList;
      WHILE o#NIL DO
        (* store the linked list of ModelEntry *)
        w.WriteNum(obMODEL);
        IF o.object#NIL THEN ok:=o.object.Store(w)
        ELSE w.WriteNum(0)
        END;
        o:=o.next
      END;
      w.WriteNum(obEOL);
      RETURN TRUE
    END;
    RETURN FALSE
  END Store;

  (**
    Remove an object from an model (not yet implemented)
  **)

  PROCEDURE (m : Model) RemoveObject*(object : MsgObject);

  VAR
    help,
    last  : ModelEntry;

  BEGIN
    help:=m.objectList;
    last:=NIL;
    WHILE (help#NIL) & (help.object#object) DO
      last:=help;
      help:=help.next;
    END;
    IF help#NIL THEN
      IF last=NIL THEN
        m.objectList:=m.objectList.next;
      ELSE
        last.next:=help.next;
      END;
    END;
  END RemoveObject;

  (**
    A model should call this method with an optional resynmessage
    when you want your viewers to resync themselfs.
  **)

  PROCEDURE (m : Model) Notify*(msg : ResyncMsg);

  VAR
    entry : ModelEntry;

  BEGIN
    entry:=m.objectList;
    WHILE entry#NIL DO
      entry.object.Resync(m,msg);
      entry:=entry.next;
    END;
  END Notify;

  (**
    Deattach an object from the given model.
  **)

  PROCEDURE (m : MsgObject) UnattachModel*(model : Model);

  BEGIN
    model.RemoveObject(m);
  END UnattachModel;

  (**
    Use this function to attach an object to a model.
    Normaly you should not call this method directly, the
    object should offer special methods for this.
  **)

  PROCEDURE (m : MsgObject) AttachModel*(model : Model);

  BEGIN
    model.AddObject(m);
    m.Resync(model,NIL);
  END AttachModel;

BEGIN
  refCnt:=0;
  refList:=NIL;  (* each object needs a unique reference id *)
END VOObject.