(**
   A layouting class for grouping objects horizontaly or verticaly.
   Does some complex resizing operations to make the arrangement
   of its children look lice.
**)

MODULE VOPanel;

(*
    Implements a layouting group.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D := VODisplay,
       G := VOGUIObject,
       U := VOUtil;

CONST
  horizontal* = 0;
  vertical*   = 1;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
               END;


  Panel*     = POINTER TO PanelDesc;
  PanelDesc* = RECORD (G.GroupDesc)
                 prefs : Prefs;
                 type  : LONGINT;
               END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) SetPrefs(pa : Panel);

  BEGIN
    pa.prefs:=p;   (* We set the prefs *)

    IF p.background#NIL THEN
      pa.SetBackgroundObject(p.background.Copy());
      pa.backgroundObject.source:=pa;
    END;
  END SetPrefs;

  PROCEDURE (p : Panel) Init*;

  BEGIN
    p.Init^;

    prefs.SetPrefs(p);

    p.type:=vertical;
  END Init;

  (**
    Set the direction (horizontal or vertical) of the panel.
  **)

  PROCEDURE (p : Panel) Set*(type : LONGINT);

  BEGIN
    p.type:=type;
  END Set;

  PROCEDURE (p : Panel) CalcSize*(display : D.Display);

  VAR
    object : G.Object;

  BEGIN
    p.width:=0;
    p.height:=0;
    p.minWidth:=0;
    p.minHeight:=0;

    object:=p.list;
    WHILE object#NIL DO

      object.CalcSize(display);

      IF p.type=vertical THEN
        p.width:=G.MaxLong(p.width,object.oWidth);
        INC(p.height,object.oHeight);

        p.minWidth:=G.MaxLong(p.minWidth,object.oMinWidth);
        INC(p.minHeight,object.oMinHeight);
      ELSE
        INC(p.width,object.oWidth);
        p.height:=G.MaxLong(p.height,object.oHeight);

        INC(p.minWidth,object.oMinWidth);
        p.minHeight:=G.MaxLong(p.minHeight,object.oMinHeight);
      END;

      object:=object.next;
    END;
    p.CalcSize^(display);
  END CalcSize;

  PROCEDURE (p : Panel) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    size,
    count,
    old,
    pos    : LONGINT;
    object : G.Object;

  BEGIN
    p.Draw^(x,y,draw);

    IF p.type=horizontal THEN
      size:=0;
      object:=p.list;
      WHILE object#NIL DO
        object.Resize(object.oMinWidth,p.height);
        INC(size,object.oMinWidth);
        object:=object.next;
      END;
    ELSE (* vertical *)
      size:=0;
      object:=p.list;
      WHILE object#NIL DO
        object.Resize(p.width,object.oMinHeight);
        INC(size,object.oMinHeight);
        object:=object.next;
      END;
    END;

    IF p.type=horizontal THEN
      LOOP
        (* count number of resizable objects *)
        count:=0;
        object:=p.list;
        WHILE object#NIL DO
          IF object.CanResize(p.width>=size,TRUE) THEN
            INC(count);
          END;
          object:=object.next;
        END;

        (* we leave the loop if there are no objects (anymore) that can be resized *)
        IF count=0 THEN
          EXIT;
        END;

        object:=p.list;
        WHILE object#NIL DO
          IF object.CanResize(p.width>=size,TRUE) THEN
            old:=object.oWidth;
            object.Resize(object.oWidth+U.UpDiv(p.width-size,count),-1);
            INC(size,object.oWidth-old);
            DEC(count);
          END;
          object:=object.next;
        END;

        (* no space left to delegate *)
        IF size=p.width THEN
          EXIT;
        END;
      END;
    ELSE (* vertical *)
      LOOP
        (* count number of resizable objects *)
        count:=0;
        object:=p.list;
        WHILE object#NIL DO
          IF object.CanResize(p.height>size,FALSE) THEN
            INC(count);
          END;
          object:=object.next;
        END;

        (* we leave the loop if there are no objects (anymore) that can be resized *)
        IF count=0 THEN
          EXIT;
        END;

        object:=p.list;
        WHILE object#NIL DO
          IF object.CanResize(p.height>size,FALSE) THEN
            old:=object.oHeight;
            object.Resize(-1,object.oHeight+U.UpDiv(p.height-size,count));
            INC(size,object.oHeight-old);
            DEC(count);
          END;
          object:=object.next;
        END;

        (* no space left to delegate *)
        IF size=p.height THEN
          EXIT;
        END;
      END;
    END;

    p.DrawBackground(p.x,p.y,p.width,p.height);

    IF p.type=horizontal THEN
      pos:=p.x;
      object:=p.list;
      WHILE object#NIL DO
        object.Draw(pos,p.y+(p.height-object.oHeight) DIV 2,draw);
        INC(pos,object.width);
        object:=object.next;
      END;
    ELSE (* vertical *)
      pos:=p.y;
      object:=p.list;
      WHILE object#NIL DO
        object.Draw(p.x+(p.width-object.oWidth) DIV 2,pos,draw);
        INC(pos,object.height);
        object:=object.next;
      END;
    END;
  END Draw;

  PROCEDURE (p : Panel) Refresh*(x,y,w,h : LONGINT);

  VAR
    help : G.Object;

  BEGIN
    IF p.visible & p.Intersect(x,y,w,h) THEN
        p.RestrictToBounds(x,y,w,h);
        p.DrawBackground(x,y,w,h);
      help:=p.list;
      WHILE help#NIL DO
        help.Refresh(x,y,w,h);
        help:=help.next;
      END;
    END;
  END Refresh;


  PROCEDURE (p : Panel) Hide*;

  VAR
    help : G.Object;

  BEGIN
    IF p.visible THEN
      help:=p.list;
      WHILE help#NIL DO
        help.Hide;
        help:=help.next;
      END;
      p.DrawHide;
      p.Hide^;
    END;
  END Hide;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOPanel.