/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

/*
 * sockets.c - functions to deal with sockets.
 */

#include <X11/Xatom.h>
#include "global.h"

#include "vncsockets.h"


//constructors

VNCSockets::VNCSockets(){}

VNCSockets::VNCSockets(char *nomServeur, int Port=5901){

  strcpy(ServerName, nomServeur);
  port=Port;
  buffered=0;
  bufoutptr=buf;
  rfbsock = -1;
  if(!StringToIPAddr()) 
    trace(DBG_FORCE, "VNC: constructor: StringToIPAddr");
}

VNCSockets::VNCSockets(unsigned int IPAddr, int Port=5901){
  port=Port;
  buffered=0;
  bufoutptr=buf;
  rfbsock = -1;
  host = IPAddr;
}

/*
 * ReadFromRFBServer is called whenever we want to read some data from the RFB
 * server.  It is non-trivial for two reasons:
 *
 * 1. For efficiency it performs some intelligent buffering, avoiding invoking
 *    the read() system call too often.  For small chunks of data, it simply
 *    copies the data out of an internal buffer.  For large amounts of data it
 *    reads directly into the buffer provided by the caller.
 *
 * 2. Whenever read() would block, it invokes the Xt event dispatching
 *    mechanism to process X events.  In fact, this is the only place these
 *    events are processed, as there is no XtAppMainLoop in the program.
 */

/*
 *Read bytes from the sever
 */
bool VNCSockets::ReadFromRFBServer(char *out, unsigned int n)
{
  if (n <= buffered) {
    memcpy(out, bufoutptr, n);
    bufoutptr += n;
    buffered -= n;
    return true;
  }

  memcpy(out, bufoutptr, buffered);

  out += buffered;
  n -= buffered;

  bufoutptr = buf;
  buffered = 0;

  if (n <= BUF_SIZE) {

    while (buffered < n) {
      int i = read(rfbsock, buf + buffered, BUF_SIZE - buffered);
      if (i <= 0) {
	if (i < 0) {
	  if (errno == EWOULDBLOCK || errno == EAGAIN) {
	    i = 0;
	  } else {
	    trace(DBG_FORCE, "VNC: read");
	    return false;
	  }
	} else {
	    trace(DBG_FORCE, "VNC: VNC server closed connection");
	  return false;
	}
      }
      buffered += i;
    }
    memcpy(out, bufoutptr, n);
    bufoutptr += n;
    buffered -= n;
    return true;

  } else {

    while (n > 0) {
      int i = read(rfbsock, out, n);
      if (i <= 0) {
	if (i < 0) {
	  if (errno == EWOULDBLOCK || errno == EAGAIN) {
	    i = 0;
	  } else {
	    trace(DBG_FORCE, "VNC: read");
	    return false;
	  }
	} else {
	    trace(DBG_FORCE, "VNC: VNC server closed connection");
	  return false;
	}
      }
      out += i;
      n -= i;
    }
    return true;
  }
}

/*
 * Write an exact number of bytes, and don't return until you've sent them.
 */
bool VNCSockets::WriteExact(char *buf, int n)
{
  fd_set fds;
  int i = 0;
  int j;

  while (i < n) {
    j = write(rfbsock, buf + i, (n - i));
    if (j <= 0) {
      if (j < 0) {
	if (errno == EWOULDBLOCK || errno == EAGAIN) {
	  FD_ZERO(&fds);
	  FD_SET(rfbsock,&fds);

	  if (select(rfbsock+1, NULL, &fds, NULL, NULL) <= 0) {
	    trace(DBG_FORCE, "VNC: select");
	    return false;
	  }
	  j = 0;
	} else {
	  trace(DBG_FORCE, "VNC: write");
	  return false;
	}
      } else {
	trace(DBG_FORCE, "VNC: write failed");
	return false;
      }
    }
    i += j;
  }
  return true;
}

/*
 * ConnectToTcpAddr connects to the given TCP port.
 */
int VNCSockets::ConnectToTcpAddr()
{
  int sock;
  struct sockaddr_in sa;
  int one = 1;

  sa.sin_family = AF_INET;
  sa.sin_port = htons(port);
  sa.sin_addr.s_addr = host;

  if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    trace(DBG_FORCE, "VNC: ConnectToTcpAddr: socket");
    return -1;
  }
  trace(DBG_FORCE, "connecting to %s:%i %x",
                   ServerName, sa.sin_port, sa.sin_addr.s_addr);

  if (connect(sock, (struct sockaddr *)&sa, sizeof(sa)) < 0) {
    perror("VNC: connect");
    trace(DBG_FORCE, "VNC: ConnectToTcpAddr: connect");
    close(sock);
    return -1;
  }

  if (setsockopt(sock, IPPROTO_TCP, TCP_NODELAY,
		 (char *)&one, sizeof(one)) < 0) {
    trace(DBG_FORCE, "VNC: ConnectToTcpAddr: setsockopt");
    close(sock);
    return -1;
  }

  rfbsock = sock;
  return sock;
}

/*
 * SetNonBlocking sets a socket into non-blocking mode.
 */
bool VNCSockets::SetNonBlocking()
{
  if (fcntl(rfbsock, F_SETFL, O_NONBLOCK) < 0) {
    trace(DBG_FORCE, "VNC: AcceptTcpConnection: fcntl");
    return false;
  }
  return true;
}

/*
 * StringToIPAddr - convert a host string to an IP address.
 */
bool VNCSockets::StringToIPAddr()
{
  struct hostent *hp;

  host = inet_addr(ServerName);
  if (host != -1)
    return true;

  if ((hp = gethostbyname(ServerName)) !=0 ) {
    memcpy(&host,hp->h_addr,hp->h_length);
    return true;
  }
  return false;
}

/*
 * Test if the other end of a socket is on the same machine.
 */
bool VNCSockets::SameMachine()
{
  struct sockaddr_in peeraddr, myaddr;
  socklen_t addrlen = sizeof(struct sockaddr_in);

  getpeername(rfbsock, (struct sockaddr *)&peeraddr, &addrlen);
  getsockname(rfbsock, (struct sockaddr *)&myaddr, &addrlen);

  return (peeraddr.sin_addr.s_addr == myaddr.sin_addr.s_addr);
}

/*
 * Print out the contents of a packet for debugging.
 */
void VNCSockets::PrintInHex(char *buf, int len)
{
  int i, j;
  char c, str[17];

  str[16] = 0;

  trace(DBG_VNC, "ReadExact: ");

  for (i = 0; i < len; i++)
    {
      if ((i % 16 == 0) && (i != 0)) {
	fprintf(stderr,"           ");
      }
      c = buf[i];
      str[i % 16] = (((c > 31) && (c < 127)) ? c : '.');
      fprintf(stderr,"%02x ",(unsigned char)c);
      if ((i % 4) == 3)
	fprintf(stderr," ");
      if ((i % 16) == 15)
	{
	  fprintf(stderr,"%s\n",str);
	}
    }
  if ((i % 16) != 0)
    {
      for (j = i % 16; j < 16; j++)
	{
	  fprintf(stderr,"   ");
	  if ((j % 4) == 3) fprintf(stderr," ");
	}
      str[i % 16] = 0;
      fprintf(stderr,"%s\n",str);
    }

  fflush(stderr);
}

/*
 *Returns the socket used
 */
int VNCSockets::GetSock()
{
  return rfbsock;
}
