#include "global.h"

#include "net.h"
#include "netdefs.h"
#include "netImpl.h"
#include "dates.h"	/* diffDates */
#include "payload.h"	/* putPayload */
#include "channel.h"	/* pchannel_curr */
#include "pkt.h"	/* sendPayload */

#include "world.h"	/* isValidType replicateObject */
#include "wobject.h"	/* WObject */
#include "user.h"	/* USER_TYPE */


/****************************/
/*** send commands        ***/ 
/****************************/

/*
 * envoi d'un message de type 'd' = Delta, sur le mcast
 * concernant la propriete prop_id de l'objet donne.
 * On fait le resetDates (pas version) 
 * Format actuel d'un Delta: 
 * 'd', nom de l'objet (n), no de propriete (c), version (h),
 * donnees du world.
 */
// M.S. static
void sendObjDelta(NetObject *pn, u_int8 prop_id)
{
  NetProperty *pprop;  
  Payload payload;

  if (!pn || !pn->prop) {
    trace(DBG_FORCE, "sendObjDelta: pn NULL");
    return;
  }
  pprop = pn->prop + prop_id;
  pprop->responsible = RESPONSIBLE;
  resetDates(pprop);
  resetPayload(&payload);
  putPayload(&payload, "cnch", VREP_OBJ_DELTA, pn->noid, prop_id, pprop->version);

  /*** TODO: the timevals ***/

  if (prop_id >= countProperties(pn->type)) {
    trace(DBG_FORCE, "sendObjDelta: prop_id wrong");
    return;
  }
  getProperty(pn, prop_id, &payload);
  sendPayload(pchannel_curr->sa[SA_RTP], &payload);
}

/*
 * send a 'C' to mentioned address for the mentionned object
 * we don't touch version and dates
 * C format:
 * 'C', object type (c), object name (n), permanent (c)
 * then the initialization payload, obtained by a getAllProperties()
 * then the version vector (nprop *pn)
 */
static
void sendObjCreate(struct sockaddr_in *to, NetObject *pn)
{
  Payload payload;
  int i, nprop;

  resetPayload(&payload);
  putPayload(&payload, "ccnc", VREP_OBJ_CREATE, pn->type, pn->noid, pn->permanent);
  getAllProperties(pn, &payload);  
  nprop = countProperties(pn->type);
  for (i = 0; i < nprop; i++)
    putPayload(&payload, "h", pn->prop[i].version);
  trace(DBG_NET, "sendCreate: netobj=%s to=%x", getNetNameById(pn->noid), to);
  sendPayload(to, &payload);
}

/*
 * send a query '?'
 * we try to avoid to send too many
 * Format:
 * '?', object name (n)
 * called by incomingDelta
 */
static
void sendObjQuery(struct sockaddr_in *sender, NetObjectId noid)
{
  Payload payload;
  static NetObjectId oldnoid;
  static int count = 0;

  /* Heuristique pour eviter d'envoyer des rafales de ? */
  if (equalNetObjectId(noid, oldnoid)) { 
    /* last request was on this name */
    if (++count <= 15)
      /* 15: 10 proprietes en moyenne, avec 15 on saute donc
       * sans doute un "bloc" de deltas, mais sans doute pas deux 
       */ 
      return;    /* cancel this request */
  }
  else {
    /* it's another one */
    oldnoid = noid;
  }
  count = 0;
  resetPayload(&payload);
  putPayload(&payload, "cn", VREP_OBJ_QUERY, noid);
  trace(DBG_RTP, "sendObjQuery: netobj=%s sender=%x",
                 getNetNameById(noid), sender);
  sendPayload(sender, &payload);
}

/*
 * send a Delete 'D'
 * Format: 'D', object name (n) 
 */
static
void sendObjDelete(struct sockaddr_in *to, NetObjectId noid)
{
  Payload payload;

  resetPayload(&payload);
  putPayload(&payload, "cn", VREP_OBJ_DELETE, noid);
  trace(DBG_NET, "sendObjDelete: netobj=%s to=%x", getNetNameById(noid), to);
  sendPayload(to, &payload);
} 

static
void sendPersistSet(struct sockaddr_in *to, NetObjectId noid)
{
  Payload payload;

  resetPayload(&payload);
  putPayload(&payload, "cn", VREP_PERSIST_SET, noid);
  trace(DBG_NET, "sendObjPersist: netobj=%s to=%x", getNetNameById(noid), to);
  sendPayload(to, &payload);
} 

static
void sendPersistInfo(struct sockaddr_in *to, NetObjectId noid)
{
  Payload payload;

  resetPayload(&payload);
  putPayload(&payload, "cn", VREP_PERSIST_INFO, noid);
  trace(DBG_NET, "sendPersistInfo: netobj=%s to=%x", getNetNameById(noid), to);
  sendPayload(to, &payload);
} 

static
void sendPersistReset(struct sockaddr_in *to, NetObjectId noid)
{
  Payload payload;

  resetPayload(&payload);
  putPayload(&payload, "cn", VREP_PERSIST_RESET, noid);
  trace(DBG_NET, "sendPersistReset: netobj=%s to=%x", getNetNameById(noid), to);
  sendPayload(to, &payload);
} 

static
void sendWorldSet(struct sockaddr_in *to, NetObjectId noid)
{
  Payload payload;

  resetPayload(&payload);
  putPayload(&payload, "cn", VREP_WORLD_SET, noid);
  trace(DBG_NET, "sendWorldSet: netobj=%s to=%x", getNetNameById(noid), to);
  sendPayload(to, &payload);
} 

static
void sendWorldInfo(struct sockaddr_in *to, NetObjectId noid)
{
  Payload payload;

  resetPayload(&payload);
  putPayload(&payload, "cn", VREP_WORLD_INFO, noid);
  trace(DBG_NET, "sendWorldInfo: netobj=%s to=%x", getNetNameById(noid), to);
  sendPayload(to, &payload);
} 

static
void sendWorldReset(struct sockaddr_in *to, NetObjectId noid)
{
  Payload payload;

  resetPayload(&payload);
  putPayload(&payload, "cn", VREP_WORLD_RESET, noid);
  trace(DBG_NET, "sendWorldReset: netobj=%s to=%x", getNetNameById(noid), to);
  sendPayload(to, &payload);
} 


/*****************************************/
/*** functions "declare" our "request" ***/
/*****************************************/

/*
 * we assume the header yet initialized with createNetObject()
 * Exported to wmgt, to call for each new objects
 */
void declareObjCreation(NetObject *pn)
{
  if (!pn) {
    trace(DBG_FORCE, "declareObjCreation: pn NULL");
    return;
  }
  if (getNetObjectId(pn->noid) == NULL) {
    warning("declareObjCreation: unnamed/deleted object (type=%d)", pn->type);
    return;
  }
  if (ntohl(pn->noid.src_id) == 1) {
    warning("declareObjCreation: not a new object (type=%d)", pn->type);
    return;
  }
  sendObjCreate(pchannel_curr->sa[SA_RTP], pn);
}  

/*
 * Update object version
 * Exported to wmgt, to call at each modification
 */
void declareObjDelta(NetObject *pn, u_int8 prop_id)
{
  NetProperty *pprop;  

  if (!pn) {
    trace(DBG_FORCE, "declareObjDelta: pn NULL");
    return;
  }
  if (pn->noid.src_id == 0) {
    warning("declareObjDelta: unnamed object (type=%d)", pn->type);
    return;
  }
  if (prop_id >= countProperties(pn->type)) {
    warning("declareObjDelta: invalid prop_id=%d (type=%d)",
             prop_id, pn->type); 
    return;
  }
  pprop = pn->prop + prop_id;
  pprop->responsible = RESPONSIBLE;
  pprop->version += 1 + abs(rand() % MAX_VERSION_JUMP); /* %10 */
  sendObjDelta(pn, prop_id);
}

/*
 * Exported to wmgt 
 * Destroy the object (local copy)
 * object must be valid (nom & co) in input
 * Au retour, il faut faire un deleteNetObject et le free() final.
 */
void declareObjDeletion(NetObject *pn)
{
  if (!pn) {
    trace(DBG_FORCE, "declareObjDeletion: pn NULL");
    return;
  }
  if (!getNetObjectId(pn->noid)) {
    return;
  }
  if (pn->permanent) {
    warning("declareObjDeletion: on permanent object (type=%d)", pn->type);
    return;
  }
  sendObjDelete(pchannel_curr->sa[SA_RTP], pn->noid);
}

void declarePersistSet(NetObject *pn)
{
  if (!pn) {
    trace(DBG_FORCE, "declarePersistSet: pn NULL");
    return;
  }
  sendPersistSet(pchannel_manager->sa[SA_RTP], pn->noid);
}

void declarePersistInfo(NetObject *pn)
{
  if (!pn) {
    trace(DBG_FORCE, "declarePersistInfo: pn NULL");
    return;
  }
  sendPersistInfo(pchannel_manager->sa[SA_RTP], pn->noid);
}

void declarePersistReset(NetObject *pn)
{
  if (!pn) {
    trace(DBG_FORCE, "declarePersistReset: pn NULL");
    return;
  }
  sendPersistReset(pchannel_manager->sa[SA_RTP], pn->noid);
}

void declareWorldSet(NetObject *pn)
{
  if (!pn) {
    trace(DBG_FORCE, "declareWorldSet: pn NULL");
    return;
  }
  sendWorldSet(pchannel_manager->sa[SA_RTP], pn->noid);
}

void declareWorldInfo(NetObject *pn)
{
  if (!pn) {
    trace(DBG_FORCE, "declareWorldInfo: pn NULL");
    return;
  }
  sendWorldInfo(pchannel_manager->sa[SA_RTP], pn->noid);
}

void declareWorldReset(NetObject *pn)
{
  if (!pn) {
    trace(DBG_FORCE, "declareWorldReset: pn NULL");
    return;
  }
  sendWorldReset(pchannel_manager->sa[SA_RTP], pn->noid);
}


/****************/	  
/*** incoming ***/
/****************/	  

/*
 * Incoming Delta:
 *         format (3 bytes)
 *         noid (8 bytes: 4 + 2 + 2)
 *         prop_id (1 byte)
 *         version_id (2 bytes)
 *         total (14 bytes)
 */
static
void incomingDelta(struct sockaddr_in *sender, Payload *pp) 
{
  NetObjectId noid;	/* name received */
  NetObject *pn;	/* header of object */
  NetProperty *pprop;	/* property found */
  u_int8 prop_id;	/* property number */
  int16 vers_id;	/* version received */
  int16 d;		/* versions difference */
  int nprop;		/* number of properties */

  if (getPayload(pp, "nch", &noid, &prop_id, &vers_id) < 0) {
    if (getPayload(pp, "nhh", &noid, &prop_id, &vers_id) < 0)
      return;
  }
  if ((pn = getNetObjectId(noid)) == NULL) {
    /* delta on an unknown object */
    trace(DBG_RTP, "incomingDelta sendQuery on: %s, to=%x, p=%d, v=%d",
                   getNetNameById(noid), sender, prop_id, vers_id);
    /* send to sender in unicast */
    sendObjQuery(sender, noid);
    return;
  }

  /* verify prop_id */
  nprop = countProperties(pn->type);
  if (prop_id >= nprop) {
    warning("incomingDelta: invalid property id (%d)"
	    "(type is %d, nprop is %d)", prop_id, pn->type, nprop);
    return;
  }
  pprop = pn->prop + prop_id;

  /*
   * depends on version
   */
  /* in complement to 2: d gives the distance, same throught the boundary */
  d = pprop->version - vers_id;
  if (abs(d) > 5000) { /* very far */
    warning("incomingDelta: very different versions: mine is %d, received %d",
             pprop->version, vers_id);
  }
  if (d > 0)
    /* mine is more recent */
    return;

  resetDates(pprop);
  if (d < 0) {
    /* its own is more recent */
    setProperty(pn, prop_id, pp);
    pprop->responsible = NOT_RESPONSIBLE; /* leave responsibility to an other */
    pprop->version = vers_id;
    return;
  }
  else if (pprop->responsible) {
    /* same versions, 2 responsibles: conflict
     * resolved by getting a new random version
     */
    /* publish new version to sender in unicast */
    declareObjDelta(pn, prop_id);
    warning("Conflict resol: obj=%s, prop=%d, changing vers_num %d->%d",
	   getNetNameById(pn->noid), prop_id, vers_id, pn->prop[prop_id].version);
  }
  /* else, it's just a "recall" (then nothing to do) */
}

/*
 * Incoming Create:
 *         format (3 bytes)
 *         type_id (1 byte)
 *         noid (8 bytes: 4 + 2 + 2)
 *         permanent_flag (1 byte)
 *	   properties (nprop * 20 bytes: 2 + 2 + 8 + 8)
 * Create the object's local copy if it doesn't exist
 */
static
void incomingCreate(struct sockaddr_in *sender, Payload *pp) 
{
  NetObjectId noid;
  NetObject *pn;
  u_int8 type_id, permanent;
  int i, nprop;

  if (getPayload(pp, "cnc", &type_id, &noid, &permanent) < 0) {
    if (getPayload(pp, "dnc", &type_id, &noid, &permanent) < 0)
      return;
  }
  if (getNetObjectId(noid) != NULL)
    return;		/* local copy already exists -> ignore this request */

  trace(DBG_NET, "incomingCreate: netobj=%s (type=%d), perm=%d", 
	         getNetNameById(noid), type_id, permanent);
  
  pn = replicateObject(type_id, noid, pp);
  if (!pn)
    return;
  if (pn->type != type_id) {
    trace(DBG_FORCE, "incomingCreate: bad type");
    return;
  }
  if (!equalNetObjectId(pn->noid, noid)) {
    trace(DBG_FORCE, "incomingCreate: bad noid");
    return;
  }
  pn->permanent = permanent;
  initProperties(pn, NOT_RESPONSIBLE); /* 0 == we are not responsible */
  insertNetObject(pn);
  nprop = countProperties(type_id);
  /* dumpPayload(stdout, pp); */
  for (i = 0; i < nprop; i++) {
    /* get properties */
    if (getPayload(pp, "h", &(pn->prop[i].version)) < 0)
      return;
  }
}

/*
 * Incoming Query:
 *         format (1 byte)
 *         noid (8 bytes: 4 + 2 + 2)
 *         total (9 bytes)
 */
static
void incomingQuery(struct sockaddr_in *sender, Payload *pp) 
{
  NetObjectId noid;
  NetObject *pn;

  if (getPayload(pp, "n", &noid) < 0)
    return;
  trace(DBG_NET, "incomingQuery: netobj=%s from=%x",
                 getNetNameById(noid), sender);
  if ((pn = getNetObjectId(noid)) == NULL)
    /* unknown object: may be we have deleted it, we advertize the requester */
    sendObjDelete(sender, noid);
  else {
    /* object known, but not properties, ask them to sender */
    trace(DBG_RTP, "incomingQuery: sendCreate netobj=%s to=%x",
                   getNetNameById(noid), sender);
    sendObjCreate(sender, pn);
  }
}

/*
 * Incoming Delete:
 *         format (1 byte)
 *         noid (8 bytes: 4 + 2 + 2)
 *         total (9 bytes)
 */
static
void incomingDelete(struct sockaddr_in *sender, Payload *pp) 
{
  NetObjectId noid;
  NetObject *pn;

  if (getPayload(pp, "n", &noid) < 0)
    return;
  trace(DBG_NET, "incomingDelete: netobj=%s from=%x",
                 getNetNameById(noid), sender);
  if ((pn = getNetObjectId(noid)) != NULL)
     requestDeletionFromNetwork(pn);
}

void incomingPersist(struct sockaddr_in *sender, Payload *pp) 
{
  NetObjectId noid;
#if 0
  NetObject *pn;
#endif

  if (getPayload(pp, "n", &noid) < 0)
    return;
  trace(DBG_FORCE, "incomingPersist: netobj=%s from=%x",
                   getNetNameById(noid), sender);
#if 0
  if (pn = getNetObjectId(noid))
     requestPersistFromNetwork(pn);
#endif
}

void incomingWorld(struct sockaddr_in *sender, Payload *pp) 
{
  NetObjectId noid;
#if 0
  NetObject *pn;
#endif

  if (getPayload(pp, "n", &noid) < 0)
    return;
  trace(DBG_FORCE, "incomingWorld: netobj=%s from=%x",
                   getNetNameById(noid), sender);
}

void incomingOther(int fd, struct sockaddr_in *sender, Payload *pp, int len) 
{
  u_int8 *pkt = pp->data;

  if (pp->len == 0)
    return;	/* ignore */
  else
    trace(DBG_FORCE, "IncomingOther: len=%d on fd=%d from %lx/%x",
          len, fd, ntohl(sender->sin_addr.s_addr), ntohs(sender->sin_port));
  trace(DBG_FORCE, "%02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x",
    pkt[0], pkt[1], pkt[2], pkt[3],
    pkt[4], pkt[5], pkt[6], pkt[7],
    pkt[8], pkt[9], pkt[10], pkt[11],
    pkt[12], pkt[13], pkt[14], pkt[15],
    pkt[16], pkt[17], pkt[18], pkt[19]);
}

/*
 * When something is available on file-descriptor "on"
 * Exported to gui
 */
void incoming(int on) 
{
  fd_set set;
  int r;
  u_int8 cmd;
  struct timeval timeout;
  static Payload payload;
  static struct sockaddr_in sender;	/* sender unicast address */

  while (1) {
    FD_ZERO(&set);
    FD_SET(on, &set);
    timeout.tv_sec = timeout.tv_usec = 0;
    if (select(FD_SETSIZE, &set, NULL, NULL, &timeout) == 0)
      break;

    /*
     * Read the packet
     */
    if ((r = recvPacket(on, &sender, &payload)) <= 0)
      return;	/* read error <0 or ignore == 0 */

    /*
     * Test if this packet belongs to VREP (VREng Protocol)
     */
    if (payload.len > 0 && getPayload(&payload, "c", &cmd) >= 0) {
      /*
       * Parse VREP
       */
      switch (cmd) {
      case VREP_OBJ_CREATE:
      case VREP_OBJ_CREATE_V1:
           incomingCreate(&sender, &payload);
           break;
      case VREP_OBJ_DELTA:
      case VREP_OBJ_DELTA_V1:
           incomingDelta(&sender, &payload);
           break;
      case VREP_OBJ_QUERY:
      case VREP_OBJ_QUERY_V1:
           incomingQuery(&sender, &payload);
           break;
      case VREP_OBJ_DELETE:
      case VREP_OBJ_DELETE_V1:
           incomingDelete(&sender, &payload);
           break;
      case VREP_PERSIST_INFO:
      case VREP_PERSIST_RESET:
           incomingPersist(&sender, &payload);
      case VREP_WORLD_INFO:
      case VREP_WORLD_RESET:
           incomingWorld(&sender, &payload);
           break;
      default:
           warning("Incoming unknown cmd: X'%02x'", cmd);
           trace(DBG_FORCE, "Incoming on fd=%d from %lx/%x (mine is %lx/%x)",
	         on, ntohl(sender.sin_addr.s_addr), ntohs(sender.sin_port),
	         my_host_id, my_port_id);
           break;
      }
    }
    else {
      /* empty or invalid payload */
      incomingOther(on, &sender, &payload, r);
      return;
    }
  }
}


/***************/
/*** Timeout ***/ 
/***************/

/*
 * Check if some responsibilities must not to be taken
 * check if some objects must not to be deleted do the refresh. 
 * return the delay in ms before recall
 */
int networkTimeout()
{
  NetObject *pn, *next;
  NetProperty *pprop;
  int i, nprop;
  float refresh = REFRESH_TIMEOUT;
  struct timeval now;
  
  /* statAdjust(); */
  gettimeofday(&now, NULL);

  /* timeout grows as log(nbsources) to keep bandwidth confined */
#ifdef REDUCE_BW
  nsrc = RtpGetSourcesNumber() - 1;
  nsrc = (nsrc <= 1) ? 1 : nsrc;
  refresh = (float) (REFRESH_TIMEOUT * ((1.0 + (float) log((double) nsrc)) / 2));
  trace(DBG_FORCE, "refresh=%.2f nsrc=%d", refresh, nsrc);
#endif
  
  /* 
   * for each object in objects list
   */
  for (pn = getNetObjectsList() ; pn != NULL; ) {
    if (!isValidType(pn->type))
      return -1;

    next = pn->next;	/* save it now, pn may disapear if death */

    /*
     * scan its properties
     */
    nprop = countProperties(pn->type);
    for (i = 0; i < nprop; i++) {
      if (pn->type == USER_TYPE) {
        if (i < USER_PROPBEGINVAR || i > USER_PROPENDVAR) {
          /* skip static properties */
          trace(DBG_NET, "skip property %d", i);
          continue;
        }
      }
      pprop = pn->prop + i;

      /* test if refresh timeout reached */
      if (pprop->responsible &&
          diffDates(pprop->last_seen, now) > refresh) {
	/* now - last_seen > refresh: we are responsible, we must refresh */
	sendObjDelta(pn, i);
      }

      if (diffDates(pprop->assume_at, now) > 0) {
	/* now > assume_at: assume's timeout on this property */
	if (pn->permanent) {
          /* permanent object (door, lift,...) */
	  if (pprop->version != 0) {
	    /* if permanent prop hasn't its initial value,
	     * somebody must assume responsibility */
	    notice("Assuming responsibility for %s, prop=%d unseen for %5.2fs", 
		    getNetNameById(pn->noid), i, diffDates(pprop->last_seen, now));
	    if (pprop->responsible) {
	      warning("networkTimeout: should assume responsibility "
		    "of something I am responsible for");
              return -1;
            }
	    declareObjDelta(pn, i);   /* assume responsibility: publish */ 
	  }
	}
        else { 
	  /* volatile object (avatar, rocket, ball,...) */
	  notice("Assuming death of %s (unseen for %5.2f s)", 
                  getNetNameById(pn->noid), diffDates(pprop->last_seen, now));
	  if (pprop->responsible) {
	    warning("networkTimeout: Should assume death "
                    "of something I am responsible for");
            return -1;
	  } 
	  declareObjDeletion(pn);
	  requestDeletionFromNetwork(pn);  /* discard the dead */
	  /* no reason to continue after a requestDeletion */
          break;
	} 
      }
    } /* end scan properties */
    /* next object */
    pn = next;
  }
  return (int) ceil(refresh * 1000);
}

