// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
// 
// $Id: MatternGVTManager.cpp
// 
//---------------------------------------------------------------------------

#include "MatternGVTManager.h"
#include "GVTManagerObjectRecord.h"
#include "SchedulingManager.h"
#include "CommunicationManager.h"
#include "TimeWarpSimulationManager.h"
#include "MatternGVTMessage.h"
#include "GVTUpdateMessage.h"
#include <sstream>
using std::istringstream;
using std::ostringstream;
using std::cout;
using std::cerr;
using std::endl;

MatternGVTManager::MatternGVTManager( TimeWarpSimulationManager *simMgr,
				      unsigned int period) :
  GVTManagerImplementationBase( simMgr, period ){
  gVTTokenPending = false;
  ASSERT( mySimulationManager != NULL );
  ASSERT( mySimulationManager->getSchedulingManager() != NULL );
  myScheduler = mySimulationManager->getSchedulingManager();
  ASSERT( mySimulationManager->getCommunicationManager() != NULL );
  myCommunicationManager = mySimulationManager->getCommunicationManager();
}

MatternGVTManager::~MatternGVTManager(){}

// This is called from SimulationObject::receiveEvent(...)
void
MatternGVTManager::calculateGVTInfo() {
  unsigned int tokenColor = objectRecord.getColor();

  // Note that SimulationObject::receiveEvent(...) is also called during the
  // coast-forward process, at which time this part of the code should NOT
  // be executed.
  if (((TimeWarpSimulationManager *)mySimulationManager)->getSuppressMessageFlag() !=
      COASTFORWARD && tokenColor == RED) {
    const VTime& minimumTime =
      MIN_FUNC(objectRecord.getMinTimeStamp(),
	       myScheduler->getLastEventScheduledTime());
    /*
      VTime minimumTime =
      MIN_FUNC(objectRecord.getMinTimeStamp(),
      sendEvent->getReceiveTime());
    */
  objectRecord.setMinTimeStamp(minimumTime);
  }
}

// This is called from SimulationObjectProxy::receiveEvent(...)
const string
MatternGVTManager::getGVTInfo(unsigned int srcSimMgr,
                              unsigned int destSimMgr,
			      const VTime &sendTime){
  string retval;
  int tokenColor = objectRecord.getColor();
  retval = intToString( tokenColor );
  if(tokenColor == WHITE){
    if (destSimMgr == 0) {
      // this means the kernel is sending an EventMessage across the network
      // we need to update the dest simulation manager's record of messages
      // sent
      
      objectRecord.incrementNumberOfWhiteMessages();
    }
  }
  else {
    const VTime& minimumTime =
      MIN_FUNC(objectRecord.getMinTimeStamp(),
	       sendTime);
    //VTime minimumTime = myScheduler->getLastEventScheduledTime();
    objectRecord.setMinTimeStamp(minimumTime);
  }
  return retval;
}

void
MatternGVTManager::updateEventRecord( const string &infoStream,
				      unsigned int receivingSimMgr ){
  // for mattern only color information will be present in the
  // info stream received from the kernel
  istringstream inputStream(infoStream);
  unsigned int tokenColor;
  inputStream >> tokenColor;

  if(tokenColor == WHITE){
    // myID == receivingSimMgr
    if (receivingSimMgr == 0) {
      objectRecord.decrementNumberOfWhiteMessages();
    }
  }
}

void 
MatternGVTManager::calculateGVT(){
  // a few assumptions to state first:
  //
  // a. SimulationManager(0) is assumed to be the initiator of the
  //    gvt estimation procedure.
  //
  // b. if numberOfSimulationManagers < 2, then the gVT is simply the
  //    last scheduled event.
  if( mySimulationManager->getSimulationManagerID() == 0 ){
    if( mySimulationManager->getNumberOfSimulationManagers() > 1){
      // okay, time to start off the gvt token
      if(gVTTokenPending == false){
	// make sure there isn't a token already out there
	// sending gvt token into the ring
	sendGVTToken( myScheduler->getLastEventScheduledTime(),
		      mySimulationManager->getPositiveInfinity());
	objectRecord.setColor(RED);
	objectRecord.setNumberOfWhiteMessages(0);
	objectRecord.setMinTimeStamp(mySimulationManager->getPositiveInfinity());
	gVTTokenPending = true;
      }
    }
    else { // numberOfSimulationManagers < 2
      // okay, we dont need to launch a distributed gvt estimation step
      // gVT is simply the last scheduled event's timestamp
      setGVT( myScheduler->getLastEventScheduledTime() );
      cout << "GVT = " << getGVT() << endl;
      //mySimulationManager->garbageCollect(gVT);
    }
  }
  else { // myID != 0
    return;
    // do nothing; as SimulationManager(0) will inform us of the gVT value.
    // we do not need to do anything other than process the GVTTokenMessage.
  }
   
}

// ----------------------------------------------------------------
// A GVTTokenMessage contains:
//   1. VTime mClock;             // minimum of the local clocks
//   2. VTime mSend;              // minimum of timestamps
//   3. unsigned int count;       // number of messages in transit
// ----------------------------------------------------------------
void
MatternGVTManager::sendGVTToken(const VTime &lastScheduledEventTime,
                                const VTime &minimumTimeStamp){
  // My(malolan) changes here The guy, whom i am going to send to has to
  // know, how many white messages, i have sent and not the white messages
  // that, i have received so changing
  // objectRecordArray[myID].getNumberOfWhiteMessages() to
  // objectRecordArray[destination].getNumberOfWhiteMessages( send the
  // token to the next guy;
  unsigned int destination = 0;
  if( mySimulationManager->getSimulationManagerID() != 
      mySimulationManager->getNumberOfSimulationManagers() - 1 ){
    destination = mySimulationManager->getSimulationManagerID() + 1;
  }

  // increment the token iteration number
  objectRecord.incrementTokenIterationNumber();
   
  KernelMessage *messageToSend =
    new MatternGVTMessage( mySimulationManager->getSimulationManagerID(),
			   destination, 
			   lastScheduledEventTime,
			   minimumTimeStamp,
			   objectRecord.getNumberOfWhiteMessages() );

  ASSERT(myCommunicationManager != NULL);
  myCommunicationManager->sendMessage(messageToSend, destination);
}

void
MatternGVTManager::sendGVTUpdate(){
  // send out the latest gVT value to everyone
  cerr << "GVT = " << getGVT() << "\n";
   
  // send everybody except ourselves the gVT update message
  for( unsigned int count = 0; 
       count < mySimulationManager->getNumberOfSimulationManagers(); 
       count++){
    if( mySimulationManager->getSimulationManagerID() != count ){
      KernelMessage *messageToSend =
	new GVTUpdateMessage( mySimulationManager->getSimulationManagerID(), 
			      count, 
			      getGVT() );
      ASSERT(myCommunicationManager != NULL);
      myCommunicationManager->sendMessage(messageToSend, count);
    }
  }
}

// the MatternGVTManager must register the following message types with
// the communication manager: GVTTokenMessage and GVTUpdateMessage
void
MatternGVTManager::registerWithCommunicationManager(){
  ASSERT(myCommunicationManager != NULL);
  myCommunicationManager->registerMessageType( GVTUpdateMessage::getGVTUpdateMessageType(), 
					       this );
  myCommunicationManager->registerMessageType( MatternGVTMessage::getMatternGVTMessageType(), 
					       this );
}

// the communication manager will call this method to deliver a
// GVTTokenMessage or a GVTUpdateMessage.
void
MatternGVTManager::receiveKernelMessage(KernelMessage *msg){
  ASSERT(msg != NULL);
  if( dynamic_cast<MatternGVTMessage *>(msg) != 0 ){
    const MatternGVTMessage *gVTMessage = dynamic_cast<MatternGVTMessage *>( msg );
    const unsigned int count = gVTMessage->getNumMessagesInTransit();
    if( mySimulationManager->getSimulationManagerID() == 0 ){ 
      // initiator has received the control message
      // check to see if the count is zero and this is at least the second
      // round of the token
      if( objectRecord.getTokenIterationNumber() > 1 &&
	  (objectRecord.getNumberOfWhiteMessages() + count == 0)){
	// okay, so now calculate the global virtual time
	setGVT( MIN_FUNC( gVTMessage->getLastScheduledEventTime(),
			  gVTMessage->getMinimumTimeStamp() ) );
	// time to inform everybody about the new value
	// reset yourself and everybody else back to color WHITE
	objectRecord.setColor(WHITE);
	sendGVTUpdate();
	gVTTokenPending = false; // end of this gvt calculation cycle
	objectRecord.setTokenIterationNumber(0);
	objectRecord.setNumberOfWhiteMessages(0);
      }
      else {
	// nope; not yet ready to calculate gvt;
	// send the token around again ...
	objectRecord.setNumberOfWhiteMessages(objectRecord.getNumberOfWhiteMessages() + 
					      count);
	sendGVTToken( myScheduler->getLastEventScheduledTime(),
		      MIN_FUNC(gVTMessage->getMinimumTimeStamp(), 
			       objectRecord.getMinTimeStamp() ));
	objectRecord.setNumberOfWhiteMessages(0);
      }
    }
    else {
      // the gvt token has been received by another simulation manager
      // okay, two things to do:
      // [a] set color of this sim mgr to RED; set tMin = positive infinity
      // [b] pass on the token to processor(i mod n) + 1
      if (objectRecord.getColor() == WHITE){
	objectRecord.setColor(RED);
	objectRecord.setMinTimeStamp(mySimulationManager->getPositiveInfinity());
      }
      objectRecord.setNumberOfWhiteMessages(objectRecord.getNumberOfWhiteMessages() + 
					    count);
      sendGVTToken(MIN_FUNC(gVTMessage->getLastScheduledEventTime(),
			    myScheduler->getLastEventScheduledTime()),
		   MIN_FUNC(gVTMessage->getMinimumTimeStamp(),
			    objectRecord.getMinTimeStamp()));
      objectRecord.setNumberOfWhiteMessages(0);
    }
  }
  else if( dynamic_cast<GVTUpdateMessage *>(msg) != 0 ){
    const GVTUpdateMessage *gVTMessage = dynamic_cast<GVTUpdateMessage *>(msg);
    setGVT( gVTMessage->getNewGVT() );

    // reset color to white
    objectRecord.setColor(WHITE);
    objectRecord.setNumberOfWhiteMessages(0);
  }
  else {
    cerr << "MatternGVTManager::receiveKernelMessage() received"
	 << " unknown (" << msg->getDataType() << ") message type" << endl;
    cerr << "Aborting simulation ..." << endl;
    abort();
  }
  // we are done with this kernel message
  delete msg;
}
