/*
 * Proto.h
 *
 * Copyright (C) 1999 Stephen F. White
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */

#ifndef _PROTO_H
#define _PROTO_H

#ifndef _ARRAY_H
#include "Array.h"
#endif
#ifndef _DUNE_STRING_H
#include "MyString.h"
#endif
#ifndef _NODE_H
#include "Node.h"
#endif

#define INVALID_INDEX (-1)

class Element;
class EventIn;
class EventOut;
class Field;
class ExposedField;
class FieldValue;
class Scene;
class Node;
class NodeList;
typedef Array<MyString> StringArray;

class NodeScript;
class ScriptDialog;

class FieldIndex {
// in principle, this is a readonly integer
// it is used to contain the index of field in a VRML node
// e.g. field "center" of the VRML Transform node is 0, field children is 1 etc.
public:
    FieldIndex()         { _written = 0; }
    void set(int i)      { 
                         if (_written == 0) {
                             _written = 1;
                             if (i < 0) {
                                 assert(false);
                             } else 
                                 _fieldIndex = i;
                         } else 
                             assert(false);
                         }
    operator int() const { 
                         if (_written != 1) assert(false); 
                             return _fieldIndex; 
                         }
protected:
    int _written;
    int _fieldIndex;
};

class Proto {
public:

    // unlike other nodes, script nodes need to change their Proto dynamically
    friend class NodeScript;
    friend class ScriptDialog;

				Proto(Scene *scene, const MyString &name);
    virtual		       ~Proto();

    virtual Node	       *create(Scene *scene);

    EventIn		       *getEventIn(int index) const { return _eventIns[index]; }
    EventOut		       *getEventOut(int index) const { return _eventOuts[index]; }
    Field		       *getField(int index) const { return _fields[index]; }
    ExposedField	       *getExposedField(int index) const { return _exposedFields[index]; }

    int				getNumFields() const { return _fields.size(); }
    int				getNumEventIns() const { return _eventIns.size(); }
    int				getNumEventOuts() const { return _eventOuts.size(); }

    int				lookupEventIn(const MyString &name) const;
    int				lookupEventOut(const MyString &name) const;
    int				lookupField(const MyString &name) const;
    int				lookupExposedField(const MyString &name) const;

    const MyString	       &getName() const { return _name; }

    int				write(int filedes, int indent) const;
    int				writeWithoutFields(int filedes, int indent) const;
    int	                        write_define(int filedes, int indent) const;

    int				addElement(Element *element);
    int				addExposedField(ExposedField *exposedField);

    void			define(Node *primaryNode, NodeList *nodes);
    bool			isPROTO(void);

protected:
    Proto		       *copy() const { return new Proto(*this); }
    int				addField(int fieldType, const MyString &name,
    					 FieldValue *defaultValue,
					 FieldValue *min = NULL,
					 FieldValue *max = NULL);
    int				addField(int fieldType, const MyString &name,
    					 FieldValue *defaultValue,
					 int nodeType);
    int				addField(int fieldType, const MyString &name,
					 FieldValue *defaultValue,
					 int flags,
					 const char **strings);
    void			addEventIn(int fieldType, const MyString &name);
    void			addEventIn(int fieldType, const MyString &name,
                                           int flags);
    void			addEventIn(int fieldType, const MyString &name,
                                           int flags, int field);
    void			addEventOut(int fieldType, const MyString &name);
    void			addEventOut(int fieldType, const MyString &name,
                                            int flags);
    int				addExposedField(int fieldType,
						const MyString &name,
    						FieldValue *defaultValue,
						FieldValue *min = NULL,
						FieldValue *max = NULL);
    int				addExposedField(int fieldType,
						const MyString &name,
    						FieldValue *defaultValue,
						int nodeType);
    int				addExposedField(int fieldType,
						const MyString &name,
    						FieldValue *defaultValue,
						int flags,
						const char **strings);
    int				lookupSimpleEventIn(const MyString &name) const;
    int				lookupSimpleEventOut(const MyString &name) const;

protected:
    Scene		       *_scene;
    MyString			_name;
    Array<EventIn *>		_eventIns;
    Array<EventOut *>		_eventOuts;
    Array<Field *>		_fields;
    Array<ExposedField *>	_exposedFields;

    // for PROTO's:
    NodeList		       *_nodes;
    Node		       *_primaryNode;

};

class ProtoNode : public Node
{
public:
			ProtoNode(Scene *scene, Proto *proto);

    virtual int		getType() const { return -1; }
    virtual Node       *copy() const { return new ProtoNode(*this); }
};

#endif // _PROTO_H
