/*
 * xalf_launch - overloaded Xlib function XMapWindow
 * 
 * To be used with xalf. 
 *
 * Peter strand <astrand@lysator.liu.se> 2000. GPLV2. 
 *
 * Based on scwm_set_pid_property.c (C) 1999 Toby Sargeant and Greg J. Badros
 * 
 * */
 
#undef DEBUG 
/* Uncomment below for debugging */
/* #define DEBUG */

#ifdef DEBUG
#   define DPRINTF(args) fprintf args
#else
#   define DPRINTF(args) 
#endif

#define _GNU_SOURCE

#include "config.h"
#include <dlfcn.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>


#define PID_PROPERTY_NAME "XALF_LAUNCH_PID"
#define PRELOAD_LIBRARY LIBDIR"/libxalflaunch.so.0"

static long int launch_pid = 0;
static void restore_env();

#ifdef __FreeBSD__
static void _init(void) __attribute__ ((section (".init")));
#endif /* __FreeBSD__ */

void
_init () 
{
    char *pid_string;
    void *dlh = NULL;

    DPRINTF ((stderr, "_init xalf_launch\n"));

    pid_string = getenv (PID_PROPERTY_NAME);
    if (pid_string) 
	launch_pid = atol (pid_string);
    DPRINTF ((stderr, "launch_pid is %ld\n", launch_pid));
    
    if (launch_pid == 0) 
	{
	    /* PID_PROPERTY_NAME was not defined. 
	       Restore env, so that we don't get loaded again. */
	    restore_env ();
	    return;
	}

    dlh = dlopen (NULL, RTLD_GLOBAL | RTLD_NOW);
    if (dlsym (dlh, "XSync") == NULL) 
	{
	    /* This is not an X11 app. Maybe our app is started via a shellscript. 
	       Hold on... */
	    DPRINTF ((stderr, "no XSync\n"));
	    return;
	}
    
    restore_env ();
}


extern int 
XMapWindow (Display* display, Window w)                               
{                                                                               
    int i = 0;
    static int (*fptr)() = 0;

    DPRINTF ((stderr, "XMapWindow\n"));
    
    if (fptr == 0) 
	{
	    fptr = (int (*)())dlsym(RTLD_NEXT, "XMapWindow");
	    if (fptr == NULL) 
		{
		    fprintf(stderr, "dlopen: %s\n", dlerror());
		    return (0);
		}
	}

    i = ((*fptr)(display, w));
    
    /* we don't want to kill neither 0 nor init */
    if (launch_pid > 1) 
	{
	    DPRINTF ((stderr, "XMapWindow: Sending signal to process %d\n", launch_pid));
	    kill (launch_pid, SIGUSR1);
	    launch_pid = 0;
	    /* Restore env. */
	    restore_env ();
	}
    
    return i;
}


extern int 
XMapRaised (Display* display, Window w)                               
{                                                                               
    int i = 0;
    static int (*fptr)() = 0;

    DPRINTF ((stderr, "XMapRaised\n"));
    
    if (fptr == 0) 
	{
	    fptr = (int (*)())dlsym(RTLD_NEXT, "XMapRaised");
	    if (fptr == NULL) {
		fprintf(stderr, "dlopen: %s\n", dlerror());
		return (0);
	    }
	}

    i = ((*fptr)(display, w));
    
    /* we don't want to kill neither 0 nor init */
    if (launch_pid > 1) 
	{
	    DPRINTF ((stderr, "XMapRaised: Sending signal to process %d\n", launch_pid));
	    kill (launch_pid, SIGUSR1);
	    launch_pid = 0;
	    /* Restore env. */
	    restore_env ();
	}
	
    return i;                                                                   
}


#ifdef HAVE_UNSETENV
 #define PRELOAD_UNSETTER unsetenv ("LD_PRELOAD")
 #define PID_UNSETTER unsetenv (PID_PROPERTY_NAME)
#else
 #define PRELOAD_UNSETTER putenv ("LD_PRELOAD=")
 #define PID_UNSETTER putenv (PID_PROPERTY_NAME"=")
#endif /* HAVE_UNSETENV */


static char *
next_colon_or_space(char *p)
{
    char *rest_colon;
    char *rest_space;
    rest_colon = strchr (p, ':'); 
    rest_space = strchr (p, ' '); 
    
    if (rest_colon && rest_space) 
	/* Both exists. Take the first one. */
	{
	    if (rest_colon < rest_space)
		return rest_colon;
	    else
		return rest_space;
	}
    else if (rest_colon)
	return rest_colon;
    else if (rest_space)
	return rest_space;
    else
	return NULL;
}


/* Remove our PRELOAD_LIBRARY from LD_PRELOAD 
   and unset PID_PROPERTY_NAME */
static void
restore_env()
{
#ifdef MULTI_PRELOAD
    char *env_ptr = NULL;
    char *p = NULL;
    char *rest = NULL;

    env_ptr = getenv ("LD_PRELOAD");
    
#ifdef DEBUG
    {
	char *pid_ptr;
	
	if (env_ptr)
	    DPRINTF((stderr, "LD_PRELOAD was |%s|\n", env_ptr));
	else
	    DPRINTF((stderr, "LD_PRELOAD was null\n"));
	
	pid_ptr = getenv(PID_PROPERTY_NAME);
	if (pid_ptr)
	    DPRINTF((stderr, PID_PROPERTY_NAME" was |%s|\n", pid_ptr));
	else
	    DPRINTF((stderr, PID_PROPERTY_NAME" was null\n"));
    }
#endif /* DEBUG */
    
    if (!env_ptr || !(*env_ptr)) 
	{
	    DPRINTF ((stderr, "No or empty LD_PRELOAD.\n"));
	    return;
	}
    
    p = strstr (env_ptr, PRELOAD_LIBRARY);
    if (!p) 
	{
	    /* PRELOAD library is not in LD_PRELOAD. Do nothing. */
	    return;
	}
    
    rest = next_colon_or_space (p);
    
    if (rest) 
	/* There was a colon or space. */
	{
	    rest++;
	    if (*rest) 
		/* ...and something after it. */
		{
		    /* Move the string after the colon or space to the place where 
		       PRELOAD_LIBRARY used to be. */
		    while ( (*p++ = *rest++) );
		}
	    else
		/* ...but nothing after it. Remove it. */
		{
		    rest--;
		    *rest = '\0';
		}
	}
    
    /* Remove the colon or space, if we are not at start of string. */
    if (p != env_ptr) {
	p--;
    }
    /* Truncate. */
    *p = '\0';
    
    if (! (*env_ptr)) 
	{
	    DPRINTF ((stderr, "unsetting LD_PRELOAD\n")); 
	    PRELOAD_UNSETTER;
	}
    
#else 
    
    PRELOAD_UNSETTER;    

#endif /* MULTI_PRELOAD */

    PID_UNSETTER;


#ifdef DEBUG
    {
	char *env_ptr;
	char *pid_ptr;

	env_ptr = getenv("LD_PRELOAD");
	pid_ptr = getenv(PID_PROPERTY_NAME);
	
	if (env_ptr)
	    DPRINTF((stderr, "LD_PRELOAD is now |%s|\n", env_ptr));
	else
	    DPRINTF((stderr, "LD_PRELOAD is now null\n"));

	if (pid_ptr)
	    DPRINTF((stderr, PID_PROPERTY_NAME" was |%s|\n", pid_ptr));
	else
	    DPRINTF((stderr, PID_PROPERTY_NAME" was null\n"));
    }
#endif /* DEBUG */

    return;
}

#undef PRELOAD_UNSETTER
#undef PID_UNSETTER

