// Simple test that we can load the xapian module and run a simple test
//
// Copyright (C) 2004,2005,2006 Olly Betts
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
// USA

// The Portable.NET compiler has a bug which prevents it finding static
// member functions such as Xapian.Version.Major():
//
// http://savannah.gnu.org/bugs/?func=detailitem&item_id=12231
//
// The workaround is to add an explicit "using Xapian;" here:
using Xapian;

class TestMatchDecider : Xapian.MatchDecider {
    public override int Apply(Xapian.Document doc) {
	return (doc.GetValue(0) == "yes") ? 1 : 0;
    }
}

class SmokeTest {
    public static void Main() {
	try {
	    // Test the version number reporting functions give plausible
	    // results.
	    string v = "";
	    v += Xapian.Version.Major();
	    v += ".";
	    v += Xapian.Version.Minor();
	    v += ".";
	    v += Xapian.Version.Revision();
	    string v2 = Xapian.Version.String();
	    if (v != v2) {
		System.Console.WriteLine("Unexpected version output (" + v + " != " + v2 + ")");
		System.Environment.Exit(1);
	    }

	    Xapian.Stem stem = new Xapian.Stem("english");
	    Xapian.Document doc = new Xapian.Document();
	    // Currently SWIG doesn't generate zero-byte clean code for
	    // transferring strings between C# and C++.
	    /*
	    doc.SetData("a\0b");
	    if (doc.GetData() == "a") {
		System.Console.WriteLine("GetData+SetData truncates at a zero byte");
		System.Environment.Exit(1);
	    }
	    if (doc.GetData() != "a\0b") {
		System.Console.WriteLine("GetData+SetData doesn't transparently handle a zero byte");
		System.Environment.Exit(1);
	    }
	    */
	    doc.SetData("is there anybody out there?");
	    doc.AddTerm("XYzzy");
	    doc.AddPosting(stem.StemWord("is"), 1);
	    doc.AddPosting(stem.StemWord("there"), 2);
	    doc.AddPosting(stem.StemWord("anybody"), 3);
	    doc.AddPosting(stem.StemWord("out"), 4);
	    doc.AddPosting(stem.StemWord("there"), 5);

	    Xapian.WritableDatabase db = Xapian.InMemory.Open();
	    db.AddDocument(doc);
	    if (db.GetDocCount() != 1) {
		System.Environment.Exit(1);
	    }

	    if (doc.TermListCount() != 5) {
		System.Environment.Exit(1);
	    }
	    int count = 0;
	    Xapian.TermIterator i = doc.TermListBegin();
	    while (i != doc.TermListEnd()) {
		++count;
		++i;
	    }
	    if (count != 5) {
		System.Environment.Exit(1);
	    }

	    // Check exception handling for Xapian::DocNotFoundError.
	    try {
		Xapian.Document doc2 = db.GetDocument(2);
		System.Console.WriteLine("Retrieved non-existent document: " + doc2.ToString());
		System.Environment.Exit(1);
	    } catch (System.Exception e) {
		// We expect DocNotFoundError
		if (e.Message.Substring(0, 16) != "DocNotFoundError") {
		    System.Console.WriteLine("Unexpected exception from accessing non-existent document: " + e.Message);
		    System.Environment.Exit(1);
		}
	    }

	    // Check QueryParser parsing error.
	    try {
		Xapian.QueryParser qp = new Xapian.QueryParser();
		qp.ParseQuery("test AND");
		System.Console.WriteLine("Successfully parsed bad query");
		System.Environment.Exit(1);
	    } catch (System.Exception e) {
		if (e.Message != "QueryParserError: Syntax: <expression> AND <expression>") {
		    System.Console.WriteLine("Exception string not as expected, got: '" + e.Message + "'");
		    System.Environment.Exit(1);
		}
	    }

	    // Check that OP_ELITE_SET works (in 0.9.6 and earlier it had the
	    // wrong value in C#).
	    try {
		Xapian.Query foo = new Xapian.Query(Xapian.Query.op.OP_OR, "hello", "world");
		Xapian.Query foo2 = new Xapian.Query(Xapian.Query.op.OP_ELITE_SET, foo);
		foo = foo2; // Avoid "unused variable" warning.
	    } catch (System.Exception e) {
		System.Console.WriteLine("Using OP_ELITE_SET cause exception '" + e.Message + "'");
		System.Environment.Exit(1);
	    }

	    // Feature test for MatchDecider.
	    doc = new Xapian.Document();
	    doc.SetData("Two");
	    doc.AddPosting(stem.StemWord("out"), 1);
	    doc.AddPosting(stem.StemWord("source"), 2);
	    doc.AddValue(0, "yes");
	    db.AddDocument(doc);

	    Xapian.Query query = new Xapian.Query(stem.StemWord("out"));
	    Xapian.Enquire enquire = new Xapian.Enquire(db);
	    enquire.SetQuery(query);
	    Xapian.MSet mset = enquire.GetMSet(0, 10, null, new TestMatchDecider());
	    if (mset.Size() != 1) {
		System.Console.WriteLine("MatchDecider found " + mset.Size().ToString() + " documents, expected 1");
		System.Environment.Exit(1);
	    }
	    if (mset.GetDocId(0) != 2) {
		System.Console.WriteLine("MatchDecider mset has wrong docid in");
		System.Environment.Exit(1);
	    }
	} catch (System.Exception e) {
	    System.Console.WriteLine("Exception: " + e.ToString());
	    System.Environment.Exit(1);
	}
    }
}
