/*
 * $Id: db.c,v 1.102 2001/05/01 14:33:43 kg4ijb Exp $
 *
 * XASTIR, Amateur Station Tracking and Information Reporting
 * Copyright (C) 1999,2000  Frank Giannandrea
 * Copyright (C) 2000,2001  The Xastir Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Look at the README for more information on the program.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
#include <math.h>

#include <Xm/XmAll.h>

#include "xastir.h"
#include "main.h"
#include "draw_symbols.h"
#include "alert.h"
#include "util.h"
#include "bulletin_gui.h"
#include "fcc_data.h"
#include "rac_data.h"
#include "interface.h"
#include "wx.h"
#include "igate.h"
#include "list_gui.h"
#include "track_gui.h"

#ifdef  WITH_DMALLOC
#include <dmalloc.h>
#endif


//#define STATION_DEBUG 1

#define STATION_REMOVE_CYCLE 60         /* check station remove in seconds (every minute) */
#define MESSAGE_REMOVE_CYCLE 60         /* check message remove in seconds (every minute) */
#define MAX_TRAIL_SEG_LEN    60l        /* max length of displayed trail segments in minutes (1 deg) */
#define IN_VIEW_MIN          30l        /* margin for off-screen stations, with possible trails on screen, in minutes */
#define TRAIL_POINT_MARGIN   10l        /* margin for off-screen trails points, for segment to be drawn, in minutes */
#define TRAIL_MAX_SPEED     900         /* max. acceptible speed for drawing trails, in mph */

Widget station_list;
Widget si_text;
Widget db_station_info = (Widget)NULL;
Widget db_station_popup   = (Widget)NULL;

static xastir_mutex db_station_info_lock;
static xastir_mutex db_station_popup_lock;

void redraw_symbols(Widget w);
int  delete_weather(DataRow *fill);
void Station_data_destroy_track(Widget widget, XtPointer clientData, XtPointer callData);
void my_station_gps_change(char *pos_long, char *pos_lat, char *course, char *speed, char speedu, char *alt, char *sats);

int  extract_speed_course(char *info, char *speed, char *course);

int tracked_stations = 0;       // A count variable used in debug code only
void track_station(Widget w, char *call_tracked, DataRow *p_station);

int  new_message_data;
time_t last_message_remove;     // last time we did a check for message removing

char packet_data[16][300];
int  packet_data_display;
int  redraw_on_new_packet_data;

long stations;                  // number of stored stations
DataRow *n_first;               // pointer to first element in name sorted station list
DataRow *n_last;                // pointer to last  element in name sorted station list
DataRow *t_first;               // pointer to first element in time sorted station list
DataRow *t_last;                // pointer to last  element in time sorted station list
time_t last_station_remove;     // last time we did a check for station removing
time_t last_sec,curr_sec;       // for comparing if seconds in time have changed
int next_time_sn;               // time serial number for unique time index

DataRow *p_trail_del;           // trail that should be deleted
void draw_trail(Widget w, DataRow *fill, int solid);



void db_init(void)
{
    init_critical_section( &db_station_info_lock );
    init_critical_section( &db_station_popup_lock );
}



///////////////////////////////////  Utilities  ////////////////////////////////////////////////////

/*
 *  Look if call is mine, exact looks on SSID too
 */
int is_my_call(char *call, int exact) {
    char *p_del;
    int ok;

    if (exact) {
        ok = (int)( !strcmp(call,my_callsign) );
    } else {
        int len1,len2;
        p_del = index(call,'-');
        if (p_del == NULL)
            len1 = (int)strlen(call);
        else
            len1 = p_del - call;
        p_del = index(my_callsign,'-');
        if (p_del == NULL)
            len2 = (int)strlen(my_callsign);
        else
            len2 = p_del - my_callsign;
        ok = (int)(len1 == len2 && !strncmp(call,my_callsign,(size_t)len1));
    }
    return(ok);
}
        

char *remove_trailing_spaces(char *data) {
    int i;

    for(i=strlen(data)-1;i>=0;i--)
        if(data[i] == ' ')
            data[i] = '\0';
        else
            break;

        return(data);
}



char *remove_trailing_asterisk(char *data) {
    int i;

    for(i=strlen(data)-1;i>0;i--) {
        if(data[i] == '*')
            data[i] = '\0';
    }
    return(data);
}



/////////////////////////////////////////// Messages ///////////////////////////////////////////

static long *msg_index;
static long msg_index_end;
static long msg_index_max;
static Message *msg_data;
time_t last_message_update = 0;

// How often update_messages() will run, in seconds.
// This is necessary because routines like UpdateTime()
// call update_messages() VERY OFTEN.
static int message_update_delay = 5;
    


void init_message_data(void) {          // called at start of main

    new_message_data = 0;
    message_counter = 0;
    last_message_remove = sec_now();
}



#ifdef MSG_DEBUG
void msg_clear_data(Message *clear) {
    int size;
    int i;
    unsigned char *data_ptr;

    data_ptr = (unsigned char *)clear;
    size=sizeof(Message);
    for(i=0;i<size;i++)
        *data_ptr++ = 0;
}



void msg_copy_data(Message *to, Message *from) {
    int size;
    int i;
    unsigned char *data_ptr;
    unsigned char *data_ptr_from;

    data_ptr = (unsigned char *)to;
    data_ptr_from = (unsigned char *)from;
    size=sizeof(Message);
    for(i=0;i<size;i++)
        *data_ptr++ = *data_ptr_from++;
}
#endif /* MSG_DEBUG */



// Returns 1 if it's time to update the messages again
int message_update_time () {
    if ( sec_now() > (last_message_update + message_update_delay) )
        return(1);
    else
        return(0);
}



int msg_comp_active(const void *a, const void *b) {
    char temp_a[MAX_CALLSIGN+MAX_CALLSIGN+MAX_MESSAGE_ORDER+2];
    char temp_b[MAX_CALLSIGN+MAX_CALLSIGN+MAX_MESSAGE_ORDER+2];

    sprintf(temp_a, "%c%s%s%s", ((Message*)a)->active, ((Message*)a)->call_sign, ((Message*)a)->from_call_sign,
            ((Message*)a)->seq);
    sprintf(temp_b, "%c%s%s%s", ((Message*)b)->active, ((Message*)b)->call_sign, ((Message*)b)->from_call_sign,
            ((Message*)b)->seq);
    return(strcmp(temp_a, temp_b));
}



int msg_comp_data(const void *a, const void *b) {
    char temp_a[MAX_CALLSIGN+MAX_CALLSIGN+MAX_MESSAGE_ORDER+1];
    char temp_b[MAX_CALLSIGN+MAX_CALLSIGN+MAX_MESSAGE_ORDER+1];

    sprintf(temp_a, "%s%s%s", msg_data[*(long*)a].call_sign, msg_data[*(long *)a].from_call_sign, msg_data[*(long *)a].seq);
    sprintf(temp_b, "%s%s%s", msg_data[*(long*)b].call_sign, msg_data[*(long *)b].from_call_sign, msg_data[*(long *)b].seq);
    return(strcmp(temp_a, temp_b));
}



void msg_input_database(Message *m_fill) {
#define MSG_INCREMENT 10
    void *m_ptr;
    long i;

    if (msg_index_end == msg_index_max) {
        for (i = 0; i < msg_index_end; i++) {
            if (msg_data[msg_index[i]].active == RECORD_NOTACTIVE) {
                memcpy(&msg_data[msg_index[i]], m_fill, sizeof(Message));
                qsort(msg_data, (size_t)msg_index_end, sizeof(Message), msg_comp_active);
                for (i = 0; i < msg_index_end; i++) {
                    msg_index[i] = i;
                    if (msg_data[i].active == RECORD_NOTACTIVE) {
                        msg_index_end = i;
                        break;
                    }
                }
                qsort(msg_index, (size_t)msg_index_end, sizeof(long *), msg_comp_data);
                return;
            }
        }
        m_ptr = realloc(msg_data, (msg_index_max+MSG_INCREMENT)*sizeof(Message));
        if (m_ptr) {
            msg_data = m_ptr;
            m_ptr = realloc(msg_index, (msg_index_max+MSG_INCREMENT)*sizeof(Message *));
            if (m_ptr) {
                msg_index = m_ptr;
                msg_index_max += MSG_INCREMENT;
            } else {
                XtWarning("Unable to allocate message index.\n");
            }
        } else {
            XtWarning("Unable to allocate message database.\n");
        }
    }
    if (msg_index_end < msg_index_max) {
        msg_index[msg_index_end] = msg_index_end;
        memcpy(&msg_data[msg_index_end++], m_fill, sizeof(Message));
        qsort(msg_index, (size_t)msg_index_end, sizeof(long *), msg_comp_data);
    }
}



long msg_find_data(Message *m_fill) {
    long record_start, record_mid, record_end, return_record, done;
    char tempfile[MAX_CALLSIGN+MAX_CALLSIGN+MAX_MESSAGE_ORDER+1];
    char tempfill[MAX_CALLSIGN+MAX_CALLSIGN+MAX_MESSAGE_ORDER+1];

    sprintf(tempfill, "%s%s%s", m_fill->call_sign, m_fill->from_call_sign, m_fill->seq);
    return_record = -1L;
    if (msg_index && msg_index_end >= 1) {
        /* more than one record */
         record_start=0L;
         record_end = (msg_index_end - 1);
         record_mid=(record_end-record_start)/2;

         done=0;
         while (!done) {
            /* get data for record start */
            sprintf(tempfile, "%s%s%s", msg_data[msg_index[record_start]].call_sign,
                        msg_data[msg_index[record_start]].from_call_sign,
                        msg_data[msg_index[record_start]].seq);
            if (strcmp(tempfill, tempfile) < 0) {
                /* filename comes before */
                /*printf("Before No data found!!\n");*/
                done=1;
                break;
            } else { /* get data for record end */
                sprintf(tempfile, "%s%s%s", msg_data[msg_index[record_end]].call_sign,
                            msg_data[msg_index[record_end]].from_call_sign,
                            msg_data[msg_index[record_end]].seq);
                if (strcmp(tempfill,tempfile)>=0) { /* at end or beyond */
                    if (strcmp(tempfill, tempfile) == 0)
                        return_record = record_end;
                    done=1;
                    break;
                } else if ((record_mid == record_start) || (record_mid == record_end)) {
                    /* no mid for compare check to see if in the middle */
                    done=1;
                    sprintf(tempfile, "%s%s%s", msg_data[msg_index[record_mid]].call_sign,
                                msg_data[msg_index[record_mid]].from_call_sign,
                                msg_data[msg_index[record_mid]].seq);
                    if(strcmp(tempfill,tempfile)==0)
                        return_record = record_mid;
                }
            }
            if (!done) { /* get data for record mid */
                sprintf(tempfile, "%s%s%s", msg_data[msg_index[record_mid]].call_sign,
                            msg_data[msg_index[record_mid]].from_call_sign,
                            msg_data[msg_index[record_mid]].seq);
                if (strcmp(tempfill, tempfile) == 0) {
                    return_record = record_mid;
                    done = 1;
                    break;
                }
                if(strcmp(tempfill, tempfile)<0)
                    record_end = record_mid;
                else
                    record_start = record_mid;

                record_mid = record_start+(record_end-record_start)/2;
            }
        }
    }
    return(return_record);
}



void msg_replace_data(Message *m_fill, long record_num) {
    memcpy(&msg_data[msg_index[record_num]], m_fill, sizeof(Message));
}



void msg_get_data(Message *m_fill, long record_num) {
    memcpy(m_fill, &msg_data[msg_index[record_num]], sizeof(Message));
}


void msg_data_add(char *call_sign, char *from_call, char *data, char *seq, char type, char from) {
    Message m_fill;
    long record;
    char time_data[MAX_TIME];

    substr(m_fill.call_sign, call_sign, MAX_CALLSIGN);
    (void)remove_trailing_asterisk(m_fill.call_sign);

    substr(m_fill.from_call_sign, from_call, MAX_CALLSIGN);
    substr(m_fill.seq, seq, MAX_MESSAGE_ORDER);

    record = msg_find_data(&m_fill);
    msg_clear_data(&m_fill);
    if(record != -1L) /* fill old data */
        msg_get_data(&m_fill, record);

    m_fill.sec_heard = sec_now();

    /* FROM */
    m_fill.data_via=from;
    m_fill.active=RECORD_ACTIVE;
    m_fill.type=type;
    if (m_fill.heard_via_tnc != VIA_TNC)
        m_fill.heard_via_tnc = (from == 'T') ? VIA_TNC : NOT_VIA_TNC;

    substr(m_fill.call_sign,call_sign,MAX_CALLSIGN);
    substr(m_fill.from_call_sign,from_call,MAX_CALLSIGN);
    substr(m_fill.message_line,data,MAX_MESSAGE_LENGTH);
    substr(m_fill.seq,seq,MAX_MESSAGE_ORDER);
    strcpy(m_fill.packet_time,get_time(time_data));

    if(record == -1L)
        msg_input_database(&m_fill);
    else
        msg_replace_data(&m_fill, record);

    /* display messages */

    if (type == MESSAGE_MESSAGE)
        all_messages(from,call_sign,from_call,data);
}



// WE7U:  What I'd like to do for the following routine:  Use
// XmTextGetInsertionPosition() or XmTextGetCursorPosition() to
// find the last of the text.  Could also save the position for
// each SendMessage window.  Compare the timestamps of messages
// found with the last update time.  If newer, then add them to
// the end.  This should stop the incessant scrolling.

// Another idea, easier method:  Create a buffer.  Snag out the
// messages from the array and sort by time.  Put them into a
// buffer.  Figure out the length of the text widget, and append
// the extra length of the buffer onto the end of the text widget.
// Once the message data is turned into a linked list, it might
// be sorted already by time, so this window will look better
// anyway.

// Calling update_messages with force == 1 will cause an update
// no matter what message_update_time() says.
void update_messages(int force) {
    static XmTextPosition pos;
    char temp1[MAX_CALLSIGN+1];
    char temp2[500];
    char stemp[20];
    long i;
    int mw_p;

    if ( message_update_time() || force) {

        //printf("Um %d\n",(int)sec_now() );

        /* go through all mw_p's! */

        // Perform this for each message window
        for (mw_p=0; msg_index && mw_p < MAX_MESSAGE_WINDOWS; mw_p++) {
            //pos=0;

begin_critical_section(&send_message_dialog_lock, "db.c:update_messages" );

            if (mw[mw_p].send_message_dialog!=NULL/* && mw[mw_p].message_group==1*/) {

                // Snag the callsign you're dealing with from the message dialogue
                strcpy(temp1,XmTextFieldGetString(mw[mw_p].send_message_call_data));

                new_message_data--;
                if (new_message_data<0)
                    new_message_data=0;

                if(strlen(temp1)>0) {   // We got a callsign from the dialogue
                    (void)remove_trailing_spaces(temp1);
                    (void)to_upper(temp1);

                    pos = 0;
                    // Loop through looking for messages to/from that callsign
                    for (i = 0; i < msg_index_end; i++) {
                        if (msg_data[msg_index[i]].active == RECORD_ACTIVE &&
                                (strcmp(temp1, msg_data[msg_index[i]].from_call_sign) == 0 ||
                                  strcmp(temp1,msg_data[msg_index[i]].call_sign) == 0) &&
                                ( is_my_call(msg_data[msg_index[i]].from_call_sign, 1) ||
                                  is_my_call(msg_data[msg_index[i]].call_sign, 1) ) ) {
 
                            // Message matches so snag the important pieces into a string
                            sprintf(stemp,"%c%c/%c%c %c%c:%c%c",
                                msg_data[msg_index[i]].packet_time[0], msg_data[msg_index[i]].packet_time[1],
                                msg_data[msg_index[i]].packet_time[2], msg_data[msg_index[i]].packet_time[3],
                                msg_data[msg_index[i]].packet_time[8], msg_data[msg_index[i]].packet_time[9],
                                msg_data[msg_index[i]].packet_time[10], msg_data[msg_index[i]].packet_time[11]
                            );

                            // Figure out whether it is from or to, put the arrow the right way
                            if(strcmp(temp1,msg_data[msg_index[i]].call_sign)==0)
                                sprintf(temp2,"%s  %-9s>%s\n",stemp,msg_data[msg_index[i]].from_call_sign,
                                    msg_data[msg_index[i]].message_line);
                            else
                                sprintf(temp2,"%s  %-9s<%s\n",stemp,msg_data[msg_index[i]].call_sign,
                                    msg_data[msg_index[i]].message_line);

                            // Replace the text from pos to pos+strlen(temp2) by the string "temp2"
                            XmTextReplace(mw[mw_p].send_message_text, pos, pos+strlen(temp2), temp2);
                            pos += strlen(temp2);
                        }
                    }
                    if (pos > 0)
                        XmTextReplace(mw[mw_p].send_message_text, --pos, XmTextGetLastPosition(mw[mw_p].send_message_text), "");
                }
            }

end_critical_section(&send_message_dialog_lock, "db.c:update_messages" );

        }
        last_message_update = sec_now();
    }
}



void mdelete_messages_from(char *from) {
    long i;

    for (i = 0; msg_index && i < msg_index_end; i++)
        if (strcmp(msg_data[i].call_sign, my_callsign) == 0 && strcmp(msg_data[i].from_call_sign, from) == 0)
            msg_data[i].active = RECORD_NOTACTIVE;
}



void mdelete_messages_to(char *to) {
    long i;

    for (i = 0; msg_index && i < msg_index_end; i++)
        if (strcmp(msg_data[i].call_sign, to) == 0)
            msg_data[i].active = RECORD_NOTACTIVE;
}



void mdelete_messages(char *to_from) {
    long i;

    for (i = 0; msg_index && i < msg_index_end; i++)
        if (strcmp(msg_data[i].call_sign, to_from) == 0 || strcmp(msg_data[i].from_call_sign, to_from) == 0)
            msg_data[i].active = RECORD_NOTACTIVE;
}



void mdata_delete_type(const char msg_type, const time_t reference_time) {
    long i;

    for (i = 0; msg_index && i < msg_index_end; i++)
        if ((msg_type == '\0' || msg_type == msg_data[i].type) &&
                msg_data[i].active == RECORD_ACTIVE && msg_data[i].sec_heard < reference_time)
            msg_data[i].active = RECORD_NOTACTIVE;
}



void check_message_remove(void) {       // called in timing loop

    if (last_message_remove < sec_now() - MESSAGE_REMOVE_CYCLE) {
        mdata_delete_type('\0', sec_now()-sec_remove);
        last_message_remove = sec_now();
    }
}



void mscan_file(char msg_type, void (*function)(Message *)) {
    long i;

    for (i = 0; msg_index && i < msg_index_end; i++)
        if ((msg_type == '\0' || msg_type == msg_data[msg_index[i]].type) &&
                msg_data[msg_index[i]].active == RECORD_ACTIVE)
            function(&msg_data[msg_index[i]]);
}



void mprint_record(Message *m_fill) {
    printf("%-9s>%-9s seq:%5s type:%c :%s\n", m_fill->from_call_sign, m_fill->call_sign,
        m_fill->seq, m_fill->type, m_fill->message_line);
}



void mdisplay_file(char msg_type) {
    printf("\n\n");
    mscan_file(msg_type, mprint_record);
    printf("\tmsg_index_end %ld, msg_index_max %ld\n", msg_index_end, msg_index_max);
}



/////////////////////////////////////// Station Data ///////////////////////////////////////////


void pad_callsign(char *callsignout, char *callsignin) {
    int i,l;

    l=(int)strlen(callsignin);
    for(i=0; i<9;i++)
        if(i<l)
            if(isalnum((int)callsignin[i]) || callsignin[i]=='-')
                callsignout[i]=callsignin[i];
            else
                callsignout[i] = ' ';
        else
            callsignout[i] = ' ';

    callsignout[i] = '\0';
}



void overlay_symbol(char symbol, char data, DataRow *fill) {
    if ( data != '/' && data !='\\') {
        /* Symbol overlay */
        fill->aprs_symbol.aprs_type = '\\';
        fill->aprs_symbol.special_overlay = data;
    } else {
        fill->aprs_symbol.aprs_type = data;
        fill->aprs_symbol.special_overlay=' ';
    }
    fill->aprs_symbol.aprs_symbol = symbol;
}



APRS_Symbol *id_callsign(char *call_sign, char * to_call) {
    char *ptr;
    char *id = "/aUfbYX's><OjRkv";
    char hold[MAX_CALLSIGN+1];
    int index;
    static APRS_Symbol symbol;

    symbol.aprs_symbol = '/';
    symbol.special_overlay = '\0';
    symbol.aprs_type ='/';
    ptr=strchr(call_sign,'-');
    if(ptr!=NULL)                      /* get symbol from SSID */
        if((index=atoi(ptr+1))<= 15)
            symbol.aprs_symbol = id[index];

    if (strncmp(to_call, "GPS", 3) == 0 || strncmp(to_call, "SPC", 3) == 0 || strncmp(to_call, "SYM", 3) == 0) {
        substr(hold, to_call+3, 3);
        if ((ptr = strpbrk(hold, "->,")) != NULL)
            *ptr = '\0';

        if (strlen(hold) >= 2) {
            switch (hold[0]) {
                case 'A':
                    symbol.aprs_type = '\\';

                case 'P':
                    if (('0' <= hold[1] && hold[1] <= '9') || ('A' <= hold[1] && hold[1] <= 'Z'))
                        symbol.aprs_symbol = hold[1];

                    break;

                case 'O':
                    symbol.aprs_type = '\\';

                case 'B':
                    switch (hold[1]) {
                        case 'B':
                            symbol.aprs_symbol = '!';
                            break;
                        case 'C':
                            symbol.aprs_symbol = '"';
                            break;
                        case 'D':
                            symbol.aprs_symbol = '#';
                            break;
                        case 'E':
                            symbol.aprs_symbol = '$';
                            break;
                        case 'F':
                            symbol.aprs_symbol = '%';
                            break;
                        case 'G':
                            symbol.aprs_symbol = '&';
                            break;
                        case 'H':
                            symbol.aprs_symbol = '\'';
                            break;
                        case 'I':
                            symbol.aprs_symbol = '(';
                            break;
                        case 'J':
                            symbol.aprs_symbol = ')';
                            break;
                        case 'K':
                            symbol.aprs_symbol = '*';
                            break;
                        case 'L':
                            symbol.aprs_symbol = '+';
                            break;
                        case 'M':
                            symbol.aprs_symbol = ',';
                            break;
                        case 'N':
                            symbol.aprs_symbol = '-';
                            break;
                        case 'O':
                            symbol.aprs_symbol = '.';
                            break;
                        case 'P':
                            symbol.aprs_symbol = '/';
                            break;
                    }
                    break;

                case 'D':
                    symbol.aprs_type = '\\';

                case 'H':
                    switch (hold[1]) {
                        case 'S':
                            symbol.aprs_symbol = '[';
                            break;
                        case 'T':
                            symbol.aprs_symbol = '\\';
                            break;
                        case 'U':
                            symbol.aprs_symbol = ']';
                            break;
                        case 'V':
                            symbol.aprs_symbol = '^';
                            break;
                        case 'W':
                            symbol.aprs_symbol = '_';
                            break;
                        case 'X':
                            symbol.aprs_symbol = '`';
                            break;
                    }
                    break;

                case 'N':
                    symbol.aprs_type = '\\';

                case 'M':
                    switch (hold[1]) {
                        case 'R':
                            symbol.aprs_symbol = ':';
                            break;
                        case 'S':
                            symbol.aprs_symbol = ';';
                            break;
                        case 'T':
                            symbol.aprs_symbol = '<';
                            break;
                        case 'U':
                            symbol.aprs_symbol = '=';
                            break;
                        case 'V':
                            symbol.aprs_symbol = '>';
                            break;
                        case 'W':
                            symbol.aprs_symbol = '?';
                            break;
                        case 'X':
                            symbol.aprs_symbol = '@';
                            break;
                    }
                    break;

                case 'Q':
                    symbol.aprs_type = '\\';

                case 'J':
                    switch (hold[1]) {
                        case '1':
                            symbol.aprs_symbol = '{';
                            break;
                        case '2':
                            symbol.aprs_symbol = '|';
                            break;
                        case '3':
                            symbol.aprs_symbol = '}';
                            break;
                        case '4':
                            symbol.aprs_symbol = '~';
                            break;
                    }
                    break;

                case 'S':
                    symbol.aprs_type = '\\';

                case 'L':
                    if ('A' <= hold[1] && hold[1] <= 'Z')
                        symbol.aprs_symbol = tolower((int)hold[1]);

                    break;
            }
            if (hold[2])
                symbol.special_overlay = hold[2];
        }
    }
    return(&symbol);
}



/******************************** Sort begin *************************** ****/

void  clear_sort_file(char *filename) {
    char ptr_filename[400];

    sprintf(ptr_filename,"%s-ptr",filename);
    (void)unlink(filename);
    (void)unlink(ptr_filename);
}



void sort_reset_pointers(FILE *pointer,long new_data_ptr,long records, int type, long start_ptr) {
    long cp;
    long temp[13000];
    long buffn,start_buffn;
    long cp_records;
    long max_buffer;
    int my_size;

    my_size=(int)sizeof(new_data_ptr);
    max_buffer=13000l;
    if(type==0) {
        /* before start_ptr */
        /* copy back pointers */
        cp=start_ptr;
        for(buffn=records; buffn > start_ptr; buffn-=max_buffer) {
            start_buffn=buffn-max_buffer;
            if(start_buffn<start_ptr)
                start_buffn=start_ptr;

            cp_records=buffn-start_buffn;
            (void)fseek(pointer,(my_size*start_buffn),SEEK_SET);
            if(fread(&temp,(my_size*cp_records),1,pointer)==1) {
                (void)fseek(pointer,(my_size*(start_buffn+1)),SEEK_SET);
                (void)fwrite(&temp,(my_size*cp_records),1,pointer);
            }
        }
        /* copy new pointer in */
        (void)fseek(pointer,(my_size*start_ptr),SEEK_SET);
        (void)fwrite(&new_data_ptr,(size_t)my_size,1,pointer);
    }
}



long sort_input_database(char *filename, char *fill, int size) {
    FILE *my_data;
    FILE *pointer;
    char file_data[2000];

    char ptr_filename[400];

    char tempfile[2000];
    char tempfill[2000];

    int ptr_size;
    long data_ptr;
    long new_data_ptr;
    long return_records;
    long records;
    long record_start;
    long record_end;
    long record_mid;
    int done;

    ptr_size=(int)sizeof(new_data_ptr);
    sprintf(ptr_filename,"%s-ptr",filename);
    /* get first string to sort on */
    (void)sscanf(fill,"%1999s",tempfill);

    data_ptr=0l;
    my_data=NULL;
    return_records=0l;
    pointer = fopen(ptr_filename,"r+");
    /* check if file is there */
    if(pointer == NULL)
      pointer = fopen(ptr_filename,"a+");

    if(pointer!=NULL) {
        my_data = fopen(filename,"a+");
        if(my_data!=NULL) {
            new_data_ptr = data_ptr = ftell(my_data);
            (void)fwrite(fill,(size_t)size,1,my_data);
            records = (data_ptr/size);
            return_records=records+1;
            if(records<1) {
                /* no data yet */
                (void)fseek(pointer,0l,SEEK_SET);
                (void)fwrite(&data_ptr,(size_t)ptr_size,1,pointer);
            } else {
                /* more than one record*/
                (void)fseek(pointer,(ptr_size*records),SEEK_SET);
                (void)fwrite(&data_ptr,(size_t)ptr_size,1,pointer);
                record_start=0l;
                record_end=records;
                record_mid=(record_end-record_start)/2;
                done=0;
                while(!done) {
                    /*printf("Records Start %ld, Mid %ld, End %ld\n",record_start,record_mid,record_end);*/
                    /* get data for record start */
                    (void)fseek(pointer,(ptr_size*record_start),SEEK_SET);
                    (void)fread(&data_ptr,(size_t)ptr_size,1,pointer);
                    (void)fseek(my_data,data_ptr,SEEK_SET);
                    if(fread(file_data,(size_t)size,1,my_data)==1) {
                        /* COMPARE HERE */
                        (void)sscanf(file_data,"%1999s",tempfile);
                        if(strcasecmp(tempfill,tempfile)<0) {
                            /* file name comes before */
                            /*printf("END - Before start\n");*/
                            done=1;
                            /* now place pointer before start*/
                            sort_reset_pointers(pointer,new_data_ptr,records,0,record_start);
                        } else {
                            /* get data for record end */
                            (void)fseek(pointer,(ptr_size*record_end),SEEK_SET);
                            (void)fread(&data_ptr,(size_t)ptr_size,1,pointer);
                            (void)fseek(my_data,data_ptr,SEEK_SET);
                            if(fread(file_data,(size_t)size,1,my_data)==1) {
                                /* COMPARE HERE */
                                (void)sscanf(file_data,"%1999s",tempfile);
                                if(strcasecmp(tempfill,tempfile)>0) {
                                    /* file name comes after */
                                    /*printf("END - After end\n");*/
                                    done=1;
                                    /* now place pointer after end */
                                } else {
                                    if((record_mid==record_start) || (record_mid==record_end)) {
                                        /* no mid for compare check to see if in the middle */
                                        /*printf("END - NO Middle\n");*/
                                        done=1;
                                        /* now place pointer before start*/
                                        if (record_mid==record_start)
                                            sort_reset_pointers(pointer,new_data_ptr,records,0,record_mid+1);
                                        else
                                            sort_reset_pointers(pointer,new_data_ptr,records,0,record_mid-1);
                                    } else {
                                        /* get data for record mid */
                                        (void)fseek(pointer,(ptr_size*record_mid),SEEK_SET);
                                        (void)fread(&data_ptr,(size_t)ptr_size,1,pointer);
                                        (void)fseek(my_data,data_ptr,SEEK_SET);
                                        if(fread(file_data,(size_t)size,1,my_data)==1) {
                                            /* COMPARE HERE */
                                            (void)sscanf(file_data,"%1999s",tempfile);
                                            if(strcasecmp(tempfill,tempfile)<0) {
                                                /* checking comes before */
                                                /*record_start=0l;*/
                                                record_end=record_mid;
                                                record_mid=record_start+(record_end-record_start)/2;
                                                /*printf("TOP %ld, mid %ld\n",record_mid,record_end);*/
                                            } else {
                                                /* checking comes after*/
                                                record_start=record_mid;
                                                /*record_end=end*/
                                                record_mid=record_start+(record_end-record_start)/2;
                                                /*printf("BOTTOM start %ld, mid %ld\n",record_start,record_mid);*/
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        } else
            printf("Could not open file %s\n",filename);
    } else
        printf("Could not open file %s\n",filename);

    if(my_data!=NULL)
        (void)fclose(my_data);

    if(pointer!=NULL)
        (void)fclose(pointer);

    return(return_records);
}

/******************** sort end **********************/


void display_station(Widget w, DataRow *fill, int single) {
    char temp_altitude[20];
    char temp_course[20];
    char temp_speed[20];
    char temp_call[20+1];
    char wx_tm[50];
    char temp_wx_temp[30];
    char temp_wx_wind[40];
    char temp_my_distance[20];
    char temp_my_course[20];
    char temp1_my_course[20];
    time_t temp_sec_heard;
    char unit[6];
    long l_lon, l_lat;
    WeatherRow *weather;
    char orient;
    float value;


    if (!symbol_display)
        return;

    /* setup call string for display */
    if (symbol_callsign_display)
        strcpy(temp_call,fill->call_sign);
    else
        strcpy(temp_call,"");

    /* setup altitude string for display */
    if (symbol_alt_display && strlen(fill->altitude)>0) {
        if(units_english_metric)
            sprintf(temp_altitude,"%.0fft",atof(fill->altitude)*3.28084);
        else
            sprintf(temp_altitude,"%.0fm",atof(fill->altitude));
    } else
        strcpy(temp_altitude,"");

    /* setup speed and course strings for display */
    strcpy(temp_speed,"");
    strcpy(temp_course,"");

    if (!(strlen(fill->course)>0 && strlen(fill->speed)>0 && atof(fill->course) == 0 && atof(fill->speed) == 0)) {
        /* don't display 'fixed' stations speed and course */
        if (symbol_speed_display && strlen(fill->speed)>0) {
            if (units_english_metric)
                sprintf(temp_speed,"%.0fmph",atof(fill->speed)*1.1508);
            else
                sprintf(temp_speed,"%.0fkm/h",atof(fill->speed)*1.852);
        }

        if(symbol_course_display && strlen(fill->course)>0)
            sprintf(temp_course,"%.0f",atof(fill->course));
    }

    /* setup distance and direction strings for display */
    if (symbol_dist_course_display && strcmp(fill->call_sign,my_callsign)!=0) {
        l_lat = convert_lat_s2l(my_lat);
        l_lon = convert_lon_s2l(my_long);
        value = (float)calc_distance_course(l_lat,l_lon,fill->coord_lat,fill->coord_lon,temp1_my_course); /* knots */
        if (units_english_metric) {
            value *= 1.15078;
            strcpy(unit,"mi");
        } else {
            value *= 1.852;
            strcpy(unit,"km");
        }
        if (value < 5.0)
            sprintf(temp_my_distance,"%0.1f%s",value,unit);
        else
            sprintf(temp_my_distance,"%0.0f%s",value,unit);

        sprintf(temp_my_course,"%.0f",atof(temp1_my_course));
    } else {
        strcpy(temp_my_distance,"");
        strcpy(temp_my_course,"");
    }

    /* setup weather strings for display */
    strcpy(temp_wx_temp,"");
    strcpy(temp_wx_wind,"");
    if (symbol_weather_display && fill->weather_data != NULL) {
        weather = fill->weather_data;
        if (strlen(weather->wx_temp) > 0) {
            if (units_english_metric)
                sprintf(temp_wx_temp,"T:%.0fF ",atof(weather->wx_temp));
            else
            sprintf(temp_wx_temp,"T:%.0fC ",((atof(weather->wx_temp)-32.0)*5.0)/9.0);
        }

        if (strlen(weather->wx_hum) > 0) {
            sprintf(wx_tm,"H:%.0f%%",atof(weather->wx_hum));
            strcat(temp_wx_temp,wx_tm);
        }

        if (temp_wx_temp[strlen(temp_wx_temp)-1] == ' ')
            temp_wx_temp[strlen(temp_wx_temp)-1] = '\0';  // delete blank at EOL

        if (strlen(weather->wx_speed) > 0) {
            if (units_english_metric)
                sprintf(temp_wx_wind,"S:%.0fmph ",atof(weather->wx_speed));
            else
                sprintf(temp_wx_wind,"S:%.0fkm/h ",atof(weather->wx_speed)*1.6094);
        }

        if (strlen(weather->wx_gust) > 0) {
            if (units_english_metric)
                sprintf(wx_tm,"G:%.0fmph ",atof(weather->wx_gust));
            else
                sprintf(wx_tm,"G:%.0fkm/h ",atof(weather->wx_gust)*1.6094);
            strcat(temp_wx_wind,wx_tm);
        }

        if (strlen(weather->wx_course) > 0) {
            sprintf(wx_tm,"C:%.0f",atof(weather->wx_course));
            strcat(temp_wx_wind,wx_tm);
        }

        if (temp_wx_wind[strlen(temp_wx_wind)-1] == ' ')
            temp_wx_wind[strlen(temp_wx_wind)-1] = '\0';  // delete blank at EOL
    }

    (void)remove_trailing_asterisk(fill->call_sign);  // DK7IN: is this needed here?

    if (symbol_rotate)
        orient = symbol_orient(fill->course);
    else
        orient = ' ';

    temp_sec_heard = (strcmp(fill->call_sign, my_callsign) == 0) ? sec_now(): fill->sec_heard;
    if (single) {
        draw_symbol(w,fill->aprs_symbol.aprs_type,fill->aprs_symbol.aprs_symbol,fill->aprs_symbol.special_overlay,
        fill->coord_lon,fill->coord_lat,temp_call,temp_altitude,temp_course,temp_speed,temp_my_distance,
                temp_my_course,temp_wx_temp,temp_wx_wind,temp_sec_heard,XtWindow(da),orient);

        temp_sec_heard = fill->sec_heard;    // DK7IN: ???
    }
    draw_symbol(w,fill->aprs_symbol.aprs_type,fill->aprs_symbol.aprs_symbol,fill->aprs_symbol.special_overlay,
        fill->coord_lon,fill->coord_lat,temp_call,temp_altitude,temp_course,temp_speed,temp_my_distance,
            temp_my_course,temp_wx_temp,temp_wx_wind,temp_sec_heard,pixmap_final,orient);

    if (show_phg && strlen(fill->power_gain)>3) {
        /*printf("PHG:%s\n",fill->power_gain);*/
        draw_phg(fill->coord_lon,fill->coord_lat,fill->power_gain,temp_sec_heard,pixmap_final);
        if (single) { // data_add       ????
            draw_phg(fill->coord_lon,fill->coord_lat,fill->power_gain,temp_sec_heard,XtWindow(w));
        }
    }
}


/*
 *  Display all stations on screen (trail, symbol, info text)
 */
void display_file(Widget w) {
    DataRow *p_station;         // pointer to station data
    char temp_call[20+1];
    time_t temp_sec_heard;      // time last heard
    int show_station;
    int altnet_match;
    char *net_ptr;
    time_t t_clr, t_old;

    if(debug_level & 1)
        printf("Display File Start\n");

    t_old = sec_now() - sec_old;        // precalc compare times
    t_clr = sec_now() - sec_clear;
    temp_sec_heard = 0l;
    p_station = t_first;                // start with oldest station, have newest on top
    while (p_station != NULL) {
        if ((p_station->flag & ST_ACTIVE) != '\0') {       // ignore deleted objects
            substr(temp_call, p_station->node_path, MAX_CALL);
            if ((net_ptr = strchr(temp_call, ',')))
                *net_ptr = '\0';

            for (altnet_match = (int)strlen(altnet_call); altnet && altnet_call[altnet_match-1] == '*'; altnet_match--);
                show_station = altnet ? (!strncmp(temp_call, altnet_call, (size_t)altnet_match) || !strcmp(temp_call, "local") ||
                                     !strncmp(temp_call, "SPC", 3) || !strcmp(temp_call, "SPECL") ||
                                     is_my_call(p_station->call_sign,1)): (int)TRUE;

            temp_sec_heard = (is_my_call(p_station->call_sign,1))? sec_now(): p_station->sec_heard;

            if ((p_station->flag & ST_INVIEW) != '\0') {  // skip far away stations...
                // we make better use of the In View flag in the future
                if (show_station) {
                    if (station_trails && p_station->track_data != NULL) {
                        // ????????????   what is the difference? :
                        if (temp_sec_heard > t_clr) {      // Not too old, so draw trail
                            if (temp_sec_heard > t_old)    // New trail, so draw solid trail
                                draw_trail(w,p_station,1);
                            else
                                draw_trail(w,p_station,0);
                        }
                    }
                    display_station(w,p_station,0);
                }
            }
        }
        p_station = p_station->t_next;  // next station
    }
    if (debug_level & 1)
        printf("Display File Stop\n");
}



/* delete track of current station */
void Station_data_destroy_track( /*@unused@*/ Widget widget, /*@unused@*/ XtPointer clientData, /*@unused@*/ XtPointer callData) {

    if (delete_trail(p_trail_del))
        redraw_on_new_data = 2;         // redraw immediately
}



/******** station info box **********/

void Station_data_destroy_shell(/*@unused@*/ Widget widget, XtPointer clientData, /*@unused@*/ XtPointer callData) {
    Widget shell = (Widget) clientData;
    XtPopdown(shell);

begin_critical_section(&db_station_info_lock, "db.c:Station_data_destroy_shell" );

    XtDestroyWidget(shell);
    db_station_info = (Widget)NULL;

end_critical_section(&db_station_info_lock, "db.c:Station_data_destroy_shell" );

}



void Station_data_add_fcc(Widget w, XtPointer clientData, /*@unused@*/ XtPointer calldata) {
    char temp[500];
    FccAppl my_data;
    char *station = (char *) clientData;

    (void)check_fcc_data();
    busy_cursor(XtParent(w));
    if (search_fcc_data_appl(station, &my_data)==1) {
        /*printf("FCC call %s\n",station);*/
        sprintf(temp,"%s\n%s %s\n%s %s %s\n%s %s, %s %s, %s %s\n\n",
        langcode("STIFCC0001"),
        langcode("STIFCC0003"),my_data.name_licensee,
        langcode("STIFCC0004"),my_data.text_street,my_data.text_pobox,
        langcode("STIFCC0005"),my_data.city,
        langcode("STIFCC0006"),my_data.state,
        langcode("STIFCC0007"),my_data.zipcode);
        XmTextInsert(si_text,0,temp);
        XmTextShowPosition(si_text,0);
    }
}



void Station_data_add_rac(Widget w, XtPointer clientData, /*@unused@*/ XtPointer calldata) {
    char temp[512];
    char club[512];
    rac_record my_data;
    char *station = (char *) clientData;

    strncpy(temp," ",sizeof(temp));
    (void)check_rac_data();
    busy_cursor(XtParent(w));
    if (search_rac_data(station, &my_data)==1) {
        /*printf("IC call %s\n",station);*/
        /*   sprintf(temp,"RAC Database Lookup\n%15.15s %20.20s\n%31.31s\n%20.20s\n%2.2s %7.7s\n", */
        sprintf(temp,"%s\n%s %s\n%s\n%s, %s\n%s\n",langcode("STIFCC0002"),my_data.first_name,my_data.last_name,my_data.address,
                my_data.city,my_data.province,my_data.postal_code);

        if (my_data.qual_a[0] == 'A')
            strcat(temp,langcode("STIFCC0008"));

        if (my_data.qual_d[0] == 'D')
            strcat(temp,langcode("STIFCC0009"));

        if (my_data.qual_b[0] == 'B' && my_data.qual_c[0] != 'C')
            strcat(temp,langcode("STIFCC0010"));

        if (my_data.qual_c[0] == 'C')
            strcat(temp,langcode("STIFCC0011"));

        strcat(temp,"\n");
        if (strlen(my_data.club_name) > 1){
            sprintf(club,"%s\n%s\n%s, %s\n%s\n",my_data.club_name,my_data.club_address,my_data.club_city,my_data.club_province,my_data.club_postal_code);
            strcat(temp,club);
        }
        strcat(temp,"\n");
        XmTextInsert(si_text,0,temp);
        XmTextShowPosition(si_text,0);
    }
}



void Station_query_trace(/*@unused@*/ Widget w, XtPointer clientData, /*@unused@*/ XtPointer calldata) {
    char *station = (char *) clientData;
    char temp[50];
    char call[25];

    pad_callsign(call,station);
    sprintf(temp,":%s:?APRST",call);
    transmit_message_data(station,temp);
}



void Station_query_messages(/*@unused@*/ Widget w, XtPointer clientData, /*@unused@*/ XtPointer calldata) {
    char *station = (char *) clientData;
    char temp[50];
    char call[25];

    pad_callsign(call,station);
    sprintf(temp,":%s:?APRSM",call);
    transmit_message_data(station,temp);
}



void Station_query_direct(/*@unused@*/ Widget w, XtPointer clientData, /*@unused@*/ XtPointer calldata) {
    char *station = (char *) clientData;
    char temp[50];
    char call[25];

    pad_callsign(call,station);
    sprintf(temp,":%s:?APRSD",call);
    transmit_message_data(station,temp);
}



void Station_query_version(/*@unused@*/ Widget w, XtPointer clientData, /*@unused@*/ XtPointer calldata) {
    char *station = (char *) clientData;
    char temp[50];
    char call[25];

    pad_callsign(call,station);
    sprintf(temp,":%s:?VER",call);
    transmit_message_data(station,temp);
}



void General_query(/*@unused@*/ Widget w, XtPointer clientData, /*@unused@*/ XtPointer calldata) {
    char *location = (char *) clientData;
    char temp[50];

    sprintf(temp,"?APRS?%s",location);
    output_my_data(temp,-1,0);
}



void IGate_query(/*@unused@*/ Widget w, /*@unused@*/ XtPointer clientData, /*@unused@*/ XtPointer calldata) {
    output_my_data("?IGATE?",-1,0);
}



void WX_query(/*@unused@*/ Widget w, /*@unused@*/ XtPointer clientData, /*@unused@*/ XtPointer calldata) {
    output_my_data("?WX?",-1,0);
}



/* List station info and trail */
void Station_data(/*@unused@*/ Widget w, XtPointer clientData, /*@unused@*/ XtPointer calldata) {
    DataRow *p_station;
    char *station = (char *) clientData;
    char temp[300];
    int pos, last_pos;
    unsigned int n;
    Atom delw;
    static Widget  pane, form, button_cancel, button_message, button_fcc,
      button_rac, button_clear_track, button_trace, button_messages,
      button_direct, button_version, station_icon, station_call;
#ifdef STATION_DEBUG
    static Widget station_type;
#endif /* STATION_DEBUG */
    Arg args[20];
    int last;
    char temp_my_distance[20];
    char temp_my_course[20];
    char temp1_my_course[20];
    float temp_out_C, e, humidex;
    Pixmap icon;
    long l_lat, l_lon;
    float value;
    WeatherRow *weather;

    temp_out_C=0;
    pos=0;

    if (db_station_info != NULL)
        Station_data_destroy_shell(db_station_info, db_station_info, NULL);

    if (db_station_info == NULL) {

begin_critical_section(&db_station_info_lock, "db.c:Station_data" );

        db_station_info = XtVaCreatePopupShell(langcode("WPUPSTI001"),xmDialogShellWidgetClass,Global.top,
                        XmNdeleteResponse,XmDESTROY,
                        XmNdefaultPosition, FALSE,
                        NULL);

        pane = XtVaCreateWidget("pane",xmPanedWindowWidgetClass, db_station_info,
                        XmNbackground, colors[0xff],
                        NULL);

        form =  XtVaCreateWidget("form2",xmFormWidgetClass, pane,
                        XmNfractionBase, 4,
                        XmNbackground, colors[0xff],
                        XmNautoUnmanage, FALSE,
                        XmNshadowThickness, 1,
                        NULL);

        if (search_station_name(&p_station,station,1)           // find call
                  && (p_station->flag & ST_ACTIVE) != '\0') {      // ignore deleted objects
            icon = XCreatePixmap(XtDisplay(appshell),RootWindowOfScreen(XtScreen(appshell)),
                    20,20,DefaultDepthOfScreen(XtScreen(appshell)));

            symbol(db_station_info,0,p_station->aprs_symbol.aprs_type,
                p_station->aprs_symbol.aprs_symbol,
                p_station->aprs_symbol.special_overlay,icon,0,0,0,' ');

            station_icon = XtVaCreateManagedWidget("icon", xmLabelWidgetClass, form,
                                XmNtopAttachment, XmATTACH_FORM,
                                XmNtopOffset, 2,
                                XmNbottomAttachment, XmATTACH_NONE,
                                XmNleftAttachment, XmATTACH_FORM,
                                XmNleftOffset, 5,
                                XmNrightAttachment, XmATTACH_NONE,
                                XmNlabelType, XmPIXMAP,
                                XmNlabelPixmap,icon,
                                XmNbackground, colors[0xff],
                                NULL);
#ifdef STATION_DEBUG
            station_type = XtVaCreateManagedWidget("type", xmTextFieldWidgetClass, form,
                                XmNeditable,   FALSE,
                                XmNcursorPositionVisible, FALSE,
                                XmNshadowThickness,       0,
                                XmNcolumns,15,
                                XmNwidth,((15*7)+2),
                                XmNbackground, colors[0xff],
                                XmNtopAttachment,XmATTACH_FORM,
                                XmNtopOffset, 2,
                                XmNbottomAttachment,XmATTACH_NONE,
                                XmNleftAttachment, XmATTACH_WIDGET,
                                XmNleftWidget,station_icon,
                                XmNleftOffset,10,
                                XmNrightAttachment,XmATTACH_NONE,
                                NULL);

            sprintf(temp, "%c%c%c", p_station->aprs_symbol.aprs_type, 
                p_station->aprs_symbol.aprs_symbol,
                    p_station->.aprs_symbol.special_overlay);

            XmTextFieldSetString(station_type, temp);
            XtManageChild(station_type);
#endif /* STATION_DEBUG */

            station_call = XtVaCreateManagedWidget("call", xmTextFieldWidgetClass, form,
                                XmNeditable,   FALSE,
                                XmNcursorPositionVisible, FALSE,
                                XmNshadowThickness,       0,
                                XmNcolumns,15,
                                XmNwidth,((15*7)+2),
                                XmNbackground, colors[0xff],
                                XmNtopAttachment,XmATTACH_FORM,
                                XmNtopOffset, 2,
                                XmNbottomAttachment,XmATTACH_NONE,
                                XmNleftAttachment, XmATTACH_WIDGET,
#ifdef STATION_DEBUG
                                XmNleftWidget,station_type,
#else
                                XmNleftWidget,station_icon,
#endif /* STATION_DEBUG */
                                XmNleftOffset,10,
                                XmNrightAttachment,XmATTACH_NONE,
                                NULL);

            XmTextFieldSetString(station_call,p_station->call_sign);
            XtManageChild(station_call);

            n=0;
            XtSetArg(args[n], XmNrows, 10); n++;
            XtSetArg(args[n], XmNcolumns, 100); n++;
            XtSetArg(args[n], XmNeditable, FALSE); n++;
            XtSetArg(args[n], XmNeditMode, XmMULTI_LINE_EDIT); n++;
            XtSetArg(args[n], XmNwordWrap, TRUE); n++;
            XtSetArg(args[n], XmNbackground, colors[0xff]); n++;
            XtSetArg(args[n], XmNscrollHorizontal, FALSE); n++;
            XtSetArg(args[n], XmNcursorPositionVisible, FALSE); n++;
            XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
            XtSetArg(args[n], XmNtopWidget, station_icon); n++;
            XtSetArg(args[n], XmNtopOffset, 5); n++;
            XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE); n++;
            XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
            XtSetArg(args[n], XmNleftOffset, 5); n++;
            XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
            XtSetArg(args[n], XmNrightOffset, 5); n++;

            si_text = NULL;
            si_text = XmCreateScrolledText(form,"Station Data",args,n);

            /* Packets received ... */
            sprintf(temp,langcode("WPUPSTI005"),p_station->num_packets);
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            strncpy(temp,p_station->packet_time,2);
            temp[2]='/';
            temp[3]='\0';
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            strncpy(temp,p_station->packet_time+2,2);
            temp[2]='/';
            temp[3]='\0';
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            strncpy(temp,p_station->packet_time+4,4);
            temp[4]=' ';
            temp[5]='\0';
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            strncpy(temp,p_station->packet_time+8,2);
            temp[2]=':';
            temp[3]='\0';
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            strncpy(temp,p_station->packet_time+10,2);
            temp[2]=':';
            temp[3]='\0';
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            strncpy(temp,p_station->packet_time+12,2);
            temp[2]='\n';
            temp[3]='\0';
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            // Object
            if (strlen(p_station->origin) > 0) {
                sprintf(temp,langcode("WPUPSTI000"),p_station->origin);
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
                sprintf(temp,"\n");
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
            }

            /* Heard via TNC ... */
            if ((p_station->flag & ST_VIATNC) != '\0') {        // test "via TNC" flag
                sprintf(temp,langcode("WPUPSTI006"),p_station->heard_via_tnc_port);
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
            } else {
                sprintf(temp,langcode("WPUPSTI007"));
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
            }

            switch(p_station->data_via) {
                case('L'):
                    sprintf(temp,langcode("WPUPSTI008"));
                    break;

                case('T'):
                    sprintf(temp,langcode("WPUPSTI009"),p_station->last_port_heard);
                    break;

                case('I'):
                    sprintf(temp,langcode("WPUPSTI010"),p_station->last_port_heard);
                    break;

                case('F'):
                    sprintf(temp,langcode("WPUPSTI011"));
                    break;

                default:
                    sprintf(temp,langcode("WPUPSTI012"));
                    break;
            }
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            if (p_station->track_data != NULL) {
                sprintf(temp,langcode("WPUPSTI013"));
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
            }
            sprintf(temp,"\n");
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            /* Data Path ... */
            sprintf(temp,langcode("WPUPSTI043"),p_station->node_path);
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);
            sprintf(temp,"\n");
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            /* Comments ... */
            if(strlen(p_station->comments)>0) {
                sprintf(temp,langcode("WPUPSTI044"),p_station->comments);
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
                sprintf(temp,"\n");
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
            }

            /* Current Power Gain ... */
            sprintf(temp,langcode("WPUPSTI014"),p_station->power_gain);
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);
            sprintf(temp,"\n");
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            /* Altitude ... */
            last_pos=pos;
            if (strlen(p_station->altitude) > 0) {
                if (units_english_metric)
                    sprintf(temp,langcode("WPUPSTI016"),atof(p_station->altitude)*3.28084,"ft");
                else
                    sprintf(temp,langcode("WPUPSTI016"),atof(p_station->altitude),"m");
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
            }

            /* Course ... */
            if (strlen(p_station->course) > 0) {
                sprintf(temp,langcode("WPUPSTI017"),p_station->course);
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
            }

            /* Speed ... */
            if (strlen(p_station->speed) > 0) {
                if (units_english_metric)
                    sprintf(temp,langcode("WPUPSTI019"),atof(p_station->speed)*1.1508);
                else
                    sprintf(temp,langcode("WPUPSTI018"),atof(p_station->speed)*1.852);
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
            }

            if (last_pos!=pos) {
                sprintf(temp,"\n");
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
            }

            /* Distance ... */
            last_pos = pos;
            /* do my course */
            if (!is_my_call(p_station->call_sign,1)) {
                l_lat = convert_lat_s2l(my_lat);
                l_lon = convert_lon_s2l(my_long);
                value = (float)calc_distance_course(l_lat,l_lon,p_station->coord_lat,
                             p_station->coord_lon,temp1_my_course); /* knots */
                if (units_english_metric)
                    sprintf(temp_my_distance,langcode("WPUPSTI020"),(value*1.15078));
                else
                    sprintf(temp_my_distance,langcode("WPUPSTI021"),(value*1.852));

                sprintf(temp_my_course,"%s",temp1_my_course);
                sprintf(temp,langcode("WPUPSTI022"),temp_my_distance,temp_my_course);
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
            }

            if(last_pos!=pos) {
                sprintf(temp,"\n");
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
            }

            /* Last Position ... */
            sprintf(temp,langcode("WPUPSTI023"));
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            convert_lat_l2s(p_station->coord_lat, temp, CONVERT_HP_NORMAL);
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            sprintf(temp,"  ");
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            convert_lon_l2s(p_station->coord_lon, temp, CONVERT_HP_NORMAL);
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            sprintf(temp,"\n");
            XmTextInsert(si_text,pos,temp);
            pos += strlen(temp);

            // list rest of trail data
            if (p_station->track_data != NULL) {
                if ((p_station->track_data->trail_inp - p_station->track_data->trail_out-1+MAX_TRACKS)
                          %MAX_TRACKS +1 > 1) {       // we need at least a second point
                    for (last=(p_station->track_data->trail_inp-2+MAX_TRACKS)%MAX_TRACKS;
                                (last+1-p_station->track_data->trail_out+MAX_TRACKS)%MAX_TRACKS > 0;
                                last = (last-1+MAX_TRACKS)%MAX_TRACKS) {
                        sprintf(temp,"               ");
                        XmTextInsert(si_text,pos,temp);
                        pos += strlen(temp);
                        convert_lat_l2s(p_station->track_data->trail_lat_pos[last], temp, CONVERT_HP_NORMAL);
                        XmTextInsert(si_text,pos,temp);
                        pos += strlen(temp);
                        sprintf(temp,"  ");
                        XmTextInsert(si_text,pos,temp);
                        pos += strlen(temp);
                        convert_lon_l2s(p_station->track_data->trail_long_pos[last], temp, CONVERT_HP_NORMAL);
                        XmTextInsert(si_text,pos,temp);
                        pos += strlen(temp);
                        sprintf(temp,"\n");
                        XmTextInsert(si_text,pos,temp);
                        pos += strlen(temp);
                    }
                }
            }

            /* Weather Data ... */
            if (p_station->weather_data != NULL) { 
                weather = p_station->weather_data;
                sprintf(temp,"\n");
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
                sprintf(temp,langcode("WPUPSTI024"),weather->wx_type,weather->wx_station);
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
                sprintf(temp,"\n");
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);
                if (units_english_metric)
                    sprintf(temp,langcode("WPUPSTI026"),weather->wx_course,weather->wx_speed);
                else
                    sprintf(temp,langcode("WPUPSTI025"),weather->wx_course,(int)(atof(weather->wx_speed)*1.6094));
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);

                if (strlen(weather->wx_gust) > 0) {
                    if (units_english_metric)
                        sprintf(temp,langcode("WPUPSTI028"),weather->wx_gust);
                    else
                        sprintf(temp,langcode("WPUPSTI027"),(int)(atof(weather->wx_gust)*1.6094));
                    strcat(temp,"\n");
                } else
                    sprintf(temp,"\n");
                XmTextInsert(si_text, pos, temp);
                pos += strlen(temp);

                if (strlen(weather->wx_temp) > 0) {
                    if (units_english_metric)
                        sprintf(temp,langcode("WPUPSTI030"),weather->wx_temp);
                    else {
                        temp_out_C =(((atof(weather->wx_temp)-32)*5.0)/9.0);
                        sprintf(temp,langcode("WPUPSTI029"),temp_out_C);
                    }
                    XmTextInsert(si_text,pos,temp);
                    pos += strlen(temp);
                }

                if (strlen(weather->wx_hum) > 0) {
                    sprintf(temp,langcode("WPUPSTI031"),weather->wx_hum);
                    XmTextInsert(si_text,pos,temp);
                    pos += strlen(temp);
                }

                // DK7IN: ??? units_english ???
                if (strlen(weather->wx_hum) > 0 && strlen(weather->wx_temp) > 0 && (!units_english_metric) &&
                        (atof(weather->wx_hum) > 0.0) ) {

                    e = (float)(6.112 * pow(10,(7.5 * temp_out_C)/(237.7 + temp_out_C)) * atof(weather->wx_hum) / 100.0);
                    humidex = (temp_out_C + ((5.0/9.0) * (e-10.0)));

                    sprintf(temp,langcode("WPUPSTI032"),humidex);
                    XmTextInsert(si_text,pos,temp);
                    pos += strlen(temp);
                }

                if (strlen(weather->wx_baro) > 0) {
                    sprintf(temp,langcode("WPUPSTI033"),weather->wx_baro);
                    XmTextInsert(si_text,pos,temp);
                    pos += strlen(temp);
                    sprintf(temp,"\n");
                    XmTextInsert(si_text,pos,temp);
                    pos += strlen(temp);
                } else {
                    if(last_pos!=pos) {
                        sprintf(temp,"\n");
                        XmTextInsert(si_text,pos,temp);
                        pos += strlen(temp);
                    }
                }

                if (strlen(weather->wx_snow) > 0) {
                    if(units_english_metric)
                        sprintf(temp,langcode("WPUPSTI035"),weather->wx_snow);
                    else
                        sprintf(temp,langcode("WPUPSTI034"),atof(weather->wx_snow)*2.54);
                    XmTextInsert(si_text,pos,temp);
                    pos += strlen(temp);
                    sprintf(temp,"\n");
                    XmTextInsert(si_text,pos,temp);
                    pos += strlen(temp);
                }

                if (strlen(weather->wx_rain) > 0 || strlen(weather->wx_prec_00) > 0
                        || strlen(weather->wx_prec_24) > 0) {
                    sprintf(temp,langcode("WPUPSTI036"));
                    XmTextInsert(si_text,pos,temp);
                    pos += strlen(temp);
                }

                if (strlen(weather->wx_rain) > 0) {
                    if (units_english_metric)
                        sprintf(temp,langcode("WPUPSTI038"),weather->wx_rain);
                    else
                        sprintf(temp,langcode("WPUPSTI037"),atof(weather->wx_rain)*.254);
                    XmTextInsert(si_text,pos,temp);
                    pos += strlen(temp);
                }

                if (strlen(weather->wx_prec_24) > 0) {
                    if(units_english_metric)
                        sprintf(temp,langcode("WPUPSTI040"),weather->wx_prec_24);
                    else
                        sprintf(temp,langcode("WPUPSTI039"),atof(weather->wx_prec_24)*.254);
                    XmTextInsert(si_text,pos,temp);
                    pos += strlen(temp);
                }

                if (strlen(weather->wx_prec_00) > 0) {
                    if (units_english_metric)
                        sprintf(temp,langcode("WPUPSTI042"),weather->wx_prec_00);
                    else
                        sprintf(temp,langcode("WPUPSTI041"),atof(weather->wx_prec_00)*.254);
                    XmTextInsert(si_text,pos,temp);
                    pos += strlen(temp);
                }

                if (strlen(weather->wx_rain_total) > 0) {
                    sprintf(temp,"\n%s",langcode("WPUPSTI046"));
                    XmTextInsert(si_text,pos,temp);
                    pos += strlen(temp);
                    if (units_english_metric)
                        sprintf(temp,langcode("WPUPSTI048"),weather->wx_rain_total);
                    else
                        sprintf(temp,langcode("WPUPSTI047"),atof(weather->wx_rain_total)*.254);
                    XmTextInsert(si_text,pos,temp);
                    pos += strlen(temp);
                }
                sprintf(temp,"\n\n");
                XmTextInsert(si_text,pos,temp);
                pos += strlen(temp);

            }
        }
        button_cancel = XtVaCreateManagedWidget(langcode("UNIOP00003"),xmPushButtonGadgetClass, form,
                            XmNtopAttachment, XmATTACH_WIDGET,
                            XmNtopWidget,XtParent(si_text),
                            XmNtopOffset,10,
                            XmNbottomAttachment, XmATTACH_NONE,
                            XmNleftAttachment, XmATTACH_POSITION,
                            XmNleftPosition, 3,
                            XmNrightAttachment, XmATTACH_POSITION,
                            XmNrightPosition, 4,
                            XmNrightOffset, 5,
                            XmNbackground, colors[0xff],
                            NULL);

        XtAddCallback(button_cancel, XmNactivateCallback, Station_data_destroy_shell, db_station_info);

        button_clear_track = XtVaCreateManagedWidget(langcode("WPUPSTI045"),xmPushButtonGadgetClass, form,
                                    XmNtopAttachment, XmATTACH_WIDGET,
                                    XmNtopWidget,XtParent(si_text),
                                    XmNtopOffset,10,
                                    XmNbottomAttachment, XmATTACH_NONE,
                                    XmNleftAttachment, XmATTACH_FORM,
                                    XmNleftOffset,5,
                                    XmNrightAttachment, XmATTACH_POSITION,
                                    XmNrightPosition, 1,
                                    XmNbackground, colors[0xff],
                                    NULL);

        p_trail_del = p_station;
        XtAddCallback(button_clear_track, XmNactivateCallback, Station_data_destroy_track ,NULL);

        button_message = XtVaCreateManagedWidget(langcode("WPUPSTI002"),xmPushButtonGadgetClass, form,
                                XmNtopAttachment, XmATTACH_WIDGET,
                                XmNtopWidget,XtParent(si_text),
                                XmNtopOffset,10,
                                XmNbottomAttachment, XmATTACH_NONE,
                                XmNleftAttachment, XmATTACH_POSITION,
                                XmNleftPosition, 1,
                                XmNrightAttachment, XmATTACH_POSITION,
                                XmNrightPosition, 2,
                                XmNbackground, colors[0xff],
                                NULL);

        XtAddCallback(button_message, XmNactivateCallback, Send_message_call ,(XtPointer)p_station->call_sign);

//        if (fcc_data_available>=1 && strncmp(station,"V",1)) {
        if (strncmp(station,"V",1)) {
            button_fcc = XtVaCreateManagedWidget(langcode("WPUPSTI003"),xmPushButtonGadgetClass, form,
                                XmNtopAttachment, XmATTACH_WIDGET,
                                XmNtopWidget,XtParent(si_text),
                                XmNtopOffset,10,
                                XmNbottomAttachment, XmATTACH_NONE,
                                XmNleftAttachment, XmATTACH_POSITION,
                                XmNleftPosition, 2,
                                XmNrightAttachment, XmATTACH_POSITION,
                                XmNrightPosition, 3,
                                XmNbackground, colors[0xff],
                                NULL);

            XtAddCallback(button_fcc, XmNactivateCallback, Station_data_add_fcc,(XtPointer)p_station->call_sign);

            if ( (! fcc_data_available)
                || (! (   (! strncmp(station,"A",1)) || (! strncmp(station,"K",1))
                       || (! strncmp(station,"N",1)) || (! strncmp(station,"W",1)) ) ) )
                XtSetSensitive(button_fcc,FALSE);
        }

//        if (rac_data_available==1 && (!strncmp(station,"VE",2) || !strncmp(station,"VA",2))) {
        if (!strncmp(station,"VE",2) || !strncmp(station,"VA",2)) {
            button_rac = XtVaCreateManagedWidget(langcode("WPUPSTI004"),xmPushButtonGadgetClass, form,
                                XmNtopAttachment, XmATTACH_WIDGET,
                                XmNtopWidget,XtParent(si_text),
                                XmNtopOffset,10,
                                XmNbottomAttachment, XmATTACH_NONE,
                                XmNleftAttachment, XmATTACH_POSITION,
                                XmNleftPosition, 2,
                                XmNrightAttachment, XmATTACH_POSITION,
                                XmNrightPosition, 3,
                                XmNbackground, colors[0xff],
                                NULL);

            XtAddCallback(button_rac, XmNactivateCallback, Station_data_add_rac,(XtPointer)p_station->call_sign);

            if (! rac_data_available)
                XtSetSensitive(button_rac,FALSE);
        }

        button_trace = XtVaCreateManagedWidget(langcode("WPUPSTI049"),xmPushButtonGadgetClass, form,
                                XmNtopAttachment, XmATTACH_WIDGET,
                                XmNtopWidget,button_clear_track,
                                XmNtopOffset,1,
                                XmNbottomAttachment, XmATTACH_FORM,
                                XmNbottomOffset,5,
                                XmNleftAttachment, XmATTACH_FORM,
                                XmNleftOffset,5,
                                XmNrightAttachment, XmATTACH_POSITION,
                                XmNrightPosition, 1,
                                XmNbackground, colors[0xff],
                                NULL);

        XtAddCallback(button_trace, XmNactivateCallback, Station_query_trace ,(XtPointer)p_station->call_sign);

        button_messages = XtVaCreateManagedWidget(langcode("WPUPSTI050"),xmPushButtonGadgetClass, form,
                                XmNtopAttachment, XmATTACH_WIDGET,
                                XmNtopWidget,button_clear_track,
                                XmNtopOffset,1,
                                XmNbottomAttachment, XmATTACH_FORM,
                                XmNbottomOffset,5,
                                XmNleftAttachment, XmATTACH_POSITION,
                                XmNleftPosition, 1,
                                XmNrightAttachment, XmATTACH_POSITION,
                                XmNrightPosition, 2,
                                XmNbackground, colors[0xff],
                                NULL);

        XtAddCallback(button_messages, XmNactivateCallback, Station_query_messages ,(XtPointer)p_station->call_sign);

        button_direct = XtVaCreateManagedWidget(langcode("WPUPSTI051"),xmPushButtonGadgetClass, form,
                                XmNtopAttachment, XmATTACH_WIDGET,
                                XmNtopWidget,button_clear_track,
                                XmNtopOffset,1,
                                XmNbottomAttachment, XmATTACH_FORM,
                                XmNbottomOffset,5,
                                XmNleftAttachment, XmATTACH_POSITION,
                                XmNleftPosition, 2,
                                XmNrightAttachment, XmATTACH_POSITION,
                                XmNrightPosition, 3,
                                XmNbackground, colors[0xff],
                                NULL);

        XtAddCallback(button_direct, XmNactivateCallback, Station_query_direct ,(XtPointer)p_station->call_sign);

        button_version = XtVaCreateManagedWidget(langcode("WPUPSTI052"),xmPushButtonGadgetClass, form,
                                XmNtopAttachment, XmATTACH_WIDGET,
                                XmNtopWidget,button_clear_track,
                                XmNtopOffset,1,
                                XmNbottomAttachment, XmATTACH_FORM,
                                XmNbottomOffset,5,
                                XmNleftAttachment, XmATTACH_POSITION,
                                XmNleftPosition, 3,
                                XmNrightAttachment, XmATTACH_POSITION,
                                XmNrightPosition, 4,
                                XmNrightOffset,5,
                                XmNbackground, colors[0xff],
                                NULL);

        XtAddCallback(button_version, XmNactivateCallback, Station_query_version ,(XtPointer)p_station->call_sign);

        pos_dialog(db_station_info);

        delw = XmInternAtom(XtDisplay(db_station_info),"WM_DELETE_WINDOW", FALSE);
        XmAddWMProtocolCallback(db_station_info, delw, Station_data_destroy_shell, (XtPointer)db_station_info);

        XtManageChild(form);
        XtManageChild(si_text);
        XtVaSetValues(si_text, XmNbackground, colors[0x0f], NULL);
        XtManageChild(pane);

end_critical_section(&db_station_info_lock, "db.c:Station_data" );

        XtPopup(db_station_info,XtGrabNone);

        fix_dialog_size(db_station_info);
        XmTextShowPosition(si_text,0);
    }
}



/****** station popup info box *********/

void Station_info_destroy_shell(/*@unused@*/ Widget widget, XtPointer clientData, /*@unused@*/ XtPointer callData) {
    Widget shell = (Widget) clientData;

    if (db_station_info!=NULL)
        Station_data_destroy_shell(db_station_info, db_station_info, NULL);

    XtPopdown(shell);

begin_critical_section(&db_station_popup_lock, "db.c:Station_info_destroy_shell" );

    XtDestroyWidget(shell);
    db_station_popup = (Widget)NULL;

end_critical_section(&db_station_popup_lock, "db.c:Station_info_destroy_shell" );
}



void Station_info_select_destroy_shell(Widget widget, /*@unused@*/ XtPointer clientData, /*@unused@*/ XtPointer callData) {
    int i,x;
    char *temp;
    char temp2[50];
    XmString *list;
    int found;
    Widget shell = (Widget) clientData;

    found=0;

begin_critical_section(&db_station_popup_lock, "db.c:Station_info_select_destroy_shell" );

    if (db_station_popup) {
        XtVaGetValues(station_list,XmNitemCount,&i,XmNitems,&list,NULL);

        for (x=1; x<=i;x++) {
            if (XmListPosSelected(station_list,x)) {
                found=1;
                if (XmStringGetLtoR(list[(x-1)],XmFONTLIST_DEFAULT_TAG,&temp))
                    x=i+1;
            }
        }

        if (found) {
            sprintf(temp2,"%s",temp);
            Station_data(widget,temp2,NULL);
        }
        XtPopdown(shell);                   // Get rid of the station chooser popup here
        XtDestroyWidget(shell);             // and here
        db_station_popup = (Widget)NULL;    // and here
    }

end_critical_section(&db_station_popup_lock, "db.c:Station_info_select_destroy_shell" );

}



void Station_info(Widget w, /*@unused@*/ XtPointer clientData, /*@unused@*/ XtPointer calldata) {
    DataRow *p_station;
    DataRow *p_found;
    int num_found;
    unsigned long min_diff,diff;
    XmString str_ptr;
    unsigned int n;
    Atom delw;
    static Widget pane, form, button_ok, button_cancel;
    Arg al[20];                    /* Arg List */
    register unsigned int ac = 0;           /* Arg Count */

    num_found=0;
    min_diff=32000000ul;
    p_found = NULL;
    p_station = n_first;
    while (p_station != NULL) {    // search through database for nearby stations
        if ((p_station->flag & ST_INVIEW) != '\0') {       // only test stations in view
            diff = (unsigned long)( labs((x_long_offset+(menu_x*size)) - p_station->coord_lon)
                 + labs((y_lat_offset+(menu_y*size))  - p_station->coord_lat) );
            if ((diff/size) < min_diff) {
                min_diff = (diff/size);
                p_found = p_station;
                num_found = 1;
            } else {
                if ((diff/size) <= (min_diff+2))
                    num_found++;
            }
        }
        p_station = p_station->n_next;
    }

    if (p_found != NULL && (min_diff < 250)) {
        if (num_found==1) { // We only found one station, so it's easy
            Station_data(w,p_found->call_sign,NULL);
        } else {            // We found more: create dialog to choose a station
            if (db_station_popup!=NULL)
                Station_info_destroy_shell(db_station_popup, db_station_popup, NULL);

begin_critical_section(&db_station_popup_lock, "db.c:Station_info" );

            if (db_station_popup==NULL) {

                db_station_popup = XtVaCreatePopupShell(langcode("STCHO00001"),xmDialogShellWidgetClass,Global.top,
                                    XmNdeleteResponse,XmDESTROY,
                                    XmNdefaultPosition, FALSE,
                                    XmNbackground, colors[0xff],
                                    NULL);

                pane = XtVaCreateWidget("pane",xmPanedWindowWidgetClass, db_station_popup,
                            XmNbackground, colors[0xff],
                            NULL);

                form =  XtVaCreateWidget("form2",xmFormWidgetClass, pane,
                            XmNfractionBase, 5,
                            XmNbackground, colors[0xff],
                            XmNautoUnmanage, FALSE,
                            XmNshadowThickness, 1,
                            NULL);

                /*set args for color */
                ac=0;
                XtSetArg(al[ac], XmNbackground, colors[0xff]); ac++;
                XtSetArg(al[ac], XmNvisibleItemCount, 6); ac++;
                XtSetArg(al[ac], XmNtraversalOn, FALSE); ac++;
                XtSetArg(al[ac], XmNshadowThickness, 3); ac++;
                XtSetArg(al[ac], XmNselectionPolicy, XmSINGLE_SELECT); ac++;
                XtSetArg(al[ac], XmNscrollBarPlacement, XmBOTTOM_RIGHT); ac++;
                XtSetArg(al[ac], XmNtopAttachment, XmATTACH_FORM); ac++;
                XtSetArg(al[ac], XmNtopOffset, 5); ac++;
                XtSetArg(al[ac], XmNbottomAttachment, XmATTACH_NONE); ac++;
                XtSetArg(al[ac], XmNrightAttachment, XmATTACH_FORM); ac++;
                XtSetArg(al[ac], XmNrightOffset, 5); ac++;
                XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM); ac++;
                XtSetArg(al[ac], XmNleftOffset, 5); ac++;

                station_list = XmCreateScrolledList(form,"list",al,ac);

                /*printf("What station\n");*/
                n=1;
                p_station = n_first;
                while (p_station != NULL) {    // search through database for nearby stations
                    if ((p_station->flag & ST_INVIEW) != '\0') {       // only test stations in view
                        diff = (unsigned long)( labs((x_long_offset+(menu_x*size)) - p_station->coord_lon)
                             + labs((y_lat_offset+(menu_y*size))  - p_station->coord_lat) );
                        if ((diff/size) <= min_diff+2) {
                            /*printf("Station %s\n",p_station->call_sign);*/
                            XmListAddItem(station_list, str_ptr = XmStringCreateLtoR(p_station->call_sign,
                                    XmFONTLIST_DEFAULT_TAG), (int)n++);
                            XmStringFree(str_ptr);
                        }
                    }
                    p_station = p_station->n_next;
                }
                button_ok = XtVaCreateManagedWidget("Info",xmPushButtonGadgetClass, form,
                                XmNtopAttachment, XmATTACH_WIDGET,
                                XmNtopWidget,XtParent(station_list),
                                XmNtopOffset,5,
                                XmNbottomAttachment, XmATTACH_FORM,
                                XmNbottomOffset, 5,
                                XmNleftAttachment, XmATTACH_POSITION,
                                XmNleftPosition, 1,
                                XmNrightAttachment, XmATTACH_POSITION,
                                XmNrightPosition, 2,
                                NULL);

                button_cancel = XtVaCreateManagedWidget(langcode("UNIOP00003"),xmPushButtonGadgetClass, form,
                                    XmNtopAttachment, XmATTACH_WIDGET,
                                    XmNtopWidget,XtParent(station_list),
                                    XmNtopOffset,5,
                                    XmNbottomAttachment, XmATTACH_FORM,
                                    XmNbottomOffset, 5,
                                    XmNleftAttachment, XmATTACH_POSITION,
                                    XmNleftPosition, 3,
                                    XmNrightAttachment, XmATTACH_POSITION,
                                    XmNrightPosition, 4,
                                    NULL);

                XtAddCallback(button_cancel, XmNactivateCallback, Station_info_destroy_shell, db_station_popup);
                XtAddCallback(button_ok, XmNactivateCallback, Station_info_select_destroy_shell, db_station_popup);

                pos_dialog(db_station_popup);

                delw = XmInternAtom(XtDisplay(db_station_popup),"WM_DELETE_WINDOW", FALSE);
                XmAddWMProtocolCallback(db_station_popup, delw, Station_info_destroy_shell, (XtPointer)db_station_popup);

                XtManageChild(form);
                XtManageChild(station_list);
                XtVaSetValues(station_list, XmNbackground, colors[0x0f], NULL);
                XtManageChild(pane);

                XtPopup(db_station_popup,XtGrabNone);

            }

end_critical_section(&db_station_popup_lock, "db.c:Station_info" );

        }
    }
}



int heard_via_tnc_in_past_hour(char *call) {
    DataRow *p_station;
    int in_hour;

    in_hour=0;
    if (search_station_name(&p_station,call,1)) {  // find call
        if((p_station->flag & ST_VIATNC) != '\0') {        // test "via TNC" flag
            /* check to see if the last packet was message capable? */
            /* ok look to see if it is within the hour */
            in_hour = (int)((p_station->heard_via_tnc_last_time+3600l) > sec_now());
            if(debug_level & 2)
                printf("Call %s: %ld %ld ok %d\n",call,(long)(p_station->heard_via_tnc_last_time),(long)sec_now(),in_hour);
        } else {
            if (debug_level & 2)
                printf("Call %s Not heard via tnc\n",call);
        }
    } else {
        if (debug_level & 2)
            printf("IG:station not found\n");
    }
    return(in_hour);
}



//////////////////////////////////// Weather Data //////////////////////////////////////////////////



/* valid characters for APRS weather data fields */
int is_aprs_chr(char ch) {

    if (isdigit((int)ch) || ch==' ' || ch=='.')
    return(1);
    else
    return(0);
}



/* check data format    123 ___ ... */
int is_weather_data(char *data, int len) {
    int ok = 1;
    int i;

    for (i=0;ok && i<len;i++)
        if (!is_aprs_chr(data[i]))
            ok = 0;
    return(ok);
}



/* extract single weather data item from information field */
int extract_weather_item(char *data, char type, int datalen, char *temp) {
    int i,ofs,found,len;

    found=0;
    len = (int)strlen(data);
    for(ofs=0; !found && ofs<len-datalen; ofs++)      // search for start sequence
        if (data[ofs]==type) {
            found=1;
            if (!is_weather_data(data+ofs+1, datalen))
                found=0;
        }
    if (found) {   // ofs now points after type character
        substr(temp,data+ofs,datalen);
        for (i=ofs-1;i<len-datalen;i++)        // delete item from info field
            data[i] = data[i+datalen+1];
    } else
        temp[0] = '\0';
    return(found);
}



// raw weather report            in information field
// positionless weather report   in information field
// complete weather report       with lat/lon
//  see APRS Reference page 62ff

int extract_weather(DataRow *fill, char *data) {
    char time_data[MAX_TIME];
    char temp[5];
    int  ok = 1;
    WeatherRow *weather;
    char course[4];
    char speed[4];

    if (strlen(data)>=15 && data[3]=='/'
                 && is_weather_data(data,3) && is_weather_data(&data[4],3) 
                 && data[7] =='g' && is_weather_data(&data[8], 3)
                 && data[11]=='t' && is_weather_data(&data[12],3)) {    // Complete Weather Report
        (void)extract_speed_course(data,speed,course);
        //    printf("found Complete Weather Report\n");
    } else
        // need date/timestamp
        if (strlen(data)>=16
                && data[0] =='c' && is_weather_data(&data[1], 3)
                && data[4] =='s' && is_weather_data(&data[5], 3)
                && data[8] =='g' && is_weather_data(&data[9], 3)
                && data[12]=='t' && is_weather_data(&data[13],3)) { // Positionless Weather Data
            (void)extract_weather_item(data,'c',3,course); // wind direction (in degress)
            (void)extract_weather_item(data,'s',3,speed);  // sustained one-minute wind speed (in mph)
            //        printf("found weather2\n");
    } else
        ok = 0;
    if (ok) {
        ok = get_weather_record(fill);    // get existing or create new weather record
    }
    if (ok) {
        weather = fill->weather_data;
        strcpy(weather->wx_speed, speed);
        strcpy(weather->wx_course,course);
    
        (void)extract_weather_item(data,'g',3,weather->wx_gust);      // gust (peak wind speed in mph in the last 5 minutes)

        (void)extract_weather_item(data,'t',3,weather->wx_temp);      // temperature (in deg Fahrenheit), could be negative

        (void)extract_weather_item(data,'r',3,weather->wx_rain);      // rainfall (1/100 inch) in the last hour

        (void)extract_weather_item(data,'p',3,weather->wx_prec_24);   // rainfall (1/100 inch) in the last 24 hours

        (void)extract_weather_item(data,'P',3,weather->wx_prec_00);   // rainfall (1/100 inch) since midnight

        if (extract_weather_item(data,'h',2,weather->wx_hum))   // humidity (in %, 00 = 100%)
        sprintf(weather->wx_hum,"%03d",(atoi(weather->wx_hum)+99)%100+1);

        if (extract_weather_item(data,'b',5,weather->wx_baro))  // barometric pressure (1/10 mbar / 1/10 hPascal)
        sprintf(weather->wx_baro,"%0.1f",(float)(atoi(weather->wx_baro)/10.0));

        (void)extract_weather_item(data,'s',3,weather->wx_snow);      // snowfall (in inches) in the last 24 hours
                                                                      // was 1/100 inch, APRS reference says inch! ??

        (void)extract_weather_item(data,'L',3,temp);                  // luminosity (in watts per square meter) 999 and below

        (void)extract_weather_item(data,'l',3,temp);                  // luminosity (in watts per square meter) 1000 and above

        (void)extract_weather_item(data,'#',3,temp);                  // raw rain counter
    
//    extract_weather_item(data,'w',3,temp);                          // ?? text wUII
    
    // now there should be the name of the weather station...

        strcpy(weather->wx_time,get_time(time_data));
        weather->wx_sec_time=sec_now();
//        weather->wx_data=1;  // we don't need this

//        case ('.'):/* skip */
//            wx_strpos+=4;
//            break;

//        default:
//            wx_done=1;
//            weather->wx_type=data[wx_strpos];
//            if(strlen(data)>wx_strpos+1)
//                strncpy(weather->wx_station,data+wx_strpos+1,MAX_WXSTATION);
//            break;
    }
    return(ok);
}                



////////////////////////////////////////////////////////////////////////////////////////////////////



void init_weather(WeatherRow *weather) {    // clear weather data

    weather->wx_sec_time      = 0; // ?? is 0 ok ??
    weather->wx_time[0]       = '\0';
    weather->wx_course[0]     = '\0';
    weather->wx_speed[0]      = '\0';
    weather->wx_speed_sec_time = 0; // ??
    weather->wx_gust[0]       = '\0';
    weather->wx_temp[0]       = '\0';
    weather->wx_rain[0]       = '\0';
    weather->wx_rain_total[0] = '\0';
    weather->wx_snow[0]       = '\0';
    weather->wx_prec_24[0]    = '\0';
    weather->wx_prec_00[0]    = '\0';
    weather->wx_hum[0]        = '\0';
    weather->wx_baro[0]       = '\0';
    weather->wx_type          = '\0';
    weather->wx_station[0]    = '\0';
}



int get_weather_record(DataRow *fill) {    // get or create weather storage
    int ok=1;

    if (fill->weather_data == NULL) {      // new weather data, allocate storage and init
        fill->weather_data = malloc(sizeof(WeatherRow));
        if (fill->weather_data == NULL)
            ok = 0;
        else
            init_weather(fill->weather_data);
    }
    return(ok);
}



int delete_weather(DataRow *fill) {    // delete weather storage, if allocated

    if (fill->weather_data != NULL) {
        free(fill->weather_data);
        fill->weather_data = NULL;
        return(1);
    }
    return(0);
}



////////////////////////////////////////// Trails //////////////////////////////////////////////////


/*
 *  Get new trail color for call
 */
int new_trail_color(char *call) {
    int color;

    if (is_my_call(call,0)) {
        color= 0;    // It's one of my calls, use special color 0 (currently yellow)
    } else {
        // all other callsigns get some other color out of the color table
        current_trail_color = (current_trail_color + 1) % max_trail_colors;
        if (current_trail_color == 0)    // Skip the first (special) color
            current_trail_color++;
        color = current_trail_color;
    }
    return(color);
}



// Trail storage in ring buffer (1 is the oldest point)
// inp         v                   
//      [.][1][.][.][.][.][.][.][.]   at least 1 point in buffer,
// out      v                         else no allocated trail memory

// inp                  v
//      [.][1][2][3][4][.][.][.][.]   half full
// out      v                      

// inp      v  
//      [9][1][2][3][4][5][6][7][8]   buffer full (wrap around)
// out      v


/*
 *  Store one trail point, allocate trail storage if necessary
 */
int store_trail_point(DataRow *fill, long lon, long lat) {
    int ok = 1;

    if (fill->track_data == NULL) {       // new trail, allocate storage and do initialisation
        fill->track_data = malloc(sizeof(TrackRow));
        if (fill->track_data == NULL)
            ok = 0;
        else {
            fill->track_data->trail_out   = fill->track_data->trail_inp = 0;
            fill->track_data->trail_color = new_trail_color(fill->call_sign);
            tracked_stations++;
        }
    } else
        if (fill->track_data->trail_out == fill->track_data->trail_inp)                  // ring buffer is full
            fill->track_data->trail_out = (fill->track_data->trail_out+1) % MAX_TRACKS;  // increment and keep inside buffer
    if (ok) {
        fill->track_data->trail_long_pos[fill->track_data->trail_inp]=lon;
        fill->track_data->trail_lat_pos[fill->track_data->trail_inp] =lat;
        fill->track_data->trail_inp = (fill->track_data->trail_inp+1) % MAX_TRACKS;
    }
    return(ok);
}


/*
 *  Delete trail and free memory
 */
int delete_trail(DataRow *fill) {
    if (fill->track_data != NULL) {
        free(fill->track_data);
        fill->track_data = NULL;
        tracked_stations--;
        return(1);
    }
        return(0);
}


/*
 *  Draw trail on screen
 */
void draw_trail(Widget w, DataRow *fill, int solid) {
    int i;
    unsigned char short_dashed[2]  = {(unsigned char)1,(unsigned char)5};
    unsigned char medium_dashed[2] = {(unsigned char)5,(unsigned char)5};
    long lat0, lon0, lat1, lon1;        // trail segmant points
    long marg_lat, marg_lon;

    if (fill->track_data != NULL) {
        // trail should have at least two points:
        if ((fill->track_data->trail_inp-fill->track_data->trail_out+MAX_TRACKS)%MAX_TRACKS != 1) {
            (void)XSetForeground(XtDisplay(w),gc,trail_colors[fill->track_data->trail_color]);
            if (solid)
                // Used to be "JoinMiter" and "CapButt" below
                (void)XSetLineAttributes(XtDisplay(w), gc, 3, LineSolid, CapRound, JoinRound);
            else {
                // Another choice is LineDoubleDash
                (void)XSetLineAttributes(XtDisplay(w), gc, 3, LineOnOffDash, CapRound, JoinRound);
                (void)XSetDashes(XtDisplay(w), gc, 0, short_dashed , 2);
            }

            marg_lat = screen_height * size;                    // setup area for acceptable trail points
            if (marg_lat < TRAIL_POINT_MARGIN*60*100)           // on-screen plus 50% margin
                marg_lat = TRAIL_POINT_MARGIN*60*100;           // but at least a minimum margin
            marg_lon = screen_width  * size;
            if (marg_lon < TRAIL_POINT_MARGIN*60*100)
                marg_lon = TRAIL_POINT_MARGIN*60*100;

            for (i = fill->track_data->trail_out;
                       (fill->track_data->trail_inp -1 - i +MAX_TRACKS)%MAX_TRACKS > 0;
                       i = ++i%MAX_TRACKS) {                                    // loop over trail points
                lon0 = fill->track_data->trail_long_pos[(i)];                   // Trail segment start
                lat0 = fill->track_data->trail_lat_pos[(i)];
                lon1 = fill->track_data->trail_long_pos[(i+1)%MAX_TRACKS];      // Trail segment end
                lat1 = fill->track_data->trail_lat_pos[(i+1)%MAX_TRACKS];

                if ( (abs(lon0-mid_x_long_offset) < marg_lon) &&                // trail points have to
                     (abs(lon1-mid_x_long_offset) < marg_lon) &&                // be in margin area
                     (abs(lat0-mid_y_lat_offset)  < marg_lat) &&
                     (abs(lat1-mid_y_lat_offset)  < marg_lat) ) {
                     
                    if (abs(lon1-lon0) < MAX_TRAIL_SEG_LEN*60*100 &&            // drop very long segments
                        abs(lat1-lat0) < MAX_TRAIL_SEG_LEN*60*100) {            //   mostly wrong GPS data

                        // WE7U: possible drawing problems, if numbers too big ????
                        lon0 = (lon0-x_long_offset)/size;                       // check arguments for
                        lat0 = (lat0-y_lat_offset)/size;                        // XDrawLine()
                        lon1 = (lon1-x_long_offset)/size;
                        lat1 = (lat1-y_lat_offset)/size;
                        if (abs(lon0) < 32700 && abs(lon1) < 32700 &&
                            abs(lat0) < 32700 && abs(lat1) < 32700) {           // draw trail segment
                            (void)XDrawLine(XtDisplay(w),pixmap_final,gc,lon0,lat0,lon1,lat1);
                        }
/*                      (void)XDrawLine(XtDisplay(w), pixmap_final, gc,         // draw trail segment
                            (lon0-x_long_offset)/size,(lat0-y_lat_offset)/size,
                            (lon1-x_long_offset)/size,(lat1-y_lat_offset)/size);
*/
                    }
                }
            }
            (void)XSetDashes(XtDisplay(w), gc, 0, medium_dashed , 2);
        }
    }
}    



//////////////////////////////////////  Station storage  ///////////////////////////////////////////

// Station storage is done in a double-linked list. In fact there are two such
// pointer structures, one for sorting by name and one for sorting by time.
// We store both the pointers to the next and to the previous elements.  DK7IN

/*
 *  Setup station storage structure
 */
void init_station_data(void) {

    stations = 0;                       // empty station list
    n_first = NULL;                     // pointer to next element in name sorted list
    n_last  = NULL;                     // pointer to previous element in name sorted list
    t_first = NULL;                     // pointer to next element in time sorted list (newer)
    t_last  = NULL;                     // pointer to previous element in time sorted list (older)
    last_sec = sec_now();               // check value for detecting changed seconds in time
    next_time_sn = 0;                   // serial number for unique time index
    current_trail_color = 0x00;         // first trail color used will be 0x01
    last_station_remove = sec_now();    // last time we checked for stations to remove
    track_station_on = 0;               // no tracking at startup
}


/*
 *  Initialize station data
 */        
void init_station(DataRow *p_station) {
    // the list pointers should already be set
    p_station->track_data   = NULL;             // no trail
    p_station->weather_data = NULL;             // no weather
    p_station->coord_lat    = 0l;               //  90N
    p_station->coord_lon    = 0l;               // 180W  / undefined position
    p_station->call_sign[0]      = '\0';        // ?????
    p_station->sec_heard         = 0;
    p_station->time_sn           = 0;
    p_station->flag              = '\0';
    p_station->record_type       = '\0';
    p_station->data_via          = '\0';        // L local, T TNC, I internet, F file
    p_station->heard_via_tnc_port = 0;
    p_station->heard_via_tnc_last_time = 0;
    p_station->last_heard_via_tnc = 0;
    p_station->last_port_heard   = 0;
    p_station->num_packets       = 0;
    p_station->aprs_symbol.aprs_type = '\0';
    p_station->aprs_symbol.aprs_symbol = '\0';
    p_station->aprs_symbol.special_overlay = '\0';
    p_station->speed_unit        = '\0';        // obsolete
    p_station->station_time_type = '\0';
    p_station->origin[0]         = '\0';        // no object
    p_station->packet_time[0]    = '\0';
    p_station->node_path[0]      = '\0';
    p_station->pos_time[0]       = '\0';
    p_station->altitude_time[0]  = '\0';
    p_station->altitude[0]       = '\0';
    p_station->speed_time[0]     = '\0';
    p_station->speed[0]          = '\0';
    p_station->course[0]         = '\0';
    p_station->power_gain[0]     = '\0';
    p_station->station_time[0]   = '\0';
    p_station->sats_visible[0]   = '\0';
    p_station->comments[0]       = '\0';
}


/*
 *  Remove element from name ordered list
 */
void remove_name(DataRow *p_rem) {      // todo: return pointer to next element
    if (p_rem->n_prev == NULL)          // first element
        n_first = p_rem->n_next;
    else
        p_rem->n_prev->n_next = p_rem->n_next;

    if (p_rem->n_next == NULL)          // last element
        n_last = p_rem->n_prev;
    else
        p_rem->n_next->n_prev = p_rem->n_prev;
}


/*
 *  Remove element from time ordered list
 */
void remove_time(DataRow *p_rem) {      // todo: return pointer to next element
    if (p_rem->t_prev == NULL)          // first element
        t_first = p_rem->t_next;
    else
        p_rem->t_prev->t_next = p_rem->t_next;

    if (p_rem->t_next == NULL)          // last element
        t_last = p_rem->t_prev;
    else
        p_rem->t_next->t_prev = p_rem->t_prev;
}


/*
 *  Insert existing element into name ordered list before p_name
 */
void insert_name(DataRow *p_new, DataRow *p_name) {
    p_new->n_next = p_name;
    if (p_name == NULL) {               // add to end of list
        p_new->n_prev = n_last;
        if (n_last == NULL)             // add to empty list
            n_first = p_new;
        else
            n_last->n_next = p_new;
        n_last = p_new;
    } else {
        p_new->n_prev = p_name->n_prev;
        if (p_name->n_prev == NULL)     // add to begin of list
            n_first = p_new;
        else
            p_name->n_prev->n_next = p_new;
        p_name->n_prev = p_new;
    }
}


/*
 *  Insert existing element into time ordered list before p_time
 */
void insert_time(DataRow *p_new, DataRow *p_time) {
    p_new->t_next = p_time;
    if (p_time == NULL) {               // add to end of list
        p_new->t_prev = t_last;
        if (t_last == NULL)             // add to empty list
            t_first = p_new;
        else
            t_last->t_next = p_new;
        t_last = p_new;
    } else {
        p_new->t_prev = p_time->t_prev;
        if (p_time->t_prev == NULL)     // add to begin of list
            t_first = p_new;
        else
            p_time->t_prev->t_next = p_new;
        p_time->t_prev = p_new;
    }
}


/*
 *  Free station memory for one entry
 */
void delete_station_memory(DataRow *p_del) {
    remove_name(p_del);
    remove_time(p_del);
    free(p_del);
    stations--;
}


/*
 *  Create new uninitialized element in station list
 *  and insert it before p_name and p_time entries
 */
/*@null@*/ DataRow *insert_new_station(DataRow *p_name, DataRow *p_time) {
    DataRow *p_new;

    p_new = (DataRow *)malloc(sizeof(DataRow));
    if (p_new != NULL) {                // we really got the memory
        p_new->call_sign[0] = '\0';     // just to be sure
        insert_name(p_new,p_name);      // insert element into name ordered list
        insert_time(p_new,p_time);      // insert element into time ordered list
    }
    if (p_new == NULL)
        printf("ERROR: we got no memory for station storage\n");
    return(p_new);                      // return pointer to new element
}


/*
 *  Create new initialized element for call in station list
 *  and insert it before p_name and p_time entries
 */
/*@null@*/ DataRow *add_new_station(DataRow *p_name, DataRow *p_time, char *call) {
    DataRow *p_new;
    char station_num[30];

    p_new = insert_new_station(p_name,p_time);  // allocate memory
    if (p_new != NULL) {
        init_station(p_new);                    // initialize new station record
        strcpy(p_new->call_sign,call);
        stations++;

        // this should not be here...  ??
        if (!wait_to_redraw) {          // show number of stations in status line
            sprintf(station_num, langcode("BBARSTH001"), stations);
            XmTextFieldSetString(text3, station_num);
        }

    }
    return(p_new);                      // return pointer to new element
}


/*
 *  Move station record before p_time in time ordered list
 */
void move_station_time(DataRow *p_curr, DataRow *p_time) {

    if (p_curr != NULL) {               // need a valid record
        remove_time(p_curr);
        insert_time(p_curr,p_time);
    }
}


/*
 *  Move station record before p_name in name ordered list
 */
void move_station_name(DataRow *p_curr, DataRow *p_name) {

    if (p_curr != NULL) {               // need a valid record
        remove_name(p_curr);
        insert_name(p_curr,p_name);
    }
}


/*
 *  Search station record by callsign
 *  Returns a station with a call equal or after the searched one
 */
int search_station_name(DataRow **p_name, char *call, int exact) {
    // DK7IN: we do a linear search here.
    // Maybe I setup a tree storage too, to see what is better,
    // tree should be faster in search, list faster at display time.
    // I don't look at case, objects and internet names could have lower case
    int i,j;
    int ok = 1;
    char ch0,ch1;

    (*p_name) = n_first;                                // start of alphabet
    ch0 = call[0];
    if (ch0 == '\0')
        ok = 0;
    else {
        while(1) {                                      // check first char
            if ((*p_name) == NULL)
                break;                                  // reached end of list
            if ((*p_name)->call_sign[0] >= ch0)         // we also catch the '\0'
                break;                                  // finished search
            (*p_name) = (*p_name)->n_next;              // next element
        }
        // we now probably have found the first char
        if ((*p_name) == NULL || (*p_name)->call_sign[0] != ch0)
            ok = 0;                                     // nothing found!
    }

    for (i=1;ok && i<(int)strlen(call);i++) {           // check rest of string
        ch1 = call[i];
        ch0 = call[i-1];
        while (ok) {
            if ((*p_name)->call_sign[i] >= ch1)         // we also catch the '\0'
                break;                                  // finished search
            (*p_name) = (*p_name)->n_next;              // next element
            if ((*p_name) == NULL)
                break;                                  // reached end of list
            for (j=0;ok && j<i;j++) {                   // unchanged first part?
                if ((*p_name)->call_sign[j] != call[j])
                    ok = 0;
            }
        }
        // we now probably have found the next char
        if ((*p_name) == NULL || (*p_name)->call_sign[i] != ch1)
            ok = 0;                                     // nothing found!
    }  // check next char in call

    if (exact && ok && strlen((*p_name)->call_sign) != strlen(call))
        ok = 0;
    return(ok);         // if not ok: p_name points to correct insert position in name list
}
    

/*    
 *  Search station record by time and time serial number, serial ignored if -1
 *  Returns a station that is equal or older than the search criterium
 */
int search_station_time(DataRow **p_time, time_t heard, int serial) {
    int ok = 1;

    (*p_time) = t_last;                                 // newest station
    if (heard == 0) {                                   // we want the newest station
        if (t_last == 0)
            ok = 0;                                     // empty list
    }
    else {
        while((*p_time) != NULL) {                      // check time
            if ((*p_time)->sec_heard <= heard)          // compare
                break;                                  // found time or earlier
            (*p_time) = (*p_time)->t_prev;              // next element
        }
        // we now probably have found the entry
        if ((*p_time) != NULL && (*p_time)->sec_heard == heard) {
            // we got a match, but there may be more of them
            if (serial >= 0) {                          // check serial number, ignored if -1
                while((*p_time) != NULL) {              // for unique time index
                    if ((*p_time)->sec_heard == heard && (*p_time)->time_sn <= serial)  // compare
                        break;                          // found it (same time, maybe earlier SN)
                    if ((*p_time)->sec_heard < heard)   // compare
                        break;                          // found it (earlier time)
                    (*p_time) = (*p_time)->t_prev;      // consider next element
                }
                if ((*p_time) == NULL || (*p_time)->sec_heard != heard || (*p_time)->time_sn != serial)
                    ok = 0;                             // no perfect match
            }
        } else
            ok = 0;                                     // no perfect match
    }
    return(ok);
}


/*
 *  Get pointer to next station in name sorted list
 */
int next_station_name(DataRow **p_curr) {
    
    if ((*p_curr) == NULL)
        (*p_curr) = n_first;
    else
        (*p_curr) = (*p_curr)->n_next;
    if ((*p_curr) != NULL)
        return(1);
    else
        return(0);
}
        

/*
 *  Get pointer to previous station in name sorted list
 */
int prev_station_name(DataRow **p_curr) {
    
    if ((*p_curr) == NULL)
        (*p_curr) = n_last;
    else
        (*p_curr) = (*p_curr)->n_prev;
    if ((*p_curr) != NULL)
        return(1);
    else
        return(0);
}
        

/*
 *  Get pointer to newer station in time sorted list
 */
int next_station_time(DataRow **p_curr) {

    if ((*p_curr) == NULL)
        (*p_curr) = t_first;
    else
        (*p_curr) = (*p_curr)->t_next;
    if ((*p_curr) != NULL)
        return(1);
    else
        return(0);
}


/*
 *  Get pointer to older station in time sorted list
 */
int prev_station_time(DataRow **p_curr) {
    
    if ((*p_curr) == NULL)
        (*p_curr) = t_last;
    else
        (*p_curr) = (*p_curr)->t_prev;
    if ((*p_curr) != NULL)
        return(1);
    else
        return(0);
}

/*
 *  Set flag for all stations in current view area or a margin area around it
 *  That are the stations we look at, if we want to draw symbols or trails
 */
void setup_in_view(void) {
    DataRow *p_station;
    long min_lat, max_lat;                      // screen borders plus space
    long min_lon, max_lon;                      // for trails from off-screen stations
    long marg_lat, marg_lon;                    // margin around screen

    marg_lat = (long)(3 * screen_height * size/2);
    marg_lon = (long)(3 * screen_width  * size/2);
    if (marg_lat < IN_VIEW_MIN*60*100)          // allow a minimum area, 
        marg_lat = IN_VIEW_MIN*60*100;          // there could be outside stations
    if (marg_lon < IN_VIEW_MIN*60*100)          // with trail parts on screen
        marg_lon = IN_VIEW_MIN*60*100;

    // Only screen view
    // min_lat = y_lat_offset  + (long)(screen_height * size);
    // max_lat = y_lat_offset;
    // min_lon = x_long_offset;
    // max_lon = x_long_offset + (long)(screen_width  * size);

    // Screen view plus one screen wide margin
    // There could be stations off screen with on screen trails
    // See also the use of position_on_extd_screen()
    min_lat = mid_y_lat_offset  - marg_lat;
    max_lat = mid_y_lat_offset  + marg_lat;
    min_lon = mid_x_long_offset - marg_lon;
    max_lon = mid_x_long_offset + marg_lon;

    p_station = n_first;
    while (p_station != NULL) {
        if ((p_station->flag & ST_ACTIVE) == '\0'        // ignore deleted objects
                  || p_station->coord_lon < min_lon || p_station->coord_lon > max_lon
                  || p_station->coord_lat < min_lat || p_station->coord_lat > max_lat
                  || (p_station->coord_lat == 0 && p_station->coord_lon == 0)) {
            // outside view and undefined stations:
            p_station->flag &= (~ST_INVIEW);        // clear "In View" flag
        } else
            p_station->flag |= ST_INVIEW;           // set "In View" flag
        p_station = p_station->n_next;
    }
}


/*
 *  Check if position is inside screen borders
 */
int position_on_screen(long lat, long lon) {

    if (   lon > x_long_offset && lon < (x_long_offset+(long)(screen_width*size))
        && lat > y_lat_offset  && lat < (y_lat_offset+(long)(screen_height*size)) 
        && !(lat == 0 && lon == 0))     // discard undef positions from screen
        return(1);                      // position is inside the screen
    else
        return(0);
}


/*
 *  Check if position is inside extended screen borders
 *  (real screen + one screen margin for trails)
 *  used for station "In View" flag
 */
int position_on_extd_screen(long lat, long lon) {
    long marg_lat, marg_lon;                    // margin around screen

    marg_lat = (long)(3 * screen_height * size/2);
    marg_lon = (long)(3 * screen_width  * size/2);
    if (marg_lat < IN_VIEW_MIN*60*100)          // allow a minimum area, 
        marg_lat = IN_VIEW_MIN*60*100;          // there could be outside stations
    if (marg_lon < IN_VIEW_MIN*60*100)          // with trail parts on screen
        marg_lon = IN_VIEW_MIN*60*100;

    if (    abs(lon - mid_x_long_offset) < marg_lon
         && abs(lat - mid_y_lat_offset)  < marg_lat
         && !(lat == 0 && lon == 0))    // discard undef positions from screen
        return(1);                      // position is inside the area
    else
        return(0);
}


/*
 *  Check if position is inside inner screen area
 *  (real screen + minus 1/6 screen margin)
 *  used for station tracking
 */
int position_on_inner_screen(long lat, long lon) {

    if (    lon > mid_x_long_offset-(long)(screen_width*size/3)
         && lon < mid_x_long_offset+(long)(screen_width*size/3)
         && lat > mid_y_lat_offset -(long)(screen_height*size/3)
         && lat < mid_y_lat_offset +(long)(screen_height*size/3)
         && !(lat == 0 && lon == 0))    // discard undef positions from screen
        return(1);                      // position is inside the area
    else
        return(0);
}


/*
 *  Delete single station with all its data    ?? delete messages ??
 */
void station_del(char *call) {          // used for own call only
    DataRow *p_name;                    // DK7IN: do it with move... ?

    if (search_station_name(&p_name, call, 1)) {
        (void)delete_trail(p_name);     // Free track storage if it exists.
        (void)delete_weather(p_name);   // free weather memory, if allocated
        delete_station_memory(p_name);  // free memory
    }
}


/*
 *  Delete all stations             ?? delete messages ??
 */
void delete_all_stations(void) {
    DataRow *p_name;
    DataRow *p_curr;
    
    p_name = n_first;
    while (p_name != NULL) {
        p_curr = p_name;
        p_name = p_name->n_next;
        (void)delete_trail(p_curr);     // free trail memory, if allocated
        (void)delete_weather(p_curr);   // free weather memory, if allocated
        delete_station_memory(p_curr);  // free station memory
    }
    if (stations != 0) {
        printf("ERROR: stations should be 0 after stations delete, is %ld\n",stations);
        stations = 0;
    }    
}


/*
 *  Check if we have to delete old stations
 */
void check_station_remove(void) {
    DataRow *p_station;
    time_t t_rem;
    int old = 1;

    if (last_station_remove < sec_now() - STATION_REMOVE_CYCLE) {
        t_rem = sec_now() - sec_remove;
        p_station = t_first;
        while (p_station != NULL && old) {
            if (p_station->sec_heard < t_rem) {
                if (!is_my_call(p_station->call_sign,1)) {
                    mdelete_messages(p_station->call_sign);     // delete messages
                    (void)delete_trail(p_station);              // Free track storage if it exists.
                    (void)delete_weather(p_station);            // free weather memory, if allocated
                    delete_station_memory(p_station);           // free memory
                }
            } else
                old = 0;                                        // all other stations are newer...
            p_station = p_station->t_next;
        }
        last_station_remove = sec_now();
    }
}


/*
 *  Delete an object (mark it as deleted)
 */
void delete_object(char *name) {
    DataRow *p_station;

    p_station = NULL;
    if (search_station_name(&p_station,name,1)) {       // find object name
        p_station->flag &= (~ST_ACTIVE);                // clear flag
        p_station->flag &= (~ST_INVIEW);                // clear "In View" flag
        if (position_on_screen(p_station->coord_lat,p_station->coord_lon))
            redraw_on_new_data = 2;                     // redraw now
            // there is some problem...  it is not redrawn immediately! ????
            // but deleted from list immediatetly
        redo_list = (int)TRUE;                          // and update lists
    }
}







///////////////////////////////////////  APRS Decoding  ////////////////////////////////////////////


/*
 *  Extract Uncompressed Position Report from begin of line
 */
int extract_position(DataRow *p_station, char **info, /*@unused@*/ int type) {
    int i, ok;
    char temp_lat[8+1];
    char temp_lon[9+1];
    char *my_data;

    my_data = (*info);
    ok = (int)(strlen(my_data) >= 19);
    ok = (int)(ok && my_data[4]=='.' && my_data[14]=='.'
                  && (my_data[7] =='N' || my_data[7] =='S')
                  && (my_data[17]=='W' || my_data[17]=='E'));
    // errors found:  [4]: X   [7]: n s   [17]: w e
    if (ok) {
        ok =             is_num_chr(my_data[0]);           // 5230.31N/01316.88E>
        ok = (int)(ok && is_num_chr(my_data[1]));          // 0123456789012345678
        ok = (int)(ok && is_num_or_sp(my_data[2]));
        ok = (int)(ok && is_num_or_sp(my_data[3]));
        ok = (int)(ok && is_num_or_sp(my_data[5]));
        ok = (int)(ok && is_num_or_sp(my_data[6]));
        ok = (int)(ok && is_num_chr(my_data[9]));
        ok = (int)(ok && is_num_chr(my_data[10]));
        ok = (int)(ok && is_num_chr(my_data[11]));
        ok = (int)(ok && is_num_or_sp(my_data[12]));
        ok = (int)(ok && is_num_or_sp(my_data[13]));
        ok = (int)(ok && is_num_or_sp(my_data[15]));
        ok = (int)(ok && is_num_or_sp(my_data[16]));
    }
                                            
    if (ok) {
        overlay_symbol(my_data[18], my_data[8], p_station);

        for (i=2;i<17;i++)                              // position ambiguity
            if (my_data[i] == ' ')
                my_data[i] = '0';

        strncpy(temp_lat,my_data,7);
        temp_lat[7] = my_data[7];
        temp_lat[8] = '\0';

        strncpy(temp_lon,my_data+9,8);
        temp_lon[8] = my_data[17];
        temp_lon[9] = '\0';

        if (!is_my_call(p_station->call_sign,1)) {      // don't change my position, I know it better...
            p_station->coord_lat = convert_lat_s2l(temp_lat);   // ...in case of position ambiguity
            p_station->coord_lon = convert_lon_s2l(temp_lon);
        }

        (*info) += 19;                  // delete position from comment
    }
    return(ok);
}


/*
 *  Extract Compressed Position Report Data Formats from begin of line
 *    [APRS Reference, chapter 9]
 */
int extract_comp_position(DataRow *p_station, char **info, /*@unused@*/ int type) {
    int ok;
    char x1, x2, x3, x4, y1, y2, y3, y4;
    int c = 0;
    int s = 0;
    int T = 0;
    char *my_data;
    float lon = 0;
    float lat = 0;
    float range;

    // compressed data format  /YYYYXXXX$csT  is a fixed 13-character field
    // used for ! / @ = data IDs
    //   /     Symbol Table ID
    //   YYYY  compressed latitude
    //   XXXX  compressed longitude
    //   $     Symbol Code
    //   cs    compressed
    //            course/speed
    //            radio range
    //            altitude
    //   T     compression type ID

    my_data = (*info);
    ok = (int)(strlen(my_data) >= 13);
    if (ok) {
        y1 = my_data[1] - '!';  y2 = my_data[2] - '!';  y3 = my_data[3] - '!';  y4 = my_data[4] - '!';
        x1 = my_data[5] - '!';  x2 = my_data[6] - '!';  x3 = my_data[7] - '!';  x4 = my_data[8] - '!';

        c = (int)(my_data[10] - '!');
        s = (int)(my_data[11] - '!');
        T = (int)(my_data[12] - '!');

        if ((int)x1 == -1) x1 = '\0';   if ((int)x2 == -1) x2 = '\0';   // convert ' ' to '0'
        if ((int)x3 == -1) x3 = '\0';   if ((int)x4 == -1) x4 = '\0';   // not specified in
        if ((int)y1 == -1) y1 = '\0';   if ((int)y2 == -1) y2 = '\0';   // APRS Reference!
        if ((int)y3 == -1) y3 = '\0';   if ((int)y4 == -1) y4 = '\0';   // do we need it ???

        ok = (int)(ok && (x1 >= '\0' && x1 < (char)91));                //  /YYYYXXXX$csT
        ok = (int)(ok && (x2 >= '\0' && x2 < (char)91));                //  0123456789012
        ok = (int)(ok && (x3 >= '\0' && x3 < (char)91));
        ok = (int)(ok && (x4 >= '\0' && x4 < (char)91));
        ok = (int)(ok && (y1 >= '\0' && y1 < (char)91));
        ok = (int)(ok && (y2 >= '\0' && y2 < (char)91));
        ok = (int)(ok && (y3 >= '\0' && y3 < (char)91));
        ok = (int)(ok && (y4 >= '\0' && y4 < (char)91));
        ok = (int)(ok && (c == -1 || ((c >=0 && c < 91) && (s >= 0 && s < 91) && (T >= 0 && T < 64))));

        if (ok) {
            lat = ((((int)y1 * 91 + (int)y2) * 91 + (int)y3) * 91 + (int)y4 ) / 380926.0; // in deg, 0:  90N
            lon = ((((int)x1 * 91 + (int)x2) * 91 + (int)x3) * 91 + (int)x4 ) / 190463.0; // in deg, 0: 180W
            lat *= 60 * 60 * 100;                       // in 1/100 sec
            lon *= 60 * 60 * 100;                       // in 1/100 sec
            if ((((long)(lat+4) % 60) > 8) || (((long)(lon+4) % 60) > 8))
                ok = 0;                                 // check max resolution 0.01 min
        }                                               // to catch even more errors
    }
    if (ok) {
        overlay_symbol(my_data[9], my_data[0], p_station);      // Symbol / Table
        if (!is_my_call(p_station->call_sign,1)) {      // don't change my position, I know it better...
            p_station->coord_lat = (long)((lat));               // in 1/100 sec
            p_station->coord_lon = (long)((lon));               // in 1/100 sec
        }

        if (c >= 0) {                                   // ignore csT if c = ' '
            if (c < 90) {
                if ((T & 0x18) == 0x10) {               // check for GGA (with altitude)
                    sprintf(p_station->altitude,"%06.0f",pow(1.002,(double)(c*91+s))*0.3048);  // in meters
                } else {                                // compressed course/speed
                    sprintf(p_station->course,"%03d",c*4);                           // deg
                    sprintf(p_station->speed, "%03.0f",pow(1.08,(double)s)-1);       // knots
                }
            } else {
                if (c == 90) {
                    // pre-calculated radio range
                    range = 2 * pow(1.08,(double)s);    // miles

                    // DK7IN: dirty hack...  but better than nothing
                    if (s <= 5)                         // 2.9387 mi
                        sprintf(p_station->power_gain, "PHG%s0", "000");
                    else if (s <= 17)                   // 7.40 mi
                        sprintf(p_station->power_gain, "PHG%s0", "111");
                    else if (s <= 36)                   // 31.936 mi
                        sprintf(p_station->power_gain, "PHG%s0", "222");
                    else if (s <= 75)                   // 642.41 mi
                        sprintf(p_station->power_gain, "PHG%s0", "333");
                    else                       // max 90:  2037.8 mi
                        sprintf(p_station->power_gain, "PHG%s0", "444");
                }
            }
        }
        (*info) += 13;                  // delete position from comment
    }
    return(ok);
}


/*
 *  Extract speed and course from begin of info field
 */
int extract_speed_course(char *info, char *speed, char *course) {
    int i,found,len;

    len = (int)strlen(info);
    found = 0;
    if (len >= 7) {
        found = 1;
        for(i=0; found && i<7; i++)         // check data format
            if(!(isdigit((int)info[i]) || (i==3 && info[i]=='/')))
                found=0;
    }
    if (found) {
        substr(course,info,3);
        substr(speed,info+4,3);
        for (i=0;i<=len-7;i++)        // delete speed/course from info field
            info[i] = info[i+7];
    } else {
        speed[0] ='\0';
        course[0]='\0';
    }
    return(found);
}


/*
 *  Extract altitude from APRS info field          "/A=012345"    in feet
 */
int extract_altitude(char *info, char *altitude) {
    int i,ofs,found,len;

    found=0;
    len = (int)strlen(info);
    for(ofs=0; !found && ofs<len-8; ofs++)      // search for start sequence
        if (strncmp(info+ofs,"/A=",3)==0) {
            found=1;
            if(!isdigit((int)info[ofs+3]) && info[ofs+3]!='-')  // are negative altitudes even defined?
                found=0;
            for(i=4; found && i<9; i++)         // check data format
                if(!isdigit((int)info[ofs+i]))
                    found=0;
    }
    if (found) {
        ofs--;  // was one too much on exit from for loop
        substr(altitude,info+ofs+3,6);
        for (i=ofs;i<=len-9;i++)        // delete altitude from info field
            info[i] = info[i+9];
    } else
        altitude[0] = '\0';
    return(found);
}


// Comment Field
//  Storm Data
// Bearing and Number// Range Quality


/*
 *  Extract powergain from APRS info field          "PHG1234/" "PHG1234"  from APRS data extension
 */
int extract_powergain(char *info, char *phgd) {
    int i,found,len;

    found=0;
    len = (int)strlen(info);
    if (len >= 9 && strncmp(info,"PHG",3)==0 && info[7]=='/' && info[8]!='A'  // trailing '/' not defined in Reference...
                 && isdigit((int)info[3]) && isdigit((int)info[5]) && isdigit((int)info[6])) {
        substr(phgd,info,7);
        for (i=0;i<=len-8;i++)        // delete powergain from data extension field
            info[i] = info[i+8];
        return(1);
    } else {
        if (len >= 7 && strncmp(info,"PHG",3)==0
                 && isdigit((int)info[3]) && isdigit((int)info[5]) && isdigit((int)info[6])) {
            substr(phgd,info,7);
            for (i=0;i<=len-7;i++)        // delete powergain from data extension field
                info[i] = info[i+7];
            return(1);
        } else {
            phgd[0] = '\0';
            return(0);
        }
    }
}


/*
 *  Extract omnidf from APRS info field          "DFS1234/"    from APRS data extension
 */
int extract_omnidf(char *info, char *phgd) {
    int i,found,len;

    found=0;
    len = (int)strlen(info);
    if (len >= 8 && strncmp(info,"DFS",3)==0 && info[7]=='/'    // trailing '/' not defined in Reference...
                 && isdigit((int)info[3]) && isdigit((int)info[5]) && isdigit((int)info[6])) {
        substr(phgd,info,7);
        for (i=0;i<=len-8;i++)        // delete omnidf from data extension field
            info[i] = info[i+8];
        return(1);
    } else {
        phgd[0] = '\0';
        return(0);
    }
}


/*
 *  Extract Time from begin of line      [APRS Reference, chapter 6]
 */
int extract_time(DataRow *p_station, char *data, int type) {
    int len, i;
    int ok = 0;

    // todo: better check of time data ranges
    len = (int)strlen(data);
    if (type == APRS_WX2) {
        // 8 digit time from stand-alone positionless weather stations...
        if (len > 8) {
            // MMDDHHMM   zulu time
            // MM 01-12         todo: better check of time data ranges
            // DD 01-31
            // HH 01-23
            // MM 01-59
            ok = 1;
            for (i=0;ok && i<8;i++)
                if (!isdigit((int)data[i]))
                    ok = 0;
            if (ok) {
                substr(p_station->station_time,data+2,6);
                p_station->station_time_type = 'z';
                for (i=0;i<=len-8;i++)         // delete time from data
                    data[i] = data[i+8];
            }
        }
    } else {
        if (len > 6) {
            // Status messages only with optional zulu format
            // DK7IN: APRS ref says one of 'z' '/' 'h', but I found 'c' at HB9TJM-8   ???
            if (data[6]=='z' || data[6]=='/' || data[6]=='h' || data[6]=='c')
                ok = 1;
            for (i=0;ok && i<6;i++)
                if (!isdigit((int)data[i]))
                    ok = 0;
            if (ok) {
                substr(p_station->station_time,data,6);
                p_station->station_time_type = data[6];
                for (i=0;i<=len-7;i++)         // delete time from data
                    data[i] = data[i+7];
            }
        }
    }
    return(ok);
}


// APRS Data Extensions               [APRS Reference p.27]
//  .../...  Course & Speed, may be followed by others (see p.27)
//  .../...  Wind Dir and Speed
//  PHG....  Station Power and Effective Antenna Height/Gain
//  RNG....  Pre-Calculated Radio Range
//  DFS....  DF Signal Strength and Effective Antenna Height/Gain
//  T../C..  Area Object Descriptor

/* Extract one of several possible APRS Data Extensions */
void process_data_extension(DataRow *p_station, char *data, /*@unused@*/ int type) {
    char temp1[7+1];
    char temp2[3+1];

    if (extract_speed_course(data,temp1,temp2)) {       // ... from Mic-E, etc.
        sprintf(p_station->speed,"%06.2f",atof(temp1)); // in knots/hour
        p_station->speed_unit='K';                      // obsolet
        strcpy(p_station->course,temp2);
    } else {
        if (extract_powergain(data,temp1)) {
            strcpy(p_station->power_gain,temp1);
        } else {
            if (extract_omnidf(data,temp1))
                strcpy(p_station->power_gain,temp1);
        }
    }
}



/* extract all available information from info field */
void process_info_field(DataRow *p_station, char *info, /*@unused@*/ int type) {
    char temp_data[6+1];
    char time_data[MAX_TIME];

    if (extract_altitude(info,temp_data)) {                         // get altitude
        sprintf(p_station->altitude,"%.2f",atof(temp_data)*0.3048); // feet to meter
        strcpy(p_station->altitude_time,get_time(time_data));                
    }
    // do other things...
}



////////////////////////////////////////////////////////////////////////////////////////////////////


/*
 *  Extract data for $GPRMC, it fails if there is no position!!
 */
int extract_RMC(DataRow *p_station, char *data, char *call_sign, char *path) {
    char *temp_ptr;
    char temp_data[40];         // short term string storage, MAX_CALL, ...  ???
    char lat_s[20];
    char long_s[20];
    int ok;

    // should we copy it before processing? it changes data: ',' gets substituted by '\0' !!
    ok = 0;
    p_station->record_type = NORMAL_GPS_RMC;
    strcpy(p_station->pos_time,get_time(temp_data));
    p_station->flag &= (~ST_MSGCAP);    // clear "message capable" flag

    /* check aprs type on call sign */
    p_station->aprs_symbol = *id_callsign(call_sign, path);
    if (strchr(data,',') != NULL) {                                     // there is a ',' in string
        (void)strtok(data,",");                                         // extract GPRMC
        (void)strtok(NULL,",");                                         // extract time ?
        temp_ptr = strtok(NULL,",");                                    // extract valid char
        if (temp_ptr != NULL) {
            strncpy(temp_data,temp_ptr,2);
            if (temp_data[0]=='A' || temp_data[0]=='V') {
                /* V is a warning but we can get good data still ? */   // DK7IN: got no position with 'V' !
                temp_ptr=strtok(NULL,",");                              // extract latitude
                if (temp_ptr!=NULL && temp_ptr[4]=='.') {
                    strncpy(lat_s,temp_ptr,8);
                    temp_ptr=strtok(NULL,",");                /* get N-S */
                    if (temp_ptr!=NULL) {
                        strncpy(temp_data,temp_ptr,1);
                        lat_s[8]=toupper((int)temp_data[0]);
                        lat_s[9] = '\0';
                        if (lat_s[8] =='N' || lat_s[8] =='S') {
                            temp_ptr=strtok(NULL,",");            /* get long */
                            if(temp_ptr!=NULL && temp_ptr[5] == '.') {
                                strncpy(long_s,temp_ptr,9);
                                temp_ptr=strtok(NULL,",");            /* get E-W */
                                if (temp_ptr!=NULL) {
                                    strncpy(temp_data,temp_ptr,1);
                                    long_s[9] = toupper((int)temp_data[0]);
                                    long_s[10] = '\0';
                                    if (long_s[9] =='E' || long_s[9] =='W') {
                                        p_station->coord_lat = convert_lat_s2l(lat_s);
                                        p_station->coord_lon = convert_lon_s2l(long_s);
                                        ok = 1;
                                        temp_ptr=strtok(NULL,",");        /* Get speed */
                                        if(temp_ptr!=0) {
                                            substr(p_station->speed,temp_ptr,MAX_SPEED);
                                            p_station->speed_unit='K';
                                            temp_ptr=strtok(NULL,",");    /* Get course */
                                            if (temp_ptr!=NULL) {
                                                substr(p_station->course,temp_ptr,MAX_COURSE);
                                                (void)strtok(NULL,",");                    /* get date of fix */
                                            } else
                                                strcpy(p_station->course,"000.0");
                                        } else
                                            strcpy(p_station->speed,"0");
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    return(ok);
}


/*
 *  Extract data for $GPGGA
 */
int extract_GGA(DataRow *p_station,char *data,char *call_sign, char *path) {
    char *temp_ptr;
    char temp_data[40];         // short term string storage, MAX_CALL, ...  ???
    char lat_s[20];
    char long_s[20];
    int ok;

    ok = 0;
    p_station->record_type = NORMAL_GPS_GGA;
    strcpy(p_station->pos_time,get_time(temp_data));
    strcpy(p_station->altitude_time,get_time(temp_data));
    p_station->flag &= (~ST_MSGCAP);    // clear "message capable" flag

    /* check aprs type on call sign */
    p_station->aprs_symbol = *id_callsign(call_sign, path);
    if (strchr(data,',')!=NULL) {
        if (strtok(data,",")!=NULL) {/* get gpgga*/
            if (strtok(NULL,",")!=NULL)  {/* get time */
                temp_ptr = strtok(NULL,",");                 /* get latitude */
                if (temp_ptr !=NULL) {
                    strncpy(lat_s,temp_ptr,8);
                    temp_ptr = strtok(NULL,",");                       /* get N-S */
                    if (temp_ptr!=NULL) {
                        strncpy(temp_data,temp_ptr,1);
                        lat_s[8]=toupper((int)temp_data[0]);
                        lat_s[9] = '\0';
                        if (lat_s[8] =='N' || lat_s[8] =='S') {
                            temp_ptr=strtok(NULL,",");            /* get long */
                             if(temp_ptr!=NULL /* && temp_ptr[5] == '.' */ ) {  // ??
                                strncpy(long_s,temp_ptr,9);
                                temp_ptr=strtok(NULL,",");            /* get E-W */
                                if (temp_ptr!=NULL) {
                                    strncpy(temp_data,temp_ptr,1);
                                    long_s[9]  = toupper((int)temp_data[0]);
                                    long_s[10] = '\0';
                                    if (long_s[9] =='E' || long_s[9] =='W') {
                                        temp_ptr = strtok(NULL,",");                   /* get FIX Quality */
                                        if (temp_ptr!=NULL) {
                                            p_station->coord_lat = convert_lat_s2l(lat_s);
                                            p_station->coord_lon = convert_lon_s2l(long_s);
                                            ok = 1;
                                            strncpy(temp_data,temp_ptr,2);
                                            if (temp_data[0]=='1' || temp_data[0] =='2' ) {
                                                temp_ptr=strtok(NULL,",");  /* Get sats vis */
                                                if (temp_ptr!=NULL) {
                                                    strncpy(p_station->sats_visible,temp_ptr,MAX_SAT);
                                                    temp_ptr=strtok(NULL,",");                                 /* get hoz dil */
                                                    if (temp_ptr!=NULL) {
                                                        temp_ptr=strtok(NULL,","); /* Get altitude */
                                                        if (temp_ptr!=NULL) {
                                                            strcpy(p_station->altitude,temp_ptr); /* Get altitude */
                                                            // unit is in meters, if not adjust value ???
                                                            temp_ptr=strtok(NULL,",");                /* get UNIT */
                                                            if (temp_ptr!=NULL) {
                                                                strncpy(temp_data,temp_ptr,1);                /* get UNIT */
                                                                if (temp_data[0] != 'M') {
                                                                    //printf("ERROR: should adjust altitude for meters\n");
                                                                }
                                                            }
                                                        } else
                                                            strcpy(p_station->altitude,"");
                                                    } else
                                                        strcpy(p_station->altitude,"");
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    return(ok);
}


/*
 *  Extract data for $GPGLL
 */
int extract_GLL(DataRow *p_station,char *data,char *call_sign, char *path) {
    char *temp_ptr;
    char temp_data[40];         // short term string storage, MAX_CALL, ...  ???
    char lat_s[20];
    char long_s[20];
    int ok;

    ok = 0;
    p_station->record_type = NORMAL_GPS_GLL;
    strcpy(p_station->pos_time,get_time(temp_data));
    p_station->flag &= (~ST_MSGCAP);    // clear "message capable" flag

    /* check aprs type on call sign */
    p_station->aprs_symbol = *id_callsign(call_sign, path);
    if (strchr(data,',')!=NULL) {
        (void)strtok(data,",");                    /* get gpgll */
        temp_ptr=strtok(NULL,",");                /* get latitude */
        if (temp_ptr!=NULL && temp_ptr[4]=='.') {
            strncpy(lat_s,temp_ptr,8);
            temp_ptr=strtok(NULL,",");                /* get N-S */
            if (temp_ptr!=NULL) {
                strncpy(temp_data,temp_ptr,1);
                lat_s[8]=toupper((int)temp_data[0]);
                lat_s[9] = '\0';
                if (lat_s[8] =='N' || lat_s[8] =='S') {
                    temp_ptr=strtok(NULL,",");            /* get long */
                     if(temp_ptr!=NULL && temp_ptr[5] == '.') {
                        strncpy(long_s,temp_ptr,9);
                        temp_ptr=strtok(NULL,",");            /* get E-W */
                        if (temp_ptr!=NULL) {
                            strncpy(temp_data,temp_ptr,1);
                            long_s[9]  = toupper((int)temp_data[0]);
                            long_s[10] = '\0';
                            if (long_s[9] =='E' || long_s[9] =='W') {
                                p_station->coord_lat = convert_lat_s2l(lat_s);
                                p_station->coord_lon = convert_lon_s2l(long_s);
                                ok = 1;
                                (void)strtok(NULL,",");                /* get temp */
                                temp_ptr=strtok(NULL,",");            /* get temp */
                                if (temp_ptr!=NULL) {
                                    strncpy(temp_data,temp_ptr,2);
                                    if (temp_data[0]=='A' || temp_data[0]=='V')
                                        /* V is a warning but we can get good data still ? */
                                        strcpy(p_station->course,"000.0");

                                    strcpy(p_station->speed,"0");
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    return(ok);
}


/*
 *  Add data from APRS information field to station database
 *  Returns a 1 if successful
 */
int data_add(int type ,char *call_sign, char *path, char *data, char from, int port, char *origin, int third_party) {
    DataRow *p_station;
    DataRow *p_time;
    char call[MAX_CALLSIGN+1];
    char new_station;
    long last_lat, last_lon;
    char temp_data[40];         // short term string storage, MAX_CALL, ...
    long x_diff_long, y_diff_lat;
    long l_lat, l_lon;
    double distance;
    char station_id[600];
    int found_pos;
    float value;
    WeatherRow *weather;
    int moving;
    int changed_pos;
    int ok;

    // call and path had been validated before
    weather = NULL; // only to make the compiler happy...
    found_pos = 1;
    strcpy(call,call_sign);
    p_station = NULL;
    new_station = (char)FALSE;                          // to make the compiler happy...
    last_lat = 0L;
    last_lon = 0L;
    ok = 0;

    p_time = NULL;                                      // add to end of time sorted list (newest)
    if (search_station_name(&p_station,call,1)) {
        move_station_time(p_station,p_time);            // update time, change position in time sorted list
        new_station = (char)FALSE;                      // we have seen this one before
    } else {
        p_station = add_new_station(p_station,p_time,call);     // create storage
        new_station = (char)TRUE;                       // for new station
    }
    if (p_station != NULL) {
        last_lat = p_station->coord_lat;                // remember last position
        last_lon = p_station->coord_lon;

        ok = 1;                         // succeed as default
        switch (type) {
            case (APRS_MICE):           // Mic-E format
            case (APRS_FIXED):          // '!'
            case (APRS_MSGCAP):         // '='
                ok = (int)( (extract_position(p_station,&data,type)     // uncompressed lat/lon
                    || extract_comp_position(p_station,&data,type)));   // compressed lat/lon
                if (ok) {
                    strcpy(p_station->pos_time,get_time(temp_data));
                    (void)extract_weather(p_station,data);              // look for weather data first
                    process_data_extension(p_station,data,type);        // PHG, speed, etc.
                    process_info_field(p_station,data,type);            // altitude
                    substr(p_station->comments,data,MAX_COMMENTS);
                    p_station->record_type = NORMAL_APRS;
                    if (type == APRS_MSGCAP)
                        p_station->flag |= ST_MSGCAP;           // set "message capable" flag
                    else
                        p_station->flag &= (~ST_MSGCAP);        // clear "message capable" flag
                }
                break;

            case (APRS_DOWN):           // '/'
                ok = extract_time(p_station, data, type);                       // we need a time
                ok = (int)( ok && (extract_position(p_station,&data,type)       // uncompressed lat/lon
                        || extract_comp_position(p_station,&data,type)));       // compressed lat/lon
                if (ok) {
                    strcpy(p_station->pos_time,get_time(temp_data));
                    process_data_extension(p_station,data,type);                // PHG, speed, etc.
                    process_info_field(p_station,data,type);
                    substr(p_station->comments,data,MAX_COMMENTS);
                    p_station->record_type = DOWN_APRS;
                    p_station->flag &= (~ST_MSGCAP);            // clear "message capable" flag
                }
                break;

            case (APRS_DF):             // '@'
            case (APRS_MOBILE):         // '@'
                ok = extract_time(p_station, data, type);                       // we need a time
                ok = (int)( ok && extract_position(p_station,&data,type) );     // uncompressed lat/lon
                if (ok) {
                    process_data_extension(p_station,data,type);                // PHG, speed, etc.
                    process_info_field(p_station,data,type);
                    substr(p_station->comments,data,MAX_COMMENTS);
                    if(type == APRS_MOBILE)
                        p_station->record_type = MOBILE_APRS;
                    else
                        p_station->record_type = DF_APRS;
                    p_station->flag &= (~ST_MSGCAP);            // clear "message capable" flag
                }
                break;

            case (STATION_CALL_DATA):
                p_station->record_type = NORMAL_APRS;
                found_pos = 0;
                break;

            case (APRS_STATUS):         // '>' Status Reports     [APRS Reference, chapter 16]
                (void)extract_time(p_station, data, type);              // we need a time
                // todo: could contain Maidenhead or beam heading+power
                substr(p_station->comments,data,MAX_COMMENTS);          // store status text
                p_station->flag |= (ST_STATUS);                         // set "Status" flag
                p_station->record_type = NORMAL_APRS;                   // ???
                found_pos = 0;
                break;

            case (OTHER_DATA):          // Other Packets          [APRS Reference, chapter 19]
                // non-APRS beacons, treated as status reports until we get a real one
                if ((p_station->flag & (~ST_STATUS)) == '\0') {         // only store if no status yet
                    substr(p_station->comments,data,MAX_COMMENTS);
                    p_station->record_type = NORMAL_APRS;               // ???
                }
                found_pos = 0;
                break;

            case (APRS_OBJECT):
                ok = extract_time(p_station, data, type);                       // we need a time
                ok = (int)( ok && (extract_position(p_station,&data,type)       // uncompressed lat/lon
                        || extract_comp_position(p_station,&data,type)) );      // compressed lat/lon
                if (ok) {
                    strcpy(p_station->pos_time,get_time(temp_data));
                    strcpy(p_station->origin,origin);                           // define it as object
                    process_data_extension(p_station,data,type);                // PHG, speed, etc.
                    process_info_field(p_station,data,type);
                    substr(p_station->comments,data,MAX_COMMENTS);
                    p_station->record_type = NORMAL_APRS;
                    p_station->flag &= (~ST_MSGCAP);            // clear "message capable" flag
                }
                break;

            case (APRS_WX1):
                ok = extract_time(p_station, data, type);                       // we need a time
                ok = (int)( ok && extract_position(p_station,&data,type) );     // uncompressed lat/lon
                if (ok) {                                                       // ??? do we need it ???
                    (void)extract_weather(p_station,data);                      // look for weather data first
                    p_station->record_type = (char)APRS_WX1;
                    substr(p_station->comments,data,MAX_COMMENTS);
                }
                break;

            case (APRS_WX2):            // '_'
                ok = extract_time(p_station, data, type);               // we need a time
                if (ok) {
                    (void)extract_weather(p_station,data);              // look for weather data first
                    p_station->record_type = (char)APRS_WX2;
                    found_pos = 0;
                }
                break;

            case (APRS_WX4):            // '#'          Peet Bros U-II (mph)
            case (APRS_WX6):            // '*'          Peet Bros U-II (km/h)
            case (APRS_WX3):            // '!'          Ultimeter 2000 (data logging mode)
            case (APRS_WX5):            // '$ULTW'      Ultimeter 2000 (packet mode)
                if (get_weather_record(p_station)) {    // get existing or create new weather record
                    weather = p_station->weather_data;
                    if (type == APRS_WX3)
                        decode_U2000_L(1,(unsigned char *)data,weather);
                    else if (type == APRS_WX5)
                        decode_U2000_P(1,(unsigned char *)data,weather);
                    else
                        decode_Peet_Bros(1,(unsigned char *)data,weather,type);
                    p_station->record_type = (char)type;
                    strcpy(weather->wx_time, get_time(temp_data));
                    weather->wx_sec_time = sec_now();
                    found_pos = 0;
                }
                break;

            case (GPS_RMC):             // $GPRMC
                ok = extract_RMC(p_station,data,call_sign,path);
                break;

            case (GPS_GGA):             // $GPGGA
                ok = extract_GGA(p_station,data,call_sign,path);
                break;

            case (GPS_GLL):             // $GPGLL
                ok = extract_GLL(p_station,data,call_sign,path);
                break;

            default:
                printf("ERROR: UNKNOWN TYPE in data_add\n");
                ok = 0;
                break;
        }
    }

    if (!ok) {  // non-APRS beacon, treat it as Other Packet   [APRS Reference, chapter 19]
        if (debug_level & 1) {
            char filtered_data[MAX_LINE_SIZE + 1];
            strcpy(filtered_data,data-1);
            makePrintable(filtered_data);
            printf("store non-APRS data as status: %s: |%s|\n",call,filtered_data);
        }

        // GPRMC etc. without a position is here too, but it should not be stored as status!
        
        // store it as status report until we get a real one
        if ((p_station->flag & (~ST_STATUS)) == '\0') {         // only store it if no status yet
            substr(p_station->comments,data-1,MAX_COMMENTS);
            p_station->record_type = NORMAL_APRS;               // ???
        }
        ok = 1;            
        found_pos = 0;
    }

    if (ok) {           // we now have valid data, so do the rest
        // make time index unique by adding a serial number
        curr_sec = sec_now();
        p_station->sec_heard = curr_sec;
        if (curr_sec != last_sec) {     // todo: if old time calculate time_sn from database
            last_sec = curr_sec;
            next_time_sn = 0;           // restart time serial number
        }
        p_station->time_sn = next_time_sn++;            // todo: warning if serial number too high
        if (from == DATA_VIA_TNC) {                     // heard via TNC
            p_station->flag |= ST_VIATNC;               // set "via TNC" flag
            p_station->heard_via_tnc_last_time = sec_now();
            p_station->last_heard_via_tnc = 0l;
            p_station->heard_via_tnc_port = port;
        } else {                                        // heard other than TNC
            if (new_station) {                          // new station
                p_station->last_heard_via_tnc = 0L;
                p_station->flag &= (~ST_VIATNC);        // clear "via TNC" flag
                p_station->heard_via_tnc_last_time = 0l;
            } else {                            // old station
                p_station->last_heard_via_tnc++;
                /* if the last 20 times this station was heard other than tnc clear the heard tnc data */
                if (p_station->last_heard_via_tnc > 20) {
                    p_station->last_heard_via_tnc = 0l;
                    p_station->flag &= (~ST_VIATNC);        // clear "via TNC" flag
                }
            }
        }
        p_station->last_port_heard = port;
        p_station->data_via = from;
        strcpy(p_station->packet_time,get_time(temp_data));
        substr(p_station->node_path,path,MAX_PATH);
        p_station->flag |= ST_ACTIVE;
        if (third_party)
            p_station->flag |= ST_3RD_PT;               // set "third party" flag
        else
            p_station->flag &= (~ST_3RD_PT);            // clear "third party" flag
        if (origin != NULL && strcmp(origin,"INET") == 0)  // special treatment for inet names
            strcpy(p_station->origin,origin);           // to keep them separated from calls
        if (origin != NULL && strcmp(origin,"INET-NWS") == 0)  // special treatment for NWS
            strcpy(p_station->origin,origin);           // to keep them separated from calls
        if (origin == NULL || origin[0] == '\0')        // normal call
            p_station->origin[0] = '\0';                // undefine possible former object with same name
    } else                                              // if !ok
        if (new_station)
            delete_station_memory(p_station);           // remove unused record

    if (ok) {
        if (found_pos) {        // if station has a position with the data
            /* Check Audio Alarms based on incoming packet */
            if (!is_my_call(p_station->call_sign,1)) {
                /* FG don't care if this is on screen or off get position */
                l_lat = convert_lat_s2l(my_lat);
                l_lon = convert_lon_s2l(my_long);
                value = (float)calc_distance_course(l_lat,l_lon,p_station->coord_lat,p_station->coord_lon,temp_data); /* knots */
                if (units_english_metric)
                    distance = value * 1.15078;
                else
                    distance = value * 1.852;

                /* check ranges */
                if ((distance > atof(prox_min)) && (distance < atof(prox_max)) && sound_play_prox_message) {
                    sprintf(station_id,"%s < %.3f %s",p_station->call_sign, distance,
                            units_english_metric?langcode("UNIOP00004"):langcode("UNIOP00005"));
                    statusline(station_id,0);
                    play_sound(sound_command,sound_prox_message);
                    /*printf("%s> PROX distance %f\n",p_station->call_sign, distance);*/
                }
#ifdef HAVE_FESTIVAL
                if ((distance > atof(prox_min)) && (distance < atof(prox_max)) && festival_speak_proximity_alert) {
                    sprintf(station_id,"%s distance is %.3f %s.",p_station->call_sign, distance,
                        units_english_metric?langcode("UNIOP00004"):langcode("UNIOP00005"));
                        SayText(station_id);
                }
#endif                                                                                                                               
                /* FG really should check the path before we do this and add setup for these ranges */
                if ((distance > atof(bando_min)) && (distance < atof(bando_max)) &&
                        sound_play_band_open_message && from == DATA_VIA_TNC) {
                    sprintf(station_id,"%s %s %.1f %s",p_station->call_sign, langcode("UMBNDO0001"),
                        distance, units_english_metric?langcode("UNIOP00004"):langcode("UNIOP00005"));
                    statusline(station_id,0);
                    play_sound(sound_command,sound_band_open_message);
                    /*printf("%s> BO distance %f\n",p_station->call_sign, distance);*/
                }
#ifdef HAVE_FESTIVAL
                if ((distance > atof(bando_min)) && (distance < atof(bando_max)) &&
                       festival_speak_band_opening && from == DATA_VIA_TNC) {
                    sprintf(station_id,"Heard, D X, %s, %s %.1f %s",p_station->call_sign, langcode("UMBNDO0001"),
                        distance, units_english_metric?langcode("UNIOP00004"):langcode("UNIOP00005"));
                    SayText(station_id);
                }
#endif                                                                                                                               
            }

            if (position_on_extd_screen(p_station->coord_lat,p_station->coord_lon))
                p_station->flag |= (ST_INVIEW);   // set   "In View" flag
            else
                p_station->flag &= (~ST_INVIEW);  // clear "In View" flag
        } /* end found_pos */

        if (new_station) {
            if (position_on_screen(p_station->coord_lat,p_station->coord_lon)) {
                int show_station;
                char *net_ptr;

                substr(temp_data, p_station->node_path, MAX_CALL);
                if ((net_ptr = strchr(temp_data, ',')))
                    *net_ptr = '\0';

                show_station = altnet ? (!strcmp(temp_data, altnet_call) || !strcmp(temp_data, "local") ||
                                        !strncmp(temp_data, "SPC", 3)): (int)TRUE;

                if (p_station->coord_lat == 0 && p_station->coord_lon == 0)   // discard undef positions from screen
                    show_station = 0;

                if (show_station)
                    display_station(da,p_station,1);
            }
        } else {        // we had seen this station before...
            if (found_pos && position_defined(p_station->coord_lat,p_station->coord_lon,1)) { // ignore undefined and 0N/0E
                if (p_station->track_data != NULL)
                    moving = 1;                         // it's moving if it has a trail
                else
                    if (strlen(p_station->speed) > 0 && atof(p_station->speed) > 0)
                        moving = 1;                     // declare it moving, if it has a speed
                    else
                        if (position_defined(last_lat,last_lon,1)
                                && (p_station->coord_lat != last_lat || p_station->coord_lon != last_lon))
                            moving = 1;                 // it's moving if it has changed the position
                        else
                            moving = 0;
                changed_pos = 0;
                if (moving == 1) {                      // moving station: do trails
                    if (track_station_on == 1)          // maybe we are tracking a station
                        track_station(da,tracking_station_call,p_station);
        
                    if (atoi(p_station->speed) < TRAIL_MAX_SPEED) {     // reject high speed data
                        // if neccessary, store the first trail point, we already have the 2nd now!
                        if (p_station->track_data == NULL)
                            if (position_defined(last_lat,last_lon,1))  // ignore undefined and 0N/0E
                                (void)store_trail_point(p_station,last_lon,last_lat);

                        x_diff_long = labs(p_station->coord_lon - last_lon);
                        y_diff_lat  = labs(p_station->coord_lat - last_lat);
                        /*printf("%s delta x %ld y %ld\n",p_station->call_sign,x_diff_long,y_diff_lat);*/
                        /* I need to check these numbers to see if they give good tracking -FG */
                        if ((x_diff_long != 0l) || (y_diff_lat != 0l)) {
                            // we moved since last time
                            // don't store redundant points (change this if we store time later)
                            (void)store_trail_point(p_station,p_station->coord_lon,p_station->coord_lat);
                            if (station_trails && ((p_station->flag & ST_INVIEW) != '\0')) {
                                draw_trail(da,p_station,1);
                                changed_pos = 1;
                            }
                        }
                    } else
                        if (debug_level & 1)
                            printf("Speed over %d mph\n",TRAIL_MAX_SPEED);
                } // moving...

                if (position_on_screen(p_station->coord_lat,p_station->coord_lon)) {
                    if (!position_defined(last_lat,last_lon,0)) {
                        display_station(da,p_station,1);        // first time display
                                                        // now could be after error too...???
                    } else {
                        if (changed_pos == 1) {                 // changed position
                            display_station(da,p_station,1);
                            if (size > 2048 || p_station->data_via == 'F')
                                redraw_on_new_data = 1;         // less redraw activity
                            else
                                redraw_on_new_data = 2;         // better view of moving stations
                        }
                    }
                } // we have ignored stations just going offscreen
                
            } // defined position
        }

        if (is_my_call(p_station->call_sign,1)) {
            if (!new_station && strchr(p_station->node_path,'*') != NULL)
                statusline(langcode("BBARSTA033"),0);           // Echo from digipeater
        } else {
            if (!wait_to_redraw) {
                if (new_station) {
                    if (p_station->origin[0] == '\0')   // new station
                        sprintf(station_id,langcode("BBARSTA001"),p_station->call_sign);
                    else                                // new object
                        sprintf(station_id,langcode("BBARSTA000"),p_station->call_sign);
                } else                                  // new data
                    sprintf(station_id,langcode("BBARSTA002"),p_station->call_sign);
                statusline(station_id,0);
                if (new_station && sound_play_new_station)      // announce new station
                    play_sound(sound_command,sound_new_station);
#ifdef HAVE_FESTIVAL
                if (new_station && festival_speak_new_station) {
                    sprintf(station_id,"%s, %s",langcode("WPUPCFA005"),p_station->call_sign);
 
                    SayText(station_id);
                }
#endif                                                                                                                               
            }
        }
        p_station->num_packets += 1;
        redo_list = (int)TRUE;               // we may need to update the lists
    }
    return(ok);
}


void my_station_gps_change(char *pos_long, char *pos_lat, char *course, char *speed, char speedu, char *alt, char *sats) {
    long pos_long_temp, pos_lat_temp;
    char temp_data[40];   // short temrm string storage
    char temp_lat[12];
    char temp_long[12];
    DataRow *p_station;
    DataRow *p_time;

    p_station = NULL;
    if (!search_station_name(&p_station,my_callsign,1)) {  // find call 
        p_time = NULL;          // add to end of time sorted list
        p_station = add_new_station(p_station,p_time,my_callsign);
    }
    p_station->flag |= ST_ACTIVE;
    p_station->data_via = 'L';
    p_station->flag &= (~ST_3RD_PT);            // clear "third party" flag
    p_station->record_type = NORMAL_APRS;
    substr(p_station->node_path,"local",MAX_PATH);
    strcpy(p_station->packet_time,get_time(temp_data));
    strcpy(p_station->pos_time,get_time(temp_data));
    p_station->flag |= ST_MSGCAP;               // set "message capable" flag

    /* convert to long and weed out any odd data */
    pos_long_temp = convert_lon_s2l(pos_long);
    pos_lat_temp  = convert_lat_s2l(pos_lat);

    /* convert to back to clean string for config data */
    convert_lon_l2s(pos_long_temp,temp_data,CONVERT_HP_NORMAL);
    sprintf(temp_long,"%c%c%c%c%c.%c%c%c%c",temp_data[0],temp_data[1],temp_data[2], temp_data[4],temp_data[5],
            temp_data[7],temp_data[8], temp_data[9], temp_data[10]);
    convert_lat_l2s(pos_lat_temp,temp_data,CONVERT_HP_NORMAL);
    sprintf(temp_lat,"%c%c%c%c.%c%c%c%c",temp_data[0],temp_data[1],temp_data[3],temp_data[4], temp_data[6],
            temp_data[7], temp_data[8],temp_data[9]);

    /* fill the data in */    // ???????????????
    strcpy(my_lat,temp_lat);
    strcpy(my_long,temp_long);
    p_station->coord_lat = convert_lat_s2l(my_lat);
    p_station->coord_lon = convert_lon_s2l(my_long);

    if(debug_level)
        printf("GPS MY_LAT <%s> MY_LONG <%s>\n", my_lat, my_long);

    if ((p_station->coord_lon != pos_long_temp) || (p_station->coord_lat != pos_lat_temp)) {
        /* check to see if enough to change pos on screen */
        if ((pos_long_temp>x_long_offset) && (pos_long_temp<(x_long_offset+(long)(screen_width*size)))) {
            if ((pos_lat_temp>y_lat_offset) && (pos_lat_temp<(y_lat_offset+(long)(screen_height*size)))) {
                if((labs((p_station->coord_lon+(size/2))-pos_long_temp)/size)>0 || (labs((p_station->coord_lat+(size/2))-pos_lat_temp)/size)>0) {
                    redraw_on_new_data = 1;  // redraw next chance
                    /*printf("Redraw on new gps data \n");*/
                    statusline("Position Change on My Station",0);
                }
            }
        }
    }

    p_station->coord_lat = pos_lat_temp;    // DK7IN: we have it already !??
    p_station->coord_lon = pos_long_temp;

    strcpy(p_station->altitude_time,get_time(temp_data));
    my_last_altitude_time=sec_now();
    strcpy(p_station->speed_time,get_time(temp_data));
    strcpy(p_station->speed,speed);
    p_station->speed_unit=speedu;
    strcpy(p_station->course,course);
    strcpy(p_station->altitude,alt);
    // altu;    unit should always be meters  ????

    /* get my last altitude meters to feet */
    my_last_altitude=(long)(atof(alt)*3.28084);

    /* get my last course in deg */
    my_last_course=atoi(course);

    /* get my last speed knots to mph */
    my_last_speed=(int)(atof(speed)*1.1508);
    strcpy(p_station->sats_visible,sats);
}



void my_station_add(char *my_callsign, char my_group, char my_symbol, char *my_long, char *my_lat, char *my_phg, char *my_comment) {
    DataRow *p_station;
    DataRow *p_time;
    char temp_data[40];   // short temrm string storage

    p_station = NULL;
    if (!search_station_name(&p_station,my_callsign,1)) {  // find call 
        p_time = NULL;          // add to end of time sorted list
        p_station = add_new_station(p_station,p_time,my_callsign);
    }
    p_station->flag |= ST_ACTIVE;
    p_station->data_via = 'L';
    p_station->flag &= (~ST_3RD_PT);            // clear "third party" flag
    p_station->record_type = NORMAL_APRS;
    substr(p_station->node_path,"local",MAX_PATH);
    strcpy(p_station->packet_time,get_time(temp_data));
    strcpy(p_station->pos_time,get_time(temp_data));
    p_station->flag |= ST_MSGCAP;               // set "message capable" flag

    /* Symbol overlay */
    if(my_group != '/' && my_group != '\\') {   // overlay
        p_station->aprs_symbol.aprs_type = '\\';
        p_station->aprs_symbol.special_overlay = my_group;
    } else {                                    // normal symbol
        p_station->aprs_symbol.aprs_type = my_group;
        p_station->aprs_symbol.special_overlay = ' ';
    }
    p_station->aprs_symbol.aprs_symbol = my_symbol;

    p_station->coord_lat = convert_lat_s2l(my_lat);
    p_station->coord_lon = convert_lon_s2l(my_long);

    if (position_on_extd_screen(p_station->coord_lat,p_station->coord_lon)) {
        p_station->flag |= (ST_INVIEW);   // set   "In View" flag
    } else {
        p_station->flag &= (~ST_INVIEW);  // clear "In View" flag
    }

    substr(p_station->power_gain,my_phg,7);
    substr(p_station->comments,my_comment,MAX_COMMENTS);

    my_last_course = 0;         // set my last course in deg to zero
    redo_list = (int)TRUE;      // update active station lists
}



void display_packet_data(void) {
    int i;
    int pos;

    pos=0;
    if((Display_data_dialog != NULL) && (redraw_on_new_packet_data==1)) {
        XtVaSetValues(Display_data_text,XmNcursorPosition,&pos,NULL);
        XmTextSetString(Display_data_text,"");

        for (i=0; i<packet_data_display; i++) {
            XmTextInsert(Display_data_text,pos,packet_data[i]);
            pos += strlen(packet_data[i]);
            XtVaSetValues(Display_data_text,XmNcursorPosition,pos,NULL);
        }
        XmTextShowPosition(Display_data_text,0);
    }
    redraw_on_new_packet_data=0;
}



void packet_data_add(char *from, char *line) {
    int i;
    int offset;

    offset=0;
    if (line[0]==(char)3)
        offset=1;

    if ( (Display_packet_data_type==1 && strcmp(from,langcode("WPUPDPD005"))==0) ||
            (Display_packet_data_type==2 && strcmp(from,langcode("WPUPDPD006"))==0) || Display_packet_data_type==0) {

        redraw_on_new_packet_data=1;
        if (packet_data_display<15) {
            if (strlen(line)<256) {
                strcpy(packet_data[packet_data_display],from);
                strcat(packet_data[packet_data_display],"-> ");
                strcat(packet_data[packet_data_display],line+offset);
                strcat(packet_data[packet_data_display++],"\n\0");
            } else {
                strcpy(packet_data[packet_data_display],from);
                strcat(packet_data[packet_data_display],"-> ");
                strncat(packet_data[packet_data_display],line+offset,256);
                strcat(packet_data[packet_data_display++],"\n\0");
            }
        } else {
            packet_data_display=15;
            for (i=0; i<14; i++)
                strcpy(packet_data[i],packet_data[i+1]);

            if (strlen(line)<256) {
                strcpy(packet_data[14],from);
                strcat(packet_data[14],"-> ");
                strcat(packet_data[14],line+offset);
                strcat(packet_data[14],"\n\0");
            } else {
                strcpy(packet_data[14],from);
                strcat(packet_data[14],"-> ");
                strncat(packet_data[14],line+offset,256);
                strcat(packet_data[14],"\n\0");
            }
        }
    }
}


/*
 *  Decode Mic-E encoded data
 */
int decode_Mic_E(char *call_sign,char *path,char *info,char from,int port,int third_party) {
    int  i;
    int  offset;
    unsigned char s_b1;
    unsigned char s_b2;
    unsigned char s_b3;
    unsigned char s_b4;
    unsigned char s_b5;
    unsigned char s_b6;
    unsigned char s_b7;
    int  n,w,l;
    int  d,m,h;
    char temp[MAX_TNC_LINE_SIZE+1];     // Note: Must be big in case we get long concatenated packets
    char new_info[MAX_TNC_LINE_SIZE+1]; // Note: Must be big in case we get long concatenated packets
    int  course;
    int  speed;
    int  msg1,msg2,msg3,msg;
    int  info_size;
    long alt;
    int  msgtyp;
    char rig_type[10];
    int ok;
        
    // MIC-E Data Format   [APRS Reference, chapter 10]

    // todo:  error check
    //        drop wrong positions from receive errors...
    //        drop 0N/0E position (p.25)
    
    /* First 7 bytes of info[] contains the APRS data type ID,    */
    /* longitude, speed, course.                    */
    /* The 6-byte destination field of path[] contains latitude,    */
    /* N/S bit, E/W bit, longitude offset, message code.        */
    /*

    MIC-E Destination Field Format:
    -------------------------------
    Ar1DDDD0 Br1DDDD0 Cr1MMMM0 Nr1MMMM0 Lr1HHHH0 Wr1HHHH0 CrrSSID0
    D = Latitude Degrees.
    M = Latitude Minutes.
    H = Latitude Hundredths of Minutes.
    ABC = Message bits, complemented.
    N = N/S latitude bit (N=1).
    W = E/W longitude bit (W=1).
    L = 100's of longitude degrees (L=1 means add 100 degrees to longitude
    in the Info field).
    C = Command/Response flag (see AX.25 specification).
    r = reserved for future use (currently 0).
    
    */
    /****************************************************************************
    * I still don't handle:                                                     *
    *    Custom message bits                                                    *
    *    SSID special routing                                                   *
    *    Beta versions of the MIC-E (which use a slightly different format).    *
    *                                                                           *
    * DK7IN : lat/long with custom msg works, altitude/course/speed works       *
    *****************************************************************************/

    if (debug_level & 1)
        printf("FOUND MIC-E\n");

    msg1 = (int)( ((unsigned char)path[0] & 0x40) >>4 );
    msg2 = (int)( ((unsigned char)path[1] & 0x40) >>5 );
    msg3 = (int)( ((unsigned char)path[2] & 0x40) >>6 );
    msg = msg1 | msg2 | msg3;        // We now have the complemented message number in one variable
    msg = msg ^ 0x07;               // And this is now the normal message number
    msgtyp = 0;                     // DK7IN: Std message, I have to add custom msg decoding

    /* Snag the latitude from the destination field, Asume TAPR2 */
    /* DK7IN: latitude now works with custom message */
    s_b1 = (unsigned char)( (path[0] & 0x0f)+ (char)0x2f );
    if (path[0] & 0x10)     // A-J
        s_b1 += (unsigned char)1;

    if (s_b1 > (unsigned char)0x39)        // K,L,Z
        s_b1 = (unsigned char)0x20;

    s_b2 = (unsigned char)( (path[1] & 0x0f) + (char)0x2f );
    if (path[1] & 0x10)     // A-J
        s_b2 += (unsigned char)1;

    if (s_b2 > (unsigned char)0x39)        // K,L,Z
        s_b2 = (unsigned char)0x20;

    s_b3 = (unsigned char)( (path[2] & (char)0x0f) + (char)0x2f );
    if (path[2] & 0x10)     // A-J
        s_b3 += (unsigned char)1;

    if (s_b3 > (unsigned char)0x39)        // K,L,Z
        s_b3 = (unsigned char)0x20;

    s_b4 = (unsigned char)( (path[3] & 0x0f) + (char)0x30 );
    if (s_b4 > (unsigned char)0x39)        // L,Z
        s_b4 = (unsigned char)0x20;

    s_b5 = (unsigned char)( (path[4] & 0x0f) + (char)0x30 );
    if (s_b5 > (unsigned char)0x39)        // L,Z
        s_b5 = (unsigned char)0x20;

    s_b6 = (unsigned char)( (path[5] & 0x0f) + (char)0x30 );
    if (s_b6 > (unsigned char)0x39)        // L,Z
        s_b6 = (unsigned char)0x20;

    s_b7 =  (unsigned char)path[6];        // SSID, not used here

    n = (int)((path[3] & 0x40) == (char)0x40);  // N/S Lat Indicator
    l = (int)((path[4] & 0x40) == (char)0x40);  // Longitude Offset
    w = (int)((path[5] & 0x40) == (char)0x40);  // W/E Long Indicator

    /* Put the latitude string into the temp variable */
    sprintf(temp,"%c%c%c%c.%c%c%c%c",s_b1,s_b2,s_b3,s_b4,s_b5,s_b6,
            (n ? 'N': 'S'), info[7]);

    /* Compute degrees longitude */
    strcpy(new_info,temp);
    d = (int) info[0]-28;

    if (l)
        d += 100;

    if ((180<=d)&&(d<=189))  // ??
        d -= 80;

    if ((190<=d)&&(d<=199))  // ??
        d -= 190;

    /* Compute minutes longitude */
    m = (int) info[1]-28;
    if (m>=60)
        m -= 60;

    /* Compute hundredths of minutes longitude */
    h = (int) info[2]-28;
    /* Add the longitude string into the temp variable */
    sprintf(temp,"%03d%02d.%02d%c%c",d,m,h,(w ? 'W': 'E'), info[6]);
    strcat(new_info,temp);

    /* Compute speed in knots */
    speed = (int)( ( info[3] - (char)28 ) * (char)10 );
    speed += ( (int)( (info[4] - (char)28) / (char)10) );
    if (speed >= 800)
        speed -= 800;       // in knots

    /* Compute course */
    course = (int)( ( ( (info[4] - (char)28) % 10) * (char)100) + (info[5] - (char)28) );
    if (course >= 400)
        course -= 400;

    /*  ???
        printf("info[4]-28 mod 10 - 4 = %d\n",( ( (int)info[4]) - 28) % 10 - 4);
        printf("info[5]-28 = %d\n", ( (int)info[5]) - 28 );
    */
    sprintf(temp,"%03d/%03d",course,speed);
    strcat(new_info,temp);
    offset = 8;   // start of rest of info

    /* search for rig type in Mic-E data */
    rig_type[0] = '\0';
    if (info[offset] != '\0' && (info[offset] == '>' || info[offset] == ']')) {
        /* detected type code:     > TH-D7    ] TM-D700 */
        if (info[offset] == '>')
            strcpy(rig_type,"TH-D7");
        else
            strcpy(rig_type,"TM-D700");

        offset++;
    }

    info_size = (int)strlen(info);
    /* search for compressed altitude in Mic-E data */  // {
    if (info_size >= offset+4 && info[offset+3] == '}') {  // {
        /* detected altitude  ___} */
        alt = ((((long)info[offset] - (long)33) * (long)91 +(long)info[offset+1] - (long)33) * (long)91
                    + (long)info[offset+2] - (long)33) - 10000;  // altitude in meters
        alt /= 0.3048;                                // altitude in feet, as in normal APRS
        sprintf(temp,"/A=%06ld",alt);
        offset += 4;
        strcat(new_info,temp);
    }

    /* start of comment */
    if (strlen(rig_type) > 0) {
        sprintf(temp,"%s ",rig_type);
        strcat(new_info,temp);
    }

    strcat(new_info,"Mic-E ");
    if (msgtyp == 0) {
        switch (msg) {
            case 1:
                strcat(new_info,"Enroute");
                break;

            case 2:
                strcat(new_info,"In Service");
                break;

            case 3:
                strcat(new_info,"Returning");
                break;

            case 4:
                strcat(new_info,"Committed");
                break;

            case 5:
                strcat(new_info,"Special");
                break;

            case 6:
                strcat(new_info,"Priority");
                break;

            case 7:
                strcat(new_info,"Emergency");
                break;

            default:
                strcat(new_info,"Off Duty");
        }
    } else {
        sprintf(temp,"Custom%d",msg);
        strcat(new_info,temp);
    }

    if (info[offset] != '\0') {
        /* Append the rest of the message to the expanded MIC-E message */
        for (i=offset; i<info_size; i++)
            temp[i-offset] = info[i];

        temp[info_size-offset] = '\0';
        strcat(new_info," ");
        strcat(new_info,temp);
    }

    if (debug_level & 1)
        printf("Done decoding MIC-E\n");

    ok = data_add(APRS_MICE,call_sign,path,new_info,from,port,NULL,third_party);
    return(ok);
}


/*
 *  Directed Station Query (query to my station)   [APRS Reference, chapter 15]
 */
int process_directed_query(char *call,char *path,char *message,char from) {
    DataRow *p_station;
    char from_call[MAX_CALLSIGN+1];
    char temp[100];
    int ok;

    ok = 0;
    if (!ok && strncmp(message,"APRSD",5) == 0 && from != 'F') {  // stations heard direct
        pad_callsign(from_call,call);
        sprintf(temp,":%s:Directs=",from_call);
        p_station = n_first;
        while (p_station != NULL) {
            if ((p_station->flag & ST_ACTIVE) != '\0') {        // ignore deleted objects
                if ((p_station->flag & ST_VIATNC) != '\0') {    // test "via TNC" flag
        // DK7IN: I think the implementation is wrong, we deliver stations heard via a digi too...
                    if (strlen(temp)+strlen(p_station->call_sign) < 65) {
                        strcat(temp," ");
                        strcat(temp,p_station->call_sign);
                    } else
                        break;
                }
            }
            p_station = p_station->n_next;
        }
        transmit_message_data(call,temp);
        ok = 1;
    }
    if (!ok && strncmp(message,"APRSH",5)==0) {
        ok = 1;
    }
    if (!ok && strncmp(message,"APRSM",5)==0) {
        ok = 1;
    }
    if (!ok && strncmp(message,"APRSO",5)==0) {
        ok = 1;
    }
    if (!ok && strncmp(message,"APRSP",5) == 0 && from != 'F') {
        transmit_now = 1;       //send position
        ok = 1;
    }
    if (!ok && strncmp(message,"APRSS",5)==0) {
        ok = 1;
    }
    if (!ok && (strncmp(message,"APRST",5)==0
            ||  strncmp(message,"PING?",5)==0) && from != 'F') {
        pad_callsign(from_call,call);
        sprintf(temp,":%s:PATH= %s>%s",from_call,call,path);    // correct format ?????
        transmit_message_data(call,temp);
        ok = 1;
    }
    if (!ok && strncasecmp("VER",message,3) == 0 && from != 'F') { // not in Reference !???
        pad_callsign(from_call,call);
        sprintf(temp,":%s:%s",from_call,VERSIONLABEL);
        transmit_message_data(call,temp);
    }
    return(ok);
}


/*
 *  Station Capabilities, Queries and Responses      [APRS Reference, chapter 15]
 */
int process_query( /*@unused@*/ char *call_sign, /*@unused@*/ char *path,char *message,char from,int port, /*@unused@*/ int third_party) {
    char temp[100];
    int ok;

    ok = 0;
    if (!ok && strncmp(message,"APRS?",5)==0) {
        ok = 1;
    }
    if (!ok && strncmp(message,"IGATE?",6)==0) {
        if (operate_as_an_igate && from != 'F') {
            sprintf(temp,"<IGATE,MSG_CNT=%d,LOC_CNT=%d",(int)igate_msgs_tx,stations_types(3));
            output_my_data(temp,port,0);
        }
        ok = 1;
    }
    if (!ok && strncmp(message,"WX?",3)==0) {
        ok = 1;
    }
    return(ok);
}


/*
 *  Status Reports                              [APRS Reference, chapter 16]
 */
int process_status( /*@unused@*/ char *call_sign, /*@unused@*/ char *path, /*@unused@*/ char *message, /*@unused@*/ char from, /*@unused@*/ int port, /*@unused@*/ int third_party) {

//    popup_message(langcode("POPEM00018"),message);  // What is it ???
    return(1);
}


/*
 *  Mesages, Bulletins and Announcements         [APRS Reference, chapter 14]
 */
int decode_message(char *call,char *path,char *message,char from,int port,int third_party) {
    char *temp_ptr;
    char ipacket_message[300];
    char from_call[MAX_CALLSIGN+1];
    char ack[20];
    int ok, len;
    char addr[9+1];
    char addr9[9+1];
    char msg_id[5+1];
    int done;

    // :xxxxxxxxx:____0-67____             message              printable, except '|', '~', '{'
    // :BLNn     :____0-67____             general bulletin     printable, except '|', '~'
    // :BLNnxxxxx:____0-67____           + Group Bulletin
    // :BLNX     :____0-67____             Announcement
    // :NWS-xxxxx:____0-67____             NWS Service Bulletin
    // :xxxxxxxxx:ackn1-5n               + ack
    // :xxxxxxxxx:rejn1-5n               + rej
    // :xxxxxxxxx:____0-67____{n1-5n     + message
    // :NTS....
    //  01234567890123456
    // 01234567890123456    old
    // we get message with already extracted data ID
    
    len = (int)strlen(message);
    ok = (int)(len > 9 && message[9] == ':');
    if (ok) {
        substr(addr9,message,9);                // extract addressee
        strcpy(addr,addr9);
        (void)remove_trailing_spaces(addr);
        message = message + 10;                 // pointer to message text
        temp_ptr = strstr(message,"{");         // look for message ID
        msg_id[0] = '\0';
        if (temp_ptr != NULL) {
            substr(msg_id,temp_ptr+1,5);        // extract message ID, could be non-digit
            temp_ptr[0] = '\0';                 // adjust message end
        }
        done = 0;
    } else
        done = 1;                               // fall through...
    len = (int)strlen(message);
    //--------------------------------------------------------------------------
    if (!done && len > 3 && strncmp(message,"ack",3) == 0) {              // ACK
        substr(msg_id,message+3,5);
        // printf("ACK: %s: |%s| |%s|\n",call,addr,msg_id);
        if (is_my_call(addr,1))
            clear_acked_message(call,addr,msg_id);      // got an ACK for me
        else {                                          // ACK for other station
            /* Now if I have Igate on and I allow to retransmit station data           */
            /* check if this message is to a person I have heard on my TNC within an X */
            /* time frame. If if is a station I heard and all the conditions are ok    */
            /* spit the ACK out on the TNC -FG                                         */
            if (operate_as_an_igate>1 && from==DATA_VIA_NET && !is_my_call(call,1)) {
                /*printf("Igate check o:%d f:%c myc:%s cf:%s ct:%s\n",operate_as_an_igate,from,my_callsign,call,addr); { */
                sprintf(ipacket_message,"}%s>%s:%s:%s",call,path,addr9,message);
                output_igate_rf(call,addr,path,ipacket_message,port,third_party);
                igate_msgs_tx++;
            }
        }
        done = 1;
    }
    //--------------------------------------------------------------------------
    if (!done && len > 3 && strncmp(message,"rej",3) == 0) {              // REJ
        substr(msg_id,message+3,5);
        // printf("REJ: %s: |%s| |%s|\n",call,addr,msg_id);
        // we ignore it
        done = 1;
    }
    //--------------------------------------------------------------------------
    if (!done && strncmp(addr,"BLN",3) == 0) {                       // Bulletin
        // printf("found BLN: |%s| |%s|\n",addr,message);
        bulletin_message(from,call,addr,message,sec_now());
        msg_data_add(addr,call,message,"",MESSAGE_BULLETIN,from);
        done = 1;
    }
    //--------------------------------------------------------------------------
    if (!done && strlen(msg_id) > 0 && is_my_call(addr,1)) {   // message for me
        // printf("found Msg w line to me: |%s| |%s|\n",message,msg_id);
        msg_data_add(addr,call,message,msg_id,MESSAGE_MESSAGE,from); // id_fixed
        new_message_data += 1;
        (void)check_popup_window(call, 2);
        update_messages(0); // No force
        if (sound_play_new_message)
            play_sound(sound_command,sound_new_message);
#ifdef HAVE_FESTIVAL
/* I re-use ipacket_message as my string buffer */
        if (festival_speak_new_message_alert) {
            sprintf(ipacket_message,"You have a new message from %s.",call);
            SayText(ipacket_message);
        }
        if (festival_speak_new_message_body) {
            sprintf(ipacket_message," %s",message);
            SayText(ipacket_message);
        }
 
#endif                                                                                                                               
        if (from != 'F') {
            pad_callsign(from_call,call);         /* ack the message */
            sprintf(ack,":%s:ack%s",from_call,msg_id);
            transmit_message_data(call,ack);
            if (auto_reply == 1) {
                if (debug_level & 2)
                    printf("Send autoreply to <%s> from <%s> :%s\n",call,my_callsign,auto_reply_message);
                if (!is_my_call(call,1))
                    output_message(my_callsign,call,auto_reply_message);
            }
        }
        done = 1;
    }
    //--------------------------------------------------------------------------
    if (!done && strncmp(addr,"NWS-",4) == 0) {             // NWS weather alert
        // printf("found NWS: |%s| |%s| |%s|\n",addr,message,msg_id);      // could have sort of line number
        msg_data_add(addr,call,message,"",MESSAGE_NWS,from);
        (void)alert_message_scan();
        done = 1;
        if (operate_as_an_igate>1 && from==DATA_VIA_NET && !is_my_call(call,1)) { // { for my editor...
            sprintf(ipacket_message,"}%s>%s:%s:%s",call,path,addr9,message);
            output_nws_igate_rf(call,path,ipacket_message,port,third_party);
        }
    }
    //--------------------------------------------------------------------------
    if (!done && strlen(msg_id) > 0) {          // other message with linenumber
        // printf("found Msg w line: |%s| |%s| |%s|\n",addr,message,msg_id);
        msg_data_add(addr,call,message,msg_id,MESSAGE_MESSAGE,from);
        new_message_data += look_for_open_group_data(addr);
        if ((is_my_call(call,1) && check_popup_window(addr, 2) != -1)
                     || check_popup_window(call, 0) != -1 || check_popup_window(addr, 1) != -1)
            update_messages(0); // No force
        /* Now if I have Igate on and I allow to retransmit station data           */
        /* check if this message is to a person I have heard on my TNC within an X */
        /* time frame. If if is a station I heard and all the conditions are ok    */
        /* spit the message out on the TNC -FG                                     */
        if (operate_as_an_igate>1 && from==DATA_VIA_NET && !is_my_call(call,1) && !is_my_call(addr,1)) {
            /*printf("Igate check o:%d f:%c myc:%s cf:%s ct:%s\n",operate_as_an_igate,from,my_callsign,
                        call,addr);*/     // {
            sprintf(ipacket_message,"}%s>%s:%s:%s",call,path,addr9,message);
            output_igate_rf(call,addr,path,ipacket_message,port,third_party);
            igate_msgs_tx++;
        }
        done = 1;
    }
    //--------------------------------------------------------------------------
    if (!done && len > 5 && message[0] == '?' && is_my_call(addr,1)) { // directed query
        done = process_directed_query(call,path,message+1,from);
    }
    //--------------------------------------------------------------------------
    // special treatment required?: Msg w/o line for me ???
    //--------------------------------------------------------------------------
    if (!done) {                                   // message without line number
        // printf("found Msg: |%s| |%s|\n",addr,message);
        msg_data_add(addr,call,message,"",MESSAGE_MESSAGE,from);
        new_message_data++;      // ??????
        if (check_popup_window(addr, 1) != -1)
            update_messages(0); // No force
        // done = 1;
    }
    //--------------------------------------------------------------------------
    if (ok)
        (void)data_add(STATION_CALL_DATA,call,path,message,from,port,NULL,third_party);
    return(ok);
}


/*
 *  UI-View format messages, not relevant for APRS, format is not specified in APRS Reference
 */
int decode_UI_message(char *call,char *path,char *message,char from,int port,int third_party) {
    char *temp_ptr;
    char from_call[MAX_CALLSIGN+1];
    char ack[20];
    char addr[9+1];
    int ok, len;
    char msg_id[5+1];
    int done;

    // I'm not sure, but I think they use 2 digit line numbers only
    // extract addr from path
    substr(addr,path,9);
    ok = (int)(strlen(addr) > 0);
    if (ok) {
        temp_ptr = strstr(addr,",");         // look for end of first call
        if (temp_ptr != NULL)
            temp_ptr[0] = '\0';                 // adjust addr end
        ok = (int)(strlen(addr) > 0);
    }
    len = (int)strlen(message);
    ok = (int)(len >= 2);
    if (ok) {
        temp_ptr = strstr(message,"~");         // look for message ID
        msg_id[0] = '\0';
        if (temp_ptr != NULL) {
            substr(msg_id,temp_ptr+1,2);        // extract message ID, could be non-digit
            temp_ptr[0] = '\0';                 // adjust message end
        }
        done = 0;
    } else
        done = 1;                               // fall through...
    len = (int)strlen(message);
    //--------------------------------------------------------------------------
    if (!done && msg_id[0] != '\0' && is_my_call(addr,1)) {      // message for me
        msg_data_add(addr,call,message,msg_id,MESSAGE_MESSAGE,from);
        new_message_data += 1;
        (void)check_popup_window(call, 2);
        update_messages(0); // No force
        if (sound_play_new_message)
            play_sound(sound_command,sound_new_message);
        if (from != 'F') {
            pad_callsign(from_call,call);         /* ack the message */
            sprintf(ack,":%s:ack%s",from_call,msg_id);
            transmit_message_data(call,ack);
            if (auto_reply == 1) {
                if (debug_level & 2)
                    printf("Send autoreply to <%s> from <%s> :%s\n",call,my_callsign,auto_reply_message);
                if (!is_my_call(call,1))
                    output_message(my_callsign,call,auto_reply_message);
            }
        }
        done = 1;
    }
    //--------------------------------------------------------------------------
    if (!done && len == 2 && msg_id[0] == '\0') {                // ACK
        substr(msg_id,message,5);
        if (is_my_call(addr,1))
            clear_acked_message(call,addr,msg_id);      // got an ACK for me
//        else {                                          // ACK for other station
            /* Now if I have Igate on and I allow to retransmit station data           */
            /* check if this message is to a person I have heard on my TNC within an X */
            /* time frame. If if is a station I heard and all the conditions are ok    */
            /* spit the ACK out on the TNC -FG                                         */
//            if (operate_as_an_igate>1 && from==DATA_VIA_NET && !is_my_call(call,1)) {
//                /*printf("Igate check o:%d f:%c myc:%s cf:%s ct:%s\n",operate_as_an_igate,from,my_callsign,call,addr); { */
//                sprintf(ipacket_message,"}%s>%s:%s:%s",call,path,addr9,message);
//                output_igate_rf(call,addr,path,ipacket_message,port,third_party);
//                igate_msgs_tx++;
//            }
//        }
        done = 1;
    }
    //--------------------------------------------------------------------------
    if (ok)
        (void)data_add(STATION_CALL_DATA,call,path,message,from,port,NULL,third_party);
    return(ok);
}



/*
 *  Decode APRS Information Field and dispatch it depending on the Data Type ID
 */
void decode_info_field(char *call, char *path, char *message, char *origin, char from, int port, int third_party) {
    char line[MAX_TNC_LINE_SIZE+1];
    char my_data[MAX_TNC_LINE_SIZE+1];
    int  ok_igate_net;
    int  done, ignore;
    char data_id;

    /* remember fixed format starts with ! and can be up to 24 chars in the message */ // ???
    if (debug_level & 1)
        printf("decode c:%s p:%s m:%s f:%c\n",call,path,message,from);
    if (debug_level & 1)
        printf("decode Past check\n");

    done         = 0;                                   // if 1, packet was decoded
    ignore       = 0;                                   // if 1, don't treat undecoded packets as status text
    ok_igate_net = 0;                                   // if 1, send packet to internet
    my_data[0]   = '\0';                                // not really needed...

    if (message == NULL || strlen(message) == 0) {      // we could have an empty message
        (void)data_add(STATION_CALL_DATA,call,path,NULL,from,port,origin,third_party);
        done = 1;                                       // don't report it to internet
    } else 
        strcpy(my_data,message);                        // copy message for later internet transmit

    if (!done && strlen(origin) > 0 && strncmp(origin,"INET",4)!=0) { // special treatment for object or item
        if (message[0] == '*' || message[0] == '!') {   // set object/item
            (void)data_add(APRS_OBJECT,call,path,message+1,from,port,origin,third_party);
            ok_igate_net = 1;                           // report it to internet
        } else
            if (message[0] == '_') {                    // delete object/item
                delete_object(call);                    // ?? does not vanish from map immediately !!???
                ok_igate_net = 1;                       // report it to internet
            }
        done = 1;
    }

    if (!done) {
        data_id = message[0];           // look at the APRS Data Type ID (first char in information field)
        message += 1;                   // extract data ID from information field
        ok_igate_net = 1;               // as default report packet to internet

        if (debug_level & 1)
        {
if (ok_igate_net)
printf("ok_igate_net can be read\n");
        }

        switch (data_id) {
            case '=':   // Position without timestamp (with APRS messaging)
                done = data_add(APRS_MSGCAP,call,path,message,from,port,origin,third_party);
                break;

            case '!':   // Position without timestamp (no APRS messaging) or Ultimeter 2000 WX
                if (message[0] == '!' && is_xnum_or_dash(message+1,40))   // Ultimeter 2000 WX
                    done = data_add(APRS_WX3,call,path,message+1,from,port,origin,third_party);
                else
                    done = data_add(APRS_FIXED,call,path,message,from,port,origin,third_party);
                break;

            case '/':   // Position with timestamp (no APRS messaging)
                done = data_add(APRS_DOWN,call,path,message,from,port,origin,third_party);
                break;

            case '@':   // Position with timestamp (with APRS messaging)
                if (message[29] == '/') {
                    if (message[33] == 'g' && message[37] == 't')
                        done = data_add(APRS_WX1,call,path,message,from,port,origin,third_party);
                    else
                        done = data_add(APRS_MOBILE,call,path,message,from,port,origin,third_party);
                } else
                    done = data_add(APRS_DF,call,path,message,from,port,origin,third_party);
                break;

            case 0x27:  // Mic-E  Old GPS data (or current GPS data in Kenwood TM-D700)
            case 0x60:  // Mic-E  Current GPS data (but not used in Kennwood TM-D700)
            //case 0x1c:// Mic-E  Current GPS data (Rev. 0 beta units only)
            //case 0x1d:// Mic-E  Old GPS data (Rev. 0 beta units only)
                done = decode_Mic_E(call,path,message,from,port,third_party);
                break;

            case '_':   // Positionless weather data                [APRS Reference, chapter 12]
                done = data_add(APRS_WX2,call,path,message,from,port,origin,third_party);
                break;

            case '#':   // Peet Bros U-II Weather Station (mph)     [APRS Reference, chapter 12]
                if (is_xnum_or_dash(message,13))
                    done = data_add(APRS_WX4,call,path,message,from,port,origin,third_party);
                break;

            case '*':   // Peet Bros U-II Weather Station (km/h)
                if (is_xnum_or_dash(message,13))
                    done = data_add(APRS_WX6,call,path,message,from,port,origin,third_party);
                break;

            case '$':   // Raw GPS data or Ultimeter 2000
                if (strncmp("ULTW",message,4) == 0 && is_xnum_or_dash(message+4,44))
                    done = data_add(APRS_WX5,call,path,message+4,from,port,origin,third_party);
                else if (strncmp("GPGGA",message,5) == 0)
                    done = data_add(GPS_GGA,call,path,message,from,port,origin,third_party);
                else if (strncmp("GPRMC",message,5) == 0)
                    done = data_add(GPS_RMC,call,path,message,from,port,origin,third_party);
                else if (strncmp("GPGLL",message,5) == 0)
                    done = data_add(GPS_GLL,call,path,message,from,port,origin,third_party);
                else {
                        // handle VTG and WPT too  (APRS Ref p.25)
                }
                break;

            case ':':   // Message
                done = decode_message(call,path,message,from,port,third_party);
                // there could be messages I should not retransmit to internet... ??? Queries to me...
                break;

            case '>':   // Status                                   [APRS Reference, chapter 16]
                done = data_add(APRS_STATUS,call,path,message,from,port,origin,third_party);
                break;

            case '?':   // Query
                done = process_query(call,path,message,from,port,third_party);
                ignore = 1;     // don't treat undecoded packets as status text
                break;

            case '~':   // UI-format messages, not relevant for APRS ("Do not use" in Reference)
            case ',':   // Invalid data or test packets             [APRS Reference, chapter 19]
            case '<':   // Station capabilities                     [APRS Reference, chapter 15]
            case '{':   // User-defined APRS packet format     //}
            case '%':   // Agrelo DFJr / MicroFinder
            case '&':   // Reserved -- Map Feature
            case 'T':   // Telemetrie data                          [APRS Reference, chapter 13]
            case '[':   // Maidenhead grid locator beacon (obsolet)
                ignore = 1;     // don't treat undecoded packets as status text
                break;
        }

        if (debug_level & 1)
        {
            if (done)
                printf("done = 1\n");
            else
                printf("done = 0\n");
if (ok_igate_net)
printf("ok_igate_net can be read\n");
        }

        if (debug_level & 1)
            printf("decode_info_field: done with big switch\n");

        if (!done && !ignore) {         // Other Packets        [APRS Reference, chapter 19]
            done = data_add(OTHER_DATA,call,path,message-1,from,port,origin,third_party);
            ok_igate_net = 0;           // don't put data on internet       ????
            if (debug_level & 1)
                printf("decode_info_field: done with data_add(OTHER_DATA)\n");
        }

        if (!done) {                    // data that we do ignore...
            //printf("not decoding info: Call:%s ID:%c Msg:|%s|\n",call,data_id,message);
            ok_igate_net = 0;           // don't put data on internet
            if (debug_level & 1)
                printf("decode_info_field: done with ignored data\n");
        }
    }

    if (third_party)
        ok_igate_net = 0;               // don't put third party traffic on internet
    if (is_my_call(call,1))
        ok_igate_net = 0;               // don't put my data on internet     ???

    if (ok_igate_net) {
        if (debug_level & 1)
            printf("decode_info_field: ok_igate_net start\n");

        if (from == DATA_VIA_TNC && strlen(my_data) > 0) {
            sprintf(line,"%s>%s:%s",call,path,my_data);
            //printf("IGATE>NET %s\n",line);
            output_igate_net(line, port,third_party);
        }
    }
    if (debug_level & 1)
        printf("decode_info_field: done\n");
}


/*
 *  Extract object data from information field before processing
 */
int extract_object(char *call, char **info, char *origin) {
    int ok;

    // Object and Item Reports     [APRS Reference, chapter 11]
    ok = 0;
    // todo: add station originator to database
    if ((*info)[0] == ';') {                    // object
        // fixed 9 character object name with any printable ASCII character
        if (strlen((*info)) > 10) {
            substr(call,(*info)+1,9);           // extract object name
            (*info) = (*info) + 10;
            // Remove leading spaces ? They look bad, but are allowed by the APRS Reference ???
            (void)remove_trailing_spaces(call);
            if (valid_object(call)) {
                // info length is at least 1
                ok = 1;
            }
        }
    } else {                                    // item, not implemented now
        char filtered_data[MAX_LINE_SIZE + 1];

        strcpy(filtered_data,(*info));
        makePrintable(filtered_data);
        //printf("item from %s: %s\n",origin,(*info));
        printf("item from %s: %s\n",origin,filtered_data);

        // variable-length name, 3 to 9 printable chars except '!' and '_'
        // find delimiter
        // extract item name
    }
    return(ok);
}


/*
 *  Extract third-party traffic from information field before processing
 */
int extract_third_party(char *call, char **path, char **info, char *origin) {
    int ok;
    char *call_ptr;

    call_ptr = NULL;                            // to make the compiler happy...
    ok = 0;
    if (!is_my_call(call,1)) {
        // todo: add reporting station call to database ??
        //       but only if not identical to reported call
        (*info) = (*info) +1;                   // strip '}' character
        call_ptr = strtok((*info),">");         // extract call
        if (call_ptr != NULL) {
            (*path) = strtok(NULL,":");         // extract path
            if ((*path) != NULL) {
                (*info) = strtok(NULL,"");      // rest is information field
                if ((*info) != NULL)            // the above looks dangerous, but works on same string
                    ok = 1;                     // we have found all three components
            }
        }
    }

    if ((debug_level & 1) && !ok)
        printf("extract_third_party: invalid format from %s\n",call);

    if (ok) {
        ok = valid_path((*path));               // check the path and convert it to TAPR format
        if ((debug_level & 1) && !ok) {
            char filtered_data[MAX_LINE_SIZE + 1];
            strcpy(filtered_data,(*path));
            makePrintable(filtered_data);
            printf("extract_third_party: invalid path: %s\n",filtered_data);
        }
    }

    if (ok) {                                           // check callsign
        (void)remove_trailing_asterisk(call_ptr);       // is an asterisk valid here ???
        if (valid_inet_name(call_ptr,(*info),origin)) { // accept some of the names used in internet
            strcpy(call,call_ptr);                      // treat is as object with special origin
        } else if (valid_call(call_ptr)) {              // accept real AX.25 calls
            strcpy(call,call_ptr);
        } else {
            ok = 0;
            if (debug_level & 1) {
                char filtered_data[MAX_LINE_SIZE + 1];
                strcpy(filtered_data,call_ptr);
                makePrintable(filtered_data);
                printf("extract_third_party: invalid call: %s\n",filtered_data);
            }
        }
    }
    return(ok);
}


/*
 *  Decode AX.25 line
 */
void decode_ax25_line(char *line, char from, int port) {
    char *call_sign;
    char *path;
    char *info;
    char call[MAX_CALL+1];
    char origin[MAX_CALL+1];
    int ok;
    int third_party;

    if (debug_level & 1)
        printf("decode_ax25_line: start\n");

    call_sign   = NULL;
    path        = NULL;
    info        = NULL;
    origin[0]   = '\0';
    call[0]     = '\0';
    third_party = 0;

    // CALL>PATH:APRS-INFO-FIELD                // split line into components
    //     ^    ^
    ok = 0;
    call_sign = strtok(line,">");               // extract call from AX.25 line
    if (call_sign != NULL) {
        path = strtok(NULL,":");                // extract path from AX.25 line
        if (path != NULL) {
            info = strtok(NULL,"");             // rest is info_field
            if (info != NULL)
                ok = 1;                         // we have found all three components
        }
    }

    if (ok) {
        ok = valid_path(path);                  // check the path and convert it to TAPR format
        if ((debug_level & 1) && !ok) {
            char filtered_data[MAX_LINE_SIZE + 1];
            strcpy(filtered_data,path);
            makePrintable(filtered_data);
            printf("decode_ax25_line: invalid path: %s\n",filtered_data);
        }
    }

    if (ok) {
        if (strlen(info) > 256)                 // first check if information field conforms to APRS specs
            ok = 0;                             // drop packets too long
            if ((debug_level & 1) && !ok) {
                char filtered_data[MAX_LINE_SIZE + 1];
                strcpy(filtered_data,info);
                makePrintable(filtered_data);
                printf("decode_ax25_line: info field too long: %s\n",filtered_data);
            }
    }

    if (ok) {                                                   // check callsign
        (void)remove_trailing_asterisk(call_sign);              // is an asterisk valid here ???
        if (valid_inet_name(call_sign,info,origin)) {           // accept some of the names used in internet
            strcpy(call,call_sign);                             // treat is as object with special origin
        } else if (valid_call(call_sign)) {                     // accept real AX.25 calls
            strcpy(call,call_sign);
        } else {
            ok = 0;
            if (debug_level & 1) {
                char filtered_data[MAX_LINE_SIZE + 1];
                strcpy(filtered_data,call_sign);
                makePrintable(filtered_data);
                printf("decode_ax25_line: invalid call: %s\n",filtered_data);
            }
        }
    }

    if (ok && info[0] == '}') {                                 // look for third-party traffic
        ok = extract_third_party(call,&path,&info,origin);      // extract third-party data
        third_party = 1;
    }

    if (ok && (info[0] == ';' || info[0] == ')')) {             // look for objects or items
        strcpy(origin,call);
        ok = extract_object(call,&info,origin);                 // extract object data
    }

    if (ok) {
        // decode APRS information field, always called with valid call and path
        // info is a string with 0 - 256 bytes
        // printf("dec: %s (%s) %s\n",call,origin,info);
        decode_info_field(call,path,info,origin,from,port,third_party);
    }

    if (debug_level & 1)
        printf("decode_ax25_line: exiting\n");
}
                

/*
 *  Read a line from file
 */
void  read_file_line(FILE *f) {
    char line[MAX_LINE_SIZE+1];
    char cin;
    int pos;

    pos = 0;
    strcpy(line,"");
    while (!feof(f)) {
        if (fread(&cin,1,1,f) == 1) {
            if (cin != (char)10 && cin != (char)13) {   // normal characters
                if (pos < MAX_LINE_SIZE) {
                    line[pos++] = cin;
                 }
            } else {                                    // CR or LF
                if (cin == (char)10) {                  // Found LF as EOL char
                    line[pos] = '\0';                   // Always add a terminating zero after last char
                    pos = 0;                            // start next line
                    decode_ax25_line(line,'F',-1);      // Decode the packet
                    return;                             // only read line by line
                }
            }
        }
    }
    if (feof(f)) {                                      // Close file if at the end
        (void)fclose(f);
        read_file = 0;
        statusline(langcode("BBARSTA012"),0);           // File done..
        redraw_on_new_data = 2;                         // redraw immediately after finish
    }
}


/*
 *  Center map to new position
 */
void set_map_position(Widget w, long lat, long lon) {
    // see also map_pos() in location.c
    set_last_position();
    mid_y_lat_offset  = lat;
    mid_x_long_offset = lon;
    setup_in_view();  // flag all stations in new screen view
    create_image(w);
    (void)XCopyArea(XtDisplay(w),pixmap_final,XtWindow(w),gc,0,0,screen_width,screen_height,0,0);
}


/*
 *  Search for a station to be located (for Tracking and Find Station)
 */
int locate_station(Widget w, char *call, int follow_case, int get_match, int center_map) {
    DataRow *p_station;
    char call_find[MAX_CALLSIGN+1];
    char call_find1[MAX_CALLSIGN+1];
    int ii;
    int call_len;

    call_len = 0;
    if (!follow_case) {
        for (ii=0; ii<(int)strlen(call); ii++) {
            if (isalpha((int)call[ii]))
                call_find[ii] = toupper((int)call[ii]);         // Problem with lowercase objects/names!!
            else
                call_find[ii] = call[ii];
        }
        call_find[ii] = '\0';
        strcpy(call_find1,call_find);
    } else
        strcpy(call_find1,call);

    if (search_station_name(&p_station,call_find1,get_match)) {
        if (position_defined(p_station->coord_lat,p_station->coord_lon,0)) {
            if (center_map || !position_on_inner_screen(p_station->coord_lat,p_station->coord_lon))
                 // only change map if really neccessary
                set_map_position(w, p_station->coord_lat, p_station->coord_lon);
            return(1);                  // we found it
        }
    }
    return(0);
}


/*
 *  Change map position if neccessary while tracking a station
 *      we call it with defined station call and position
 */
void track_station(Widget w, char *call_tracked, DataRow *p_station) {
    int found;
    char *call;
    char call_find[MAX_CALLSIGN+1];
    int ii;
    int call_len;
    long x_ofs, y_ofs;
    long new_lat, new_lon;

    call_len = 0;
    found = 0;
    call = p_station->call_sign;
    if (!track_case) {
        for (ii=0; ii<(int)strlen(call_tracked); ii++) {
            if (isalpha((int)call_tracked[ii]))
                call_find[ii] = toupper((int)call_tracked[ii]);
            else
                call_find[ii] = call_tracked[ii];
        }
        call_find[ii] = '\0';
    } else
        strcpy(call_find,call_tracked);

    /*printf("CALL %s %s %s\n",call_tracked, call_find, call);*/
    if (track_match) {
        if (strcmp(call_find,call) == 0)                // we want an exact match
            found=1;
    } else {
        found=0;
        call_len=(int)(strlen(call)-strlen(call_find));
        if (strlen(call_find)<=strlen(call)) {
            for (ii=0; ii<=call_len; ii++) {
                if (!track_case) {
                    if (strncasecmp(call_find,call+ii,strlen(call_find)) == 0)
                        found=1;
                } else {
                    if (strncmp(call_find,call+ii,strlen(call_find)) == 0)
                        found=1;
                }
            }
        }
    }

    if(found) {                                         // we want to track this station
        new_lat = p_station->coord_lat;                 // center map to station position as default
        new_lon = p_station->coord_lon;
        x_ofs = new_lon - mid_x_long_offset;            // current offset from screen center
        y_ofs = new_lat - mid_y_lat_offset;
        if ((labs(x_ofs) > (screen_width*size/3)) || (labs(y_ofs) > (screen_height*size/3))) {
            // only redraw map if near border (margin 1/6 of screen at each side)
            if (labs(y_ofs) < (screen_height*size/2))
                new_lat  += y_ofs/2;                    // give more space in driving direction
            if (labs(x_ofs) < (screen_width*size/2))
                new_lon += x_ofs/2;
            set_map_position(w, new_lat, new_lon);      // center map to new position
        }
    }
}


