#ifndef __EHASH_MAP_H__
#define __EHASH_MAP_H__


#include <vector>
#include "elist.h"


#define EHASH_MAP(type,field,hash,equal)  ehash_map<type,OFFSETOF(type,field),hash,equal>

template <class T>
struct int_hash
{
    size_t operator () (T t) const
    {
        const char *s = (const char*) &t;
        const char *e = s + sizeof (T);
        uint h = 0;
        for (; s < e; s++) {
            h *= 16777619UL;
            h ^= (uint) (*s);
        }
        return h;
    }
};

template <class T>
struct int_equals
{
    bool operator () (T a, T b) const
    {
        return a == b;
    }
};

typedef int_hash<guint64>   int64_hash;
typedef int_equals<guint64> int64_equals;


template <class T, class Ref, class Ptr, int offset, class A>
struct ehash_map_iterator
{
    typedef ehash_map_iterator<T, T&, T*, offset, A> iterator;
    typedef ehash_map_iterator<T, const T&, const T*, offset, A> const_iterator;
    typedef ehash_map_iterator<T, Ref, Ptr, offset, A> self;

    typedef forward_iterator_tag iterator_category;
    typedef T value_type;
    typedef value_type *pointer;
    typedef const value_type *const_pointer;
    typedef value_type &reference;
    typedef const value_type &const_reference;
    typedef size_t size_type;
    typedef ptrdiff_t difference_type;

    typedef elist<T, offset> chain_type;
    typedef vector<chain_type*,A> chain_array;
    typedef chain_type::iterator chain_iterator;

    uint bucket;
    chain_array *chains;
    chain_iterator iter;

private:
    void advance ()
    {
        while (++bucket < chains->size ()) {
            if ((*chains)[bucket] && !(*chains)[bucket]->empty ()) {
                iter = (*chains)[bucket]->begin ();
                return;
            }
        }
        bucket = (uint) -1;
        iter = NULL;
    }

public:
    ehash_map_iterator ()
        : bucket ((uint) -1),
          chains (NULL),
          iter (NULL)
    { }
    ehash_map_iterator (chain_array *a)
        : bucket ((uint) -1),
          chains (a),
          iter (NULL)
    {
        advance ();
    }
    ehash_map_iterator (const iterator &x)
        : bucket (x.bucket),
          chains (x.chains),
          iter (x.iter)
    { }

    bool operator== (const self &x) const
    {
        return (bucket == x.bucket) && (iter == x.iter);
    }
    bool operator!= (const self &x) const
    {
        return !(*this == x);
    }
    reference operator* () const
    {
        return *iter;
    }
    pointer operator-> () const
    {
        return &(**this);
    }
    self& operator++ ()
    {
        if (++iter == (*chains)[bucket]->end ()) {
            advance ();
        }
        return *this;
    }
    self operator++ (int)
    {
        self tmp (*this);
        ++*this;
        return tmp;
    }
};


template <class T, int offset, class H, class E, class A = alloc>
class ehash_map
{
    typedef T value_type;
    typedef value_type *pointer;
    typedef const value_type *const_pointer;
    typedef value_type &reference;
    typedef const value_type &const_reference;
    typedef size_t size_type;
    typedef ptrdiff_t difference_type;

    typedef elist<T, offset> chain_type;
    typedef vector<chain_type*,A> chain_array;

    typedef ehash_map_iterator<T, T&, T*, offset, A> iterator;
    typedef ehash_map_iterator<T, const T&, const T*, offset, A> const_iterator;

public:
    uint nitems;
    uint level;
    uint next;
    chain_array chains;
    H hasher;
    E compare;

public:
    ehash_map ()
        : nitems (0),
          level (0),
          next (0),
          chains (1U << level),
          hasher (),
          compare ()
    { }

    void clear ()
    {
        for (uint i = 0; i < chains.size (); i++) {
            if (chains[i]) {
                chains[i]->~chain_type ();
                A::deallocate (chains[i], sizeof (chain_type));
                chains[i] = NULL;
            }
        }
        nitems = 0;
    }

    ~ehash_map ()
    {
        clear ();
    }

    size_type size () const
    {
        return nitems;
    }

    size_type max_size () const
    {
        return (size_type) -1;
    }

    iterator begin ()
    {
        return iterator (&chains);
    }
    const_iterator begin () const
    {
        return const_iterator (&chains);
    }
    iterator end ()
    {
        return iterator ();
    }
    const_iterator end () const
    {
        return iterator ();
    }

    bool empty () const
    {
        return (nitems == 0);
    }

    void swap (ehash_map &m)
    {
        chains.swap (m.chains);
        swap (nitems, m.nitems);
        swap (level, m.level);
        swap (next, m.next);
    }

    uint hash (uint h) const
    {
        uint i = h % (1 << level);
        if (i < next) {
            i = h % (1 << (level + 1));
        }
        return i;
    }

    template <class K>
    T* find (uint b, const K &key)
    {
        chain_type *chain = chains[b];
        if (chain) {
            for (chain_type::iterator iter (chain->begin ());
                 iter != chain->end ();
                 ++iter) {
                if (compare (key, &(*iter))) {
                    return &(*iter);
                }
            }
        }
        return NULL;
    }

    template <class K>
    T* find (const K &key)
    {
        return find (hash (hasher (key)), key);
    }

    void insert (uint b, T *obj)
    {
        if (!chains[b]) {
            chains[b] = new (A::allocate (sizeof (chain_type))) chain_type;
        }
        chains[b]->push_back (obj);
    }

    void split ()
    {
        uint b = next++;
        uint nchains = (1U << level) + next;

        if (nchains > chains.size ()) {
            chains.resize (nchains);
        }

        if (chains[b]) {
            chain_type old;
            old.swap (*chains[b]);

            while (!old.empty ()) {
                T *obj = &old.back ();
                old.pop_back ();
                insert (hash (hasher (obj)), obj);
            }
        }

        if (next >= (1U << level)) {
            level += 1;
            next = 0;
        }
    }

    void insert (T *obj)
    {
        uint nchains = (1U << level) + next;
        if (++nitems > (2 * nchains)) {
            split ();
        }
        insert (hash (hasher (obj)), obj);
    }

    void erase (uint b, T *obj)
    {
        if (chains[b] && obj) {
            chains[b]->erase (obj);
	    // ASSERT: you erase something not here, nitems is wrong
	    nitems -= 1;
        }
    }

    void erase (T *obj)
    {
        erase (hash (hasher (obj)), obj);
    }

    template <class K>
    void erase (const K &key)
    {
        uint b = hash (hasher (key));
        erase (b, find (b, key));
    }
};


#endif /* __EHASH_MAP_H__ */

/*
  Local Variables:
  mode: c++
  End:
 */
