/* -*-Mode: C;-*-
 * $Id: rcswalk.h,v 1.10 2001/05/15 23:34:06 jmacd Exp $
 *
 * Copyright (C) 1998, 1999, 2000, Joshua P. MacDonald
 * <jmacd@CS.Berkeley.EDU> and The Regents of the University of
 * California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 *    Neither name of The University of California nor the names of
 *    its contributors may be used to endorse or promote products
 *    derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _RCSWALK_H_
#define _RCSWALK_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <sys/types.h>
#include <sys/time.h>
#include <glib.h>
#include <stdio.h>

typedef struct _RcsWalker               RcsWalker;
typedef struct _RcsFile                 RcsFile;
typedef struct _RcsVersion              RcsVersion;
typedef struct _RcsStats                RcsStats;
typedef struct _IntStat                 IntStat;
typedef struct _DblStat                 DblStat;
typedef struct _BinCounter              BinCounter;
typedef struct _ConfigOption            ConfigOption;

struct _RcsWalker {
  void*    (* initialize)    (void);
  int      (* finalize)      (RcsStats* stats, void* data);
  int      (* onefile)       (RcsFile* rcs, RcsStats* stats, void* data);
  int      (* dateorder)     (RcsFile* rcs, RcsVersion* v, void* data);
  int      (* delta_orig)    (RcsFile* rcs, RcsVersion* from, RcsVersion *to, void* data);
  int      (* delta_date)    (RcsFile* rcs, RcsVersion* from, RcsVersion *to, void* data);
  int      min_versions;
  int      max_versions;
  gboolean write_files;
};

struct _RcsVersion {
  RcsFile    *rcs;
  time_t      date;
  int         dateseq;
  int         chain_length;
  char       *vname;
  off_t       size;
  int         cc;
  guint8*     segment;
  char       *filename;
  RcsVersion *parent;
  GSList     *children;
  guint       on_trunk : 1;
};

struct _RcsFile {
  char       *filename;
  char       *copyname;
  char       *headname;

  int         version_count;
  int         forward_count;
  int         reverse_count;
  int         branch_count;

  RcsVersion *versions;
  RcsVersion **versions_date;

  RcsVersion *head_version;
  RcsVersion *root_version;

  off_t       total_size;

  guint       atflag : 1;
};

struct _RcsStats {
  BinCounter *avg_version_size;
  IntStat* version_stat;
  IntStat* forward_stat;
  IntStat* reverse_stat;
  IntStat* branch_stat;
  IntStat* unencoded_stat;
  IntStat* literal_stat;
};

struct _IntStat {
  const char* name;
  int count;
  long long sum;
  long long min;
  long long max;

  GArray *values;
};

struct _DblStat {
  const char* name;
  int count;
  double sum;
  double min;
  double max;

  GArray *values;
};

struct _BinCounter {
  const char *name;
  GPtrArray  *bins;
};

enum _ConfigArgument {
  CO_Required,
  CO_Optional,
  CO_None
};

typedef enum _ConfigArgument ConfigArgument;

enum _ConfigOptionType {
  CD_Bool,
  CD_Int32,
  CD_Double,
  CD_String
};

typedef enum _ConfigOptionType ConfigOptionType;

enum _ConfigStyle {
  CS_Ignore,
  CS_UseAsFile,
  CS_Use
};

typedef enum _ConfigStyle ConfigStyle;

struct _ConfigOption {
  const char       *name;
  const char       *abbrev;
  ConfigStyle       style;
  ConfigArgument    arg;
  ConfigOptionType  type;
  void             *value;
  gboolean          found;
};

/* RCS inspection stuff
 */

void                rcswalk_init   (void);
int            rcswalk        (RcsWalker *walker, const char* copy_base);
void                rcswalk_report (RcsStats* stats);

IntStat*            stat_int_new      (const char* name);
void                stat_int_add_item (IntStat* stat, long long v);
void                stat_int_report   (IntStat* stat);

DblStat*            stat_dbl_new      (const char* name);
void                stat_dbl_add_item (DblStat* stat, double v);
void                stat_dbl_report   (DblStat* stat);

BinCounter*         stat_bincount_new      (const char* name);
void                stat_bincount_add_item (BinCounter* bc, int bin, double val);
void                stat_bincount_report   (BinCounter* bc);

/* Experiment configuration stuff
 */

void                config_register   (ConfigOption *opts, int nopts);
int            config_parse      (const char* config_file);
int            config_done       (void);
void                config_help       (void);
void                config_set_string (const char* var, const char* val);
int            config_clear_dir  (const char* dir);
int            config_create_dir (const char* dir);
FILE*               config_output     (const char* fmt, ...);

#ifdef __cplusplus
}
#endif

#endif
