// render2d.cpp - definitions for functions of class Render2D

#include <stdlib.h>
#include <iostream.h>
#include <math.h>

#include <qapplication.h>
#include <qclipboard.h>
#include <qwidget.h>
#include <qbitmap.h>
#include <qpixmap.h>
#include <qcursor.h>
#include <qfont.h>
#include <qcolor.h>
#include <qfile.h>
#include <qtextstream.h>

#include "defs.h"
#include "render2d.h"
#include "chemdata.h"

#include "skullcursor.xpm"
#include "rotatecursor.xpm"

Render2D::Render2D(QWidget *parent, const char *name)
  : QWidget(parent, name)
{
  setPalette(QPalette(QColor(255,255,255)));

  // generally, width and height should allow for 0.5" margins and 100 dpi
  renderHeight = 1000;
  renderWidth = 750;

  resize(renderWidth, renderHeight);
  outputDevice = OUTPUT_SCREEN;

  mode = MODE_SELECT;
  mouse1down = false;
  mouse3down = false;
  dragging = false;
  moved = false;

  highlightpoint = 0;
  highlightobject = 0;
  localtext = 0;

  fixed_bond = true;
  fixedlength_bond = 25.0;
  fixedangle_bond = 15.0;
  fixed_arrow = true;
  fixedlength_arrow = 50.0;
  fixedangle_arrow = 15.0;
  thick = 1;

  currentFont = QFont("Helvetica", 12);
  currentColor = QColor(0,0,0);
  bgcolor = QColor(255,255,255);

  setMouseTracking(true);
  setFocusPolicy(ClickFocus);

  QPixmap sc(skullcursor_xpm);
  sc.setMask(sc.createHeuristicMask());
  skullcursor = QCursor(sc, 1, 1);

  QPixmap rc(rotatecursor_xpm);
  rc.setMask(rc.createHeuristicMask());
  rotatecursor = QCursor(rc, 1, 1);

  // set printing defaults
  page_size = PAGE_LETTER;
  page_orientation = PAGE_PORTRAIT;
  PrintSetup();  // see render2d_print.cpp
}

// this function required for good behavior if user changes tools while
// text is selected.
void Render2D::CleanUpAfterTextTool() {
  if (localtext == 0) return;  // don't need to clean up if no text.
  // save text, clean up
  localtext->DeselectAllText();
  emit TextOff();
  if (!text_exists) {
    if (localtext->getText().length() > 0)
      { c->addText(localtext); localtext = 0; }
  } else {
    if (localtext->getText().length() == 0)
      { c->Erase(localtext); localtext = 0; }
  }
  localtext = 0;
  highlightpoint = 0;
  if (highlightobject != 0) {
    highlightobject->Highlight(false);
    highlightobject = 0;
  }
}

// setMode functions (slots)

void Render2D::setMode_Select() { 
  if (mode == MODE_TEXT) CleanUpAfterTextTool();
  mode = MODE_SELECT; 
  startpoint = NULL; endpoint = NULL;
  setCursor(arrowCursor);
  c->DeselectAll();
  emit SignalSetStatusBar( tr("Select mode: left click on object to move, right click on object to edit") );
  repaint();
}

void Render2D::setMode_DrawLine() { 
  if (mode == MODE_TEXT) CleanUpAfterTextTool();
  mode = MODE_DRAWLINE; 
  startpoint = NULL; endpoint = NULL;
  setCursor(crossCursor);
  c->DeselectAll();
  emit SignalSetStatusBar( tr("Draw Line mode: left click to draw line, right click to edit") );
  repaint();
}

void Render2D::setMode_DrawDashLine() { 
  if (mode == MODE_TEXT) CleanUpAfterTextTool();
  mode = MODE_DRAWLINE_DASH; 
  startpoint = NULL; endpoint = NULL;
  setCursor(crossCursor);
  c->DeselectAll();
  emit SignalSetStatusBar( tr("Draw Dashed Line mode: left click to draw line, right click to edit") );
  repaint();
}

void Render2D::setMode_DrawUpLine() { 
  if (mode == MODE_TEXT) CleanUpAfterTextTool();
  mode = MODE_DRAWLINE_UP; 
  startpoint = NULL; endpoint = NULL;
  setCursor(crossCursor);
  c->DeselectAll();
  emit SignalSetStatusBar( tr("Draw Stereo-Up Line mode: left click to draw line, right click to edit") );
  repaint();
}

void Render2D::setMode_DrawDownLine() { 
  if (mode == MODE_TEXT) CleanUpAfterTextTool();
  mode = MODE_DRAWLINE_DOWN; 
  startpoint = NULL; endpoint = NULL;
  setCursor(crossCursor);
  c->DeselectAll();
  emit SignalSetStatusBar( tr("Draw Stereo-Down Line mode: left click to draw line, right click to edit") );
  repaint();
}

void Render2D::setMode_DrawArrow(QString btype) { 
  if (mode == MODE_TEXT) CleanUpAfterTextTool();
  mode = MODE_DRAWARROW; 
  startpoint = NULL; endpoint = NULL;
  setCursor(crossCursor);
  c->DeselectAll();
  if (btype == "REGULAR") bracket_type = ARROW_REGULAR;
  if (btype == "DASH") bracket_type = ARROW_DASH;
  if (btype == "BI1") bracket_type = ARROW_BI1;
  if (btype == "BI2") bracket_type = ARROW_BI2;
  if (btype == "RETRO") bracket_type = ARROW_RETRO;
  emit SignalSetStatusBar( tr("Draw Arrow mode: left click to draw arrow") );
  repaint();
}

void Render2D::setMode_DrawBracket(QString btype) { 
  if (mode == MODE_TEXT) CleanUpAfterTextTool();
  mode = MODE_DRAWBRACKET; 
  startpoint = NULL; endpoint = NULL;
  setCursor(crossCursor);
  c->DeselectAll();
  if (btype == "SQUARE") bracket_type = BRACKET_SQUARE;
  if (btype == "CURVE") bracket_type = BRACKET_CURVE;
  if (btype == "BRACE") bracket_type = BRACKET_BRACE;
  emit SignalSetStatusBar( tr("Draw Bracket mode: left click to draw bracket") );
  repaint();
}

void Render2D::setMode_Erase() { 
  if (mode == MODE_TEXT) CleanUpAfterTextTool();
  mode = MODE_ERASE; 
  startpoint = NULL; endpoint = NULL;
  localtext = 0;
  setCursor(skullcursor);
  c->DeselectAll();
  emit SignalSetStatusBar( tr("Erase mode: left click to erase object") );
  repaint();
}

void Render2D::setMode_DrawText() { 
  mode = MODE_TEXT; 
  startpoint = NULL; endpoint = NULL;
  setCursor(ibeamCursor);
  c->DeselectAll();
  emit SignalSetStatusBar( tr("Text mode: left click to add or edit text") );
  repaint();
}

void Render2D::setMode_DrawSymbol(QString s) { 
  if (mode == MODE_TEXT) CleanUpAfterTextTool();
  cout << s << endl;
  mode = MODE_SYMBOL; 
  startpoint = NULL; endpoint = NULL;
  symbolfile = s;
  setCursor(crossCursor);
  c->DeselectAll();
  emit SignalSetStatusBar( tr("Draw Symbol mode: left click to add symbol") );
  repaint();
}

void Render2D::setMode_DrawCurveArrow(QString s) { 
  if (mode == MODE_TEXT) CleanUpAfterTextTool();
  mode = MODE_DRAWCURVEARROW; 
  startpoint = NULL; endpoint = NULL;
  symbolfile = s;
  setCursor(crossCursor);
  c->DeselectAll();
  emit SignalSetStatusBar("Draw Curved Arrow mode: left click to draw arrow");
  repaint();
}

void Render2D::Tool(int t) {
  if (mode == MODE_TEXT) CleanUpAfterTextTool();
  mode = t;
  startpoint = NULL; endpoint = NULL;
  setCursor(arrowCursor);
  switch (mode) {
  case MODE_TOOL_CALCMW:
    emit SignalSetStatusBar( tr("Click on a molecule to calculate its molecular weight") );
    break;
  case MODE_TOOL_CALCEF:
    emit SignalSetStatusBar( tr("Click on a molecule to calculate its empirical formula") );
    break;
  case MODE_TOOL_CALCEA:
    emit SignalSetStatusBar( tr("Click on a molecule to calculate its elemental analysis") );
    break;
  case MODE_TOOL_13CNMR:
    emit SignalSetStatusBar( tr("Click on a molecule to calculate its 13C NMR") );
    break;
  case MODE_TOOL_1HNMR:
    emit SignalSetStatusBar( tr("Click on a molecule to calculate its 1H NMR") );
    break;
  case MODE_TOOL_IR:
    emit SignalSetStatusBar( tr("Click on a molecule to calculate its IR") );
    break;
  case MODE_TOOL_NAME:
    emit SignalSetStatusBar( tr("Click on a molecule to determine its IUPAC name") );
    break;
  case MODE_TOOL_TOSMILES:
    emit SignalSetStatusBar( tr("Click on a molecule to determine its SMILES string") );
    break;
  case MODE_TOOL_CLEANUPMOL:
    emit SignalSetStatusBar( tr("Click on a molecule to clean up its structure") );
    break;
  }
}

void Render2D::AutoLayout() {
  c->DeselectAll();
  c->StartUndo(0, 0);
  c->AutoLayout();
  mode = MODE_SELECT; 
  startpoint = NULL; endpoint = NULL;
  setCursor(arrowCursor);
  emit SignalSetStatusBar( tr("Select mode: left click to move, right click to edit") );
  repaint();
}

void Render2D::Cut() {
  c->StartUndo(0, 0);
  c->Cut();
  mode = MODE_SELECT;  // the selection is gone, don't draw selection box
  repaint();
}

void Render2D::Copy() {
  c->Copy();
  // copy to system clipboard
  QRect savebox = selectionBox;
  int prevmode = mode;
  mode = MODE_SELECT;
  c->DeselectAll();
  repaint();
  QPixmap pm( savebox.size() );
  bitBlt( &pm, 0, 0, this, savebox.left(), savebox.top(), 
	  savebox.width(), savebox.height() );
  c->NewSelectRect(savebox, false);
  selectionBox = c->selectionBox();
  mode = prevmode;
  QApplication::clipboard()->setImage(pm.convertToImage());
  repaint();  // do anyways, though Copy() should not alter the drawing.
}

void Render2D::Paste() {
  c->StartUndo(0, 0);
  if (c->Paste() == false) return;
  c->Move(30,30);
  mode = MODE_SELECT_MULTIPLE_SELECTED;
  selectionBox = c->selectionBox();
  repaint();
}

void Render2D::Undo() {
  c->SelectAll();
  c->EraseSelected();
  if (c->Undo())
    emit SignalSetStatusBar( tr("Undo!") );
  else
    emit SignalSetStatusBar( tr("Cannot undo, sorry!") );
  c->DeselectAll();
  mode = MODE_SELECT;
  repaint();
}

void Render2D::SelectAll() {
  mode = MODE_SELECT_MULTIPLE_SELECTED;
  c->SelectAll();
  selectionBox = c->selectionBox();
  repaint();
}

void Render2D::EraseSelected() {
  c->StartUndo(0, 0);
  c->EraseSelected();
  if (mode == MODE_SELECT_MULTIPLE_SELECTED) mode = MODE_SELECT;
  repaint();
}

void Render2D::Rotate90() {
  c->StartUndo(0, 0);
  rotateOrigin.setX( (selectionBox.right() + selectionBox.left() - 2) / 2);
  rotateOrigin.setY( (selectionBox.bottom() + selectionBox.top()) / 2);
  DPoint *dp1 = new DPoint( rotateOrigin );
  c->Rotate(dp1, -0.5 * M_PI);
  selectionBox = c->selectionBox();  
  repaint();
}

void Render2D::Rotate180() {
  c->StartUndo(0, 0);
  rotateOrigin.setX( (selectionBox.right() + selectionBox.left() - 2) / 2);
  rotateOrigin.setY( (selectionBox.bottom() + selectionBox.top()) / 2);
  DPoint *dp1 = new DPoint( rotateOrigin );
  c->Rotate(dp1, M_PI);
  selectionBox = c->selectionBox();  
  repaint();
}

void Render2D::Rotate270() {
  c->StartUndo(0, 0);
  rotateOrigin.setX( (selectionBox.right() + selectionBox.left() - 2) / 2);
  rotateOrigin.setY( (selectionBox.bottom() + selectionBox.top()) / 2);
  DPoint *dp1 = new DPoint( rotateOrigin );
  c->Rotate(dp1, 0.5 * M_PI);
  selectionBox = c->selectionBox();  
  repaint();
}

void Render2D::Flip(int d) {
  c->StartUndo(0, 0);
  rotateOrigin.setX( (selectionBox.right() + selectionBox.left() - 2) / 2);
  rotateOrigin.setY( (selectionBox.bottom() + selectionBox.top()) / 2);
  DPoint *dp1 = new DPoint( rotateOrigin );
  c->Flip(dp1, d);
  selectionBox = c->selectionBox();  
  repaint();
}

// Inserted(): highlight and box items inserted by RingDialog, etc.
void Render2D::Inserted() {
  mode = MODE_SELECT_MULTIPLE_SELECTED;
  selectionBox = c->selectionBox();
  repaint();
}

// change current font, or change font of active text
void Render2D::SetFont(QFont f) {
  if (localtext != 0) {
    localtext->setFont(f);
    repaint();
    return;
  } else {
    currentFont = f;
  }
}

// change current font, or change font of active text
void Render2D::SetColor(QColor c1) {
  if (mode != MODE_SELECT_MULTIPLE_SELECTED) {
    currentColor = c1;
    if (localtext != 0) localtext->SetColor(c1);
  } else {
    c->SetColorIfHighlighted(c1);
  }
}

QFont Render2D::GetFont() { return currentFont; }

QColor Render2D::GetColor() { return currentColor; }

void Render2D::CalcMW() {
  c->CalcMW();
  repaint();
}

// adjust endpoint to specified fixed length/angle
void Render2D::CorrectEndpoint_arrow() {
  double dx = endpoint->x - startpoint->x;
  double dy = endpoint->y - startpoint->y;
  int mysign = 1;
  if (dx < 0.0) mysign = -1;
  double ang = atan(dy/dx) * MOL_ARAD;
  double newang = ((int)( (ang + (fixedangle_arrow/2.0)) / fixedangle_arrow ) 
		   * fixedangle_arrow);
  if (fabs(newang) != 90.0) {
    dx = mysign * fixedlength_arrow * cos(newang / MOL_ARAD);
    dy = mysign * fixedlength_arrow * sin(newang / MOL_ARAD);
    endpoint->x = startpoint->x + dx;
    endpoint->y = startpoint->y + dy;
  } else {
    if (dy < 0.0) mysign = -1;
    endpoint->x = startpoint->x;
    endpoint->y = startpoint->y + mysign * fixedlength_arrow;
  }
}

// adjust endpoint to specified fixed length/angle
void Render2D::CorrectEndpoint_bond() {
  double dx = endpoint->x - startpoint->x;
  double dy = endpoint->y - startpoint->y;
  int mysign = 1;
  if (dx < 0.0) mysign = -1;
  double ang = atan(dy/dx) * MOL_ARAD;
  double newang = ((int)( (ang + (fixedangle_bond/2.0)) / fixedangle_bond ) 
		   * fixedangle_bond);
  if (fabs(newang) != 90.0) {
    dx = mysign * fixedlength_bond * cos(newang / MOL_ARAD);
    dy = mysign * fixedlength_bond * sin(newang / MOL_ARAD);
    endpoint->x = startpoint->x + dx;
    endpoint->y = startpoint->y + dy;
  } else {
    if (dy < 0.0) mysign = -1;
    endpoint->x = startpoint->x;
    endpoint->y = startpoint->y + mysign * fixedlength_bond;
  }
}

bool Render2D::SaveSVG(QString fn) {
  outputDevice = OUTPUT_SVG;
  output_file.setName(fn);
  // open file and create text stream
  if ( !output_file.open(IO_WriteOnly) ) return false;
  output_ts.setDevice(&output_file);

  QRect r;
  c->SelectAll();
  r = c->selectionBox();
  selectionBox = r;
  c->DeselectAll();
  mode = MODE_SELECT;  // so selection box will not appear
  svg_dx = r.x(); svg_dy = r.y();

  output_ts << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << endl;
  output_ts << "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.0//EN\" \"http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd\">" << endl;
  output_ts << "<svg version=\"1.0\" width=\"" << r.width() << 
    "\" height=\"" << r.height() << "\">" << endl;

  repaint();

  output_ts << "</svg>" << endl;

  output_file.close();
  // restore screen display
  outputDevice = OUTPUT_SCREEN;
  repaint();
  return true;
}

bool Render2D::SaveEPS(QString fn) {
  outputDevice = OUTPUT_EPS;
  output_file.setName(fn);
  // open file and create text stream
  if ( !output_file.open(IO_WriteOnly) ) return false;
  output_ts.setDevice(&output_file);
  // get bounding box
  c->SelectAll();
  selectionBox = c->selectionBox();
  c->DeselectAll();
  // write EPS header
  output_ts << "%!PS-Adobe-3.0 EPSF-3.0" << endl;
  output_ts << "%%BoundingBox: " << selectionBox.left() << " ";
  output_ts << selectionBox.top() << " ";
  output_ts << selectionBox.right() << " ";
  output_ts << selectionBox.bottom() << endl;
  output_ts << "%%Creator: XDrawChem" << endl;
  output_ts << "%%Title: " << fn << endl;
  output_ts << "%%EndComments" << endl;
  // write objects
  repaint();
  // finish
  output_ts << "showpage" << endl;
  output_ts << "%%EOF" << endl;
  output_file.close();
  // restore screen display
  outputDevice = OUTPUT_SCREEN;
  repaint();
  return true;
}
