#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>

#include <X11/Xlib.h>
#include <X11/Xatom.h>

#include <glib.h>
#include <glib-object.h>

#include <gdk-pixbuf/gdk-pixdata.h>

#include <libxfcegui4/dialogs.h>
#include <libxfcegui4/libxfcegui4.h>
#include <libxfcegui4/netk-screen.h>
#include <libxfcegui4/netk-window.h>
#include <libxfce4util/debug.h>
#include <libxfce4util/i18n.h>
#include <libxfce4util/util.h>

#include "ls_iconboxwin.h"
#include "inline-icon.h"

/* keep in sync with panel */
enum { ABOVE, NORMAL, BELOW };


struct _LSIconBoxWinPrivate 
{
  LSIconBox		*pIconBox;
  GtkVBox		*pVBox;
  GtkHBox		*pHBox;
  GtkFrame		*pFrame;
  GtkWidget		*pHandleRight;
  GtkWidget		*pHandleLeft;
  GtkWidget		*pMenu;
  GdkPixbuf		*pIcon;
  gboolean		bFirstOrientationChange;
  gboolean		bHideOnEmpty;
  
  glong			lOrientation;
  gint			lLayer;

  struct _LSIconBoxWinPrivateOffset
  {
    gint		nX;
    gint		nY;
  } offset;

  struct _LSIconBoxWinPrivetPosition {
    glong       horizontal;
    glong       vertical;
  } position;
  
  struct _LSIconBoxWinPrivateSignals
  {
    gulong		button_release_event[2];
    gulong		size_allocate;
    gulong		event[2];
  } signals;
};

static void 	ls_iconboxwin_init(LSIconBoxWin *plsibw);
static void 	ls_iconboxwin_class_init(LSIconBoxWinClass *plsibwc);
static void 	ls_iconboxwin_finalize(GObject *pgo);
GType   	ls_iconboxwin_get_type();
static void 	ls_iconboxwin_re_evaluate_position(LSIconBoxWin *plsibw);
/* CALLBACKS									*/
gboolean 	cb_lsibw_handle_release(GtkWidget *pWidget, GdkEventButton *pEvent, 
    		  gpointer pUserData);
gboolean 	cb_lsibw_size_allocate(GtkWidget *pWidget, GtkAllocation *pAlloc, 
    		  gpointer pData);
gboolean        cb_lsibw_handle_button2_pressed(GtkWidget *pWidget, 
    		  GdkEvent *pEvent, gpointer *pUserData);
static void 	cb_lsibw_iconbox_hide(GtkWidget *pWidget, gpointer pUserData);
static void 	cb_lsibw_iconbox_show(GtkWidget *pWidget, gpointer pUserData);
  
static gpointer parent_class;

/********************************************************************************/
/* GLIB OBJECT FUNCTIONS							*/
/********************************************************************************/
GType ls_iconboxwin_get_type()
{
  static GType object_type = 0;

  if(!object_type)
  {
    static const GTypeInfo object_info = {
      sizeof(LSIconBoxWinClass),
      (GBaseInitFunc)NULL,
      (GBaseFinalizeFunc)NULL,
      (GClassInitFunc)ls_iconboxwin_class_init,
      NULL,				/* class_finalize */
      NULL,				/* class_data */
      sizeof(LSIconBoxWin),
      0,				/* n_preallocs */
      (GInstanceInitFunc)ls_iconboxwin_init
    };

    object_type = g_type_register_static(/* GTK_TYPE_BUTTON */
      GTK_TYPE_WINDOW, "LSIconBoxWin", &object_info, 0);
  }

  return object_type;
}

static void ls_iconboxwin_init(LSIconBoxWin *plsibw) 
{
  return;
}

static void ls_iconboxwin_class_init(LSIconBoxWinClass *plsic) 
{
  GObjectClass *object_class = G_OBJECT_CLASS(plsic);
  parent_class = g_type_class_peek_parent(plsic);
  object_class->finalize = ls_iconboxwin_finalize;

  return;
}

static void ls_iconboxwin_finalize(GObject *pgo) 
{
  LSIconBoxWin *plsibw;
  
  g_return_if_fail(LS_IS_ICONBOXWIN(pgo));
  plsibw = LS_ICONBOXWIN(pgo);

  g_object_unref(plsibw->priv->pIcon);
 
  /* TODO do we really need to remove the handles for objects we destroy?	*/
  /* i think not!								*/
  /*
  g_signal_handler_disconnect(G_OBJECT(plsibw->priv->pHandleLeft), 
    plsibw->priv->signals.button_release_event[0]);
  g_signal_handler_disconnect(G_OBJECT(plsibw->priv->pHandleRight),
    plsibw->priv->signals.button_release_event[1]);
  g_signal_handler_disconnect(G_OBJECT(plsibw), 
    plsibw->priv->signals.size_allocate);
  g_signal_handler_disconnect(G_OBJECT(plsibw->priv->pHandleLeft),
    plsibw->priv->signals.event[0]);
  g_signal_handler_disconnect(G_OBJECT(plsibw->priv->pHandleRight),
    plsibw->priv->signals.event[1]);
  */
  
  /*
  g_object_unref(plsibw->priv->pIconBox);
  g_object_unref(plsibw->priv->pFrame);
  g_object_unref(plsibw->priv->pHandleRight);
  g_object_unref(plsibw->priv->pHandleLeft);
  g_object_unref(plsibw->priv->pMenu);
  */
  
  if(plsibw->priv->lOrientation == LS_ICONBOX_HORIZONTAL)
  {
    g_object_unref(plsibw->priv->pVBox);
  }
  else
  {
    g_object_unref(plsibw->priv->pHBox);
  }
  
  G_OBJECT_CLASS(parent_class)->finalize(pgo);
  
  return;
}

/********************************************************************************/
/* CONSTRUCTORS									*/
/********************************************************************************/
LSIconBoxWin *ls_iconboxwin_new(NetkScreen *pScreen) 
{
  LSIconBoxWin	*plsibw;
  
  /* INIT */
  plsibw			= LS_ICONBOXWIN(g_object_new(LS_TYPE_ICONBOXWIN, 
				    "type", GTK_WINDOW_TOPLEVEL, NULL));
  plsibw->priv			= (LSIconBoxWinPrivate *)g_malloc(
      				    sizeof(LSIconBoxWinPrivate));
  plsibw->priv->lOrientation	= LS_ICONBOX_HORIZONTAL;
  plsibw->priv->lLayer		= ABOVE;
  plsibw->priv->pIconBox	= LS_ICONBOX(ls_iconbox_new(pScreen,
				    plsibw->priv->lOrientation));
  plsibw->priv->pHBox		= GTK_HBOX(gtk_hbox_new(FALSE, 0));
  plsibw->priv->pVBox		= GTK_VBOX(gtk_vbox_new(FALSE, 0));
  plsibw->priv->pFrame		= GTK_FRAME(gtk_frame_new(NULL));
  plsibw->priv->bHideOnEmpty    = TRUE;
  plsibw->priv->pHandleLeft	= xfce_movehandler_new(GTK_WIDGET(plsibw));
  plsibw->priv->pHandleRight	= xfce_movehandler_new(GTK_WIDGET(plsibw));
  plsibw->priv->pIcon		= inline_icon_at_size(default_icon_data_48_48, 
      				    48, 48);
 
  netk_gtk_window_avoid_input(GTK_WINDOW(plsibw));
  g_signal_connect_object(G_OBJECT(plsibw->priv->pIconBox), "hide",
    G_CALLBACK(cb_lsibw_iconbox_hide), plsibw, 0);
  g_signal_connect_object(G_OBJECT(plsibw->priv->pIconBox), "show",
    G_CALLBACK(cb_lsibw_iconbox_show), plsibw, 0);
  
  ls_iconboxwin_set_offset(plsibw, 0, 0);
  ls_iconboxwin_set_vertical_position(plsibw, LS_ICONBOX_BOTTOM);
  ls_iconboxwin_set_horizontal_position(plsibw, LS_ICONBOX_RIGHT);

  gtk_window_set_icon(GTK_WINDOW(plsibw), plsibw->priv->pIcon);
  gtk_frame_set_shadow_type(GTK_FRAME(plsibw->priv->pFrame), GTK_SHADOW_OUT);
  gtk_container_set_border_width(GTK_CONTAINER(plsibw->priv->pFrame), 0);
  
  gtk_container_set_border_width(GTK_CONTAINER(plsibw->priv->pVBox), 0);
  gtk_container_set_border_width(GTK_CONTAINER(plsibw->priv->pHBox), 0);
  gtk_container_set_border_width(GTK_CONTAINER(plsibw), 0);

#if 0
  gtk_window_set_skip_taskbar_hint(GTK_WINDOW(plsibw), TRUE);
  gtk_window_set_skip_pager_hint(GTK_WINDOW(plsibw), TRUE);
#endif
  gtk_window_set_decorated(GTK_WINDOW(plsibw), FALSE);
  gtk_window_set_resizable(GTK_WINDOW(plsibw), FALSE);
  gtk_window_set_title(GTK_WINDOW(plsibw), "Lil' Star - IconBoxWindow");
  
  plsibw->priv->signals.button_release_event[0] =
    g_signal_connect(G_OBJECT(plsibw->priv->pHandleLeft), "button_release_event",
    G_CALLBACK(cb_lsibw_handle_release), (gpointer)plsibw);
  plsibw->priv->signals.button_release_event[1] =
    g_signal_connect(G_OBJECT(plsibw->priv->pHandleRight), "button_release_event",
    G_CALLBACK(cb_lsibw_handle_release), (gpointer)plsibw);
  plsibw->priv->signals.size_allocate =
    g_signal_connect(G_OBJECT(plsibw), "size_allocate",  
    G_CALLBACK(cb_lsibw_size_allocate), (gpointer)plsibw);
      
  
  gtk_container_add(GTK_CONTAINER(plsibw), GTK_WIDGET(plsibw->priv->pFrame));

  gtk_widget_show(GTK_WIDGET(plsibw->priv->pFrame));
  gtk_widget_show(GTK_WIDGET(plsibw->priv->pHandleLeft));
  gtk_widget_show(GTK_WIDGET(plsibw->priv->pHandleRight));
  gtk_widget_show(GTK_WIDGET(plsibw->priv->pIconBox));

  if(plsibw->priv->lOrientation == LS_ICONBOX_HORIZONTAL)
  {
    gtk_box_pack_start(GTK_BOX(plsibw->priv->pHBox), 
      plsibw->priv->pHandleLeft, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(plsibw->priv->pHBox), 
      GTK_WIDGET(plsibw->priv->pIconBox), FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(plsibw->priv->pHBox), 
      plsibw->priv->pHandleRight, FALSE, FALSE, 0);

    gtk_container_add(GTK_CONTAINER(plsibw->priv->pFrame), 
      GTK_WIDGET(plsibw->priv->pHBox));
    gtk_widget_show(GTK_WIDGET(plsibw->priv->pHBox));

    g_object_ref(plsibw->priv->pVBox);
  }
  else
  {
    gtk_box_pack_start(GTK_BOX(plsibw->priv->pVBox), 
      plsibw->priv->pHandleLeft, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(plsibw->priv->pVBox), 
      GTK_WIDGET(plsibw->priv->pIconBox), FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(plsibw->priv->pVBox), 
      plsibw->priv->pHandleRight, FALSE, FALSE, 0);
    
    gtk_container_add(GTK_CONTAINER(plsibw->priv->pFrame), 
      GTK_WIDGET(plsibw->priv->pVBox));
    gtk_widget_show(GTK_WIDGET(plsibw->priv->pVBox));
    
    g_object_ref(plsibw->priv->pHBox);
  }

  ls_iconboxwin_set_layer (plsibw, plsibw->priv->lLayer);
    
  /* POPUP MENU									*/
  ls_iconboxwin_create_menu(plsibw);
    
  plsibw->priv->signals.event[0] =
    g_signal_connect_object(G_OBJECT(plsibw->priv->pHandleLeft), "event",
    G_CALLBACK(cb_lsibw_handle_button2_pressed), (gpointer)plsibw, 0);
  plsibw->priv->signals.event[0] =
    g_signal_connect_object(G_OBJECT(plsibw->priv->pHandleRight), "event",
    G_CALLBACK(cb_lsibw_handle_button2_pressed), (gpointer)plsibw, 0);

  return plsibw;
}

/********************************************************************************/
/* FUNCTIONS                                                                    */
/********************************************************************************/ 
void ls_iconboxwin_set_hide_on_empty(LSIconBoxWin *plsibw, gboolean bHide)
{
  gboolean bVisible;

  if(plsibw->priv->bHideOnEmpty==bHide)
    return;
  
  ls_iconbox_set_hide_on_empty(plsibw->priv->pIconBox, bHide);
 
  if(plsibw->priv->bHideOnEmpty==TRUE && bHide==FALSE)
  {
    /* simply show them */
    gtk_widget_show(GTK_WIDGET(plsibw->priv->pHandleRight));
    gtk_widget_show(GTK_WIDGET(plsibw->priv->pHandleLeft));
    plsibw->priv->bHideOnEmpty=FALSE;
  }

  if(plsibw->priv->bHideOnEmpty==FALSE && bHide==TRUE)
  {
    g_object_get(G_OBJECT(plsibw->priv->pIconBox), "visible", &bVisible, NULL);
    if(bVisible==FALSE)
    {
      gtk_widget_hide(GTK_WIDGET(plsibw->priv->pHandleRight));
      gtk_widget_hide(GTK_WIDGET(plsibw->priv->pHandleLeft));
    }
    plsibw->priv->bHideOnEmpty=TRUE;
  }
  
  return;
}

LSIconBox *ls_iconboxwin_get_iconbox(LSIconBoxWin *plsibw)
{
  return plsibw->priv->pIconBox;
}

void ls_iconboxwin_set_horizontal_position(LSIconBoxWin *plsibw, glong lAlign)
{
  g_return_if_fail(LS_IS_ICONBOXWIN(plsibw));

  if(lAlign!=LS_ICONBOX_LEFT && lAlign!=LS_ICONBOX_RIGHT)
    return;

  plsibw->priv->position.horizontal = lAlign;
  ls_iconboxwin_re_evaluate_position(plsibw);
  return;
}

void ls_iconboxwin_set_vertical_position(LSIconBoxWin *plsibw, glong lAlign)
{
  g_return_if_fail(LS_IS_ICONBOXWIN(plsibw));

  if(lAlign!=LS_ICONBOX_TOP && lAlign!=LS_ICONBOX_BOTTOM)
    return;

  plsibw->priv->position.vertical = lAlign;
  ls_iconboxwin_re_evaluate_position(plsibw);
  return;  
}

void ls_iconboxwin_set_offset(LSIconBoxWin *plsibw, gint nX, gint nY)
{
  g_return_if_fail(LS_IS_ICONBOXWIN(plsibw));

  DBG(" X:%d Y:%d", nX, nY);

  if(nY>=0)
  {
    plsibw->priv->offset.nY = nY;
  }
  if(nX>=0)
  {
    plsibw->priv->offset.nX = nX;
  }
  
  ls_iconboxwin_re_evaluate_position(plsibw);
  return;
}

void ls_iconboxwin_get_offset(LSIconBoxWin *plsibw, gint *nX, gint *nY)
{
  g_return_if_fail(LS_IS_ICONBOXWIN(plsibw));

  *nY = plsibw->priv->offset.nY;
  *nX = plsibw->priv->offset.nX;

  return;
}

void ls_iconboxwin_show(LSIconBoxWin *plsibw)
{
  gtk_window_stick(GTK_WINDOW(plsibw));
  gtk_widget_show(GTK_WIDGET(plsibw));

  return;
}

void ls_iconboxwin_hide(LSIconBoxWin *plsibw)
{
  gtk_widget_hide(GTK_WIDGET(plsibw));

  return;
}

void ls_iconboxwin_set_layer (LSIconBoxWin * plsibw, gint lLayer)
{
    GtkWidget *win;
    Screen *xscreen;
    Window xid;
    static Atom xa_ABOVE = 0;
    static Atom xa_BELOW = 0;

    if (plsibw->priv->lLayer == lLayer)
	return;

    plsibw->priv->lLayer = lLayer;

    win = (GtkWidget *)plsibw;
    if (!GTK_WIDGET_REALIZED (win))
	gtk_widget_realize (win);

    xscreen = DefaultScreenOfDisplay (GDK_DISPLAY ());
    xid = GDK_WINDOW_XID (win->window);

    if (!xa_ABOVE)
    {
	xa_ABOVE = XInternAtom (GDK_DISPLAY (), "_NET_WM_STATE_ABOVE", False);
	xa_BELOW = XInternAtom (GDK_DISPLAY (), "_NET_WM_STATE_BELOW", False);
    }

    switch (lLayer)
    {
	case ABOVE:
	    netk_change_state (xscreen, xid, FALSE, xa_ABOVE, xa_BELOW);
	    netk_change_state (xscreen, xid, TRUE, xa_ABOVE, None);
            gtk_window_present (GTK_WINDOW (plsibw));
            break;
	case BELOW:
	    netk_change_state (xscreen, xid, FALSE, xa_ABOVE, xa_BELOW);
	    netk_change_state (xscreen, xid, TRUE, xa_BELOW, None);
	    break;
	default:
	    netk_change_state (xscreen, xid, FALSE, xa_ABOVE, xa_BELOW);
    }
}

void ls_iconboxwin_set_orientation(LSIconBoxWin *plsibw, glong lOrientation)
{
  GtkContainer	*pSrc;
  GtkContainer	*pDst;
  
  if(plsibw->priv->lOrientation == lOrientation)
    return;

  plsibw->priv->lOrientation = lOrientation;

  if(plsibw->priv->lOrientation == LS_ICONBOX_HORIZONTAL)
  {
    pSrc = GTK_CONTAINER(plsibw->priv->pVBox);
    pDst = GTK_CONTAINER(plsibw->priv->pHBox);
  }
  else
  {
    pSrc = GTK_CONTAINER(plsibw->priv->pHBox);
    pDst = GTK_CONTAINER(plsibw->priv->pVBox);
  }

  gtk_widget_ref(GTK_WIDGET(plsibw->priv->pHandleLeft));
  gtk_container_remove(pSrc, plsibw->priv->pHandleLeft);
  gtk_box_pack_start(GTK_BOX(pDst), plsibw->priv->pHandleLeft, FALSE, FALSE, 0);
  gtk_widget_unref(GTK_WIDGET(plsibw->priv->pHandleLeft));
    
  gtk_widget_ref(GTK_WIDGET(plsibw->priv->pIconBox));
  gtk_container_remove(pSrc, GTK_WIDGET(plsibw->priv->pIconBox));
  gtk_box_pack_start(GTK_BOX(pDst), GTK_WIDGET(plsibw->priv->pIconBox), FALSE,
    FALSE, 0);
  gtk_widget_unref(GTK_WIDGET(plsibw->priv->pIconBox));
    
  gtk_widget_ref(GTK_WIDGET(plsibw->priv->pHandleRight));
  gtk_container_remove(pSrc, plsibw->priv->pHandleRight);
  gtk_box_pack_start(GTK_BOX(pDst), plsibw->priv->pHandleRight, FALSE, FALSE, 0);
  gtk_widget_unref(GTK_WIDGET(plsibw->priv->pHandleRight));

  g_object_ref(pSrc);
  gtk_container_remove(GTK_CONTAINER(plsibw->priv->pFrame), 
    GTK_WIDGET(pSrc));
  gtk_container_add(GTK_CONTAINER(plsibw->priv->pFrame), 
    GTK_WIDGET(pDst));
  g_object_unref(pDst);

  gtk_widget_show(GTK_WIDGET(pDst));
  gtk_widget_hide(GTK_WIDGET(pSrc));

  ls_iconbox_set_orientation(ls_iconboxwin_get_iconbox(plsibw), lOrientation);
  ls_iconboxwin_re_evaluate_position(plsibw);

  return;
}

static void ls_iconboxwin_re_evaluate_position(LSIconBoxWin *plsibw)
{
  gint			nX;
  gint			nY;
  gint			nWidth;
  gint			nHeight;
  gint			nActX;
  gint			nActY;

  gtk_window_get_size(GTK_WINDOW(plsibw), &nWidth, &nHeight);
  gtk_window_get_position(GTK_WINDOW(plsibw), &nActX, &nActY);

  if(
    ls_iconboxwin_get_horizontal_position(plsibw) == LS_ICONBOX_RIGHT
  ) {
    nX			= (gdk_screen_width() - nWidth) - plsibw->priv->offset.nX;
  }
  else
  {
    nX			= plsibw->priv->offset.nX; 
  }

  if(
    ls_iconboxwin_get_vertical_position(plsibw) == LS_ICONBOX_BOTTOM
  ) {
    nY			= (gdk_screen_height() - nHeight) - plsibw->priv->offset.nY;
  }
  else
  {
    nY			= plsibw->priv->offset.nY;
  }

  if(nX<0)
    nX=0;
  if(nY<0)
    nY=0;
  if(nX+nWidth>gdk_screen_width())
    nX=gdk_screen_width()-nWidth;
  if(nY+nHeight>gdk_screen_height())
    nY=gdk_screen_height()-nHeight;
  
  if (nX!=nActX || nY!=nActY)
    gtk_window_move(GTK_WINDOW(plsibw), nX, nY);

  return;
}

glong ls_iconboxwin_get_horizontal_position(LSIconBoxWin *plsib)
{
  return plsib->priv->position.horizontal;
}

glong ls_iconboxwin_get_vertical_position(LSIconBoxWin *plsib)
{
  return plsib->priv->position.vertical;
}

/*******************************************************************************/
/* CALLBACKS                                                                   */
/*******************************************************************************/
gboolean cb_lsibw_size_allocate(
  GtkWidget *pWidget, 
  GtkAllocation *pAlloc, 
  gpointer pData
) {
  g_return_val_if_fail(LS_IS_ICONBOXWIN(pWidget) ,FALSE);
  
  ls_iconboxwin_re_evaluate_position(LS_ICONBOXWIN(pWidget));

  return TRUE;
}

gboolean cb_lsibw_handle_release (
  GtkWidget *pWidget, 
  GdkEventButton *pEvent, 
  gpointer pUserData
) {
  if(pEvent->type == GDK_BUTTON_RELEASE && pEvent->button == 1)
  {
    if(LS_IS_ICONBOXWIN(pUserData))
    {
      GtkWindow 		*pWin;
      gint			lXOff;
      gint			lYOff;
      gint			lXPos;
      gint			lYPos;
      gint			lWidth;
      gint			lHeight;

      pWin 			= GTK_WINDOW(pUserData);

      gtk_window_get_size(pWin, &lWidth, &lHeight);
      gtk_window_get_position(pWin, &lXPos, &lYPos);
      
      if(
	ls_iconboxwin_get_horizontal_position(LS_ICONBOXWIN(pUserData)) == 
	LS_ICONBOX_RIGHT
      ) {
	lXOff		= (gdk_screen_width() - lWidth) - lXPos;
      }
      else
      {
	lXOff		= lXPos;
      }

      if(
	ls_iconboxwin_get_vertical_position(LS_ICONBOXWIN(pUserData)) == 
	LS_ICONBOX_BOTTOM
      ) {
	lYOff		= (gdk_screen_height() - lHeight) - lYPos;
      }
      else
      {
	lYOff		= lYPos;
      }
   
      if(lXOff<0)
	lXOff=0;
      if(lYOff<0)
	lYOff=0;

      if(lXOff>gdk_screen_width())
	lXOff=gdk_screen_width();
      if(lYOff>gdk_screen_height())
	lYOff=gdk_screen_height();

      ls_iconboxwin_set_offset(LS_ICONBOXWIN(pUserData), lXOff, lYOff);
    }
    else
    {
      DBG("!GTK_IS_WINDOW(pUserData)");
      
      return FALSE;			/* event should be handled fourther	*/
    }
  }

  return FALSE;				/* event should be handled fourther	*/
}

gboolean cb_lsibw_handle_button2_pressed(
  GtkWidget *pWidget,
  GdkEvent *pEvent,
  gpointer *pUserData
) {
  if(pEvent->type!=GDK_BUTTON_PRESS)
    return FALSE;
  if(((GdkEventButton *)pEvent)->button!=3)
    return FALSE;

  g_return_val_if_fail(pUserData!=NULL, FALSE);
  g_return_val_if_fail(LS_IS_ICONBOXWIN(pUserData), FALSE);

  gtk_menu_popup (GTK_MENU(LS_ICONBOXWIN(pUserData)->priv->pMenu), NULL, NULL,
    NULL, NULL, ((GdkEventButton *)pEvent)->button,
    ((GdkEventButton *)pEvent)->time);

  return TRUE;
}

static void cb_lsibw_iconbox_hide(GtkWidget *pWidget, gpointer pUserData)
{
  g_return_if_fail(LS_IS_ICONBOXWIN(pUserData));
  
  if(LS_ICONBOXWIN(pUserData)->priv->bHideOnEmpty==TRUE)
  {
    gtk_widget_hide(GTK_WIDGET(LS_ICONBOXWIN(pUserData)->priv->pHandleRight));
    gtk_widget_hide(GTK_WIDGET(LS_ICONBOXWIN(pUserData)->priv->pHandleLeft));
    gtk_widget_hide(GTK_WIDGET(LS_ICONBOXWIN(pUserData)->priv->pFrame));
  }
  
  return;
}

static void cb_lsibw_iconbox_show(GtkWidget *pWidget, gpointer pUserData)
{
  g_return_if_fail(LS_IS_ICONBOXWIN(pUserData));
  
  gtk_widget_show(GTK_WIDGET(LS_ICONBOXWIN(pUserData)->priv->pHandleRight));
  gtk_widget_show(GTK_WIDGET(LS_ICONBOXWIN(pUserData)->priv->pHandleLeft));
  gtk_widget_show(GTK_WIDGET(LS_ICONBOXWIN(pUserData)->priv->pFrame));

  return;
}


/********************************************************************************/
/* MENU										*/
/********************************************************************************/
static void	cb_lsibwm_close_iconboxwin(gpointer pUserData, gint nAction, 
		  GtkWidget *pWidget);
static void	cb_lsibwm_save_pos(gpointer pUserData, gint nAction, 
		  GtkWidget *pWidget);
static void	cb_lsibwm_open_iconbox_config(gpointer pUserData, gint nAction, 
    		  GtkWidget *pWidget);
static void	cb_lsibwm_open_config(gpointer pUserData, gint nAction, 
    		  GtkWidget *pWidget);

static GtkItemFactoryEntry lsIconBoxWinMenu[] = 
{
  {N_("/Close"),		NULL,	cb_lsibwm_close_iconboxwin, 0, "<Item>"}, 
  {N_("/Save Position"),	NULL,	cb_lsibwm_save_pos, 0, "<Item>"}, 
  {"/Sep0",			NULL,	NULL, 0, "<Separator>"},
  {N_("/Properties..."),	NULL,   cb_lsibwm_open_iconbox_config, 0, "<Item>"},
  {N_("/Settings manager"),	NULL,   cb_lsibwm_open_config, 0, "<Item>"}
};

static gint nLsIconBoxWinMenu = sizeof(lsIconBoxWinMenu)/sizeof(lsIconBoxWinMenu[0]);

static gchar *
translator(const gchar *path, gpointer data)
{
  return(dgettext(GETTEXT_PACKAGE, path));
}

void ls_iconboxwin_create_menu(LSIconBoxWin *plsibw)
{
  GtkItemFactory	*pItemFactory;
  GtkAccelGroup		*pAccelGroup;

  pAccelGroup	= gtk_accel_group_new();
  pItemFactory	= gtk_item_factory_new(GTK_TYPE_MENU, "<main>", pAccelGroup);
  gtk_item_factory_set_translate_func(pItemFactory, translator, NULL, NULL);
  gtk_item_factory_create_items(pItemFactory, nLsIconBoxWinMenu, 
    lsIconBoxWinMenu, (gpointer)plsibw);
  
  plsibw->priv->pMenu = gtk_item_factory_get_widget(pItemFactory, "<main>");
  return;
}

/********************************************************************************/
/* MENU CALLBACKS								*/
/********************************************************************************/
static void cb_lsibwm_open_config(
  gpointer pUserData, 
  gint nAction, 
  GtkWidget *pWidget
) {
  GError	*pgError = NULL;
  gboolean	bError;

  bError = g_spawn_command_line_async("xfce-setting-show", &pgError); 
  if(bError==FALSE) 
  {
    DBG("g_spawn_command_line_async(\"xfce-setting-show\") failed!"
      "error: %s", pgError->message);
    xfce_err("%s", pgError->message);
  }
  
  return;
}

static void cb_lsibwm_open_iconbox_config(
  gpointer pUserData, 
  gint nAction, 
  GtkWidget *pWidget
) {
  GError	*pgError = NULL;
  gboolean	bError;

  bError = g_spawn_command_line_async("xfce-setting-show xfibx", &pgError); 
  if(bError==FALSE) 
  {
    DBG("g_spawn_command_line_async(\"xfce-setting-show xfibx\") "
      "failed! error: %s", pgError->message);
    xfce_err("%s", pgError->message);
  }
  
  return;
}

static void cb_lsibwm_close_iconboxwin(
  gpointer pUserData, 
  gint nAction, 
  GtkWidget *pWidget
) {
  g_return_if_fail(LS_IS_ICONBOXWIN(pUserData));
  gtk_widget_destroy(GTK_WIDGET(pUserData));

  return;
}

extern void save_offset(gint nXOff, gint nYOff);
static void cb_lsibwm_save_pos(
  gpointer pUserData, 
  gint nAction, 
  GtkWidget *pWidget
) {
  gint nXOff;
  gint nYOff;
  
  g_return_if_fail(LS_IS_ICONBOXWIN(pUserData));

  ls_iconboxwin_get_offset(LS_ICONBOXWIN(pUserData), &nXOff, &nYOff);  
  save_offset(nXOff, nYOff);
  
  return;
}

