#ifndef FILEDIALOG_H
#define FILEDIALOG_H

#include "FileList.h"
#include "DirBox.h"

class FileSelector;
class FileList;
class DirBox;

/// File selection widget
class FXAPI FileSelector : public FXPacker
{
    FXDECLARE(FileSelector)
protected:
    FileList    *filebox;         // File list widget
    DirBox      *dirbox;          // Directory hierarchy list
    FXTextField   *filename;        // File name entry field
    FXComboBox    *filefilter;      // Combobox for pattern list
    FXCheckButton *readonly;        // Open file as read only
    FXButton      *accept;          // Accept button
    FXButton      *cancel;          // Cancel button
    FXIcon        *updiricon;       // Up directory icon
    FXIcon        *listicon;        // List mode icon
    FXIcon        *detailicon;      // Detail mode icon
    FXIcon        *iconsicon;       // Icon mode icon
    FXIcon        *homeicon;        // Go home icon
    FXIcon        *workicon;        // Go home icon
    FXIcon        *shownicon;       // Files shown icon
    FXIcon        *hiddenicon;      // Files hidden icon
    FXuint         selectmode;      // Select mode
protected:
    FileSelector()
    {}
    static FXString patternFromText(const FXString& pattern);
    static FXString extensionFromPattern(const FXString& pattern);
private:
    FileSelector(const FileSelector&);
    FileSelector &operator=(const FileSelector&);
public:
    long onCmdAccept(FXObject*,FXSelector,void*);
    long onCmdFilter(FXObject*,FXSelector,void*);
    long onCmdItemDblClicked(FXObject*,FXSelector,void*);
    long onCmdItemSelected(FXObject*,FXSelector,void*);
    long onCmdItemDeselected(FXObject*,FXSelector,void*);
    long onCmdDirectoryUp(FXObject*,FXSelector,void*);
    long onUpdDirectoryUp(FXObject*,FXSelector,void*);
    long onCmdDirTree(FXObject*,FXSelector,void*);
    long onCmdHome(FXObject*,FXSelector,void*);
    long onCmdWork(FXObject*,FXSelector,void*);
public:
    enum {
        ID_FILEFILTER=FXPacker::ID_LAST,
        ID_ACCEPT,
        ID_FILELIST,
        ID_DIRECTORY_UP,
        ID_DIRTREE,
        ID_HOME,
        ID_WORK,
        ID_LAST
    };
public:

    /// Constructor
    FileSelector(FXComposite *p,FXObject* tgt=NULL,FXSelector sel=0,FXuint opts=0,FXint x=0,FXint y=0,FXint w=0,FXint h=0);

    /// Return a pointer to the "Accept" button
    FXButton *acceptButton() const
    {
        return accept;
    }

    /// Return a pointer to the "Cancel" button
    FXButton *cancelButton() const
    {
        return cancel;
    }

    /// Change file name
    void setFilename(const FXString& path);

    /// Return file name, if any
    FXString getFilename() const;

    /**
    * Return array of strings containing the selected file names, terminated
    * by an empty string.  This string array must be freed using delete [].
    * If no files were selected, a NULL is returned.
    */
    FXString* getFilenames() const;

    /// Change file pattern
    void setPattern(const FXString& ptrn);

    /// Return file pattern
    FXString getPattern() const;

    /**
    * Change the list of file patterns shown in the file dialog.
    * Each pattern comprises an optional name, followed by a pattern in
    * parentheses.  The patterns are separated by newlines.
    * For example,
    *
    *  "*\n*.cpp,*.cc\n*.hpp,*.hh,*.h"
    *
    * and
    *
    *  "All Files (*)\nC++ Sources (*.cpp,*.cc)\nC++ Headers (*.hpp,*.hh,*.h)"
    *
    * will set the same three patterns, but the former shows no pattern names.
    */
    void setPatternList(const FXString& patterns);

    /**
    * Set list of patterns as name,pattern pairs.
    * The list should be terminated with a final NULL string.
    * (DEPRECATED)
    */
    void setPatternList(const FXchar **ptrns);

    /// Return list of patterns
    FXString getPatternList() const;

    /**
    * After setting the list of patterns, this call will
    * initially select pattern n as the active one.
    */
    void setCurrentPattern(FXint n);

    /// Return current pattern number
    FXint getCurrentPattern() const;

    /// Get pattern text for given pattern number
    FXString getPatternText(FXint patno) const;

    /// Change pattern text for pattern number
    void setPatternText(FXint patno,const FXString& text);

    /// Change directory
    void setDirectory(const FXString& path);

    /// Return directory
    FXString getDirectory() const;

    /// Set the inter-item spacing (in pixels)
    void setItemSpace(FXint s);

    /// Return the inter-item spacing (in pixels)
    FXint getItemSpace() const;

    /// Change file list style
    void setFileBoxStyle(FXuint style);

    /// Return file list style
    FXuint getFileBoxStyle() const;

    /// Change file selection mode
    void setSelectMode(FXuint mode);

    /// Return file selection mode
    FXuint getSelectMode() const
    {
        return selectmode;
    }

    /// Show readonly button
    void showReadOnly(FXbool show);

    /// Return TRUE if readonly is shown
    FXbool shownReadOnly() const;

    /// Set initial state of readonly button
    void setReadOnly(FXbool state);

    /// Get readonly state
    FXbool getReadOnly() const;

    /// Save object to a stream
    virtual void save(FXStream& store) const;

    /// Load object from a stream
    virtual void load(FXStream& store);

    /// Destructor
    virtual ~FileSelector();
};


/// File Dialog object
class FXAPI FileDialog : public FXDialogBox
{
    FXDECLARE(FileDialog)
protected:
    FileSelector *filebox;
protected:
    FileDialog()
    {}
private:
    FileDialog(const FileDialog&);
    FileDialog &operator=(const FileDialog&);
public:

    /// Construct File Dialog Box
    FileDialog(FXWindow* owner,const FXString& name,FXuint opts=0,FXint x=0,FXint y=0,FXint w=500,FXint h=300);

    /// Change file name
    void setFilename(const FXString& path);

    /// Return file name, if any
    FXString getFilename() const;

    /// Return empty-string terminated list of selected file names, or NULL if none selected
    FXString* getFilenames() const;

    /// Change file pattern
    void setPattern(const FXString& ptrn);

    /// Return file pattern
    FXString getPattern() const;

    /**
    * Change the list of file patterns shown in the file dialog.
    * Each pattern comprises an optional name, followed by a pattern in
    * parentheses.  The patterns are separated by newlines.
    * For example,
    *
    *  "*\n*.cpp,*.cc\n*.hpp,*.hh,*.h"
    *
    * and
    *
    *  "All Files (*)\nC++ Sources (*.cpp,*.cc)\nC++ Headers (*.hpp,*.hh,*.h)"
    *
    * will set the same three patterns, but the former shows no pattern names.
    */
    void setPatternList(const FXString& patterns);

    /**
    * Set list of patterns as name,pattern pairs.
    * The list should be terminated with a final NULL string.
    * (DEPRECATED)
    */
    void setPatternList(const FXchar **ptrns);

    /// Return list of patterns
    FXString getPatternList() const;

    /**
    * After setting the list of patterns, this call will
    * initially select pattern n as the active one.
    */
    void setCurrentPattern(FXint n);

    /// Return current pattern number
    FXint getCurrentPattern() const;

    /// Get pattern text for given pattern number
    FXString getPatternText(FXint patno) const;

    /// Change pattern text for pattern number
    void setPatternText(FXint patno,const FXString& text);

    /// Change directory
    void setDirectory(const FXString& path);

    /// Return directory
    FXString getDirectory() const;

    /// Set the inter-item spacing (in pixels)
    void setItemSpace(FXint s);

    /// Return the inter-item spacing (in pixels)
    FXint getItemSpace() const;

    /// Change File List style
    void setFileBoxStyle(FXuint style);

    /// Return File List style
    FXuint getFileBoxStyle() const;

    /// Change file selection mode
    void setSelectMode(FXuint mode);

    /// Return file selection mode
    FXuint getSelectMode() const;

    /// Show readonly button
    void showReadOnly(FXbool show);

    /// Return TRUE if readonly is shown
    FXbool shownReadOnly() const;

    /// Set initial state of readonly button
    void setReadOnly(FXbool state);

    /// Get readonly state
    FXbool getReadOnly() const;

    /// Open existing filename
    static FXString getOpenFilename(FXWindow* owner,const FXString& caption,const FXString& path,const FXString& patterns="*",FXint initial=0);

    /// Open multiple existing files
    static FXString* getOpenFilenames(FXWindow* owner,const FXString& caption,const FXString& path,const FXString& patterns="*",FXint initial=0);

    /// Save to filename
    static FXString getSaveFilename(FXWindow* owner,const FXString& caption,const FXString& path,const FXString& patterns="*",FXint initial=0);

    /// Open directory name
    static FXString getOpenDirectory(FXWindow* owner,const FXString& caption,const FXString& path);

    /// Save to stream
    virtual void save(FXStream& store) const;

    /// Load from stream
    virtual void load(FXStream& store);

    /// Destructor
    virtual ~FileDialog();
};


#endif
