/*
 * Copyright (C) 2003 the xmms-kde team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include <config.h>

#include <klocale.h>
#include <kprocess.h>

#ifdef HAVE_XMMS
#include <xmmsctrl.h>
#include "xmmsplayer.h"

bool XMMSPlayer::checkRunning() {

  running = xmms_remote_is_running(0);
  return running;
}


XMMSPlayer::XMMSPlayer(bool hidePlayer) {
  qDebug("xmms-kde: created xmmsplayer interface");

  /*
   * test if xmms is already running
   */
  checkRunning();

  qDebug("xmms running ? %d\n",  running);

  hide = hidePlayer;

  xmmsIds.clear();

  if (running) {
    getXmmsIds(QPaintDevice::x11AppDisplay(),	       
	       RootWindow(QPaintDevice::x11AppDisplay(), QPaintDevice::x11AppScreen()),
	       &xmmsIds);

    mainWin = xmms_remote_is_main_win(0);
    plWin = xmms_remote_is_pl_win(0);
    eqWin = xmms_remote_is_eq_win(0);

    if (hide)
      hideXmms();
  }
}

XMMSPlayer::~XMMSPlayer() {
  qDebug("xmms-kde: deleted xmmsplayer interface");
}

void XMMSPlayer::startPlayer() {
  qDebug("xmms-kde: launching xmms");
  KShellProcess proc;
  proc << "xmms -m &";
  proc.start(KShellProcess::Block);

  qDebug("xmms-kde: player: %d", hide);
}

void XMMSPlayer::exitPlayer() {

  if (!checkRunning())
    return;

  qDebug("xmms-kde: killing xmms");

  running = false;
  xmms_remote_quit(0); 
}

void XMMSPlayer::jumpto(int pos) {

  if (!checkRunning())
    return;

  xmms_remote_set_playlist_pos(0, pos);
}

void XMMSPlayer::previous() {

  if (!checkRunning())
    return;

  xmms_remote_playlist_prev(0);
}

void XMMSPlayer::play() {

  if (!checkRunning())
    return;

 xmms_remote_play(0); 
}

void XMMSPlayer::pause() {

  if (!checkRunning())
    return;

  xmms_remote_pause(0);
}

void XMMSPlayer::stop() {

  if (!checkRunning())
    return;

  xmms_remote_stop(0);
}

void XMMSPlayer::next() {

  if (!checkRunning())
    return;

  xmms_remote_playlist_next(0);
}

void XMMSPlayer::toggleRepeat() {

  if (!checkRunning())
    return;

  xmms_remote_toggle_repeat(0);
}

void XMMSPlayer::toggleShuffle() {

  if (!checkRunning())
    return;

  xmms_remote_toggle_shuffle(0);
}

void XMMSPlayer::setVolume(int volume) {

  if (!checkRunning())
    return;

  xmms_remote_set_main_volume(0, volume);
}

void XMMSPlayer::setCurrentTime(int time) {

  if (!checkRunning())
    return;

  xmms_remote_jump_to_time(0, time);  
}

int XMMSPlayer::getVolume() {

  if (!checkRunning())
    return 0;

  return xmms_remote_get_main_volume(0);
}

QString XMMSPlayer::getTitle() {

  if (!checkRunning())
    return 0;

  gchar *title = 
    xmms_remote_get_playlist_title(0, xmms_remote_get_playlist_pos(0));
  
  QString t = QString::fromLocal8Bit((char *) title, -1);
  if (title) free(title);
  return t;
}

int XMMSPlayer::getPlaylistLength() { 

  if (!checkRunning())
    return 0;
  
  return xmms_remote_get_playlist_length(0);
} 

int XMMSPlayer::getPlaylistPos() { 

  if (!checkRunning())
    return 0;
  
  return xmms_remote_get_playlist_pos(0);
} 

QString XMMSPlayer::getPlaylistTitle(int pos) {

  if (!checkRunning())
    return 0;

  gchar *title = 
    xmms_remote_get_playlist_title(0, pos);
  
  QString t = QString::fromLocal8Bit((char *) title, -1);
  if (title) free(title);
  return t;
}


int XMMSPlayer::getCurrentTime() {

  if (!checkRunning())
    return 0;

  return xmms_remote_get_output_time(0);
}

int XMMSPlayer::getTrackTime() {

  if (!checkRunning())
    return 0;

  return xmms_remote_get_playlist_time(0, xmms_remote_get_playlist_pos(0));
}

bool XMMSPlayer::getRepeat() {

  if (!checkRunning())
    return 0;

  return xmms_remote_is_repeat(0);
}

bool XMMSPlayer::getShuffle() {

  if (!checkRunning())
    return 0;

  return xmms_remote_is_shuffle(0);
}

bool XMMSPlayer::isPlaying() {

  if (!checkRunning())
    return 0;

  return xmms_remote_is_playing(0);
}

bool XMMSPlayer::isPlayerRunning() {

  bool run = running;
  running = xmms_remote_is_running(0);

  // if player has started
  if (run != running && running) {

    xmmsIds.clear();
    
    getXmmsIds(QPaintDevice::x11AppDisplay(),
	       RootWindow(QPaintDevice::x11AppDisplay(), QPaintDevice::x11AppScreen()),
	       &xmmsIds);

    mainWin = xmms_remote_is_main_win(0);
    plWin = xmms_remote_is_pl_win(0);
    eqWin = xmms_remote_is_eq_win(0);

    if (hide)
      hideXmms();
  }

  return running;
}

void XMMSPlayer::openFileSelector() {

  if (!checkRunning())
    return;

  xmms_remote_eject(0);
}

void XMMSPlayer::playlistClear() {

  if (!checkRunning())
    return;

  xmms_remote_playlist_clear(0);
}

void XMMSPlayer::playlistAdd(QString files) {

  if (!checkRunning())
    return;

  printf("adding [%s]\n", files.local8Bit().data());
  xmms_remote_playlist_add_url_string(0, (gchar *) files.local8Bit().data());
}

void XMMSPlayer::playlistAdd(QStringList files) {

  playlistAdd(files.join("\r\n"));
}

void XMMSPlayer::minimize(bool player) {
  
  hide = player;
}

void XMMSPlayer::getXmmsIds(Display *dpy, Window top, std::vector<int> *idList) {

  Window *children, dummy;
  unsigned int nchildren;
  int i;
  Window w = 0;
  char *window_name = 0;
  
  if (XFetchName(dpy, top, &window_name) && !strncmp(window_name, "XMMS", 4))
    idList->push_back(top);
    
  if (window_name) XFree((char *) window_name);
  
  if (!XQueryTree(dpy, top, &dummy, &dummy, &children, &nchildren))
    return;
  
  for (i = 0; i < nchildren; i++) {
    getXmmsIds(dpy, children[i], idList);
  }

  if (children) XFree((char *) children);
}

void XMMSPlayer::showXmms() {

  qDebug("xmms-kde: show");

  if (!checkRunning())
    return;

  for (int index = 0; index < xmmsIds.size(); index ++) {
    KWin::clearState(xmmsIds[index], NET::SkipTaskbar);
  }

  if (mainWin) xmms_remote_main_win_toggle(0, true);
  if (plWin) xmms_remote_pl_win_toggle(0, true);
  if (eqWin) xmms_remote_eq_win_toggle(0, true);
}

void XMMSPlayer::hideXmms() {

  qDebug("xmms-kde: hide");

  if (!checkRunning())
    return;
 
  xmms_remote_main_win_toggle(0, false);
  xmms_remote_pl_win_toggle(0, false);
  xmms_remote_eq_win_toggle(0, false);

  for (int index = 0; index < xmmsIds.size(); index ++) {
    KWin::setState(xmmsIds[index], NET::SkipTaskbar);
  }
}

#endif
