/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "maths.h"

#include <limits.h>


CMatrix operator+(const CMatrix &m0,const CMatrix &m1)
{
	CMatrix result;

	for (int i = 0;i != CMatrix::I_ELEMENTS*CMatrix::J_ELEMENTS;i++) {
		result.m[i] = m0.m[i] + m1.m[i];
	}

	return result;
}

CMatrix operator-(const CMatrix &m0,const CMatrix &m1)
{
	CMatrix result;

	for (int i = 0;i != CMatrix::I_ELEMENTS*CMatrix::J_ELEMENTS;i++) {
		result.m[i] = m0.m[i] - m1.m[i];
	}

	return result;
}

inline CMatrix operator*(const CMatrix &m0,const CMatrix &m1)
{
	CMatrix result;

	for (int j = 0;j != CMatrix::J_ELEMENTS;j++) {
		for (int i = 0;i != CMatrix::I_ELEMENTS;i++) {
			result.m[CMatrix::GetIndex(j,i)] =
				  m0.m[CMatrix::GetIndex(j,0)] * m1.m[CMatrix::GetIndex(0,i)]
				+ m0.m[CMatrix::GetIndex(j,1)] * m1.m[CMatrix::GetIndex(1,i)]
				+ m0.m[CMatrix::GetIndex(j,2)] * m1.m[CMatrix::GetIndex(2,i)]
				+ m0.m[CMatrix::GetIndex(j,3)] * m1.m[CMatrix::GetIndex(3,i)];
		}
	}

	return result;
}


CMatrix::CMatrix()
{
}

CMatrix::~CMatrix()
{
}

int CMatrix::GetIndex(int j,int i)
{
	return j*I_ELEMENTS + i;
}

void CMatrix::RotUnit(void)
{
	m[GetIndex(0,0)] = 1.0;
	m[GetIndex(0,1)] = 0.0;
	m[GetIndex(0,2)] = 0.0;
	m[GetIndex(1,0)] = 0.0;
	m[GetIndex(1,1)] = 1.0;
	m[GetIndex(1,2)] = 0.0;
	m[GetIndex(2,0)] = 0.0;
	m[GetIndex(2,1)] = 0.0;
	m[GetIndex(2,2)] = 1.0;
}

void CMatrix::TransZero(void)
{
	m[GetIndex(3,0)] = m[GetIndex(3,1)] = m[GetIndex(3,2)] = 0;
}

void CMatrix::Unit(void)
{
	int i;
	for (i = 0; i != I_ELEMENTS*J_ELEMENTS - 1; i += I_ELEMENTS+1) {
		m[i+0] = 1.0;
		m[i+1] = m[i+2] = m[i+3] = m[i+4] = 0.0;
	}
	m[i] = 1.0;
}

void CMatrix::Transpose(void)
{
	float tmp;

	tmp = m[1];
	m[1] = m[4];
	m[4] = tmp;

	tmp = m[2];
	m[2] = m[8];
	m[8] = tmp;

	tmp = m[6];
	m[6] = m[9];
	m[9] = tmp;
}

void CMatrix::GetModelView(void)
{
	glGetFloatv(GL_MODELVIEW_MATRIX,m);
}

void CMatrix::MulRotX(float sine,float cosine)
{
	float tmp0,tmp1;

	tmp0 = m[GetIndex(0,1)];
	tmp1 = m[GetIndex(0,2)];
	m[GetIndex(0,1)] = tmp0 * cosine + tmp1 * -sine;
	m[GetIndex(0,2)] = tmp0 * sine + tmp1 * cosine;

	tmp0 = m[GetIndex(1,1)];
	tmp1 = m[GetIndex(1,2)];
	m[GetIndex(1,1)] = tmp0 * cosine + tmp1 * -sine;
	m[GetIndex(1,2)] = tmp0 * sine + tmp1 * cosine;

	tmp0 = m[GetIndex(2,1)];
	tmp1 = m[GetIndex(2,2)];
	m[GetIndex(2,1)] = tmp0 * cosine + tmp1 * -sine;
	m[GetIndex(2,2)] = tmp0 * sine + tmp1 * cosine;

	tmp0 = m[GetIndex(3,1)];
	tmp1 = m[GetIndex(3,2)];
	m[GetIndex(3,1)] = tmp0 * cosine + tmp1 * -sine;
	m[GetIndex(3,2)] = tmp0 * sine + tmp1 * cosine;
}

void CMatrix::MulRotX(float radian)
{
	float sine,cosine;

	sine = Sin(radian);
	cosine = Cos(radian);
	MulRotX(sine,cosine);
}

void CMatrix::MulRotX(fix16_16 radian)
{
	float sine,cosine;

	sine = Sin(radian);
	cosine = Cos(radian);
	MulRotX(sine,cosine);
}


void CMatrix::MulRotY(float sine,float cosine)
{
	float tmp0,tmp1;

	tmp0 = m[GetIndex(0,0)];
	tmp1 = m[GetIndex(0,2)];
	m[GetIndex(0,0)] = tmp0 * cosine + tmp1 * sine;
	m[GetIndex(0,2)] = tmp0 * -sine + tmp1 * cosine;

	tmp0 = m[GetIndex(1,0)];
	tmp1 = m[GetIndex(1,2)];
	m[GetIndex(1,0)] = tmp0 * cosine + tmp1 * sine;
	m[GetIndex(1,2)] = tmp0 * -sine + tmp1 * cosine;

	tmp0 = m[GetIndex(2,0)];
	tmp1 = m[GetIndex(2,2)];
	m[GetIndex(2,0)] = tmp0 * cosine + tmp1 * sine;
	m[GetIndex(2,2)] = tmp0 * -sine + tmp1 * cosine;

	tmp0 = m[GetIndex(3,0)];
	tmp1 = m[GetIndex(3,2)];
	m[GetIndex(3,0)] = tmp0 * cosine + tmp1 * sine;
	m[GetIndex(3,2)] = tmp0 * -sine + tmp1 * cosine;
}

void CMatrix::MulRotY(float radian)
{
	float sine,cosine;

	sine = Sin(radian);
	cosine = Cos(radian);
	MulRotY(sine,cosine);
}

void CMatrix::MulRotY(fix16_16 radian)
{
	float sine,cosine;

	sine = Sin(radian);
	cosine = Cos(radian);
	MulRotY(sine,cosine);
}


void CMatrix::MulRotZ(float sine,float cosine)
{
	float tmp0,tmp1;

	tmp0 = m[GetIndex(0,0)];
	tmp1 = m[GetIndex(0,1)];
	m[GetIndex(0,0)] = tmp0 * cosine + tmp1 * -sine;
	m[GetIndex(0,1)] = tmp0 * sine + tmp1 * cosine;

	tmp0 = m[GetIndex(1,0)];
	tmp1 = m[GetIndex(1,1)];
	m[GetIndex(1,0)] = tmp0 * cosine + tmp1 * -sine;
	m[GetIndex(1,1)] = tmp0 * sine + tmp1 * cosine;

	tmp0 = m[GetIndex(2,0)];
	tmp1 = m[GetIndex(2,1)];
	m[GetIndex(2,0)] = tmp0 * cosine + tmp1 * -sine;
	m[GetIndex(2,1)] = tmp0 * sine + tmp1 * cosine;

	tmp0 = m[GetIndex(3,0)];
	tmp1 = m[GetIndex(3,1)];
	m[GetIndex(3,0)] = tmp0 * cosine + tmp1 * -sine;
	m[GetIndex(3,1)] = tmp0 * sine + tmp1 * cosine;
}

void CMatrix::MulRotZ(float radian)
{
	float sine,cosine;

	sine = Sin(radian);
	cosine = Cos(radian);
	MulRotZ(sine,cosine);
}

void CMatrix::MulRotZ(fix16_16 radian)
{
	float sine,cosine;

	sine = Sin(radian);
	cosine = Cos(radian);
	MulRotZ(sine,cosine);
}


void CMatrix::SetTrans(float tx,float ty,float tz)
{
	m[GetIndex(3,0)] = tx;
	m[GetIndex(3,1)] = ty;
	m[GetIndex(3,2)] = tz;
}

void CMatrix::AddTrans(float tx,float ty,float tz)
{
	m[GetIndex(3,0)] += tx;
	m[GetIndex(3,1)] += ty;
	m[GetIndex(3,2)] += tz;
}

void CMatrix::MulScale(float sx,float sy,float sz)
{
	m[GetIndex(0,0)] *= sx;
	m[GetIndex(1,1)] *= sy;
	m[GetIndex(2,2)] *= sz;
	m[GetIndex(3,0)] *= sx;
	m[GetIndex(3,1)] *= sy;
	m[GetIndex(3,2)] *= sz;
}

void CMatrix::MulVector(const float src[],float dst[])
{
	float vx,vy,vz,vw;

	vx = src[0];
	vy = src[1];
	vz = src[2];
	vw = 1.0;

	dst[0] = (m[GetIndex(0,0)] * vx)
	    + (m[GetIndex(1,0)] * vy)
	    + (m[GetIndex(2,0)] * vz)
	    + (m[GetIndex(3,0)] * vw);
	dst[1] = (m[GetIndex(0,1)] * vx)
	    + (m[GetIndex(1,1)] * vy)
	    + (m[GetIndex(2,1)] * vz)
	    + (m[GetIndex(3,1)] * vw);
	dst[2] = (m[CMatrix::GetIndex(0,2)] * vx)
	    + (m[GetIndex(1,2)] * vy)
	    + (m[GetIndex(2,2)] * vz)
	    + (m[GetIndex(3,2)] * vw);
	dst[3] = (m[GetIndex(0,3)] * vx)
	    + (m[GetIndex(1,3)] * vy)
	    + (m[GetIndex(2,3)] * vz)
	    + (m[GetIndex(3,3)] * vw);
}

void CMatrix::MulVector(const float src[],float dst[],int n)
{
	int i;
	for (i = 0;i != n;i++) {
		MulVector(&src[i * 4],&dst[i * 4]);
	}
}

float CMatrix::MulVectorZ(const float src[])
{
	float vx,vy,vz,vw;

	vx = src[0];
	vy = src[1];
	vz = src[2];
	vw = 1.0;

	return (m[GetIndex(0,2)] * vx)
	    + (m[GetIndex(1,2)] * vy)
	    + (m[GetIndex(2,2)] * vz)
	    + (m[GetIndex(3,2)] * vw);
}

void CMatrix::MulVectorZ(const float src[],float dst[],int n)
{
	int i;
	for (i = 0;i != n;i++) {
		dst[i] = MulVectorZ(&src[i * 4]);
	}
}

void CMatrix::SetMatrix(const float src[])
{
	int i;
	for (i = 0;i != I_ELEMENTS * J_ELEMENTS;i++) {
		m[i] = src[i];
	}
}
