/*-
# MOTIF/X-BASED CUBES
#
#  xcubes.c
#
###
#
#  Copyright (c) 1993 - 2006	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/11/15 X/Windows
  Version 5: 95/10/01 Xt/Motif
  Version 4: 94/06/07 Xt
  Version 3: 93/04/01 Motif
  Version 2: 92/12/19 XView
  Version 1: 91/03/19 SunView
*/

#include "file.h"
#ifdef WINVER
#include "CubesP.h"
#include "wcubes.h"
#define TITLE "wcubes"

static CubesRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "c:/WINDOWS"
#else
#define SCOREPATH "c:\\WINDOWS"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->cubes.started = b
#else
#include "xwin.h"
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Cubes.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/cubes.xpm"
#endif
#include "icons/cubes.xbm"
#ifndef SCOREPATH
#ifdef VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef WINVER
static const char aboutHelp[] = {
"Cubes Version 7.2.4\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd@tux.org>\n"
"The latest version is at: "
"http://www.tux.org/~bagleyd/puzzles.html\n"
};

static const char optionsHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-block {color}] [-{border|bd} {color}] [-[no]install]\n"
"[-picture {filename}] [-delay msecs] [-[no]sound]\n"
"[-bumpSound {filename}] [-dripSound {filename}]\n"
"[-{font|fn} {fontname}] [-size{x|y|z} {int}] [-base {int}]\n"
"[-userName {string}] [-scoreFile {filename}] [-scores]\n"
"[-version]\n"
};
#endif

#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
"A 3D  sliding block puzzle.  The cubes represented are  "
"flattened out on the plane.  If the Blocks Z slider is\n"
"greater than 1, then each group of boxes is a layer of the "
"puzzle.  If the Blocks Z slider = 1, then this is a\n"
"simple 15 tile puzzle.  The 15 puzzle was originally "
"made around 1879 (the 14-15  puzzle of Sam Loyd  was made\n"
"soon after.)  Similar 3D puzzles include  BloxBox; "
"Qrazy Qube (2x2x2) by P. Hein of Denmark and Varikon Box 'L\n"
"Box 'L (3x3x3) (Hungary) and Mad Marbles (3x3x3) by "
"Toys & Games International USA.\n"
};

static const char featuresHelp[] = {
"Click \"mouse-left\" button to move a block.  Clicks on "
"a space, or clicks on blocks  that are not in line\n"
"with a space, will not move the blocks.  Notice, one can move "
"more than  one block at a time by clicking\n"
"on the block farthest from the space but still in line "
"with the space.  Since you only have a 2-D screen\n"
"the next level the down is  either the box to the right of "
"the current box  or the bottom of the current\n"
"box.\n"
"\n"
"Click \"mouse-right\" button, or press \"C\" or \"c\" "
"keys, to clear the puzzle.\n"
"\n"
"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save (write) a puzzle.\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"R\" or \"r\" keys to redo a move.\n"
"\n"
"Press \"Z\" or \"z\" keys to randomize the puzzle.  New records "
"can only be set from a cleared state.\n"
"\n"
"Press \"S\" or \"s\" keys to start the auto-solver.  Only "
"works on lxmx1 where l > 3 and m > 3.\n"
"\n"
"Press \"I\" or \"i\" keys to move a block in.\n"
"\n"
"Press \"O\" or \"o\" keys to move a block out.\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of blocks.\n"
"\n"
"Press \"<\" or \",\" keys to slow down the movement of blocks.\n"
"\n"
"Press \"@\" key to toggle the sound.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Use the key pad or arrow keys to move without the mouse.\n"
"Key pad is defined for Cubes as:\n"
"  /    Out\n"
"  8    Up\n"
"  ^\n"
"4<5>6  Left, In, Right\n"
"  v\n"
"  2    Down\n"
"\n"
"The control key allows you to change the complexity of "
"xcubes.  Hold down control key and use the keypad\n"
"or arrow keys to reduce or enlarge puzzle complexity.\n"
};

static const char referencesHelp[] = {
"L. E. Horden, Sliding Piece Puzzles  (Recreations in "
"Mathematics Series), Oxford University Press  1986,\n"
"pp 1, 157-159.\n"
"Jerry Slocum & Jack Botermans, Puzzles Old & New (How to Make "
"and Solve Them),  University of Washington\n"
"Press, Seattle 1987, p 126, 127.\n"
};
#endif

static const char solveHelp[] = {
"Auto-solver: sorry, only implemented for lxmx1 where "
"l > 3 and m > 3.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "cubes.scores"
#endif

#define MAXCUBES 8
#define NEVER (-1)
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} GameRecord;

static GameRecord cubesRecord[MAXCUBES - MINCUBES + 1][MAXCUBES - MINCUBES + 1]
[MAXCUBES - MINCUBES + 1];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";

#ifdef HAVE_MOTIF
#define MINSPEED 1
#define MAXSPEED 50
static Widget moves, record, message, blockX, blockY, blockZ, speed;
static char buff[21];
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, clearDialog;
static Arg arg[3];
#else
static char titleDsp[TITLELEN] = "";
#endif
static char scoreFileName[FILENAMELEN] = SCOREFILE;
static char fileName[FILENAMELEN];
static Boolean randomized = False;
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char userNameDsp[USERNAMELEN] = "Guest";
#else
static Pixmap icon = None;
static Widget topLevel, cubes;
static char *progDsp;
static char userNameDsp[USERNAMELEN] = "";

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w, XmNlabelString, xmstr, NULL);
}
#endif

static void
PrintRecords(void)
{
	int i, j, k;

	(void) printf("          CUBES  HALL OF FAME\n\n");
	(void) printf("   Z   Y   X USER             MOVES\n");
	for (i = 0; i < MAXCUBES - MINCUBES + 1; i++)
		for (j = i; j < MAXCUBES - MINCUBES + 1; j++)
			for (k = j; k < MAXCUBES - MINCUBES + 1; k++) {
				if (cubesRecord[k][j][i].score > 0)
					(void) printf("%4d%4d%4d %-12s%9d\n",
						i + 1, j + 1, k + 1,
						cubesRecord[k][j][i].name,
						cubesRecord[k][j][i].score);
			}
}
#endif

static void
InitRecords(void)
{
	int i, j, k;

	for (i = 0; i < MAXCUBES - MINCUBES + 1; i++)
		for (j = i; j < MAXCUBES - MINCUBES + 1; j++)
			for (k = j; k < MAXCUBES - MINCUBES + 1; k++) {
				cubesRecord[k][j][i].score = cubesRecord[k][i][j].score =
					cubesRecord[j][k][i].score = cubesRecord[j][i][k].score =
					cubesRecord[i][k][j].score = cubesRecord[i][j][k].score = NEVER;
				(void) strncpy(cubesRecord[k][j][i].name,
					NOACCESS, USERNAMELEN);
				(void) strncpy(cubesRecord[k][i][j].name,
					NOACCESS, USERNAMELEN);
				(void) strncpy(cubesRecord[j][k][i].name,
					NOACCESS, USERNAMELEN);
				(void) strncpy(cubesRecord[j][i][k].name,
					NOACCESS, USERNAMELEN);
				(void) strncpy(cubesRecord[i][k][j].name,
					NOACCESS, USERNAMELEN);
				(void) strncpy(cubesRecord[i][j][k].name,
					NOACCESS, USERNAMELEN);
			}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int i, j, k, n;
	char userName[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USERNAMELEN);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USERNAMELEN);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (i = 0; i < MAXCUBES - MINCUBES + 1; i++)
		for (j = i; j < MAXCUBES - MINCUBES + 1; j++)
			for (k = j; k < MAXCUBES - MINCUBES + 1; k++) {
				(void) fscanf(fp, "%d %s", &n, userName);
				if (n <= cubesRecord[i][j][k].score ||
					  cubesRecord[i][j][k].score <= NEVER) {
					cubesRecord[k][j][i].score = cubesRecord[k][i][j].score =
						cubesRecord[j][k][i].score = cubesRecord[j][i][k].score =
						cubesRecord[i][k][j].score = cubesRecord[i][j][k].score = n;
					(void) strncpy(cubesRecord[k][j][i].name,
						userName, USERNAMELEN);
					(void) strncpy(cubesRecord[k][i][j].name,
						userName, USERNAMELEN);
					(void) strncpy(cubesRecord[j][k][i].name,
						userName, USERNAMELEN);
					(void) strncpy(cubesRecord[j][i][k].name,
						userName, USERNAMELEN);
					(void) strncpy(cubesRecord[i][k][j].name,
						userName, USERNAMELEN);
					(void) strncpy(cubesRecord[i][j][k].name,
						userName, USERNAMELEN);
				}
			}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int i, j, k;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Can not write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILENAMELEN];

		(void) strncpy(lockFile, fileName, FILENAMELEN - 6);
		(void) strcat(lockFile, ".lock");
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (i = 0; i < MAXCUBES - MINCUBES + 1; i++) {
			for (j = i; j < MAXCUBES - MINCUBES + 1; j++) {
				for (k = j; k < MAXCUBES - MINCUBES + 1; k++)
					(void) fprintf(fp, "%d %s\n",
						cubesRecord[i][j][k].score,
						cubesRecord[i][j][k].name);
				(void) fprintf(fp, "\n");
			}
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
PrintRecord(int sizeX, int sizeY, int sizeZ)
{
	int i = sizeX - MINCUBES, j = sizeY - MINCUBES, k = sizeZ - MINCUBES;

	if (sizeX > MAXCUBES || sizeY > MAXCUBES || sizeZ > MAXCUBES) {
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGELEN);
	} else if (cubesRecord[i][j][k].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			cubesRecord[i][j][k].score, cubesRecord[i][j][k].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static void
PrintStatus(char *msg, int nMoves
#ifndef HAVE_MOTIF
		, int sizeX, int sizeY, int sizeZ
#endif
		)
{
#ifdef HAVE_MOTIF
	PrintState(message, msg);
	(void) sprintf(buff, "%d", nMoves);
	PrintState(moves, buff);
#else
	(void) sprintf(titleDsp, "%s: %dx%dx%d@ (%d/%s) - %s",
		progDsp, sizeX, sizeY, sizeZ, nMoves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
	XtVaSetValues(XtParent(cubes), XtNtitle, titleDsp, NULL);
#endif
#endif
}

static Boolean
HandleSolved(int counter, int sizeX, int sizeY, int sizeZ)
{
	int i = sizeX - MINCUBES, j = sizeY - MINCUBES, k = sizeZ - MINCUBES;

	if (sizeX <= MAXCUBES && sizeY <= MAXCUBES && sizeZ <= MAXCUBES &&
			(counter < cubesRecord[i][j][k].score ||
			cubesRecord[i][j][k].score <= NEVER)) {
		ReadRecords();	/* Maybe its been updated by another */

		cubesRecord[i][j][k].score = cubesRecord[i][k][j].score =
			cubesRecord[j][i][k].score =
			cubesRecord[j][k][i].score =
			cubesRecord[k][i][j].score =
			cubesRecord[k][j][i].score =
			counter;
		(void) strncpy(cubesRecord[i][j][k].name, userNameDsp,
			USERNAMELEN);
		(void) strncpy(cubesRecord[i][k][j].name, userNameDsp,
			USERNAMELEN);
		(void) strncpy(cubesRecord[j][i][k].name, userNameDsp,
			USERNAMELEN);
		(void) strncpy(cubesRecord[j][k][i].name, userNameDsp,
			USERNAMELEN);
		(void) strncpy(cubesRecord[k][i][j].name, userNameDsp,
			USERNAMELEN);
		(void) strncpy(cubesRecord[k][j][i].name, userNameDsp,
			USERNAMELEN);
		WriteRecords();
		PrintRecord(sizeX, sizeY, sizeZ);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
CubesWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	int sizeX, sizeY, sizeZ;
	char *userName, *scoreFile;

#ifdef WINVER
	InitializePuzzle(w, brush);

	sizeX = w->cubes.sizeX;
	sizeY = w->cubes.sizeY;
	sizeZ = w->cubes.sizeZ;
	userName = w->cubes.userName;
	scoreFile = w->cubes.scoreFile;
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILENAMELEN);
#else
	Boolean scoreOnly, versionOnly;

	XtVaGetValues(w,
		XtNuserName, &userName,
		XtNscoreFile, &scoreFile,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNsizeZ, &sizeZ,
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf(aboutHelp);
		exit(0);
	}
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILENAMELEN);
	if (scoreOnly) {
		InitRecords();
		ReadRecords();
		PrintRecords();
		exit(0);
	}
#ifdef HAVE_MOTIF
	if (sizeX > MAXCUBES)
		XtVaSetValues(blockX, XmNmaximum, sizeX, NULL);
	XmScaleSetValue(blockX, sizeX);
	if (sizeY > MAXCUBES)
		XtVaSetValues(blockY, XmNmaximum, sizeY, NULL);
	XmScaleSetValue(blockY, sizeY);
	if (sizeZ > MAXCUBES)
		XtVaSetValues(blockZ, XmNmaximum, sizeZ, NULL);
	XmScaleSetValue(blockZ, sizeZ);
	{
		int delay;

		XtVaGetValues(w, XtNdelay, &delay, NULL);
		XmScaleSetValue(speed, MAXSPEED + MINSPEED - delay - 1);
	}
#endif
#endif
	SET_STARTED(w, False);
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USERNAMELEN);
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USERNAMELEN);
#else
		char *login = getlogin();

		if (login == NULL) {
			(void) strcpy(userNameDsp, "");
		} else {
			(void) sprintf(userNameDsp, "%s", login);
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(userNameDsp, "%s", "guest");
#endif
	}
	PrintRecord(sizeX, sizeY, sizeZ);
	PrintStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, sizeX, sizeY, sizeZ
#endif
		);
}

#ifdef WINVER
void
setPuzzle(CubesWidget w, int reason)
#else
static void
CallbackPuzzle(Widget w, caddr_t clientData, cubesCallbackStruct *callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	int sizeX, sizeY, sizeZ;
	Boolean cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	sizeX = w->cubes.sizeX;
	sizeY = w->cubes.sizeY;
	sizeZ = w->cubes.sizeZ;
	cheat = w->cubes.cheat;
#else
	XtVaGetValues(w,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNsizeZ, &sizeZ,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case PUZZLE_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
			break;
#ifndef WINVER
		case PUZZLE_CLEAR_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(clearDialog);
#else
			XtVaSetValues(cubes, XtNmenu, MENU_CLEAR, NULL);
#endif
			break;
#endif
		case PUZZLE_SOLVE_MESSAGE:
#ifdef WINVER
			(void) MessageBox(w->core.hWnd, solveHelp,
				"Auto-Solve", MB_OK);
#else

#ifdef HAVE_MOTIF
			XtManageChild(solveDialog);
#else
			(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
			break;
		case PUZZLE_RESTORE:
		case PUZZLE_RESET:
			movesDsp = 0;
			randomized = False;
			break;
		case PUZZLE_BLOCKED:
			(void) strncpy(messageDsp, "Blocked", MESSAGELEN);
			break;
		case PUZZLE_SPACE:
#if 0
			/* Too annoying */
			(void) strncpy(messageDsp, "A space can not slide",
				MESSAGELEN);
#endif
			break;
		case PUZZLE_IGNORE:
			(void) strncpy(messageDsp, "Clear to start",
				MESSAGELEN);
			break;
		case PUZZLE_MOVED:
			movesDsp++;
			SET_STARTED(w, True);
			break;
		case PUZZLE_SOLVED:
			if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", userNameDsp);
			else if (!randomized && HandleSolved(movesDsp,
					sizeX, sizeY, sizeZ))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", userNameDsp);
			else
				(void) strncpy(messageDsp, "Solved!",
					MESSAGELEN);
			SET_STARTED(w, False);
			break;
		case PUZZLE_RANDOMIZE:
			movesDsp = 0;
			randomized = True;
			SET_STARTED(w, False);
			break;
		case PUZZLE_DEC_X:
			movesDsp = 0;
			sizeX--;
			PrintRecord(sizeX, sizeY, sizeZ);
#ifdef WINVER
			w->cubes.sizeX = sizeX;
#else
			XtVaSetValues(w, XtNsizeX, sizeX, NULL);
#ifdef HAVE_MOTIF
			XmScaleSetValue(blockX, sizeX);
			if (sizeX >= MAXCUBES)
				XtVaSetValues(blockX, XmNmaximum, sizeX, NULL);
#endif
#endif
			break;
		case PUZZLE_INC_X:
			movesDsp = 0;
			sizeX++;
			PrintRecord(sizeX, sizeY, sizeZ);
#ifdef WINVER
			w->cubes.sizeX = sizeX;
#else
			XtVaSetValues(w, XtNsizeX, sizeX, NULL);
#ifdef HAVE_MOTIF
			if (sizeX > MAXCUBES)
				XtVaSetValues(blockX, XmNmaximum, sizeX, NULL);
			XmScaleSetValue(blockX, sizeX);
#endif
#endif
			break;
		case PUZZLE_DEC_Y:
			movesDsp = 0;
			sizeY--;
			PrintRecord(sizeX, sizeY, sizeZ);
#ifdef WINVER
			w->cubes.sizeY = sizeY;
#else
			XtVaSetValues(w, XtNsizeY, sizeY, NULL);
#ifdef HAVE_MOTIF
			XmScaleSetValue(blockY, sizeY);
			if (sizeY >= MAXCUBES)
				XtVaSetValues(blockY, XmNmaximum, sizeY, NULL);
#endif
#endif
			break;
		case PUZZLE_INC_Y:
			movesDsp = 0;
			sizeY++;
			PrintRecord(sizeX, sizeY, sizeZ);
#ifdef WINVER
			w->cubes.sizeY = sizeY;
#else
			XtVaSetValues(w, XtNsizeY, sizeY, NULL);
#ifdef HAVE_MOTIF
			if (sizeY > MAXCUBES)
				XtVaSetValues(blockY, XmNmaximum, sizeY, NULL);
			XmScaleSetValue(blockY, sizeY);
#endif
#endif
			break;
		case PUZZLE_DEC_Z:
			movesDsp = 0;
			sizeZ--;
			PrintRecord(sizeX, sizeY, sizeZ);
#ifdef WINVER
			w->cubes.sizeZ = sizeZ;
#else
			XtVaSetValues(w, XtNsizeZ, sizeZ, NULL);
#ifdef HAVE_MOTIF
			XmScaleSetValue(blockZ, sizeZ);
			if (sizeZ >= MAXCUBES)
				XtVaSetValues(blockZ, XmNmaximum, sizeZ, NULL);
#endif
#endif
			break;
		case PUZZLE_INC_Z:
			movesDsp = 0;
			sizeZ++;
			PrintRecord(sizeX, sizeY, sizeZ);
#ifdef WINVER
			w->cubes.sizeZ = sizeZ;
#else
			XtVaSetValues(w, XtNsizeZ, sizeZ, NULL);
#ifdef HAVE_MOTIF
			if (sizeZ > MAXCUBES)
				XtVaSetValues(blockZ, XmNmaximum, sizeZ, NULL);
			XmScaleSetValue(blockZ, sizeZ);
#endif
#endif
			break;
		case PUZZLE_COMPUTED:
			SET_STARTED(w, False);
			break;
		case PUZZLE_UNDO:
			movesDsp--;
			SET_STARTED(w, True);
			break;
		case PUZZLE_REDO:
			movesDsp++;
			SET_STARTED(w, True);
			break;
#ifdef HAVE_MOTIF
		case PUZZLE_SPEED:
			{
				int oldDelay, delay;

				XtVaGetValues(cubes, XtNdelay, &delay, NULL);
				oldDelay = delay;
				if (delay > MAXSPEED - MINSPEED)
					delay = MAXSPEED - MINSPEED;
				if (delay < 0)
					delay = 0;
				if (delay != oldDelay) {
					XtVaSetValues(w,
						XtNdelay, delay, NULL);
				}
				XmScaleSetValue(speed,
					MAXSPEED + MINSPEED - delay - 1);
			}
			return;
#endif
	}
	PrintStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, sizeX, sizeY, sizeZ
#endif
		);
}

#ifdef WINVER
static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.cubes.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			EnterPuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.cubes.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			LeavePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyPuzzle(&widget, brush);
			break;
		case WM_SIZE:
			ResizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			ExposePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			ClearPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			SelectPuzzle(&widget, LOWORD(lParam), HIWORD(lParam));
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			ReleasePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
		case WM_MOUSEWHEEL:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			{
				int zDelta = ((short) HIWORD(wParam));
				POINT cursor, origin;

				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				if (zDelta > (WHEEL_DELTA >> 1)) {
					MovePuzzle(&widget, TOP,
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
					if (GetKeyState(VK_CONTROL) >> 1) {
						SizePuzzle(&widget);
						(void) InvalidateRect(hWnd, NULL, TRUE);
					}
				} else if (zDelta < -(WHEEL_DELTA >> 1)) {
					MovePuzzle(&widget, BOTTOM,
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
					if (GetKeyState(VK_CONTROL) >> 1) {
						SizePuzzle(&widget);
						(void) InvalidateRect(hWnd, NULL, TRUE);
					}
				}
			}
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#endif
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDM_GET:
					GetPuzzle(&widget);
					ResizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WritePuzzle(&widget);
					break;
				case IDM_EXIT:
					DestroyPuzzle(&widget, brush);
					break;
				case IDM_HIDE:
					HidePuzzle(&widget);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					UndoPuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_REDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					RedoPuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_CLEAR:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					ClearPuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_RANDOMIZE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					RandomizePuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					SolvePuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_SPEED:
					SpeedPuzzle(&widget);
					break;
				case IDM_SLOW:
					SlowPuzzle(&widget);
					break;
				case IDM_SOUND:
					SoundPuzzle(&widget);
					break;
				case IDM_UP:
				case IDM_RIGHT:
				case IDM_DOWN:
				case IDM_LEFT:
				case IDM_IN:
				case IDM_OUT:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					(void) MovePuzzle(&widget, (int) LOWORD(wParam) - IDM_UP, FALSE);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_DECY:
				case IDM_INCX:
				case IDM_INCY:
				case IDM_DECX:
				case IDM_DECZ:
				case IDM_INCZ:
					if (MovePuzzle(&widget, (int) LOWORD(wParam) - IDM_DECY, TRUE)) {
						SizePuzzle(&widget);
						(void) InvalidateRect(hWnd, NULL, TRUE);
					}
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
						"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
						"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
						"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
#if 0
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
#else
		512 + 10, 512 + 10 + 34,
#endif
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
Usage(char *programName)
{
	unsigned int i;

	(void) fprintf(stderr, "usage: %s\n", programName);
	for (i = 0; i < strlen(optionsHelp); i++) {
		if (i == 0 || optionsHelp[i - 1] == '\n')
			(void) fprintf(stderr, "\t");
		(void) fprintf(stderr, "%c", (optionsHelp[i]));
	}
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*cubes.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*cubes.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*cubes.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*cubes.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*cubes.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*cubes.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*cubes.foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*cubes.foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*cubes.blockBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*cubes.blockBorder", XrmoptionSepArg, NULL},
	{(char *) "-frame", (char *) "*cubes.frameColor", XrmoptionSepArg, NULL},
	{(char *) "-block", (char *) "*cubes.blockColor", XrmoptionSepArg, NULL},
	{(char *) "-install", (char *) "*cubes.install", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noinstall", (char *) "*cubes.install", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-picture", (char *) "*cubes.picture", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*cubes.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*cubes.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*cubes.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-bumpSound", (char *) "*cubes.bumpSound", XrmoptionSepArg, NULL},
	{(char *) "-dripSound", (char *) "*cubes.dripSound", XrmoptionSepArg, NULL},
	{(char *) "-base", (char *) "*cubes.base", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*cubes.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*cubes.font", XrmoptionSepArg, NULL},
	{(char *) "-sizex", (char *) "*cubes.sizeX", XrmoptionSepArg, NULL},
	{(char *) "-sizey", (char *) "*cubes.sizeY", XrmoptionSepArg, NULL},
	{(char *) "-sizez", (char *) "*cubes.sizeZ", XrmoptionSepArg, NULL},
	{(char *) "-userName", (char *) "*cubes.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*cubes.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*cubes.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*cubes.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
CallbackPuzzleClear(Widget w, XtPointer clientData, XmAnyCallbackStruct *cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(cubes, XtNmenu, MENU_CLEAR, NULL);
	}
}

static void
BlockXSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int sizeX = cbs->value, sizeY, sizeZ, oldX;

	XtVaGetValues(cubes,
		XtNsizeX, &oldX,
		XtNsizeY, &sizeY,
		XtNsizeZ, &sizeZ, NULL);
	if (oldX == sizeX)
		return;
	XtVaSetValues(cubes,
		XtNsizeX, sizeX, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(sizeX, sizeY, sizeZ);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
BlockYSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int sizeX, sizeY = cbs->value, sizeZ, oldY;

	XtVaGetValues(cubes,
		XtNsizeX, &sizeX,
		XtNsizeY, &oldY,
		XtNsizeZ, &sizeZ, NULL);
	if (oldY == sizeY)
		return;
	XtVaSetValues(cubes,
		XtNsizeY, sizeY, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(sizeX, sizeY, sizeZ);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
BlockZSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int sizeX, sizeY, sizeZ = cbs->value, oldZ;

	XtVaGetValues(cubes,
		XtNsizeX, &sizeX,
		XtNsizeY, &sizeY,
		XtNsizeZ, &oldZ, NULL);
	if (oldZ == sizeZ)
		return;
	XtVaSetValues(cubes,
		XtNsizeZ, sizeZ, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(sizeX, sizeY, sizeZ);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
SpeedSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int delay = MAXSPEED + MINSPEED - cbs->value - 1, oldDelay;

	XtVaGetValues(cubes,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
		XtVaSetValues(cubes, XtNdelay, delay, NULL);
	}
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == MENU_EXIT)
		exit(0);
	XtVaSetValues(cubes, XtNmenu, val, NULL); /* GWQ */
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	XtVaSetValues(cubes, XtNmenu, val + MENU_EXIT + 1, NULL);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}

static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, blocksRowCol, speedRowCol, messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString undoString, redoString;
	XmString clearString, randomizeString, solveString;
	XmString speedString, slowString, soundString;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Cubes",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#ifdef HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) cubes_xpm, &icon, NULL,
			&xpmAttributes);
	}
	if (icon == (Pixmap) NULL)
#endif
		icon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) cubes_bits,
			cubes_width, cubes_height);
#ifdef HAVE_MOTIF
	XtVaSetValues(topLevel,
		XtNiconPixmap, icon,
		XmNkeyboardFocusPolicy, XmPOINTER, NULL); /* not XmEXPLICIT */
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	undoString = XmStringCreateSimple((char *) "Undo");
	redoString = XmStringCreateSimple((char *) "Redo");
	clearString = XmStringCreateSimple((char *) "Clear");
	randomizeString = XmStringCreateSimple((char *) "RandomiZe");
	solveString = XmStringCreateSimple((char *) "Solve");
	speedString = XmStringCreateSimple((char *) ">Speed");
	slowString = XmStringCreateSimple((char *) "<Slow");
	soundString = XmStringCreateSimple((char *) "@Sound");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, redoString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'Z', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
		XmVaPUSHBUTTON, soundString, '@', NULL, NULL,
		NULL);
	XmStringFree(undoString);
	XmStringFree(redoString);
	XmStringFree(clearString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(speedString);
	XmStringFree(slowString);
	XmStringFree(soundString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	clearDialog = createQuery(topLevel,
		(char *) "Are you sure you want to start over?",
		(char *) "Clear Query",
		(XtCallbackProc) CallbackPuzzleClear);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move block", 11, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Clear", 6, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

	blocksRowCol = XtVaCreateManagedWidget("blocksRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	blockX = XtVaCreateManagedWidget("blockX",
		xmScaleWidgetClass, blocksRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Blocks X", 9,
		XmNminimum, MINCUBES,
		XmNmaximum, MAXCUBES,
		XmNvalue, DEFAULTCUBESX,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(blockX, XmNvalueChangedCallback,
		(XtCallbackProc) BlockXSlider, (XtPointer) NULL);
	blockY = XtVaCreateManagedWidget("blockY",
		xmScaleWidgetClass, blocksRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Blocks Y", 9,
		XmNminimum, MINCUBES,
		XmNmaximum, MAXCUBES,
		XmNvalue, DEFAULTCUBESY,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(blockY, XmNvalueChangedCallback,
		(XtCallbackProc) BlockYSlider, (XtPointer) NULL);
	blockZ = XtVaCreateManagedWidget("blockZ",
		xmScaleWidgetClass, blocksRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Blocks Z", 9,
		XmNminimum, MINCUBES,
		XmNmaximum, MAXCUBES,
		XmNvalue, DEFAULTCUBESZ,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(blockZ, XmNvalueChangedCallback,
		(XtCallbackProc) BlockZSlider, (XtPointer) NULL);
	speedRowCol = XtVaCreateManagedWidget("speedRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	speed = XtVaCreateManagedWidget("Animation Speed",
		xmScaleWidgetClass, speedRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MINSPEED,
		XmNmaximum, MAXSPEED,
		XmNvalue, MAXSPEED - DEFAULTDELAY,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speed, XmNvalueChangedCallback,
		(XtCallbackProc) SpeedSlider, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Cubes! (use mouse or keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);
	cubes = XtCreateManagedWidget("cubes",
		cubesWidgetClass, mainPanel, NULL, 0);
#else
	XtVaSetValues(topLevel,
		XtNiconPixmap, icon,
		XtNinput, True, NULL);
	cubes = XtCreateManagedWidget("cubes",
		cubesWidgetClass, topLevel, NULL, 0);
#endif
	XtAddCallback(cubes, XtNselectCallback,
		(XtCallbackProc) CallbackPuzzle, (XtPointer) NULL);
	Initialize(cubes);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(cubes), (unsigned int) AnyButton, AnyModifier,
		XtWindow(cubes), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(cubes),
		XCreateFontCursor(XtDisplay(cubes), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
