/*-
# MOTIF/X-BASED DINOSAUR CUBE
#
#  xdino.c
#
###
#
#  Copyright (c) 1995 - 2006	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/12/15 Xt/Motif
  Version 5: 95/10/06 Xt/Motif
  Version 4: 94/05/30 Xt
*/

#include "file.h"
#ifdef WINVER
#include "DinoP.h"
#include "wdino.h"
#define TITLE "wdino"

static DinoRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "c:/WINDOWS"
#else
#define SCOREPATH "c:\\WINDOWS"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->dino.started = b
#else
#include "xwin.h"
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/ToggleB.h>
#include <Xm/ToggleBG.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Dino.h"
#include "Dino2d.h"
#include "Dino3d.h"
#ifdef HAVE_OPENGL
#include "DinoGL.h"
#endif
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/dino.xpm"
#endif
#include "icons/dino.xbm"
#ifndef SCOREPATH
#ifdef VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef WINVER
static const char aboutHelp[] = {
"Dino Version 7.2.4\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd@tux.org>\n"
"The latest version is at: "
"http://www.tux.org/~bagleyd/puzzles.html\n"
};

static const char optionsHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-face{0|1|2|3|4|5} {color}] [-{border|bd} {color}]\n"
"[-{font|fn} {fontname}] [-view {int}]\n"
"[-{mode {int} | both}] [-[no]orient] [-[no]practice]\n"
"[-userName {string}] [-scoreFile {filename}] [-scores]\n"
"[-version]\n"
};
#endif

#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
"The period 2 turning (i.e. the edges turn with 180 "
"degree intervals) was suggested by Derek Bosch\n"
"<bosch@sgi.com>.  The physical puzzle has period 3 "
"turning (i.e. the points turn with 120 degree\n"
"intervals) is called a Triangle - 4 Cube - Dinosaur "
"with colored dinosaurs on it, (or Triangle - 4 Cube -\n"
"6 Colors with solid colors, (a 4 color and a 2 color "
"are also available with point oriented coloring))\n"
"manufactured by The International Puzzle & Games "
"(Great Animation International Corporation).\n"
};

static const char featuresHelp[] = {
"Press \"mouse-left\" button to move a piece.  Release "
"\"mouse-left\" button on a piece on the same face.\n"
"The pieces will then turn towards where the mouse "
"button was released.\n"
"\n"
"Click \"mouse-center\" button, or press \"P\" or \"p\" "
"keys, to toggle the practice mode (in practice mode the\n"
"record should say \"practice\").  This is good for learning "
"moves and experimenting.\n"
"\n"
"Click \"mouse-right\" button, or press \"Z\" or \"z\" "
"keys, to randomize the puzzle (this must be done first\n"
"to set a new record).\n"
"\n"
"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save (write) a puzzle.\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"R\" or \"r\" keys to redo a move.\n"
"\n"
"Press \"C\" or \"c\" keys to clear the puzzle.\n"
"\n"
"\"S\" or \"s\" keys reserved for the auto-solver "
"(not implemented).\n"
"\n"
"Press \"O\" or \"o\" keys to toggle the orient mode.  One "
"has to orient the faces in orient mode, besides\n"
"getting all the faces to be the same color.  To do this "
"one has to get the lines to be oriented in\n"
"the same direction.  This does add complexity (ever so "
"slightly, there are only 2 possibilities) so there\n"
"are 2 sets of records.\n"
"\n"
"Press \"2\", \"3\", \"B\", or \"b\" keys (not the keypad "
"2, 3) to change modes to Period 2, Period 3, or Both.\n"
"\n"
"Press \"V\" or \"v\" keys to change the view of the cube.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Use the key pad or arrow keys to move without the mouse.\n"
"Key pad is defined for Dino2d as:\n"
"7 8 9  Upper Left, Up, Upper Right\n"
"  ^\n"
"4< >6  Left, Right\n"
"  v\n"
"1 2 3  Lower Left, Down, Lower Right\n"
"Note: Up, Left, Right, and Down only work when the "
"control key is pressed and there is no analog for\n"
"Dino3d.\n"
"Key pad for Dino3d, use must use your intuition (is this "
"a cop out or what?).  The key pad is defined\n"
"differently depending on which side of the cube your mouse "
"is pointing at.\n"
"\n"
"Use the alt key and the left mouse button, keypad, or arrow "
"keys to move the center of the cube.  The 2 opposite\n"
"corners do not move.\n"
"\n"
"Use the shift keys and the left mouse button, keypad, "
"or arrow key to access \"Period 2\" turns from \"Both\" mode,\n"
"otherwise it assumes \"Period 3\" turning.  Edges turn in "
"\"Period 2\" and corners turn in \"Period 3\".  The\n"
"\"Period 2\" mode has extra cuts around the faces.\n"
"\n"
"Use the control key and the left mouse button, keypad, or "
"arrow keys to move the whole cube.  This is\n"
"not recorded as a turn.\n"
};

static const char referencesHelp[] = {
"International Puzzles & Games Catalog.\n"
};
#endif

static const char solveHelp[] = {
"Auto-solver: sorry, not implemented.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "dino.scores"
#endif

#define NEVER (-1)
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} GameRecord;

static GameRecord dinoRecord[MAXMODES][2];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";
#ifndef HAVE_MOTIF
static char titleDsp[TITLELEN] = "";
#endif
static char scoreFileName[FILENAMELEN] = SCOREFILE;
static char fileName[FILENAMELEN];
static Boolean randomized = False;
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char userNameDsp[USERNAMELEN] = "Guest";
#else
#ifdef HAVE_MOTIF
static Widget moves, record, message, modes[MAXMODES];
static Widget orientSwitch, practiceSwitch;
static char buff[21];
static const char *modeString[] =
{
	"Period 2", "Period 3", "Both"
};
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, practiceDialog, randomizeDialog;
static Arg arg[3];
#else
static Widget shell;
#ifdef HAVE_OPENGL
static Widget shellGL;
#endif
#endif
static Pixmap icon = None;
static Widget topLevel, dino2d, dino3d;
#ifdef HAVE_OPENGL
static Widget dinoGL;
#endif
static char *progDsp;
static char userNameDsp[USERNAMELEN] = "";

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w, XmNlabelString, xmstr, NULL);
}
#endif

static void
PrintRecords(void)
{
	int mode, orient;

	(void) printf("          DINO  HALL OF FAME\n\n");
	(void) printf("MODE ORIENT USER            MOVES\n");
	for (mode = 0; mode < MAXMODES; mode++)
		for (orient = 0; orient < 2; orient++) {
			if (dinoRecord[mode][orient].score > 0)
				(void) printf("%4d%7d %-12s%9d\n",
					mode + 2, orient,
					dinoRecord[mode][orient].name,
					dinoRecord[mode][orient].score);
		}
}
#endif

static void
InitRecords(void)
{
	int mode, orient;

	for (mode = 0; mode < MAXMODES; mode++)
		for (orient = 0; orient < 2; orient++) {
			dinoRecord[mode][orient].score = NEVER;
			(void) strncpy(dinoRecord[mode][orient].name, NOACCESS,
				USERNAMELEN);
		}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int n, mode, orient;
	char userName[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USERNAMELEN);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USERNAMELEN);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (mode = 0; mode < MAXMODES; mode++)
		for (orient = 0; orient < 2; orient++) {
			(void) fscanf(fp, "%d %s\n", &n, userName);
			if (n <= dinoRecord[mode][orient].score ||
					dinoRecord[mode][orient].score <= NEVER) {
				dinoRecord[mode][orient].score = n;
				(void) strncpy(dinoRecord[mode][orient].name,
					userName, USERNAMELEN);
			}
		}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int mode, orient;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Can not write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILENAMELEN];

		(void) strncpy(lockFile, fileName, FILENAMELEN - 6);
		(void) strcat(lockFile, ".lock");
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (mode = 0; mode < MAXMODES; mode++) {
			for (orient = 0; orient < 2; orient++)
				(void) fprintf(fp, "%d %s\n",
					dinoRecord[mode][orient].score,
					dinoRecord[mode][orient].name);
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
PrintRecord(int mode, Boolean orient, Boolean practice)
{
	int i = mode - PERIOD2;
	int j = (orient) ? 1 : 0;

	if (practice) {
		(void) strncpy(recordDsp, "practice", MESSAGELEN);
	} else if (dinoRecord[i][j].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			dinoRecord[i][j].score, dinoRecord[i][j].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

#ifndef WINVER
/* There's probably a better way to assure that they are the same but I don't
   know it off hand. */
static void
MakeEquivalent(String *userName, String *scoreFile, int *mode,
		Boolean *orient, Boolean *practice)
{
	Boolean mono, reverse;
	Boolean scoreOnly, versionOnly;
	Pixel foreground, background, pieceBorder;
	String faceColor[MAXFACES];

	XtVaGetValues(dino2d,
		XtNuserName, userName,
		XtNscoreFile, scoreFile,
		XtNmode, mode,
		XtNorient, orient,
		XtNpractice, practice,
		XtNmono, &mono,
		XtNreverse, &reverse,
		XtNforeground, &foreground,
		XtNbackground, &background,
		XtNpieceBorder, &pieceBorder,
		XtNfaceColor0, &(faceColor[0]),
		XtNfaceColor1, &(faceColor[1]),
		XtNfaceColor2, &(faceColor[2]),
		XtNfaceColor3, &(faceColor[3]),
		XtNfaceColor4, &(faceColor[4]),
		XtNfaceColor5, &(faceColor[5]),
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf(aboutHelp);
		exit(0);
	}
	if (strcmp(*scoreFile, ""))
		(void) strncpy(scoreFileName, *scoreFile, FILENAMELEN);
	if (scoreOnly) {
		InitRecords();
		ReadRecords();
		PrintRecords();
		exit(0);
	}
	XtVaSetValues(dino2d,
		XtNdirection, PUZZLE_IGNORE,
		XtNstart, False, NULL);
	XtVaSetValues(dino3d,
		XtNuserName, *userName,
		XtNmode, *mode,
		XtNorient, *orient,
		XtNpractice, *practice,
		XtNmono, mono,
		XtNreverse, reverse,
		XtNdirection, PUZZLE_IGNORE,
		XtNstart, False,
		XtNforeground, foreground,
		XtNbackground, background,
		XtNpieceBorder, pieceBorder,
		XtNfaceColor0, faceColor[0],
		XtNfaceColor1, faceColor[1],
		XtNfaceColor2, faceColor[2],
		XtNfaceColor3, faceColor[3],
		XtNfaceColor4, faceColor[4],
		XtNfaceColor5, faceColor[5], NULL);
#ifdef HAVE_OPENGL
	XtVaSetValues(dinoGL,
		XtNuserName, *userName,
		XtNorient, *orient,
		XtNpractice, *practice,
		XtNmono, mono,
		XtNreverse, reverse,
		XtNdirection, PUZZLE_IGNORE,
		XtNstart, False,
		XtNforeground, foreground,
		XtNbackground, background,
		XtNpieceBorder, pieceBorder,
		XtNfaceColor0, faceColor[0],
		XtNfaceColor1, faceColor[1],
		XtNfaceColor2, faceColor[2],
		XtNfaceColor3, faceColor[3],
		XtNfaceColor4, faceColor[4],
		XtNfaceColor5, faceColor[5], NULL);
#endif
}
#endif

static void
PrintStatus(char *msg, int nMoves
#ifndef HAVE_MOTIF
		, int dim, int mode
#endif
		)
{
#ifdef HAVE_MOTIF
	PrintState(message, msg);
	(void) sprintf(buff, "%d", nMoves);
	PrintState(moves, buff);
#else
	char mb[10];

	if (mode == BOTH)
		(void) strcpy(mb, "both");
	else
		(void) sprintf(mb, "%d", mode);
#ifdef HAVE_OPENGL
	if (dim == 4)
		(void) sprintf(titleDsp, "%sGL.%s: (%d/%s) - %s",
			progDsp, mb, nMoves, recordDsp, msg);
	else
#endif
		(void) sprintf(titleDsp, "%s%dd.%s: (%d/%s) - %s",
			progDsp, dim, mb, nMoves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
#ifdef HAVE_OPENGL
	if (dim == 4)
		XtVaSetValues(XtParent(dinoGL), XtNtitle, titleDsp, NULL);
	else
#endif
	XtVaSetValues(XtParent((dim == 2) ? dino2d : dino3d),
		XtNtitle, titleDsp, NULL);
#endif
#endif
}

static Boolean
HandleSolved(int counter, int mode, Boolean orient)
{
	int i = mode - PERIOD2;
	int j = (orient) ? 1 : 0;

	if (counter < dinoRecord[i][j].score ||
			dinoRecord[i][j].score <= NEVER) {
		ReadRecords();	/* Maybe its been updated by another */
		dinoRecord[i][j].score = counter;
		(void) strncpy(dinoRecord[i][j].name, userNameDsp,
			USERNAMELEN);
		if (orient && (counter < dinoRecord[!i][j].score ||
				dinoRecord[!i][j].score <= NEVER)) {
			dinoRecord[!i][j].score = counter;
			(void) strncpy(dinoRecord[!i][j].name, userNameDsp,
				USERNAMELEN);
		}
		WriteRecords();
		PrintRecord(mode, orient, False);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
DinoWidget w, HBRUSH brush
#else
void
#endif
)
{
	int mode;
	Boolean orient, practice;
	char *userName, *scoreFile;
#ifdef WINVER
	int dim;

	InitializePuzzle(w, brush);

	mode = w->dino.mode;
	orient = w->dino.orient;
	practice = w->dino.practice;
	userName = w->dino.userName;
	scoreFile = w->dino.scoreFile;
	dim = w->dino.dim;
	SET_STARTED(w, False);
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILENAMELEN);
#else
	MakeEquivalent(&userName, &scoreFile, &mode, &orient, &practice);
#ifdef HAVE_MOTIF
	XmToggleButtonSetState(modes[mode - PERIOD2], True, False);
	XmToggleButtonSetState(orientSwitch, orient, True);
	XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
#endif
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USERNAMELEN);
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USERNAMELEN);
#else
		char *login = getlogin();

		if (login == NULL) {
			(void) strcpy(userNameDsp, "");
		} else {
			(void) sprintf(userNameDsp, "%s", login);
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(userNameDsp, "%s", "guest");
#endif
	}
	PrintRecord(mode, orient, practice);
#ifdef WINVER
	PrintStatus(messageDsp, movesDsp, dim, mode);
#else
#ifdef HAVE_MOTIF
	PrintStatus(messageDsp, movesDsp);
#else
	PrintStatus(messageDsp, movesDsp, 2, mode);
	PrintStatus(messageDsp, movesDsp, 3, mode);
#ifdef HAVE_OPENGL
	PrintStatus(messageDsp, movesDsp, 4, mode);
#endif
#endif
#endif
}

#ifdef WINVER
void
setPuzzle(DinoWidget w, int reason)
#else
static void
CallbackPuzzle(Widget w, caddr_t clientData, dinoCallbackStruct *callData)
#endif
{
	int mode;
	Boolean orient, practice, start, cheat;
#ifdef WINVER
	int dim = 0;
#else
	int reason = callData->reason;
	Widget otherw1 = (Widget) NULL;
#ifdef HAVE_OPENGL
	Widget otherw2 = (Widget) NULL;
#endif
#ifndef HAVE_MOTIF
	int dim = 0, otherdim1 = 0;
#ifdef HAVE_OPENGL
	int otherdim2 = 0;
#endif
#endif

	if (w == dino2d) {
		otherw1 = dino3d;
#ifdef HAVE_OPENGL
		otherw2 = dinoGL;
#endif
#ifndef HAVE_MOTIF
		dim = 2;
		otherdim1 = 3;
#ifdef HAVE_OPENGL
		otherdim2 = 4;
#endif
#endif
	} else if (w == dino3d) {
		otherw1 = dino2d;
#ifdef HAVE_OPENGL
		otherw2 = dinoGL;
#endif
#ifndef HAVE_MOTIF
		dim = 3;
		otherdim1 = 2;
#ifdef HAVE_OPENGL
		otherdim2 = 4;
#endif
#endif
	}
#ifdef HAVE_OPENGL
	else if (w == dinoGL) {
		otherw1 = dino2d;
		otherw2 = dino3d;
#ifndef HAVE_MOTIF
		dim = 4;
		otherdim1 = 2;
		otherdim2 = 3;
#endif
	}
#endif
#endif
	(void) strcpy(messageDsp, "");
#ifdef WINVER
	mode = w->dino.mode;
	orient = w->dino.orient;
	practice = w->dino.practice;
	cheat = w->dino.cheat;
	start = w->dino.started;
	dim = w->dino.dim;
#else
	XtVaGetValues(w,
		XtNorient, &orient,
		XtNmode, &mode,
		XtNpractice, &practice,
		XtNstart, &start,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case PUZZLE_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#ifndef HAVE_MOTIF
			(void) XIconifyWindow(XtDisplay(shell),
				XtWindow(shell),
				XScreenNumberOfScreen(XtScreen(shell)));
#ifdef HAVE_OPENGL
			(void) XIconifyWindow(XtDisplay(shellGL),
				XtWindow(shellGL),
				XScreenNumberOfScreen(XtScreen(shellGL)));
#endif
#endif
#endif
			break;
#ifndef WINVER
		case PUZZLE_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(practiceDialog);
#else
			XtVaSetValues(dino2d, XtNmenu, MENU_PRACTICE, NULL);
#endif
			break;
		case PUZZLE_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(randomizeDialog);
#else
			XtVaSetValues(dino2d, XtNmenu, MENU_RANDOMIZE, NULL);
#endif
			break;
#endif
		case PUZZLE_SOLVE_MESSAGE:
#ifdef WINVER
			(void) MessageBox(w->core.hWnd, solveHelp,
				"Auto-Solve", MB_OK);
#else
#ifdef HAVE_MOTIF
			XtManageChild(solveDialog);
#else
			(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
			break;
		case PUZZLE_RESTORE:
			if (practice) {
				(void) strncpy(recordDsp, "practice",
					MESSAGELEN);
#ifdef HAVE_MOTIF
				PrintState(record, recordDsp);
#endif
			}
			movesDsp = 0;
			randomized = False;
#ifndef WINVER
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNdirection, PUZZLE_RESTORE, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNdirection, PUZZLE_RESTORE, NULL);
			XtVaSetValues(w,
				XtNdirection, PUZZLE_RESTORE, NULL);
#endif
			break;
		case PUZZLE_CLEAR:
			movesDsp = 0;
			randomized = False;
#ifndef WINVER
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNdirection, PUZZLE_CLEAR, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNdirection, PUZZLE_CLEAR, NULL);
			XtVaSetValues(w,
				XtNdirection, PUZZLE_CLEAR, NULL);
#endif
			break;
		case PUZZLE_RESET:
			movesDsp = 0;
			randomized = False;
			break;
		 case PUZZLE_AMBIGUOUS:
			(void) strncpy(messageDsp, "Ambiguous move",
				MESSAGELEN);
			break;
		case PUZZLE_ILLEGAL:
			if (practice || randomized)
				(void) strncpy(messageDsp, "Illegal move",
					MESSAGELEN);
			else
				(void) strncpy(messageDsp,
					"Randomize to start", MESSAGELEN);
			break;
#ifndef WINVER
		case PUZZLE_MOVED:
			movesDsp++;
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNstart, True,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNstyle, callData->style, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNstart, True,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNstyle, callData->style, NULL);
			SET_STARTED(w, True);
			break;
		case PUZZLE_CONTROL:
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNstyle, callData->style, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNstyle, callData->style, NULL);
			break;
#endif
		case PUZZLE_SOLVED:
			if (practice)
				movesDsp = 0;
			else if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", userNameDsp);
			else if (HandleSolved(movesDsp, mode, orient))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", userNameDsp);
			else
				(void) strncpy(messageDsp, "Solved!",
					MESSAGELEN);
			SET_STARTED(w, False);
#ifndef WINVER
#ifdef HAVE_OPENGL
			SET_STARTED(otherw2, False);
#endif
			SET_STARTED(otherw1, False);
#endif
			randomized = False;
			break;
		case PUZZLE_PRACTICE:
			movesDsp = 0;
			randomized = False;
			practice = !practice;
			if (!practice)
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			PrintRecord(mode, orient, practice);
#ifdef WINVER
			w->dino.practice = practice;
			w->dino.started = False;
#else
			XtVaSetValues(w,
				XtNpractice, practice,
				XtNstart, False, NULL);
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNpractice, practice,
				XtNstart, False, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNpractice, practice,
				XtNstart, False, NULL);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
#endif
			break;
		case PUZZLE_RANDOMIZE:
			movesDsp = 0;
			randomized = True;
#ifdef WINVER
			w->dino.practice = False;
			w->dino.started = False;
#else
			XtVaSetValues(w,
				XtNpractice, False,
				XtNstart, False, NULL);
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNpractice, False,
				XtNstart, False, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNpractice, False,
				XtNstart, False, NULL);
#endif
			break;
#ifdef HAVE_OPENGL
		case PUZZLE_VIEW:
			{
				int view;

				XtVaGetValues(dinoGL,
					XtNview, &view, NULL);
				view = (view + 1) % (2 * MAXVIEWS);
				XtVaSetValues(dinoGL,
					XtNview, view, NULL);
			}
			break;
#endif
		case PUZZLE_ORIENTIZE:
			movesDsp = 0;
			orient = !orient;
			PrintRecord(mode, orient, practice);
#ifdef WINVER
			w->dino.orient = orient;
#else
			XtVaSetValues(w,
				XtNorient, orient, NULL);
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNorient, orient, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNorient, orient, NULL);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(orientSwitch, orient, True);
#endif
#endif
			break;
		case PUZZLE_PERIOD2:
		case PUZZLE_PERIOD3:
		case PUZZLE_BOTH:
			movesDsp = 0;
			mode = reason - PUZZLE_PERIOD2 + PERIOD2;
			PrintRecord(mode, orient, practice);
#ifdef WINVER
			w->dino.mode = mode;
#else
			XtVaSetValues(w,
				XtNmode, mode, NULL);
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNmode, mode, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNmode, mode, NULL);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(modes[mode - PERIOD2], True, True);
#endif
#endif
			break;
		case PUZZLE_COMPUTED:
			SET_STARTED(w, False);
#ifndef WINVER
#ifdef HAVE_OPENGL
			SET_STARTED(otherw2, False);
#endif
			SET_STARTED(otherw1, False);
#endif
			break;
#ifndef WINVER
		case PUZZLE_UNDO:
			movesDsp--;
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNstart, True,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNstyle, callData->style, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNstart, True,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNstyle, callData->style, NULL);
			SET_STARTED(w, False);
			break;
#endif
#ifndef WINVER
		case PUZZLE_REDO:
			movesDsp++;
#ifdef HAVE_OPENGL
			XtVaSetValues(otherw2,
				XtNstart, True,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNstyle, callData->style, NULL);
#endif
			XtVaSetValues(otherw1,
				XtNstart, True,
				XtNface, callData->face,
				XtNpos, callData->position,
				XtNdirection, callData->direction,
				XtNstyle, callData->style, NULL);
			SET_STARTED(w, True);
			break;
#endif
#ifdef WINVER
		case PUZZLE_DIM:
			dim++;
#ifdef HAVE_OPENGL
			if (dim > 4)
#else
			if (dim > 3)
#endif
				dim = 2;
			w->dino.dim = dim;
			break;
#endif
	}
#ifdef WINVER
	PrintStatus(messageDsp, movesDsp, dim, mode);
#else
#ifdef HAVE_MOTIF
	PrintStatus(messageDsp, movesDsp);
#else
	PrintStatus(messageDsp, movesDsp, dim, mode);
	PrintStatus(messageDsp, movesDsp, otherdim1, mode);
#ifdef HAVE_OPENGL
	PrintStatus(messageDsp, movesDsp, otherdim2, mode);
#endif
#endif
#endif
}

#ifdef WINVER
void
setPuzzleMove(DinoWidget w, int reason, int face, int position, int direction,
		int style)
{
	int mode;
	/* Boolean orient, practice, cheat; */
	int dim = 0;

	(void) strcpy(messageDsp, "");
	mode = w->dino.mode;
#if 0
	orient = w->dino.orient;
	practice = w->dino.practice;
	cheat = w->dino.cheat;
#endif
	dim = w->dino.dim;
	switch (reason) {
		case PUZZLE_MOVED:
			movesDsp++;
			SET_STARTED(w, True);
			break;
		case PUZZLE_CONTROL:
			break;
		case PUZZLE_UNDO:
			movesDsp--;
			SET_STARTED(w, True);
			break;
		case PUZZLE_REDO:
			movesDsp++;
			SET_STARTED(w, True);
			break;
	}
	PrintStatus(messageDsp, movesDsp, dim, mode);
}

static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;
	int shift = 0;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.dino.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			EnterPuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.dino.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			LeavePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyPuzzle(brush);
			break;
		case WM_SIZE:
			ResizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			ExposePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			RandomizePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			SelectPuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			ReleasePuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0,
				(GetKeyState(VK_TAB) >> 1) ? 1 : 0);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
		case WM_MOUSEWHEEL:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			{
				int zDelta = ((short) HIWORD(wParam));
				POINT cursor, origin;

				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				if (zDelta > (WHEEL_DELTA >> 1)) {
					MovePuzzleInput(&widget,
						cursor.x - origin.x, cursor.y - origin.y,
						TOP,
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0,
				(GetKeyState(VK_TAB) >> 1) ? 1 : 0);
				} else if (zDelta < -(WHEEL_DELTA >> 1)) {
					MovePuzzleInput(&widget,
						cursor.x - origin.x, cursor.y - origin.y,
						BOTTOM,
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0,
				(GetKeyState(VK_TAB) >> 1) ? 1 : 0);
				}
			}
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#endif
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDM_GET:
					GetPuzzle(&widget);
					ResizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WritePuzzle(&widget);
					break;
				case IDM_EXIT:
					DestroyPuzzle(brush);
					break;
				case IDM_HIDE:
					HidePuzzle(&widget);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					UndoPuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_REDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					RedoPuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_CLEAR:
					ClearPuzzle(&widget);
					SizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_RANDOMIZE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					RandomizePuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_PRACTICE:
					PracticePuzzle(&widget);
					SizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC,
						GetStockObject(NULL_PEN));
					SolvePuzzle(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_ORIENTIZE:
					OrientizePuzzle(&widget);
					SizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DIM:
					(void) DimPuzzle(&widget);
					ResizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SHIFT_TR:
				case IDM_SHIFT_BR:
				case IDM_SHIFT_BL:
				case IDM_SHIFT_TL:
				case IDM_SHIFT_CW:
				case IDM_SHIFT_CCW:
				case IDM_SHIFT_TOP:
				case IDM_SHIFT_RIGHT:
				case IDM_SHIFT_BOTTOM:
				case IDM_SHIFT_LEFT:
					shift = 1;
				case IDM_TR:
				case IDM_BR:
				case IDM_BL:
				case IDM_TL:
				case IDM_CW:
				case IDM_CCW:
				case IDM_TOP:
				case IDM_RIGHT:
				case IDM_BOTTOM:
				case IDM_LEFT:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC,
							GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						shift = shift || (GetKeyState(VK_CAPITAL) & 1);
						(void) MovePuzzleInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - ((shift) ? IDM_SHIFT_TR : IDM_TR),
							shift, FALSE, FALSE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}
					break;
				case IDM_SHIFTCONTROL_TR:
				case IDM_SHIFTCONTROL_BR:
				case IDM_SHIFTCONTROL_BL:
				case IDM_SHIFTCONTROL_TL:
				case IDM_SHIFTCONTROL_CW:
				case IDM_SHIFTCONTROL_CCW:
				case IDM_SHIFTCONTROL_TOP:
				case IDM_SHIFTCONTROL_RIGHT:
				case IDM_SHIFTCONTROL_BOTTOM:
				case IDM_SHIFTCONTROL_LEFT:
					shift = 1;
				case IDM_CONTROL_TR:
				case IDM_CONTROL_BR:
				case IDM_CONTROL_BL:
				case IDM_CONTROL_TL:
				case IDM_CONTROL_CW:
				case IDM_CONTROL_CCW:
				case IDM_CONTROL_TOP:
				case IDM_CONTROL_RIGHT:
				case IDM_CONTROL_BOTTOM:
				case IDM_CONTROL_LEFT:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC,
							GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MovePuzzleInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - ((shift) ? IDM_SHIFTCONTROL_TR : IDM_CONTROL_TR),
							shift, TRUE, FALSE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}
					break;
				case IDM_CONTROLALT_TR:
				case IDM_CONTROLALT_BR:
				case IDM_CONTROLALT_BL:
				case IDM_CONTROLALT_TL:
				case IDM_CONTROLALT_CW:
				case IDM_CONTROLALT_CCW:
				case IDM_CONTROLALT_TOP:
				case IDM_CONTROLALT_RIGHT:
				case IDM_CONTROLALT_BOTTOM:
				case IDM_CONTROLALT_LEFT:
					shift = 1;
				case IDM_ALT_TR:
				case IDM_ALT_BR:
				case IDM_ALT_BL:
				case IDM_ALT_TL:
				case IDM_ALT_CW:
				case IDM_ALT_CCW:
				case IDM_ALT_TOP:
				case IDM_ALT_RIGHT:
				case IDM_ALT_BOTTOM:
				case IDM_ALT_LEFT:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC,
							GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MovePuzzleInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - ((shift) ? IDM_CONTROLALT_TR : IDM_ALT_TR),
							FALSE, shift, TRUE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}
					break;
				case IDM_PERIOD2:
				case IDM_PERIOD3:
				case IDM_BOTH:
					PeriodModePuzzle(&widget, (int) LOWORD(wParam) - IDM_PERIOD2);
					SizePuzzle(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
						"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
						"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
						"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else
static void
Usage(char *programName)
{
	unsigned int i;

	(void) fprintf(stderr, "usage: %s\n", programName);
	for (i = 0; i < strlen(optionsHelp); i++) {
		if (i == 0 || optionsHelp[i - 1] == '\n')
			(void) fprintf(stderr, "\t");
		(void) fprintf(stderr, "%c", (optionsHelp[i]));
	}
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*dino.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*dino.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*dino.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*dino.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*dino.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*dino.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*dino.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*dino.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*dino.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*dino.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-face0", (char *) "*dino.faceColor0", XrmoptionSepArg, NULL},
	{(char *) "-face1", (char *) "*dino.faceColor1", XrmoptionSepArg, NULL},
	{(char *) "-face2", (char *) "*dino.faceColor2", XrmoptionSepArg, NULL},
	{(char *) "-face3", (char *) "*dino.faceColor3", XrmoptionSepArg, NULL},
	{(char *) "-face4", (char *) "*dino.faceColor4", XrmoptionSepArg, NULL},
	{(char *) "-face5", (char *) "*dino.faceColor5", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*dino.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*dino.font", XrmoptionSepArg, NULL},
	{(char *) "-view", (char *) "*dino.view", XrmoptionSepArg, NULL},
	{(char *) "-mode", (char *) "*dino.mode", XrmoptionSepArg, NULL},
	{(char *) "-both", (char *) "*dino.mode", XrmoptionNoArg, (char *) "4"},
	{(char *) "-orient", (char *) "*dino.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*dino.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-practice", (char *) "*dino.practice", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopractice", (char *) "*dino.practice", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-userName", (char *) "*dino.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*dino.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*dino.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*dino.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
CallbackPuzzlePractice(Widget w, XtPointer clientData, XmAnyCallbackStruct *cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(dino2d, XtNmenu, MENU_PRACTICE, NULL);
	}
}

static void
CallbackPuzzleRandomize(Widget w, XtPointer clientData, XmAnyCallbackStruct *cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(dino2d, XtNmenu, MENU_RANDOMIZE, NULL);
	}
}
static void
ModeToggle(Widget w, int mode, XmToggleButtonCallbackStruct *cbs)
{
	Boolean orient, practice;

	if (cbs->set) {
		XtVaGetValues(dino2d,
			XtNorient, &orient,
			XtNpractice, &practice, NULL);
		XtVaSetValues(dino2d,
			XtNmode, mode + PERIOD2, NULL);
		XtVaSetValues(dino3d,
			XtNmode, mode + PERIOD2, NULL);
#ifdef HAVE_OPENGL
		XtVaSetValues(dinoGL,
			XtNmode, mode + PERIOD2, NULL);
#endif
		movesDsp = 0;
		(void) sprintf(buff, "%d", movesDsp);
		PrintState(moves, buff);
		PrintRecord(mode + PERIOD2, orient, practice);
		(void) strcpy(messageDsp, "");
		PrintState(message, messageDsp);
	}
}

static void
OrientToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct *cbs)
{
	int mode;
	Boolean orient = cbs->set, practice;

	XtVaGetValues(dino2d,
		XtNmode, &mode,
		XtNpractice, &practice, NULL);
	XtVaSetValues(dino2d,
		XtNorient, orient, NULL);
	XtVaSetValues(dino3d,
		XtNorient, orient, NULL);
#ifdef HAVE_OPENGL
	XtVaSetValues(dinoGL,
		XtNorient, orient, NULL);
#endif
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(mode, orient, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
PracticeToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct *cbs)
{
	int mode;
	Boolean orient, practice = cbs->set;

	XtVaSetValues(dino2d,
		XtNpractice, practice,
		XtNstart, False, NULL);
	XtVaSetValues(dino3d,
		XtNpractice, practice,
		XtNstart, False, NULL);
#ifdef HAVE_OPENGL
	XtVaSetValues(dinoGL,
		XtNpractice, practice,
		XtNstart, False, NULL);
#endif
	XtVaGetValues(dino2d,
		XtNmode, &mode,
		XtNpractice, &orient, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(mode, orient, practice);
	if (practice)
		(void) strcpy(messageDsp, "");
	else
		(void) strncpy(messageDsp, "Randomize to start", MESSAGELEN);
	PrintState(message, messageDsp);
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == MENU_EXIT)
		exit(0);
	XtVaSetValues(dino2d, XtNmenu, val, NULL); /* GWQ */
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	XtVaSetValues(dino2d, XtNmenu, val + MENU_EXIT + 1, NULL);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, labelRowCol, radioRowCol, switchRowCol;
	Widget messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString undoString, redoString;
	XmString clearString, randomizeString, practiceString, solveString;
	XmString orientString, viewString;
	unsigned int i;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Dino",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#ifdef HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) dino_xpm, &icon, NULL,
			&xpmAttributes);
	}
	if (icon == (Pixmap) NULL)
#endif
		icon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) dino_bits, dino_width, dino_height);
#ifdef HAVE_MOTIF
	XtVaSetValues(topLevel,
		XtNiconPixmap, icon,
		XmNkeyboardFocusPolicy, XmPOINTER, NULL); /* not XmEXPLICIT */
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	undoString = XmStringCreateSimple((char *) "Undo");
	redoString = XmStringCreateSimple((char *) "Redo");
	clearString = XmStringCreateSimple((char *) "Clear");
	randomizeString = XmStringCreateSimple((char *) "RandomiZe");
	practiceString = XmStringCreateSimple((char *) "Practice");
	solveString = XmStringCreateSimple((char *) "(Solve)");
	orientString = XmStringCreateSimple((char *) "Orientize");
	viewString = XmStringCreateSimple((char *) "View");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, redoString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'Z', NULL, NULL,
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, orientString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, viewString, 'V', NULL, NULL,
		NULL);
	XmStringFree(undoString);
	XmStringFree(redoString);
	XmStringFree(clearString);
	XmStringFree(randomizeString);
	XmStringFree(practiceString);
	XmStringFree(solveString);
	XmStringFree(orientString);
	XmStringFree(viewString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	practiceDialog = createQuery(topLevel,
		(char *) "Are you sure you want to toggle the practice mode?",
		(char *) "Practice Query",
		(XtCallbackProc) CallbackPuzzlePractice);
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) CallbackPuzzleRandomize);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("Rowcol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move", 5, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

	labelRowCol = XtVaCreateManagedWidget("labelRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	(void) XtVaCreateManagedWidget("PeriodText",
		xmLabelGadgetClass, labelRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Periods:", 9, NULL);
	radioRowCol = XtVaCreateManagedWidget("radioRowCol",
		xmRowColumnWidgetClass, labelRowCol,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN,
		XmNradioBehavior, True, NULL);
	for (i = 0; i < XtNumber(modeString); i++) {
		modes[i] = XtVaCreateManagedWidget(modeString[i],
			xmToggleButtonGadgetClass, radioRowCol,
			XmNradioBehavior, True, NULL);
		XtAddCallback(modes[i], XmNvalueChangedCallback,
			(XtCallbackProc) ModeToggle, (XtPointer) i);
	}
	switchRowCol = XtVaCreateManagedWidget("switchRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	orientSwitch = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, switchRowCol, NULL);
	XtAddCallback(orientSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) OrientToggle, (XtPointer) NULL);
	practiceSwitch = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, switchRowCol, NULL);
	XtAddCallback(practiceSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) PracticeToggle, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	message = XtVaCreateManagedWidget("Play Dino! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);

	dino2d = XtCreateManagedWidget("dino",
		dino2dWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(dino2d,
		XtNheight, 200, NULL);
	XtAddCallback(dino2d, XtNselectCallback,
		(XtCallbackProc) CallbackPuzzle, (XtPointer) NULL);
	dino3d = XtCreateManagedWidget("dino",
		dino3dWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(dino3d,
		XtNheight, 200, NULL);
#ifdef HAVE_OPENGL
	dinoGL = XtCreateManagedWidget("dino",
		dinoGLWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(dinoGL,
		XtNheight, 200, NULL);
#endif
#else
	XtVaSetValues(topLevel,
		XtNiconPixmap, icon,
		XtNinput, True, NULL);
	shell = XtCreateApplicationShell(argv[0],
		topLevelShellWidgetClass, NULL, 0);
	XtVaSetValues(shell,
		XtNiconPixmap, icon,
		XtNinput, True, NULL);
#ifdef HAVE_OPENGL
	shellGL = XtCreateApplicationShell(argv[0],
		topLevelShellWidgetClass, NULL, 0);
	XtVaSetValues(shellGL,
		XtNiconPixmap, icon,
		XtNinput, True, NULL);
#endif
	dino2d = XtCreateManagedWidget("dino",
		dino2dWidgetClass, topLevel, NULL, 0);
	XtAddCallback(dino2d, XtNselectCallback,
		(XtCallbackProc) CallbackPuzzle, (XtPointer) NULL);
	dino3d = XtCreateManagedWidget("dino",
		dino3dWidgetClass, shell, NULL, 0);
#ifdef HAVE_OPENGL
	dinoGL = XtCreateManagedWidget("dino",
		dinoGLWidgetClass, shellGL, NULL, 0);
#endif
#endif
	XtAddCallback(dino3d, XtNselectCallback,
		(XtCallbackProc) CallbackPuzzle, (XtPointer) NULL);
#ifdef HAVE_OPENGL
	XtAddCallback(dinoGL, XtNselectCallback,
		(XtCallbackProc) CallbackPuzzle, (XtPointer) NULL);
#endif
	Initialize();
	XtRealizeWidget(topLevel);
#ifndef HAVE_MOTIF
	XtRealizeWidget(shell);
#ifdef HAVE_OPENGL
	XtRealizeWidget(shellGL);
#endif
#endif
	XGrabButton(XtDisplay(dino2d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(dino2d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(dino2d),
		XCreateFontCursor(XtDisplay(dino2d), XC_crosshair));
	XGrabButton(XtDisplay(dino3d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(dino3d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(dino3d),
		XCreateFontCursor(XtDisplay(dino3d), XC_crosshair));
#ifdef HAVE_OPENGL
	XGrabButton(XtDisplay(dinoGL), (unsigned int) AnyButton, AnyModifier,
		XtWindow(dinoGL), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(dinoGL),
		XCreateFontCursor(XtDisplay(dinoGL), XC_crosshair));
#endif
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
