/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "swidget/STextEdit.h"
#include "stoolkit/SEncoder.h"

SDClick::SDClick (void)
{
  clickCount = 0;
  timer = 0;
}

SDClick::~SDClick ()
{
  if (timer) delete timer;
}

void
SDClick::start (unsigned int millisec)
{
  if (timer == 0)
  {
    timer = STimer::newTimer(300, this);
    clickCount = 0;
  }
  else
  {
    clickCount++;
  }
}

/**
 * The timer event
 */
bool
SDClick::timeout (const SEventSource* s)
{
  if (timer == 0)
  {
    fprintf (stderr, "strange. timeout..\n");
    return false;
  }
  if (clickCount == 1) return true; /* again */
  clickCount = 0;
  delete timer;
  timer = 0;
  return false; /* dont call me again */
}

#define SS_LEFT_MARGIN 2
/**
 * A text area is an editable area of text.
 * This one creates one with empty text.
 */
STextEdit::STextEdit (void) : textView ("")
{
  currentHistorySize = 0;
  historySize = 0;
  sliderListener = 0;
  yuditInput = 0;
  editable = true;
  focused = false;
  selecting = false;
  editor.setInterface (this);
  column = 0;

  textView.move (SLocation((int)border.getBorderSize().width + SS_LEFT_MARGIN, 
        (int)border.getBorderSize().height));
  clip (true);
  textView.setWindowInterface (this);
  caret.resize (SDimension(textView.lineHeight, textView.lineHeight));
  moveCaret ();
  caret.setWindowInterface (this);
}

bool
STextEdit::isFocused () const
{
  return focused;
}

/**
 * Deletes a text area.
 */
STextEdit::~STextEdit ()
{
  if (yuditInput)
  {
    //fprintf (stderr, "deleting yuditinput\n");
    delete yuditInput;
  }
}

/**
 * Create a text area with pre-set utf8 text.
 * @param utf8 is the initial text.
 */
STextEdit::STextEdit (const SString& utf8) : textView (utf8)
{
  currentHistorySize = 0;
  historySize = 0;
  sliderListener = 0;
  yuditInput = 0;
  editable = true;
  focused = false;

  editor.setInterface (this);
  textView.textData.move (STextIndex(0,0));
  selecting = false;
  column = 0;
  textView.move (SLocation((int)border.getBorderSize().width + SS_LEFT_MARGIN, 
        (int)border.getBorderSize().height));
  clip (true);
  textView.textData.fireEvent ();
  textView.setWindowInterface (this);
  caret.resize (SDimension(textView.lineHeight, textView.lineHeight));
  moveCaret ();
  caret.setWindowInterface (this);
}


/**
 * Set the editor that works on this widget.
 */
void
STextEdit::setEditor (const SEditor& _editor)
{
  editor = _editor;
  if (yuditInput) yuditInput->clear();
  endSelect();
  deselectText();
  editor.setInterface (this);
  if (textView.textData.setLineType(editor.getLineBreak()))
  {
    for (unsigned i=0; i<listeners.size(); i++)
    {
      listeners[i]->textChanged (this);
    }
    window->redraw (true, 0, 0, size.width, size.height);
  }
}

/**
 * Add a text edit listener to the list.
 * @param ls is the listener to be added.
 */
void
STextEdit::addTextEditLS (STextEditLS* ls)
{
  listeners.append (ls);
}

/**
 * Remove the text edit listener from the list.
 * @param ls is the listener to be removed.
 */
void
STextEdit::removeTextEditLS (STextEditLS* ls)
{
  for (unsigned int i=0; i<listeners.size(); i++)
  {
    if (listeners[i] == ls)
    {
      listeners.remove (i);
      return;
    }
  }
  return;
}

/**
 * Set the overall background. This will effectively be the border's 
 * background.
 * @param bg is the backgroud.
 */
void
STextEdit::setBackground (const SColor& bg)
{
  border.setBackground (bg);
}

/**
 * Set the background of the text itself.
 * @param bg is the backgroud.
 */
void
STextEdit::setTextBackground (const SColor& bg)
{
  SPanel::setBackground (bg);
  textView.setBackground (bg);
  caret.setBackground (bg);
}

/**
 * Set the foreground of the text itself.
 * @param bg is the foreground.
 */
void
STextEdit::setForeground (const SColor& lrfg, const SColor& rlfg)
{
  textView.setForeground (lrfg, rlfg);
  caret.setForeground (lrfg, rlfg);
}

/**
 * Set the foreground of the text itself.
 * @param bg is the foreground.
 */
void
STextEdit::setCaretForeground (const SColor& lrfg, const SColor& rlfg)
{
  caret.setForeground (lrfg, rlfg);
}

/**
 * Set the text alignment align is true if it is right aligned.
 */
void
STextEdit::setAlignment (bool align)
{
  textView.setAlignment (align);
}

/**
 */
void
STextEdit::setMultiline (bool _multiline)
{
  textView.setMultiline (_multiline);
}

/**
 */
bool
STextEdit::isMultiline () const
{
  return textView.isMultiline ();
}


/**
 * Redraw the Component on a canvas 
 * @param canvas is where we redraw this.
 */
void
STextEdit::redraw (SWindow *w, int x, int y, unsigned int width, unsigned int height)
{
  clip (false);
  border.redraw (w, x, y, width, height);
  clip (true);
  textView.redraw (w, x, y, width, height);
  caret.redraw (w);
  if (imWaiting.size()!=0 && window->isVisible())
  {
     setInputMethod(imWaiting);
  }
}

/**
 * Turn clipping at the border on and off.
 * @param on is true if we turn on clipping.
 */
void
STextEdit::clip (bool on)
{
  if (!on)
  {
    window->removeClippingArea ();
    textView.setClippingArea (0,0,0,0);
    return;
  }
  window->setClippingArea (
       (int) border.getBorderSize().width ,
       (int) border.getBorderSize().height,
       size.width - 2 * border.getBorderSize().width,
       size.height - 2 * border.getBorderSize().height);
  textView.setClippingArea (
       (int) border.getBorderSize().width ,
       (int) border.getBorderSize().height,
       size.width - 2 * border.getBorderSize().width,
       size.height - 2 * border.getBorderSize().height);
}

/**
 * Pass this to the editor.
 */
void
STextEdit::keyPressed (SWindow * w, SKey key, const SString& s,
          bool ctrl, bool shift, bool meta)
{
  if (yuditInput)
  {
    yuditInput->keyPressed (key, s, ctrl, shift, meta);
  }
  else
  {
    editor.keyPressed (key, s, ctrl, shift, meta);
  }
}


/**
 * Pass this to the editor.
 */
void
STextEdit::keyReleased (SWindow * w, SKey key, const SString& s,
          bool ctrl, bool shift, bool meta)
{
  if (!yuditInput) editor.keyReleased (key, s, ctrl, shift, meta);
}

/**
 * Pass this to the editor. Start a timer to measure
 * the double click. Get the focus.
 */
void
STextEdit::buttonPressed (SWindow * w, int button, int x, int y)
{
  if (yuditInput) yuditInput->clear();
  window->getKeyboardFocus();
  STextIndex textIndex = textView.getTextIndex (SLocation (x, y));
  setFocus ();

  /* double click check */
  clicks.start (500);    
  editor.buttonPressed (button, textIndex);
}

/**
 * A button was released.
 */
void
STextEdit::buttonReleased (SWindow * w, int button, int x, int y)
{
  if (yuditInput) yuditInput->clear();
  editor.buttonReleased (button, textView.getTextIndex (SLocation (x, y)));
  if (clicks.clickCount == 1)
  {
    editor.multiClicked (button, textView.getTextIndex (SLocation (x, y)), 2);
  }
  else if (clicks.clickCount == 2)
  {
    editor.multiClicked (button, textView.getTextIndex (SLocation (x, y)), 3);
  }
}

void
STextEdit::buttonDragged (SWindow * w, int button, int x, int y)
{
  if (yuditInput) yuditInput->clear();
  editor.buttonDragged (button, textView.getTextIndex (SLocation (x, y)));
}

/**
 * keyboard focus gained.
 */
void 
STextEdit::lostKeyboardFocus (SWindow* w)
{
  focused = false;
  caret.animate (false);
  for (unsigned int i=0; i<listeners.size(); i++)
  {
    listeners[i]->focusChanged (this, false);
  }
}

/**
 * Text has gone from clipboard. 
 */ 
void
STextEdit::lostClipSelection (SWindow* w)
{
  deselectText();
}

/**
 * keyboard focus lost.
 */
void 
STextEdit::gainedKeyboardFocus (SWindow* w)
{
  focused = true;
  if (editable)
  {
    caret.animate (true);
  }
  for (unsigned int i=0; i<listeners.size(); i++)
  {
    listeners[i]->focusChanged (this, true);
  }
}

/**
 * Resize the component
 * @param d is the new size
 */
void 
STextEdit::resize(const SDimension& d)
{
  if (getSize() == d) return;
  SPanel::resize (d);
  border.resize (d);
  SDimension td;
  if (size.width + 2 * SS_LEFT_MARGIN> border.getBorderSize().width * 2)
  {
    td.width  = size.width - border.getBorderSize().width * 2 - 2 * SS_LEFT_MARGIN;
  }
  if (size.height > border.getBorderSize().height * 2)
  {
    td.height  = size.height - border.getBorderSize().height * 2;
  }
  clip (true);
  textView.resize (td);
  moveCaret();
}

/**
 * Move the component
 * @param l is the new location
 */
void 
STextEdit::move(const SLocation& l)
{
  SPanel::move (l);
}

/**
 * getPreferredSize.
 * This is calculated form the preferred size of the textView and
 * adding the border size to it.
 */
const SDimension&
STextEdit::getPreferredSize()
{
  preferredSize = textView.getPreferredSize();
  SDimension d = border.getBorderSize();
  preferredSize.width += d.width * 2;
  preferredSize.height += d.height * 2;
  return preferredSize;
}

/**
 * Gain keyboard focus
 * @return true if focus could be gained.
 */
void
STextEdit::setFocus ()
{
  window->getKeyboardFocus();
}

/**
 * Scroll the area up or down.
 * Move text up -
 * Move text down +
 * @param value is positive if up.
 * @param value is negative is down.
 */
void
STextEdit::scrollVertical (int value, bool notify)
{
  //fprintf (stderr, "Moving viewport: %d\n", value);
  if (value ==0) return;

  int eheight = (int)size.height - (int)2 * border.getBorderSize().height;
  if (eheight <= 0) return;
  int ewidth = (int)size.width - (int)2 * border.getBorderSize().width 
       - 2 * SS_LEFT_MARGIN;
  if (ewidth <= 0) return;
  SLocation v = textView.getViewPort();
  v.y = v.y + value;
  textView.setViewPort (v);
  /* outof range. */
  if ((value < 0 && -value >= eheight) || (value > 0 && value >= eheight))
  {
    window->redraw (true,
       (int) border.getBorderSize().width ,
       (int) border.getBorderSize().height,
       ewidth + 2 * SS_LEFT_MARGIN,
       eheight);
    //fprintf (stderr, "STextEdit::scrollVertical all: %d %d %u %u\n",
     //  (int) border.getBorderSize().width ,
      // (int) border.getBorderSize().height,
       //ewidth, eheight);
  }
  /* copy the area and redraw the rest */
  else if (value < 0)
  {
    value = -value;
    /* copy up */
    window->copy ((int) border.getBorderSize().width, 
        value + (int) border.getBorderSize().height,
        ewidth, eheight-value, 
        border.getBorderSize().width,
        border.getBorderSize().height);
    /* redraw down */
    window->redraw (true,
       (int) border.getBorderSize().width ,
       (int) border.getBorderSize().height + eheight - value,
       ewidth,
       value);
 //   fprintf (stderr, "STextEdit::scrollVertical lower: %d %d %u %u\n",
  //       (int) border.getBorderSize().width,
   //    (int) border.getBorderSize().height + eheight - value,
    //   ewidth,  value);
       
  }
  else
  {
    /* copy down */
    window->copy ((int) border.getBorderSize().width, 
        (int) border.getBorderSize().height,
        ewidth, eheight-value, 
        border.getBorderSize().width,
        (int) border.getBorderSize().height + value);
    /* redraw up */
    window->redraw (true,
       (int) border.getBorderSize().width ,
       (int) border.getBorderSize().height,
       ewidth,
       value);
    //fprintf (stderr, "STextEdit::scrollVertical upper: %d %d %u %u\n",
     //  (int) border.getBorderSize().width ,
      //(int) border.getBorderSize().height,
       //ewidth, value);
  }
  if (notify)
  {
    notifySlider();
  }
}

void
STextEdit::scrollHorizontal (int value, bool notify)
{
  //fprintf (stderr, "Moving viewport: %d\n", value);
  if (value ==0) return;
  SLocation v = textView.getViewPort();
  v.x = v.x + value;
  textView.setViewPort (v);

  int eheight = (int)size.height - (int)2 * border.getBorderSize().height;
  if (eheight <= 0) return;
  int ewidth = (int)size.width - (int)2 * border.getBorderSize().width 
       + 2 * SS_LEFT_MARGIN;
  if (ewidth <= 0) return;
  /* outof range. */
  if ((value < 0 && -value >= ewidth) || (value > 0 && value >= ewidth))
  {
    window->redraw (true,
       (int) border.getBorderSize().width ,
       (int) border.getBorderSize().height,
       ewidth + 2 * SS_LEFT_MARGIN,
       eheight);
   // fprintf (stderr, "STextEdit::scrollHorizonatl all: %d %d %u %u\n",
    //   (int) border.getBorderSize().width ,
     //  (int) border.getBorderSize().height,
      // ewidth, eheight);
  }
  /* copy the area and redraw the rest */
  else if (value < 0)
  {
    value = -value;
    /* copy up */
    window->copy (value + (int) border.getBorderSize().width, 
        (int) border.getBorderSize().height,
        ewidth-value, eheight, 
        border.getBorderSize().width,
        border.getBorderSize().height);
    /* redraw down */
    window->redraw (true,
       (int) border.getBorderSize().width + ewidth-value,
       (int) border.getBorderSize().height,
       value,
       eheight);
//    fprintf (stderr, "STextEdit::scrollHorizonatl lower: %d %d %u %u\n",
 //        (int) border.getBorderSize().width,
  //     (int) border.getBorderSize().height + eheight - value,
   //    ewidth,  value);
       
  }
  else
  {
    /* copy down */
    window->copy ((int) border.getBorderSize().width, 
        (int) border.getBorderSize().height,
        ewidth-value, eheight, 
        border.getBorderSize().width + value,
        (int) border.getBorderSize().height);
    /* redraw up */
    window->redraw (true,
       (int) border.getBorderSize().width ,
       (int) border.getBorderSize().height,
       value,
       ewidth);
   // fprintf (stderr, "STextEdit::scrollHorizonatl upper: %d %d %u %u\n",
    //   (int) border.getBorderSize().width ,
     //  (int) border.getBorderSize().height,
      // ewidth, value);
  }
  if (notify)
  {
    notifySlider();
  }
}

/**
 * Shift the text, and caret so that caret is always visible.
 */
void
STextEdit::setCaretLocationVisible ()
{
  if (isMultiline())
  {
    setCaretVisibleVertical ();
  }
  else
  {
    setCaretVisibleHorizontal ();
  }
}

/**
 * Do the vertical part of caret visible trickies.
 */
void
STextEdit::setCaretVisibleVertical()
{
  /* t=text-widget c=caret tf=text-full*/
  SLocation cb = caret.getLocation();
  SLocation tb = textView.getLocation();

  SLocation ce = SLocation (cb.x, cb.y + textView.lineHeight);
  SLocation te = SLocation (tb.x + textView.getSize().width,
                 tb.y + textView.getSize().height);

  /* It happens before resize. */
  if (textView.getSize().height==0) return;
  SLocation tfe = SLocation (0,
      textView.getLocation().y + (int)textView.getDocumentHeight() 
      + textView.getViewPort().y);
  SLocation tfb = textView.getTextLocation (STextIndex(0, 0));

  /* if full text fits  move to origin. */
//fprintf (stderr, "2.  tfe.y=%d te.y=%d tfb.y=%d tb.y=%d\n", 
 //      tfe.y, te.y, tfb.y , tb.y);
  if (tfe.y < te.y && tfb.y < tb.y)
  {
    if (tb.y - tfb.y < te.y - tfe.y)
    {
      scrollVertical (tb.y - tfb.y, true);
    }
    else
    {
      scrollVertical (te.y - tfe.y, true);
    }
    return;
  }
//fprintf (stderr, "2.  cb.y=%d tb.y=%d\n", cb.y, tb.y);
  /* moved up too much. */
  if (cb.y < tb.y)
  {
    scrollVertical (tb.y-cb.y, true);
    return;
  }
  /* move down too much */
//fprintf (stderr, "3. te.y=%d ce.y=%d\n", te.y, ce.y);
  if (te.y <= ce.y)
  {
    scrollVertical (te.y-ce.y, true);
    return;
  }
  return;
}

/**
 * Do the horizontal part of caret visible trickies.
 */
void
STextEdit::setCaretVisibleHorizontal()
{
  /* t=text-widget c=caret tf=text-full*/
  SLocation cb = caret.getLocation();
  SLocation tb = textView.getLocation();

  SLocation ce = SLocation (cb.x, cb.y);
  SLocation te = SLocation (tb.x + textView.getSize().width, tb.y);

  /* It happens before resize. */
  if (textView.getSize().width==0) return;

  SLocation tfe = textView.getTextLocation (
      STextIndex (0, textView.textData.size(0)));
  SLocation tfb = textView.getTextLocation (STextIndex(0, 0));

  /* if full text fits  move to origin. */
  if (tfe.x < te.x && tfb.x < tb.x)
  {
    if (tb.x - tfb.x < te.x - tfe.x)
    {
      scrollHorizontal (tb.x - tfb.x, true);
    }
    else
    {
      scrollHorizontal (te.x - tfe.x, true);
    }
    return;
  }

  /* moved up too much. */
  if (cb.x < tb.x)
  {
    scrollHorizontal (tb.x-cb.x, true);
    return;
  }
  /* move down too much */
  if (te.x < ce.x)
  {
    scrollHorizontal (te.x-ce.x, true);
    return;
  }
  return;
}

/**
 * Add a property for XInput
 * @param key is the property.
 * @param p is the property to be modified.
 */
SProperties
STextEdit::getProperties ()
{
  SProperties p;
  char arr[64];

  p.put ("InputStyle", "over-the-spot");

  SLocation l = caret.getLocation();
  sprintf (arr, "%d,%d", l.x, l.y + textView.lineAscent);
  p.put ("InputSpot", arr);

  SColor bg = textView.getBackground();
  SColor fg = textView.getForeground (caret.getDirection () == '>');
  sprintf (arr, "%lu,%lu", 
      (unsigned long)bg.getValue(), (unsigned long)fg.getValue());
  p.put ("Color", arr);
  return SProperties (p);
}

/**
 * If show <- newline characters
 * @param lineend is true if lineend is shown.
 */
void
STextEdit::setLineEndMark (bool lineend)
{
  textView.setLineEndMark (lineend);
}

/**
 * Is new line shown?
 * @return true if newline characters are shown.
 */
bool
STextEdit::getLineEndMark () const
{
  return textView.getLineEndMark ();
}


/**
 * Chose between fonts.
 * @param fnt is the font to chose.
 * @param size is the size of the font or zero if old size is used.
 */
void
STextEdit::setFont (const SString& fnt, double size)
{
  textView.setFont (fnt, size);
  caret.resize (SDimension(textView.lineHeight, textView.lineHeight));
  moveCaret ();
  redraw ();
}

/**
 * Set the size of the font.
 */
void
STextEdit::setFontSize (double size)
{
  textView.setFontSize (size);
  caret.resize (SDimension(textView.lineHeight, textView.lineHeight));
  moveCaret ();
  redraw ();
}

/**
 * redraw the whole thing 
 */
void
STextEdit::redraw ()
{
  int eheight = (int)size.height - (int)2 * border.getBorderSize().height;
  if (eheight <= 0) return;
  int ewidth = (int)size.width - (int)2 * border.getBorderSize().width 
       + 2 * SS_LEFT_MARGIN;

  window->redraw (true, (int) border.getBorderSize().width ,
    (int) border.getBorderSize().height, ewidth + 2 * SS_LEFT_MARGIN,
     eheight);
}


/**
 * Move caret to be in sync with text.
 * @param updateColumn is true if column needs updating (dafult)
 */
void
STextEdit::moveCaret(bool updateColumn)
{
  STextIndex ti = textView.textData.getTextIndex();
  SLocation lc = textView.getTextLocation (ti);
  caret.move (lc);
  setCaretLocationVisible ();
  SLocation le = textView.getTextLocation (ti);
  caret.move (le);
  caret.on (true);
  if (updateColumn) column = ti.index;
  window->setInputMethodProperties (getProperties());
  fireCaretMoved();
  notifySlider();
}

/*----------- This is the implementation of SEditorIF ---------------*/

/**
 * Chose between fonts.
 * @param fnt is the font to chose.
 */
void
STextEdit::setFont (const SString& fnt)
{
  setFont (fnt, 0.0);
}

/**
 * Chose between input methods.
 * @patam im is the input to chose.
 */
void
STextEdit::setInputMethod (const SString& _im)
{
  //fprintf (stderr, "Starting -----------%*.*s\n", SSARGS(im));
  /* Stop current xinput method. */
  SString im = _im;
  if (!window->isVisible())
  {
     imWaiting = im;
     return;
  }

  imWaiting.clear(); 

  if (yuditInput)
  {
    yuditInput->clear();
    delete yuditInput;
    yuditInput=0;;
  }
  if (im == SS_DEFAULT_IM)
  {
    window->stopInputMethod();
  }
  else if (im == SS_KINPUT2_IM)
  {
    if (window->getInputMethod().size()>0) return;
    if (!window->startInputMethod ("_JAPANESE_CONVERSION",getProperties()))
    {
      fprintf (stderr, "Can not start kinput2 (_JAPANESE_CONVERSION) input method.\n");
    }
    else
    {
      //fprintf (stderr, "Input method kinput2 (_JAPANESE_CONVERSION) started.\n");
    }
  }
  else 
  {
    window->stopInputMethod();
    yuditInput = new SYuditInput (im, this, &editor);
    //fprintf (stderr, "new yuditinput\n");
  }
}

/**
 * Get text from clipboard and put it here.
 */
STextIndex
STextEdit::insertText ()
{
  SString str = window->getClipUTF8();
  if (str.size()==0)
  {
    return STextIndex (textView.textData.getTextIndex());
  }
  if (isMultiline())
  {
    return insertText (str);
  }
  /* multiline */
  SEncoder oc;
  SV_UCS4 s4 = oc.decode (str);
  SV_UCS4 vle;
  for (unsigned int i=0; i<s4.size(); i++)
  {
    SS_UCS4 u = s4[i];
    if (u==SGlyph::CR || u==SGlyph::LF || u==SGlyph::LS || u==SGlyph::PS)
    {
      if (!isMultiline()) break;;
    }
    vle.append ((SS_UCS4) u);
  }
  return insertText (oc.encode(vle));
}


/**
 * Put selection on clipboard.
 * Strictly speaking this is not and EditorIM
 */
void
STextEdit::clipSelection ()
{
  if (startSelection == endSelection)
  {
    return;
  }
  SString str = textView.textData.getText (startSelection, endSelection);
  window->putClipUTF8 (str);
}

/**
 * Insert the text in str.
 * @param str is an utf8 string that may contain line-breaks as well.
 * @return the index to the previous insertion point.
 */
STextIndex
STextEdit::insertDirtyText (const SString& str)
{
  if (selecting || !editable)
  {
    return STextIndex (textView.textData.getTextIndex());
  }
  deselectText();
  bool lineended = false;
  STextIndex indexNow = textView.textData.getTextIndex();
  SV_UCS4 vle;
  /* Convert text here */
  SEncoder oc;
  SV_UCS4 s4 = oc.decode (str);
  unsigned i;
  for (i=0; i<s4.size(); i++)
  {
    SS_UCS4 u = s4[i];
    if (u==SGlyph::CR || u==SGlyph::LF || u==SGlyph::LS || u==SGlyph::PS)
    {
      lineended = true;
      if (!isMultiline()) break;;
    }
    switch (u)
    {
    case SGlyph::RLO:
    case SGlyph::LRO:
    case SGlyph::PDF:
       break;
    default:
    vle.append ((SS_UCS4) u);
    }
  }

  SEncoder ic;
  SGlyph::SType gt = (
      (isMultiline() && lineended) || caret.getDirection () == '>' ) 
          ? SGlyph::LRO : SGlyph::RLO;

  vle.insert (0, (SS_UCS4)gt);
  vle.append ((SS_UCS4) SGlyph::PDF);

  STextIndex indexMovedTo = textView.textData.insert (ic.encode (vle));
  if (caret.getDirection () == '<' && (!(lineended && isMultiline())))
  {
     textView.textData.fireEvent();
     textView.textData.move (indexNow);
     moveCaret ();
  }
  else
  {
     textView.textData.fireEvent();
     moveCaret ();
  }
  for (i=0; i<listeners.size(); i++)
  {
    listeners[i]->textChanged (this);
    if (lineended)
    {
      listeners[i]->textEntered (this);
    }
  }
  return STextIndex(indexMovedTo);
}

/**
 * Insert the text in str. The text wont be affected by keys.
 * @param str is an utf8 string that may contain line-breaks as well.
 * @return the index to the previous insertion point.
 */
STextIndex
STextEdit::insertText (const SString& str)
{
  STextIndex indexNow = textView.textData.getTextIndex();
  STextIndex indexMovedTo = textView.textData.insert (str);
  //textView.textData.move (indexNow);
  textView.textData.fireEvent();
  moveCaret ();
  fireTextChanged ();
  return indexMovedTo;
}

/**
 * Scroll  it up.
 */
void
STextEdit::pageUp ()
{
  if (textView.lineHeight == 0 && !isMultiline() 
     || textView.textData.size ()==0)
  {
    return;
  }
  STextIndex indexNow = textView.textData.getTextIndex ();
  SLocation l0 = textView.getTextLocation (indexNow);
  /* half pages */
  int decrement = ((int)textView.getSize().height/2)/textView.lineHeight + 1;
  decrement = decrement * textView.lineHeight;
  column = 0;
  SLocation l1 = SLocation (0, l0.y - decrement);
  if (l0.y-l1.y > -textView.getViewPort().y)
  {
    l1.y = l0.y + textView.getViewPort().y;
  }

  int scrollVle = l0.y - l1.y;
  if (scrollVle == 0) return;

  caret.redraw();
  scrollVertical (scrollVle, true);
  STextIndex indexNext = textView.getTextIndex (l0);

  if (scrollVle <= (int)textView.lineHeight && indexNext.line > 1)
  {
    indexNext.line -= 1;
  }

  if (selecting)
  {
    selectText (indexNext);
  }
  else
  {
    deselectText();
    textView.textData.move (indexNext);
    textView.textData.fireEvent();
    moveCaret (false);
  }
  return;
}

/**
 * Scroll  it down.
 */
void
STextEdit::pageDown ()
{
  if (textView.lineHeight == 0 && !isMultiline() 
     || textView.textData.size ()==0)
  {
    return;
  }
  STextIndex indexNow = textView.textData.getTextIndex ();
  SLocation l0 = textView.getTextLocation (indexNow);
  /* half pages */
  int increment = ((int)textView.getSize().height/2)/textView.lineHeight + 1;
  increment = increment * textView.lineHeight;
  column = 0;
  SLocation l1 = SLocation (0, l0.y + increment);
  int docHeight = (int) textView.getDocumentHeight();

  /* so what if we scroll */
  int scrollVle = l1.y - l0.y;
  if (scrollVle - textView.getViewPort().y  > docHeight 
       - (int) textView.getSize().height)
  {
    scrollVle = docHeight + textView.getViewPort().y 
          - (int) textView.getSize().height;
  }
  if (scrollVle ==0) return;
  if (textView.getViewPort().y -scrollVle > 0) return;

  caret.redraw();
  scrollVertical (-scrollVle, true);
  STextIndex indexNext = textView.getTextIndex (l0);
  if (scrollVle <= (int)textView.lineHeight 
         && textView.textData.size() > indexNext.line+1)
  {
    indexNext.line += 1;
  }

  if (selecting)
  {
    selectText (indexNext);
  }
  else
  {
    deselectText();
    textView.textData.move (indexNext);
    textView.textData.fireEvent();
    moveCaret (false);
  }
  return;
}

/**
 * Move caret to index. if possible.
 * @param index is the index to go to.
 */
void
STextEdit::caretGoTo (const STextIndex& index)
{
  textView.textData.move (index);
  textView.textData.fireEvent();
  moveCaret ();
}

/**
 * Move caret up. if possible.
 */
void
STextEdit::caretUp ()
{
  STextIndex indexNow = textView.textData.getTextIndex ();
  if (indexNow.line==0)
  {
    historyUp ();
    return;
  }
  STextIndex ti = STextIndex (indexNow.line-1, column);
  unsigned int sz = textView.textData.size (ti.line);
  if (sz < ti.index) ti.index = sz;
  if (sz > 0 && ti.index == sz && textView.textData.isProperLine (ti.line))
  {
    ti.index--;
  }
  if (selecting)
  {
    selectText (ti);
  }
  else
  {
    deselectText();
    textView.textData.move (ti);
    textView.textData.fireEvent();
    moveCaret (false);
  }
  return;
}

/**
 * Move caret downwards. if possible.
 */
void
STextEdit::caretDown ()
{
  STextIndex indexNow = textView.textData.getTextIndex ();
  if (!textView.textData.isProperLine (indexNow.line) 
       || indexNow.line + 1 > textView.textData.size())
  {
    historyDown ();
    return;
  }
  STextIndex ti = STextIndex (indexNow.line+1, column);
  unsigned int sz = textView.textData.size (ti.line);
  if (sz < ti.index) ti.index = sz;
  if (sz > 0 && ti.index == sz && textView.textData.isProperLine (ti.line))
  {
    ti.index--;
  }
  if (selecting)
  {
    selectText (ti);
  }
  else
  {
    deselectText();
    textView.textData.move (ti);
    textView.textData.fireEvent();
    moveCaret (false);
  }
  historyDown ();
  return;
}

/**
 * Move caret to the left. if possible.
 */
void
STextEdit::caretLeft ()
{
  STextIndex indexNow = textView.textData.getTextIndex ();
  STextIndex indexPrev = textView.textData.getTextIndex (-1);
  if (indexPrev == indexNow) return;
  if (selecting)
  {
    selectText (indexPrev);
  }
  else
  {
    deselectText();
    textView.textData.move (indexPrev);
    textView.textData.fireEvent();
    moveCaret ();
  }
  return;
}


/**
 * Move caret to the right. if possible.
 */
void
STextEdit::caretRight ()
{
  STextIndex indexNow = textView.textData.getTextIndex ();
  STextIndex indexNext = textView.textData.getTextIndex (1);
  if (indexNext == indexNow) return; 
  if (selecting)
  {
    selectText (indexNext);
  }
  else 
  {
    deselectText();
    textView.textData.move (indexNext);
    textView.textData.fireEvent();
    moveCaret ();
  } 
  return;
}

/**
 * Put state machine into a 'selecting' state.
 */
void
STextEdit::startSelect ()
{
  startSelection = textView.textData.getTextIndex();
  endSelection = startSelection;
  selecting = true;
}

/**
 * End the selection.
 */
void
STextEdit::endSelect ()
{
  selecting = false;
  if (startSelection != endSelection)
  {
    clipSelection();
  }
}

/**
 * Drag select the text.
 * @param till indicates new endselction.
 */
void
STextEdit::selectText (const STextIndex& till)
{
  if (!selecting)
  {
    //fprintf (stderr, "STextEdit::selectText - not selecting\n");
    return;
  }

  //STextIndex textIndex = till;
  if (till == endSelection)
  {
    //fprintf (stderr, "STextEdit::selectText same index %u %u\n", till.line, till.index);
    return;
  }

  /* moving up */
  STextIndex now = till;

  //textIndex = endSelection;
  if (now > endSelection)
  {
    /* crossed startSelection - delete */
    textView.textData.move (endSelection);
    if (endSelection < startSelection)
    {
      if (now < startSelection)
      {
        textView.textData.select (now, false);
      }
      else
      {
        textView.textData.select (startSelection, false);
      }
    }
    textView.textData.select (now, true);
  }
  else /* moving down */
  {
    /* crossed startSelection - delete */
    textView.textData.move (endSelection);
    if (endSelection > startSelection)
    {
       if (now > startSelection)
       {
         textView.textData.select (now, false);
       }
       else
       {
         textView.textData.select (startSelection, false);
       }
    }
    textView.textData.select (now, true);
  }
  endSelection = now;
  textView.textData.fireEvent ();
  textView.textData.move (now);
  moveCaret ();
} 

/**
 * Wherever the current index is, select a word there.
 */
void
STextEdit::selectWord ()
{
  if (selecting) return;

  deselectText();

  STextIndex index = textView.textData.getTextIndex();
  if (index.line == textView.textData.size())
  {
     return;
  }
  unsigned int start;
  for (start=index.index; start>0; start--)
  {
    if (textView.textData.isWhiteSpace (STextIndex (index.line, start-1)))
    {
      break;
    }
  }
  unsigned int end;
  for (end=index.index; end<textView.textData.size(index.line); end++)
  {
    if (textView.textData.isWhiteSpace (STextIndex (index.line, end)))
    {
      break;
    }
  }
  if (start != end)
  {
    startSelection = (STextIndex (index.line, start));
    endSelection = (STextIndex (index.line, end));
    textView.textData.move (startSelection);
    textView.textData.select (endSelection);
    textView.textData.fireEvent ();
    textView.textData.move (index);
    clipSelection();
  }
}

/**
 * Wherever the current index is, select that line.
 */
void
STextEdit::selectLine ()
{
  if (selecting) return;

  deselectText();
  STextIndex index = textView.textData.getTextIndex();
  STextIndex nextIndex = getIndexAfterLineBreak();
  if (index==nextIndex) return;
  startSelection = STextIndex(index.line, 0);
  endSelection = nextIndex;
  textView.textData.move (startSelection);
  textView.textData.select (endSelection);
//fprintf (stderr, "start = %u %u end = %u %u\n", 
 //startSelection.line, startSelection.index,
 //endSelection.line, endSelection.index);
  textView.textData.fireEvent ();
  textView.textData.move (index);
  clipSelection();
}

/**
 * The text hoghlighted should be unhighlighted.
 */
void
STextEdit::deselectText ()
{
  selecting = false;
  if (startSelection == endSelection)
  {
    startSelection = textView.textData.getTextIndex();
    endSelection = textView.textData.getTextIndex();
    return;
  }
  STextIndex now = textView.textData.getTextIndex();

//fprintf (stderr, "deselect: %u %u -  %u %u\n", 
 // startSelection.line, startSelection.index,
  //endSelection.line, endSelection.index);

  textView.textData.move (startSelection);
  textView.textData.select (endSelection, false);
  textView.textData.fireEvent();
  textView.textData.move (now);

  startSelection = now;
  endSelection = now;
}

/**
 * getSelectedIndex 
 * @param hi is true if we want the upper index.
 */
STextIndex
STextEdit::getSelectedIndex (bool hi)
{
  if (startSelection < endSelection)
  {
    return hi ? endSelection : startSelection;
  }
  else
  {
    return hi ? startSelection : startSelection;
  }
}

/**
 * Remove the selected text.
 * Return the text that was remove - if any.
 */
SString
STextEdit::eraseSelectedText ()
{
  if (!editable) return (SString(""));

  selecting = false;
  if (startSelection == endSelection)
  {
    startSelection = textView.textData.getTextIndex();
    endSelection = textView.textData.getTextIndex();
    return SString("");
  }
  textView.textData.move (startSelection);
  SString ret = textView.textData.getText (endSelection);
  textView.textData.remove (endSelection);
  textView.textData.fireEvent();

  startSelection = textView.textData.getTextIndex();
  endSelection = textView.textData.getTextIndex();
  moveCaret ();
  fireTextChanged ();
  return SString(ret);
}


/**
 * erase a glyph - that is one glyph back.
 * text deleted are always <- direction.
 * in an LR context this should be backspace.
 * @return the text erased (if any)
 */
SString
STextEdit::backspace ()
{
  if (selecting || !editable) return SString("");

  STextIndex indexNow = textView.textData.getTextIndex ();
  STextIndex indexPrev = textView.textData.getTextIndex (-1);
  if (indexPrev == indexNow) return SString("");
  SString ret = textView.textData.getText (indexPrev);
  textView.textData.remove (indexPrev);
  textView.textData.fireEvent();
  moveCaret ();
  fireTextChanged ();
  return SString(ret);
}

/**
 * Delete text  - that is one position forward.
 * text deleted are always -> direction.
 * in an LR context this should be delete.
 * @return the text deleted (if any)
 */
SString
STextEdit::erase ()
{
  if (selecting || !editable) return (SString(""));
  
  STextIndex indexNow = textView.textData.getTextIndex ();
  STextIndex indexPrev = textView.textData.getTextIndex (-1);
  STextIndex indexNext = textView.textData.getTextIndex (1);
  if (indexNext == indexNow) return SString("");
  SString ret = textView.textData.getText (indexNext);
  textView.textData.remove (indexNext);
  textView.textData.fireEvent();
  moveCaret ();
  fireTextChanged ();
  return SString(ret);
}

/**
 * Set the direction of the caret.
 * @param lr is '>' or '<'
 */
void
STextEdit::setDirection (char lr)
{
  if (yuditInput) yuditInput->clear();
  caret.setDirection (lr);
  window->setInputMethodProperties (getProperties());
}

/**
 * Change the direction of the selected text.
 * If there is not direction change (nothing selected)
 * return false
 */

bool
STextEdit::changeDirection (char lr)
{
  if (isSelecting())
  {
    endSelect ();
  }
  /**
   * force the direction of the selected text 
   */ 
  if (startSelection == endSelection) 
  {
     return false;
  }
  STextIndex strt = startSelection;
  STextIndex send = endSelection;
  STextIndex xy = getCaret();
  SString str;
  str = textView.textData.getText (startSelection, endSelection);
  
  editor.keyPressed (SWindowListener::Key_Delete, SString(""), 
            false, false, false); 
  SEncoder enc;
  SV_UCS4 ucs4 = enc.decode (str, false);
  unsigned int start = 0;
  unsigned int end = 0;
  SString forced;
  //SGlyph::SType gt = (lr == '>') ? SGlyph::LRO : SGlyph::RLO;
  SGlyph::SType gt = (lr == '>') ? SGlyph::LRO : SGlyph::RLO;
  bool lrF = (lr == '>');
  bool reallyChanged=false;
  do
  {
    SGlyphLine gl;
    start = end;
    /* split the lines and do stoolkit composition */
    end = split (ucs4, &gl, start);

    /* put bidi into OUR visual order */
    bidiToVisual (&gl);
    unsigned int sz = gl.size();
    if (sz ==0) continue;

    SV_UCS4 ucs4Visual;
    ucs4Visual.append (gt);
    bool needpdf = true;
    unsigned int i=0;
    if (lrF)
    {
      for (i=0; i<sz; i++)
      {
        if (gl[i].isLineEnd())
        {
          needpdf = false;
          break;
        }
        else if (gl[i].lr)
        {
          ucs4Visual.append (gl[i].getChars());
          continue;
        }
        /* reverse the string */
        reallyChanged = true;
        unsigned int revend=0;
        while (i+revend < sz && !gl[i+revend].lr)
        {
          revend++;
        }
        unsigned int k = revend + i;
        while (k>i)
        {
          ucs4Visual.append (gl[k-1].getChars());
          k--;
        }
        i = i + revend -1;
      }
      if (!needpdf)
      {
        ucs4Visual.append (SGlyph::PDF);
        ucs4Visual.append (gl[sz-1].getChars());
      }
    }
    else
    {
      i= sz;
      while (i>0)
      {
        i--;
        if (gl[i].isLineEnd())
        {
          needpdf = false;
          continue;
        }
        else if (!gl[i].lr)
        {
          ucs4Visual.append (gl[i].getChars());
          continue;
        }
        /* reverse the string */
        reallyChanged = true;
        unsigned int revend=0;
        while (i-revend > 0 && gl[i-revend-1].lr)
        {
          revend++;
        }
        unsigned int k = i-revend;
        while (k<=i)
        {
          ucs4Visual.append (gl[k].getChars());
          k++;
        }
        i = i - revend;
      }
      if (!needpdf)
      {
        ucs4Visual.append (SGlyph::PDF);
        ucs4Visual.append (gl[sz-1].getChars());
      }
    }

    if (needpdf) ucs4Visual.append (SGlyph::PDF);
    forced.append (enc.encode (ucs4Visual));
  } while (start != end);
  editor.insertText (forced); 
  caretGoTo(xy);
  // confusing
  //return reallyChanged;
  return true;
}

/**
 * get the current direction of the caret.
 * @return either '<' or '>'
 */
char
STextEdit::getDirection () const
{
  return caret.getDirection();
}

/**
 * @return the current position of the caret.
 */
STextIndex
STextEdit::getCaret () const
{
  return textView.textData.getTextIndex();
}

/**
 * We should put away the focus to somewhere else.
 */
void
STextEdit::focusOut ()
{
  for (unsigned int i=0; i<listeners.size(); i++)
  {
    listeners[i]->focusOutRequest (this);
  }
}

void 
STextEdit::fireTextChanged ()
{
  for (unsigned int i=0; i<listeners.size(); i++)
  {
    listeners[i]->textChanged (this);
  }
}

void 
STextEdit::fireCaretMoved ()
{
  STextIndex ndx = textView.textData.getTextIndex();
  for (unsigned int i=0; i<listeners.size(); i++)
  {
    listeners[i]->caretMoved (this, ndx.line, ndx.index);
  }
}

void
STextEdit::setEditable (bool _editable)
{
  if (editable == _editable) return;
  editable = _editable;
  caret.animate (editable && focused);
}

bool
STextEdit::isEditable () const
{
  return editable;
}
bool
STextEdit::isSelecting () const
{
  return selecting;
}

/**
 * directly insert a glyph.
 * This is very dangerous - on SYuditInput is allowed to do it.
 */
void
STextEdit::directInsert (const SGlyph& g, STextIndex* index)
{
  STextIndex indexNow = textView.textData.getTextIndex();
  textView.textData.insert (g, index);
  if (caret.getDirection () == '<' && (!g.isLineEnd() && isMultiline()))
  {
     textView.textData.move (indexNow);
     textView.textData.fireEvent();
     moveCaret ();
  }
  else
  {
     textView.textData.fireEvent();
     moveCaret ();
  }
}

/**
 * erase text between this and 'till'.
 * @return the erased text.
 */
SString
STextEdit::eraseText (const STextIndex& till)
{
  if (selecting || !editable)
  { 
    //fprintf (stderr, "selecting=%d editable=%d\n", 
    //     (int) selecting, (int) editable);
    return SString("");
  }
  STextIndex index = textView.textData.getTextIndex();
  if (index == till)
  {
    //fprintf (stderr, "index=%u %u till=%u %u\n", 
    //     index.line, index.index, till.line, till.index);
    return SString("");
  }

  SString ret = textView.textData.getText (till);
  textView.textData.remove (till);
  textView.textData.fireEvent();
  moveCaret ();
  fireTextChanged ();
  return SString(ret);
}

/**
 * get the index just before the newline mark.
 * if there is no mark go to the real end of the line.
 */
STextIndex
STextEdit::getIndexBeforeLineBreak ()
{
  STextIndex index = textView.textData.getTextIndex();
  if (index.line == textView.textData.size())
  {
    return STextIndex(index);
  }
  index.index = textView.textData.size(index.line);
  if (textView.textData.isProperLine (index.line))
  {
    index.index = index.index-1;
  }
  return STextIndex(index);
}

/**
 * get the index after new line mark. that is already on
 * the next line, or if there is no new line, this is the same line.
 */
STextIndex
STextEdit::getIndexAfterLineBreak ()
{
  STextIndex index = textView.textData.getTextIndex();
  if (index.line == textView.textData.size())
  {
    return STextIndex(index);
  }
  if (textView.textData.isProperLine (index.line))
  {
    index.line = index.line+1;
    index.index = 0;
  }
  else
  {
    index.index = textView.textData.size(index.line);
  }
  return STextIndex(index);
}

SString
STextEdit::getText (const STextIndex& from, const STextIndex& till)
{
  return textView.textData.getText (from, till);
}

SString
STextEdit::getText ()
{
  return textView.textData.getText ();
}
void
STextEdit::setUnderlineColor (const SColor& c)
{
  textView.setUnderlineColor(c);
}

/**
 * This is called by a slibadle and it shows that vales have been changed.
 */
void
STextEdit::valueChanged (SSlidable* _slidable, SSlideType _type)
{
  SLocation diff = textView.getViewPort() + _slidable->value;
  /* it can not change anything by the viewport. */

  if (isMultiline()) /* only vertical */
  {
    scrollVertical (-diff.y, false);
  }
  else /* only horizontal */
  {
    scrollHorizontal (-diff.x, false);
  } 
  STextIndex ti = textView.textData.getTextIndex();
  SLocation lc = textView.getTextLocation (ti);
  caret.move (lc);
}

/**
 * calculate value, step, page and max and send it to 
 * the listener
 */
void 
STextEdit::notifySlider ()
{
  if (!sliderListener) return;
  /* FIXME: add horizontal one */
  SDimension d = SDimension (0, textView.getDocumentHeight());
  SSlidable s;
  s.max = d - textView.getSize();
  s.step = SDimension (0, textView.lineHeight);
  s.value = SLocation (0, -textView.getViewPort().y);
  s.page = textView.getSize();
  if (slidable==s) return;
  slidable = s;

//fprintf (stderr, "max=%u step=%u value=%d page=%d\n",
// slidable.max.height, slidable.step.height, slidable.value.y, slidable.page.height);

  if (isMultiline()) /* only vertical */
  {
    sliderListener->valueChanged (&slidable, SSliderListener::SS_VERTICAL);
  }
}

/**
 * @param l  is the listener.
 */
SSlidable*
STextEdit::setSliderListener (SSliderListener* l)
{
  sliderListener = l;
  return &slidable;
}

bool
STextEdit::undo ()
{
 return editor.undo();
}

/**
 * Clear all states 
 * return treu if it had a state.
 */
bool
STextEdit::clearState()
{
  bool had=false;
  selecting = false;
  if (yuditInput && yuditInput->clear(false))
  {
    had = true;
  }
  if (startSelection != endSelection)
  {
    had=true;
    clipSelection();
  }
  deselectText();
  return had;
}

bool
STextEdit::redo ()
{
 return editor.redo();
}

void
STextEdit::setLineBreak (const SString& lbr)
{
  editor.setLineBreak(lbr);
  if (textView.textData.setLineType(lbr))
  {
    for (unsigned i=0; i<listeners.size(); i++)
    {
      listeners[i]->textChanged (this);
    }
    window->redraw (true, 0, 0, size.width, size.height);
  }
}

void
STextEdit::clear()
{
  if (yuditInput) yuditInput->clear();
  startSelection = STextIndex(0,0);
  endSelection = STextIndex(0,0);
  selecting = false;
  editor.clear();
  textView.textData.clear();
  textView.textData.fireEvent();
  moveCaret();
  fireCaretMoved();
  fireTextChanged();
}

void
STextEdit::setText (const SString& text)
{
  clear();
  textView.textData.setText (text);
  textView.textData.fireEvent();
  //textView.textData.move(STextIndex(0,0));
  moveCaret();
  fireCaretMoved();
  fireTextChanged();
  editor.clear();
}

/**
 * Insert text through editor.
 */
void
STextEdit::insertEditorText (const SString& text)
{
    editor.keyPressed (SWindowListener::Key_Send, text, 
           false, false, false); 
}

const SStringVector&
STextEdit::getHistory()
{
  return history;
}

/**
 * history is for singles
 */
void
STextEdit::putHistory (const SString& str)
{
  if (historySize==0) return;
  if (isMultiline())  return;
  unsigned int i;
  for (i=0; i<history.size(); i++)
  {
    if (history[i] == str) break;
  } 
  if (i<history.size())
  {
    history.remove(i);
  }
  history.append (str);
  /* actually - next */
  currentHistorySize = history.size();
  if (currentHistorySize > historySize)
  {
    history.remove (0);
  }
  currentHistorySize = history.size();
}

void
STextEdit::setHistorySize (unsigned int siz)
{
   historySize = siz;
   if (history.size() > siz) history.truncate (siz);
}

/**
 * set text from history
 */
void
STextEdit::historyDown ()
{
  if (history.size() == 0 || historySize==0) return;
  if (currentHistorySize+1 >= history.size()) return;
  if (isMultiline())  return;
  editor.clear();
  clear();
//fprintf (stderr, "history down %u \n", currentHistorySize);
  currentHistorySize++;
  setText (history[currentHistorySize]);
}

/**
 * set text from history
 */
void
STextEdit::historyUp ()
{
  if (history.size() == 0 || historySize==0) return;
  if (currentHistorySize == 0) return;
  if (isMultiline())  return;
  editor.clear();
  clear();
  currentHistorySize--;
//fprintf (stderr, "history up %u \n", currentHistorySize);
  setText (history[currentHistorySize]);
}

void
STextEdit::historyEnd()
{
  currentHistorySize = history.size();
}

/**
 * If there is selected text unselect it increment the index.
 * and try find again.
 */
bool
STextEdit::find(const SString& str)
{
  if (isSelecting ()) return false;
  /* just find */
  STextIndex fs;
  if (startSelection == endSelection)
  {
    fs = textView.textData.find(str);
    if (fs == STextIndex(0,0))
    {
      caretGoTo(STextIndex(0,0));
      fs = textView.textData.find(str);
      if (fs == STextIndex(0,0))
      {
        return false;
      }
    }
  }
  /* select the text between current index and fs */
  else
  {
    caretRight();
    fs = textView.textData.find(str);
    if (fs == STextIndex(0,0))
    {
      caretGoTo(STextIndex(0,0));
      fs = textView.textData.find(str);
      if (fs == STextIndex(0,0))
      {
        return false;
      }
    }
  }
  startSelection = fs;
  endSelection = fs;
  selecting = true;
  selectText (textView.textData.getTextIndex());
  selecting = false;
  // clipSelection() ?
  return true;
}

/**
 * replace text.
 */
bool
STextEdit::replace (const SString& orig, const SString& repl)
{
  if (isSelecting())
  {
    endSelect ();
  }
  if (orig.size()==0)
  {
     return false;
  }
  SString str;
  if (startSelection != endSelection)
  {
    str = textView.textData.getText (startSelection, endSelection);
  }
  if (startSelection == endSelection || str != orig)
  {
     return find (orig);
  }
  else
  {
    editor.keyPressed (SWindowListener::Key_Delete, SString(""), 
           false, false, false); 
    if (repl.size()) editor.insertText (repl); 
  }
  return true;
}

const SGlyph*
STextEdit::glyphAt(const STextIndex & ti)
{
  if (ti.line >= textView.textData.size()) return 0;
  if (ti.index >= textView.textData.size(ti.line)) return 0;
  return &(textView.textData.glyphAt (ti));
}
