import string
from string import join, split, find, lstrip, strip, replace
import re

class NR:
   """
   New Riders Outputter
   """
   
   element_types={
      '#text': '_text',
      'StructuredTextDocument': 'document',
      'StructuredTextParagraph': 'paragraph',
      'StructuredTextExample': 'example',
      'StructuredTextBullet': 'bullet',
      'StructuredTextNumbered': 'numbered',
      'StructuredTextDescription': 'description',
      'StructuredTextDescriptionTitle': 'descriptionTitle',
      'StructuredTextDescriptionBody': 'descriptionBody',
      'StructuredTextSection': 'section',
      'StructuredTextSectionTitle': 'sectionTitle',
      'StructuredTextLiteral': 'literal',
      'StructuredTextEmphasis': 'emphasis',
      'StructuredTextStrong': 'strong',
      'StructuredTextLink': 'link',
      'StructuredTextXref': 'xref',
      'StructuredTextSGML': 'sgml',
      'StructuredTextImage': 'image',
      }      

   def dispatch(self, doc, level, output):
      getattr(self, self.element_types[doc.getNodeName()])(doc, level, output)
      
   def __call__(self, doc, level=1):
      r=[]
      self.dispatch(doc, level-1, r.append)
      return join(r,'')

   def _text(self, doc, level, output):
      if doc.getNodeName() == 'StructuredTextLiteral':
         output(doc.getNodeValue())
      else:
         output(parastrip(doc.getNodeValue()))         

   def document(self, doc, level, output):
      output('***PRODUCTION CODES USED:\n')
      output('[LB] BULLET\n')
      output('[MD] EM DASH\n')
      output('<BOLD>...</BOLD> BOLD\n')
      output('<ITAL>...</ITAL> ITALIC\n')
      output('<CODE>...</CODE> LITERAL\n\n')
      
      children=doc.getChildNodes()
      if (children and
          children[0].getNodeName() == 'StructuredTextSection'):
         title=children[0].getChildNodes()[0].getNodeValue()
         m=re.search(r'Chapter\s+(\d+)\s*:\s*(.+)', title)
         if m is None:
            m=re.search(r'Appendix\s+(\w+)\s*:\s*(.+)', title)            
         output('(a)%s\n' % m.group(1))
         output('(b)%s\n' % m.group(2))

      output('\n\n')

      for c in children:
         getattr(self, self.element_types[c.getNodeName()])(c, level, output)

   def section(self, doc, level, output):
      children=doc.getChildNodes()
      for c in children:
         getattr(self, self.element_types[c.getNodeName()])(c, level+1, output)
      
   def sectionTitle(self, doc, level, output):
      h={2 : '(c)',
         3 : '(d)',
         4 : '(e)',
         5 : '(f)',
         6 : '(g)',
         7 : '(h)',
         }
      if level==1:
         return
      output(h[level])
      for c in doc.getChildNodes():
         getattr(self, self.element_types[c.getNodeName()])(c, level, output)
      output('\n\n')

   def description(self, doc, level, output):
      p=doc.getPreviousSibling()
      if p is None or  p.getNodeName() is not doc.getNodeName():         
         output('\n\n***BEGIN DESCRIPTION LIST***\n\n')
      for c in doc.getChildNodes():
         getattr(self, self.element_types[c.getNodeName()])(c, level, output)
      n=doc.getNextSibling()
      if n is None or n.getNodeName() is not doc.getNodeName():         
         output('***END DESCRIPTION LIST***\n\n')
      
   def descriptionTitle(self, doc, level, output):
      for c in doc.getChildNodes():
         getattr(self, self.element_types[c.getNodeName()])(c, level, output)
      output('[MD]')
      
   def descriptionBody(self, doc, level, output):
      for c in doc.getChildNodes():
         getattr(self, self.element_types[c.getNodeName()])(c, level, output)
      output('\n\n')

   def bullet(self, doc, level, output):
      p=doc.getPreviousSibling()
      if p is None or p.getNodeName() is not doc.getNodeName():         
         output('\n\n***BEGIN BULLETED LIST***\n\n')
      output('[LB] ')

      for c in doc.getChildNodes():
         getattr(self, self.element_types[c.getNodeName()])(c, level, output)
      n=doc.getNextSibling()
      output('\n\n')
      if n is None or n.getNodeName() is not doc.getNodeName():         
         output('\n***END BULLETED LIST***\n')

   def numbered(self, doc, level, output):
      p=doc.getPreviousSibling()
      if p is None or p.getNodeName() is not doc.getNodeName():         
         output('\n\n***BEGIN NUMBERED LIST***\n')
      output('[LB] ')
      for c in doc.getChildNodes():
         getattr(self, self.element_types[c.getNodeName()])(c, level, output)
      n=doc.getNextSibling()
      output('\n\n')
      if n is None or n.getNodeName() is not doc.getNodeName():         
         output('\n***END NUMBERED LIST***\n')

   def example(self, doc, level, output):
      i=0
      for c in doc.getChildNodes():
         if i==0:
            output('\n\n***BEGIN PROGRAM LISTING***\n\n')
            ##
            ## eek.  A ']]>' in your body will break this...
            ##
            output(prestrip(c.getNodeValue()))
            output('\n\n***END PROGRAM LISTING***\n')
         else:
            getattr(self, self.element_types[c.getNodeName()])(
               c, level, output)

   def paragraph(self, doc, level, output):
      for c in doc.getChildNodes():
         getattr(self, self.element_types[c.getNodeName()])(
            c, level, output)
      output('\n\n')
            
   def link(self, doc, level, output):
      for c in doc.getChildNodes():
         getattr(self, self.element_types[c.getNodeName()])(c, level, output)
      output(' (%s)' % doc.href)

   def emphasis(self, doc, level, output):
      output('<ITAL>')
      for c in doc.getChildNodes():
         getattr(self, self.element_types[c.getNodeName()])(c, level, output)
      output('</ITAL> ')

   def literal(self, doc, level, output):
      output('<CODE>')
      for c in doc.getChildNodes():
         output(c.getNodeValue())
      output('</CODE>')

   def strong(self, doc, level, output):
      output('<BOLD>')
      for c in doc.getChildNodes():
         getattr(self, self.element_types[c.getNodeName()])(c, level, output)
      output('</BOLD>')

   def xref(self, doc, level, output):
      output('figure %s' % replace(doc.getNodeValue(), '-', '.'))
      pass
   
   def sgml(self, doc, level, output):
      output(doc.getNodeValue())

   def image(self, doc, level, output):
      num=replace(doc.key, '-', '.')
      output('***INSERT FIGURE %s (%s) HERE***\n' % (num, doc.href))
      output('Figure %s\n' % num)
      output('%s\n' % parastrip(doc.getNodeValue()))


def prestrip(v):
   v=string.replace(v, '\r\n', '\n')
   v=string.replace(v, '\r', '\n')
   v=string.replace(v, '\t', '        ')
   lines=string.split(v, '\n')
   indent=len(lines[0])
   for line in lines:
      if not len(line): continue
      i=len(line)-len(string.lstrip(line))
      if i < indent:
         indent=i
   nlines=[]
   for line in lines:
      nlines.append(line[indent:])
   return string.join(nlines, '\n')


def parastrip(v):
   lines=split(v, '\n')
   out=lstrip(lines[0])
   for line in lines[1:]:
      out = out + ' ' + lstrip(line)
   return out

         

