#!/usr/bin/python
import sys
import string
import types
import glob
import os.path

indent=2


class Class:

    def __init__(self, klass):
        
        self.name=klass.__name__
        self.doc=trim_doc_string(klass.__doc__)

        # inheritence information
        if hasattr(klass,'__extends__'):
            self.extends=[]
            for base in klass.__extends__:
                names=string.split(base, '.')
                self.extends.append(names[2])

        # constructor information
        if hasattr(klass, '__constructor__'):
            self.constructor=Method(klass.__constructor__)
        
        # Get info on methods and attributes, ignore special items
        self.attributes=[]
        self.methods=[]
        for k,v in klass.__dict__.items():
            if k not in ('__extends__', '__doc__', '__constructor__'):
                if type(v)==types.FunctionType:
                    self.methods.append(Method(v))
                else:
                    self.attributes.append(Attribute(k, v))
                    
    def output(self, f):
        f.write(' ' * indent)
        f.write("class '%s" % self.name)
        if hasattr(self, 'extends'):
            f.write('(%s)' % string.join(self.extends, ', '))
        f.write("'\n\n")
        f.write(indent_doc_string(self.doc, indent * 2))
        f.write('\n\n')

##        if self.attributes:
##            f.write(' ' * indent * 2)
##            f.write('Attributes\n\n')
            
        for a in self.attributes:
            a.output(f)

##        if self.methods:
##            f.write(' ' * indent * 2)
##            f.write('Methods\n\n')
        
        for m in self.methods:
            m.output(f)

        if hasattr(self, 'constructor'):
            f.write(' ' * indent * 2)
            f.write('ObjectManager Constructor\n\n')
            self.constructor.output(f, 3)

    def __cmp__(self, other):
        return cmp(self.name, other.name)
    
        
class Method:
    
    varargs=None
    kwargs=None
    
    def __init__(self, func):
        if hasattr(func, 'im_func'):
            func=func.im_func

        self.name=func.__name__
        self.doc=trim_doc_string(func.__doc__)
        
        # figure out the method arguments
        # mostly stolen from pythondoc
        CO_VARARGS = 4
        CO_VARKEYWORDS = 8
        names = func.func_code.co_varnames
        nrargs = func.func_code.co_argcount
        if func.func_defaults:
            nrdefaults = len(func.func_defaults)
        else:
            nrdefaults = 0
        self.required = names[:nrargs-nrdefaults]
        if func.func_defaults:
            self.optional = tuple(map(None, names[nrargs-nrdefaults:nrargs],
                                 func.func_defaults))
        else:
            self.optional = ()
        varargs = []
        ix = nrargs
        if func.func_code.co_flags & CO_VARARGS:
            self.varargs=names[ix]
            ix = ix+1
        if func.func_code.co_flags & CO_VARKEYWORDS:
            self.kwargs=names[ix]

    def output(self, f, level=2):
        f.write(' '* indent * level)
        f.write("'%s(" % self.name)
        args=list(self.required)
        for arg, value in self.optional:
            if value=='':
                value = '""'
            args.append('%s=%s' % (arg, value))
        if self.varargs:
            args.append('*%s' % self.varargs)
        if self.kwargs:
            args.append('**%s' % self.kwargs)
        f.write(string.join(args, ', '))
        f.write(")'\n\n")
        f.write(indent_doc_string(self.doc, indent * (level +1)))
        f.write('\n\n')


class Function(Method):

    def output(self, f, level=1):
        f.write(' '* indent * level)
        f.write("'def %s(" % self.name)
        args=list(self.required)
        for arg, value in self.optional:
            if value=='':
                value = '""'
            args.append('%s=%s' % (arg, value))
        if self.varargs:
            args.append('*%s' % self.varargs)
        if self.kwargs:
            args.append('**%s' % self.kwargs)
        f.write(string.join(args, ', '))
        f.write(")'\n\n")
        f.write(indent_doc_string(self.doc, indent * (level +1)))
        f.write('\n\n')

class Attribute:

    def __init__(self, name, value):
        self.name=name
        self.value=value

    def output(self, f):
        f.write(' ' * indent * 2)
        f.write("%s=%s\n\n" % self.name, self.value)


class Module:

    def __init__(self, filename, doc):
        self.name=module_name(filename)
        self.doc=doc
        
    def output(self, f):
        f.write("module '%s'\n" % self.name)
        f.write(indent_doc_string(self.doc, indent))
        f.write('\n')

def module_name(filename):
    file=os.path.split(filename)[1]
    return os.path.splitext(file)[0]    
        
def trim_doc_string(text):
    """
    Trims a doc string to make it format
    correctly with structured text.
    """
    text=string.strip(text)
    text=string.replace(text, '\r\n', '\n')
    lines=string.split(text, '\n')
    nlines=[lines[0]]
    if len(lines) > 1:
        min_indent=None
        for line in lines[1:]:
            if not line:
                continue
            indent=len(line) - len(string.lstrip(line))
            if indent < min_indent or min_indent is None:
                min_indent=indent   
        for line in lines[1:]:
            nlines.append(line[min_indent:])
    return string.join(nlines, '\n')


def indent_doc_string(text, indent):
    nlines=[]
    for line in string.split(text, '\n'):
        nlines.append(' ' * indent + line)
    return string.join(nlines, '\n')


def fileClasses(file):
    dict={}
    results=[]
    execfile(file, dict)
    doc=dict.get('__doc__','')
    results.append(Module(file, doc))
    for v in dict.values():
        if type(v)==types.ClassType:
            results.append(Class(v))
        elif type(v)==types.FunctionType:
            # ignore constuctor functions
            for c in dict.values():
                if hasattr(c, '__constructor__') and c.__constructor__.im_func is v:
                    break
            else:
                results.append(Function(v))
    return results

    
if __name__=="__main__":
    if not sys.argv[1:]:
        print """
Usage: appendix_b.py <path>

Builds 'AppendixB.stx' by locating API reference information in a Zope
directory. The argument should be the path to a Zope directory.

    """
        sys.exit(1)
    dir=sys.argv[1]
    if not os.path.isdir(dir):
        print "Path should be a path to you Zope directory"
        sys.exit(1)

    f=open('AppendixB.stx', 'wb')
    f.write("""\
Appendix B: API Reference

  This reference describes the interfaces to the most common set of
  basic Zope objects.  This reference is useful while writing DTML,
  Perl, and Python scripts that create and manipulate Zope objects.
  
""")
    files=[]
    products=os.listdir(os.path.join(dir, 'lib/python/Products'))
    for product in products:
        product=os.path.join(dir, 'lib/python/Products', product)
        if not os.path.isdir(product):
            continue
        api_files=glob.glob(os.path.join(product, 'help/*.py'))
        for api_file in api_files:
            files.append(os.path.join(product, 'help', api_file))
    classes=[]
    files.sort(lambda x,y: cmp(module_name(x), module_name(y)))
    for file in files:
        classes = classes + fileClasses(file)
    for c in classes:
        c.output(f)
    print "Wrote AppendixB.stx"







