"""
UserFolderStorage used for CMFMember

Contributor: Sasha Vincic, Sweden

Released as open-source under the current ___ license

$Id: userfolderstorage.py 8137 2005-04-23 09:25:18Z rafrombrc $
"""

from Products.Archetypes.Storage import StorageLayer, AttributeStorage
from Products.Archetypes.utils import shasattr
from Products.Archetypes.interfaces.field import IObjectField
#from Products.CMFMember.tools.memberdatacontainer import MemberDataContainer

from Acquisition import aq_base, aq_parent, aq_inner
from AccessControl import ClassSecurityInfo
import sys

class UserFolderStorage(StorageLayer):
    """ UserFolderStorage is a transparent StorageLayer. 
        The goal of the storage is to enable you to switch storages
        at runtime without having traverse the Schema and changing 
        storage value. Second reason is that you need to identify which
        fields to change the storage on and since we don't have the
        schema editor yet we do it this way."""

    _layer = None
    _mappings = {}
    
    
    security = ClassSecurityInfo()
        
    def __init__(self, cache=True):
        self._cache = cache
        self._userFolder = 'Users'
        self._registeredForUserUpdate = False
                
    def _registerForUserUpdate(self, instance):
        """ Tells the instance that method updateUserObject should be
            called on this storage when all set has ben set. """
        instance.registerStorageForUpdate(self)
        self._registeredForUserUpdate = True
        
    # IStorage methods
        
    def get(self, name, instance, **kwargs):
        value = None
        if not shasattr(instance, name):
            # We should have a cached value, but if not we could get one
            # value = instance.getUser().getProperty( name, None) 
            value = getattr(instance.getUser(), name, None)
        else:
            value = getattr(instance, name)
        
        return value
                
    def set(self, name, instance, value, **kwargs):     
        """ 
        """
        if kwargs and kwargs.get('_initializing_', False):
            value = getattr(instance.getUser(), name, value)
        value = aq_base(value)
        setattr(aq_base(instance), name, value)
    
        instance._p_changed = 1
        
        if not self._registeredForUserUpdate:
            self._registerForUserUpdate(instance)
            
    def unset(self, name, instance, **kwargs):
        pass
    
    # ILayer methods
    security.declarePrivate('initializeInstance')
    def initializeInstance(self, instance, item=None, container=None):
        """ Check if we are an instance in MemberDataContainer and that
            the Member has a user object before we initialize. Before
            that we can't set any values on the user object.
        """
        if container and container.__class__ == 'MemberDataContainer':
            
            if not instance.hasUser() and not instance._getUserById( instance.getUserName()):
               return
            
            try:
                instance.__initialized += (self.getName(),)
            except AttributeError:
                instance.__initialized = (self.getName(),)
            
    security.declarePrivate('cleanupInstance')
    def cleanupInstance(self, instance, item=None, container=None):
        pass

    security.declarePrivate('initializeField')
    def initializeField(self, instance, field):
        pass
       

    security.declarePrivate('cleanupField')
    def cleanupField(self, instance, field):
        pass
    
    # ...
    
    def is_initialized(self, instance):
        try:
            return self.getName() in instance.__initialized
        except AttributeError:
            return None
    
    def updateFromUserObject(self, instance):
        """ Get initial values from the user folder """
        user = instance._getUserById( instance.getUserName())
        fields = instance.Schema().fields()
        fields = [f for f in fields if IObjectField.isImplementedBy(f) \
                  and f.getStorage().__class__ is self.__class__]
        for field in fields:
            fieldName = field.getName()
            value = getattr(user, fieldName, None)
            if value is not None:
                self.set(fieldName, instance, value) 
                    
                    
    def updateUserObject(self, instance):
        if instance.hasUser():
            try:
                user = instance.getUser()
                acl_users  = aq_parent(aq_inner(user)).getUserSource(user.getUserSourceId())
                fields = instance.Schema().fields()
                kwargs = {}
                for f in fields:
                    if IObjectField.isImplementedBy(f) \
                        and f.getStorage().__class__ is self.__class__:
                        kwargs[f.getName()] = self.get(f.getName(), instance)
    
                acl_users.manage_editUser( instance.getUserName() , kwargs=kwargs)
            except:
                # The user is probably outside plone and doesn't have getUserSource method.
                pass