######################################################################
# 

import string, re

###########################################################################
# CLASS SubscriberList
###########################################################################

#class ZMISubscriberList(SubscriberList, Persistent, Item, Implicit): #?

class SubscriberList:                 
    """
    RESPONSIBILITIES
    - manage a list of email subscribers to a ZWikiPage
    - expose the ZWikiPage's subscribers as a ZMI property
    - manage a list of email subscribers to a ZWikiPage's parent folder

    COLLABORATORS
    ZWikiPage ?

    NB for convenience a ZWikiPage has this mixed-in;
    for a Folder we could install a manageable subclass of this as a
    subobject.  See if this makes sense.
    Bah too many complications. For now, overload the methods below to
    also manage a property in the parent folder.
    """

    ######################################################################
    # VARIABLES
    ######################################################################

    subscribers = ''
    #subscribers = []
    # NB this will hide the folder's subscribers property

    _properties=(
        {'id':'subscribers', 'type': 'string', 'mode': 'w'},
        #{'id':'subscribers', 'type': 'lines', 'mode': 'w'},
        )

    ######################################################################
    # METHOD CATEGORY: private
    ######################################################################

    def _resetSubscribers(self, parent=0):
        """
        Clear this page's subscriber list.
        With parent arg, manage the parent folder's subscriber list instead.
        """
        if parent:
            self.aq_parent.subscribers = ''
        else:
            self.subscribers = ''

    def _setSubscribers(self, subscriberlist, parent=0):
        """
        Set this page's email subscriber list. 
        Currently a string of comma-separated email addresses.
        Change to a list.
        With parent arg, manage the parent folder's subscriber list instead.
        """
        if parent:
            self.aq_parent.subscribers = subscriberlist
        else:
            self.subscribers = subscriberlist

    ######################################################################
    # METHOD CATEGORY: page subscription
    ######################################################################

    def subscriberList(self, parent=0):
        """
        Return this page's subscriber list (as a string).
        With parent arg, manage the parent folder's subscriber list instead.
        """
        # look for a line beginning "PageSubscribers:" within the text
        #match = re.search(pagesubscribers, self.read())
        #if match:
        #    return html_unquote(match.group(1))

        if parent:
            if hasattr(self.aq_parent,'subscribers') and \
               self.aq_parent.subscribers:
                return self.aq_parent.subscribers
            else:
                return ''
        else:
            # look for a "subscribers" property
            if hasattr(self,'subscribers') and self.subscribers:
                return self.subscribers
            else:
                return ''

    def subscriberCount(self, parent=0):
        """
        Return the number of email addresses currently subscribed to
        this page
        With parent arg, manage the parent folder's subscriber list instead.
        """
        s = self.subscriberList(parent)
        if s:
            return string.count(self.subscriberList(parent),',') + 1
        else:
            return 0

    def isSubscriber(self,email,parent=0):
        """
        Is email currently subscribed to this page ?
        With parent arg, manage the parent folder's subscriber list instead.

        >>> zc.TestPage._setSubscribers('a, b')
        >>> zc.TestPage.isSubscriber('a')
        1
        >>> zc.TestPage.isSubscriber('c')
        0
        >>> zc.TestPage.isSubscriber('')
        0
        >>> zc.TestPage.isSubscriber(' ')
        0
        
        """
        if (email and
            re.match(r'(?i).*\b%s\b.*' % (email),self.subscriberList(parent))):
            return 1
        else:
            return 0
               
    def subscribe(self, email, REQUEST=None, parent=0):
        """
        Add email as a subscriber to this page.
        With parent arg, manage the parent folder's subscriber list instead.

        >>> zc.TestPage._setSubscribers('spamandeggs')
        >>> zc.TestPage.subscriberList()
        'spamandeggs'
        >>> zc.TestPage._resetSubscribers()
        >>> zc.TestPage.subscriberList()
        ''
        >>> zc.TestPage.subscriberCount()
        0
        >>> zc.TestPage.subscribe('1@test.com')
        >>> zc.TestPage.subscriberList()
        '1@test.com'
        >>> zc.TestPage.subscriberCount()
        1
        >>> zc.TestPage.isSubscriber('1@test.com')
        1
        >>> zc.TestPage.isSubscriber('2@test.com')
        0
        >>> zc.TestPage.subscribe('2@test.com')
        >>> zc.TestPage.subscriberList()
        '1@test.com, 2@test.com'
        >>> zc.TestPage.subscriberCount()
        2
        >>> zc.TestPage.isSubscriber('2@test.com')
        1
        >>> zc.TestPage.subscribe('1@test.com')
        >>> zc.TestPage.subscriberList()
        '1@test.com, 2@test.com'
        >>> zc.TestPage.unsubscribe('1@test.com')
        >>> zc.TestPage.subscriberList()
        '2@test.com'
        >>> zc.TestPage.unsubscribe('1@test.com')
        >>> zc.TestPage.subscriberList()
        '2@test.com'
        >>> zc.TestPage.unsubscribe('2@TesT.cOm')
        >>> zc.TestPage.subscriberList()
        ''
        """
        # do some email address validation
        if email:

            s = self.subscriberList(parent)
            if not s:
                self._setSubscribers(email,parent)
            else:
                if not self.isSubscriber(email,parent):
                    self._setSubscribers(s + ', ' + email,parent)

        # redirect browser if needed
        if REQUEST:
            REQUEST.RESPONSE.redirect(
                REQUEST['URL1']+'/subscribeform?email='+email)

    def unsubscribe(self, email, REQUEST=None, parent=0):
        """
        Remove an email address from this page's mail subscriber list.
        With parent arg, manage the parent folder's subscriber list instead.
        """
        if self.isSubscriber(email,parent):
            s = self.subscriberList(parent)
            s = re.sub(r'(?i)%s(, )?' % (email), r'', s)
            if s[:2] == ', ': s = s[2:]
            if s[-2:] == ', ': s = s[:-2]
            self._setSubscribers(s,parent)

        # redirect browser if needed
        if REQUEST:
            REQUEST.RESPONSE.redirect(
                REQUEST['URL1']+'/subscribeform?email='+email)

    ######################################################################
    # METHOD CATEGORY: parent folder subscription
    ######################################################################

    def wikiSubscriberList(self):
        """
        whole-wiki version of subscriberList
        """
        return self.subscriberList(parent=1)

    def wikiSubscriberCount(self):
        """
        whole-wiki version of subscriberCount
        """
        return self.subscriberCount(parent=1)

    def isWikiSubscriber(self,email):
        """
        whole-wiki version of isSubscriber
        """
        return self.isSubscriber(email,parent=1)

    def wikiSubscribe(self, email, REQUEST=None):
        """
        whole-wiki version of subscribe
        """
        return self.subscribe(email,REQUEST,parent=1)

    def wikiUnsubscribe(self, email, REQUEST=None):
        """
        whole-wiki version of unsubscribe
        """
        return self.unsubscribe(email,REQUEST,parent=1)

    ######################################################################
    # METHOD CATEGORY: misc
    ######################################################################

    def allSubscribers(self):
        """
        This page's subscribers + whole wiki subscribers.

        >>> zc.TestPage._resetSubscribers()
        >>> zc.TestPage._resetSubscribers(parent=1)
        >>> zc.TestPage.allSubscribers()
        ''
        """
        s = self.subscriberList() + ', ' + self.wikiSubscriberList()
        if s[:2] == ', ': s = s[2:]
        if s[-2:] == ', ': s = s[:-2]
        return s
