/*
This software may only be used by you under license from AT&T Corp.
("AT&T").  A copy of AT&T's Source Code Agreement is available at
AT&T's Internet website having the URL:
<http://www.research.att.com/sw/tools/graphviz/license/source.html>
If you received this software without first entering into a license
with AT&T, you have an infringing copy of this software and cannot use
it without violating AT&T's intellectual property rights.
*/

#pragma prototyped

/* this is a rough start at a working SVG driver.
you can get image nodes by setting a node's
 [shape=webimage,shapefile="http://www.your.site.com/path/image.png"]
(of course can also be a file: reference if you run locally.
which causes a warning (FIX that) but the back end turns this into
a custom shape.  you can also set
[ URL = "http://some.place.com/whatever" ] 
to get clickable nodes (and edges for that matter).

some major areas needing work:
0. fonts including embedded font support.  is SVG finished in this area?
1. styles, including dotted/dashed lines, also "style function"
passthrough in SVG similar to what we have in postscript.
2. look at what happens in landscape mode, pagination? etc.
3. allow arbitrary user transforms via graph "style" attribute.
4. javascript hooks.  particularly, look at this in the context
of SVG animation for dynadag output (n.b. dynadag only in alpha release)
5. image node improvement, e.g. set clip path to node boundary, and
support scaling with fixed aspect ratio (this feature seems to be
broken in the current Adobe SVG plugin for Windows).  
6. can we support arbitrary HTML for node contents?
7. encode abstract graph as interleaved XML in some sort of graph XML dialect?
8. accessibility features
9. embed directions about getting plugin, if not present in browser

Stephen North
north@research.att.com
*/

#include	<stdarg.h>
#include	"render.h"

#ifdef HAVE_LIBZ
#include	"zlib.h"
#endif

#ifdef DMALLOC
#include "dmalloc.h"
#endif

#define	NONE	0
#define	NODE	1
#define	EDGE	2
#define	CLST	3

/* SVG font modifiers */
#define REGULAR 0
#define BOLD	1
#define ITALIC	2

/* SVG patterns */
#define P_SOLID	0
#define P_NONE  15
#define P_DOTTED 4	/* i wasn't sure about this */
#define P_DASHED 11 /* or this */

/* SVG bold line constant */
#define WIDTH_NORMAL 1
#define WIDTH_BOLD 3

#ifdef TESTFAILED
/* sodipodi doesn't understand style sheets */
#define DEFAULT_STYLE\
"<style type='text/css'>\n" \
"<![CDATA[\n" \
".node ellipse {fill:none;filter:URL(#MyFilter)}\n" \
".node polygon {fill:none;filter:URL(#MyFilter)}\n" \
".cluster polygon {fill:none;filter:URL(#MyFilter)}\n" \
".edge path {fill:none;stroke:black;stroke-width:1;}\n" \
"text {font-family:Times;stroke:black;stroke-width:.4;font-size:12px}\n" \
"]]>\n" \
"</style>"

#define DEFAULT_FILTER \
"<filter id=\"MyFilter\" x=\"-20%\" y=\"-20%\" width=\"160%\" height=\"160%\">\n" \
"<feGaussianBlur in=\"SourceAlpha\" stdDeviation=\"4\" result=\"blur\"/>\n" \
"<feOffset in=\"blur\" dx=\"4\" dy=\"4\" result=\"offsetBlur\"/>\n" \
"<feSpecularLighting in=\"blur\" surfaceScale=\"5\" specularConstant=\"1\"\n" \
"specularExponent=\"10\" style=\"lighting-color:white\" result=\"specOut\">\n" \
"<fePointLight x=\"-5000\" y=\"-10000\" z=\"20000\"/>\n" \
"</feSpecularLighting>\n" \
"<feComposite in=\"specOut\" in2=\"SourceAlpha\" operator=\"in\" result=\"specOut\"/>\n" \
"<feComposite in=\"SourceGraphic\" in2=\"specOut\" operator=\"arithmetic\"\n" \
"k1=\"0\" k2=\"1\" k3=\"1\" k4=\"0\" result=\"litPaint\"/>\n" \
"<feMerge>\n" \
"<feMergeNode in=\"offsetBlur\"/>\n" \
"<feMergeNode in=\"litPaint\"/>\n" \
"</feMerge>\n" \
"</filter>"
#endif

/* SVG dash array */
static char * sdarray = "5,2";
/* SVG dot array */
static char * sdotarray = "1,5";

static FILE		*Outfile;
static	int		Obj,N_pages;
/* static 	point	Pages; */
static	double	Scale;
static	pointf	Offset;
static	int		Rot;
static	box		PB;
static	graph_t		*G;
static int		onetime = TRUE;

static node_t		*Curnode;
static char		*CurURL;

typedef struct context_t {
char 	*pencolor,*fillcolor,*fontfam,fontopt,font_was_set;
char	pen,fill,penwidth,style_was_set;
float	fontsz;
} context_t;

#define MAXNEST 4
static context_t cstk[MAXNEST];
static int SP;

#ifdef HAVE_LIBZ
gzFile Zfile;
#endif

static int svg_printf(const char *format, ...)
{
	char buf[BUFSIZ];
	va_list argp;
	int len;

	va_start(argp, format);
#ifdef HAVE_VSNPRINTF
	(void)vsnprintf(buf, sizeof(buf), format, argp);
#else
	(void)vsprintf(buf, format, argp);
#endif
	va_end(argp);
	len = strlen(buf);		/* some *sprintf (e.g C99 std)
				   	don't return the number of
				   	bytes actually written */

	if (Output_lang == SVGZ)
		return gzwrite(Zfile,buf,(unsigned)len);
	else
		return fwrite(buf,sizeof(char),(unsigned)len,Outfile);
}

/* return true if *s points to &[a-z]*;  (e.g. &amp; )
 *                          or &#[0-9]*; (e.g. &#38; )
 */
static int
svg_isentity(char *s)
{
	s++;  /* already known to be '&' */
	if (*s == '#') {
		s++;
		while (*s >= '0' && *s <= '9') s++;
	}
	else {
		while (*s >= 'a' && *s <= 'z') s++;
	}
	if (*s == ';') return 1;
	return 0;
}

static char *
svg_string(char *s)
{
	static char	*buf=NULL;
	static int	bufsize=0;
	char		*p, *sub;
	int		len, pos=0;

	if (!buf) {
		bufsize = 64;
		buf = malloc(bufsize);
	}

	p = buf;
	while (*s) {
		if (pos > (bufsize-8)) {
			bufsize *= 2;
			buf = realloc(buf,bufsize);
			p = buf + pos;
		}
		/* these are safe even if string is already UTF-8 coded
		 * since UTF-8 strings won't contain '<' or '>' */
		if (*s == '<') {
			sub = "&lt;";
			len = 4;
		}
		else if (*s == '>') {
			sub = "&gt;";
			len = 4;
		}
		/* escape '&' only if not part of a legal entity sequence */
		else if (*s == '&' && ! (svg_isentity(s))) {
			sub = "&amp;";
			len = 5;
		}
		else {
			sub = s;
			len = 1;
		}
		while (len--) {
			*p++ = *sub++;
			pos++;
		}
		s++;
	}
	*p = '\0';
	return buf;
}


static void svg_reset(void)
{
onetime = TRUE;
}


static void init_svg(void)
{
SP = 0;
cstk[0].pencolor = 0;		/* SVG pencolor */
cstk[0].fillcolor = 0;		/* SVG fillcolor */
cstk[0].fontfam = "Times";	/* font family name */
cstk[0].fontopt = REGULAR;	/* modifier: REGULAR, BOLD or ITALIC */
cstk[0].pen = P_SOLID;		/* pen pattern style, default is solid */
cstk[0].fill = P_NONE;
cstk[0].penwidth = WIDTH_NORMAL;
}

/* can't hack a transform directly in SVG preamble, alas */
static point
svgpt(point p)
{
point	rv;

if (Rot == 0) {
	rv.x = p.x * Scale + Offset.x;
	rv.y = PB.UR.y - PB.LL.y - p.y * Scale + Offset.y;
} else {
	rv.x = PB.UR.x - PB.LL.x - p.y * Scale + Offset.x;
	rv.y = PB.UR.y - PB.LL.y - p.x * Scale + Offset.y;
}
return rv;
}

static void
svgbzptarray(point* A, int n)
{
int		i;
point	p;
char	*c;

c = "M";			/* first point */
for (i = 0; i < n; i++) {
	p.x = A[i].x; p.y = A[i].y;
	p = svgpt(p);
	svg_printf("%s%d %d",c,p.x,p.y);
	if (i==0) c = "C";	/* second point */
	else c = " ";		/* remaining points */
}
}

static void svg_font(context_t* cp)
{
svg_printf("style=\"font-family:%s;font-size:%.2f\"",
	cp->fontfam,Scale*cp->fontsz);
}

static char* svg_resolve_color(char *name)
{
static char *known_colors[] = {"aqua", "black",
  "blue", "fuchsia", "gray", "green", "lime",
  "maroon", "navy", "olive", "purple", "red",
  "silver", "teal", "white", "yellow", 0};
color_t	color;
char *tok, **known, buf[SMALLBUF];

tok = canontoken(name);
for (known = known_colors; *known; known++)
	if (streq(tok,*known)) break;
if (*known == 0) {
	colorxlate(name,&color,RGBA_BYTE);
	sprintf(buf,"#%02x%02x%02x",
		color.u.rgba[0],color.u.rgba[1],color.u.rgba[2]);
	tok = buf;
}
return tok;
}

static void svg_grstyle(context_t* cp, int filled)
{
svg_printf(" style=\"");
if (filled)
	svg_printf("fill:%s;",svg_resolve_color(cp->fillcolor));
else
	svg_printf("fill:none;");
svg_printf("stroke:%s",svg_resolve_color(cp->pencolor));
if (cp->penwidth!=WIDTH_NORMAL)
	svg_printf(";stroke-width:%d",cp->penwidth);
if( cp->pen == P_DASHED ) {
	svg_printf(";stroke-dasharray:%s", sdarray);
} else if( cp->pen == P_DOTTED) {
	svg_printf(";stroke-dasharray:%s", sdotarray);
}
svg_printf("\"");
}

static void svg_comment(void* obj, attrsym_t* sym)
{
char	*str;
str = late_string(obj,sym,"");
if (str[0]) svg_printf("<!-- %s -->\n",str);
}

static void
svg_begin_job(FILE *ofp, graph_t *g, char **lib, char *user, char *info[], point pages)
{
int	fd;

Outfile = ofp;
if (Output_lang == SVGZ) {
#if HAVE_LIBZ

	fd = dup(fileno(Outfile)); /* open dup so can gzclose 
				     independent of FILE close */
#ifdef HAVE_SETMODE
#ifdef O_BINARY
	/*
	 * Windows will do \n -> \r\n  translations on
	 * stdout unless told otherwise.
	 */
	setmode(fd, O_BINARY);
#endif
#endif

	Zfile = gzdopen(fd, "wb");
	if (!Zfile) {
		fprintf(stderr,"Error opening compressed output file\n");
		exit;
	}
#else
	fprintf(stderr,"No support for compressed output. Not compiled with zlib.\n");
	exit;
#endif
}

/* Pages = pages; */
N_pages = pages.x * pages.y;
#ifdef NONE_OF_THIS_REALLY_WORKS
svg_printf("<?xml version=\"1.0\" standalone=\"no\"?>\n");
svg_printf("<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 20000303 Stylable//EN\"\n");
svg_printf(" \"http://www/w3.org/TR/2000/03/WD-SVG-20000303/DTD/svg-20000303-stylable.dtd\">\n");
#endif
svg_printf("<!-- Generated by %s version %s (%s)\n", info[0],info[1],info[2]);
svg_printf("     For user: %s   Title: %s    Pages: %d-->\n",user, g->name, N_pages);
}

static  void
svg_end_job(void)
{
}

static void
svg_begin_graph(graph_t* g, box bb, point pb)
{
G = g;
PB = bb;
Offset.x = PB.LL.x + 1;
Offset.y = PB.LL.y + 1;
if (onetime) {
#if 0
fprintf(stderr,"LL %d %d UR %d %d\n", PB.LL.x,PB.LL.y,PB.UR.x, PB.UR.y);
#endif
	init_svg();
	svg_comment(g,agfindattr(g,"comment"));
	onetime = FALSE;
}
}

static void
svg_end_graph(void)
{
if (Output_lang==SVGZ) {
/*
*		if ((gzflush(Zfile,Z_FINISH)) != Z_STREAM_END) {
*			fprintf(stderr,"error flushing compressed output\n");
*			exit;
*		}
*/
	gzclose(Zfile);
}
}

static void
svg_begin_page(point page, double scale, int rot, point offset)
{
/* int		page_number; */
point	sz;

Scale = scale;
Rot = rot;
/* page_number =  page.x + page.y * Pages.x + 1; */
sz = sub_points(PB.UR,PB.LL);
svg_printf("<svg width=\"%dpx\" height=\"%dpx\"",
	PB.UR.x + PB.LL.x + 2, PB.UR.y + PB.LL.y + 2);
/* namespace of svg */
svg_printf(" xmlns='http://www.w3.org/2000/svg'");
/* namespace of xlink */
svg_printf(" xmlns:xlink='http://www.w3.org/1999/xlink'>\n");
#ifdef TESTFAILED
/* sodipodi doesn't understand style sheets */
svg_printf("%s\n",DEFAULT_STYLE);
svg_printf("<def>\n%s\n</def>\n",DEFAULT_FILTER);
#endif
/* its really just a page of the graph, but its still a graph,
* and it is the entire graph if we're not currently paging */
svg_printf("<g id=\"%s\" class=\"graph\">\n",svg_string(G->name));
}

static  void
svg_end_page(void)
{
svg_printf("</g>\n</svg>\n");
}

static  void
svg_begin_cluster(graph_t* g)
{
svg_printf("<g id=\"%s\" class=\"cluster\">\n",svg_string(g->name));
Obj = CLST;
}

static  void
svg_end_cluster (void)
{
svg_printf("</g>\n");
	Obj = NONE;
}

static void
svg_begin_nodes(void)        
{
	Obj = NONE;
}
    
static void
svg_end_nodes(void)  
{
	Obj = NONE;
}

static void 
svg_begin_edges(void)        
{
	Obj = NONE;
}            

static void
svg_end_edges(void)  
{            
	Obj = NONE;
}            

static void beginURL(void *obj)
{
	char	*s, *url, *esc;
	char 	*s0, *s1, *s2, *s3, *s4;

	if ((s = agget(obj, "URL")) && strlen(s)) {
		CurURL = s;
		s0 = url = strdup(s);
		s1 = s2 = s3 = s4 = "";
		if ((esc = strstr(url,NODENAME_ESC))) {
			*esc = 0;
			s4 = esc + strlen(NODENAME_ESC);

			switch(Obj) {
			case NODE:
				s2 = ((Agnode_t*)obj)->name;
				break;
			case CLST:
				s2 = ((Agraph_t*)obj)->name;
				break;
			case EDGE:
				s1 = ((Agedge_t*)obj)->tail->name;
				if (((Agedge_t*)obj)->tail->graph->root->kind & AGFLAG_DIRECTED)
					s2 = "-&gt;";
				else
					s2 = "--";
				s3 = ((Agedge_t*)obj)->head->name;
				break;
			}
		}

		svg_printf("<a xlink:href=\"%s%s%s%s%s\">\n",s0,s1,s2,s3,s4);
	} else CurURL = 0;
}

static void endURL()
{
	if (CurURL) svg_printf("</a>");
	CurURL = 0;
}

static  void
svg_begin_node(node_t* n)
{
	Obj = NODE;
	Curnode = n;
#if 0
	svg_printf("<!-- %s -->\n",n->name);
	svg_comment(n,N_comment);
#endif
	svg_printf("<g id=\"%s\" class=\"node\">\n",svg_string(n->name));
	beginURL(n);
}

static  void
svg_end_node (void)
{
	endURL();
	svg_printf("</g>\n");
	Obj = NONE;
}

static  void
svg_begin_edge (edge_t* e)
{
	char* edgeop;

	Obj = EDGE;
#if 0
	svg_printf("<!-- %s->%s -->\n",e->tail->name,e->head->name);
	svg_comment(e,E_comment);
#endif
	if (e->tail->graph->root->kind & AGFLAG_DIRECTED)
		edgeop = "-&gt;";
	else
		edgeop = "--";
	svg_printf("<g id=\"%s%s",svg_string(e->tail->name),edgeop);
	svg_printf("%s\" class=\"edge\">\n",svg_string(e->head->name));
	beginURL(e);
}

static  void
svg_end_edge (void)
{
	endURL();
	svg_printf("</g>\n");
	Obj = NONE;
}

static  void
svg_begin_context(void)
{
	assert(SP + 1 < MAXNEST);
	cstk[SP+1] = cstk[SP];
	SP++;
}

static  void 
svg_end_context(void)
{
	int			psp = SP - 1;
	assert(SP > 0);
	/*free(cstk[psp].fontfam);*/
	SP = psp;
}

static void 
svg_set_font(char* name, double size)
{
	char	*p,*q;
	context_t	*cp;

	cp = &(cstk[SP]);
	cp->font_was_set = TRUE;
	cp->fontsz = size;
	p = strdup(name);
	if ((q = strchr(p,'-'))) {
		*q++ = 0;
		if (strcasecmp(q,"italic") == 0)
			cp->fontopt = ITALIC;
		else if (strcasecmp(q,"bold") == 0)
			cp->fontopt = BOLD;
	}
	cp->fontfam = p;
}

static  void
svg_set_pencolor(char* name)
{
	cstk[SP].pencolor = name;
}

static  void
svg_set_fillcolor(char* name)
{
	cstk[SP].fillcolor = name;
}

static  void
svg_set_style(char** s)
{
	char		*line, *p;
	context_t	*cp;

	cp = &(cstk[SP]);
	while ((p = line = *s++)) {
		if (streq(line,"solid")) { /* no-op */ }
		else if (streq(line,"dashed")) cp->pen = P_DASHED;
		else if (streq(line,"dotted")) cp->pen = P_DOTTED;
		else if (streq(line,"invis")) cp->pen = P_NONE;
		else if (streq(line,"bold")) cp->penwidth = WIDTH_BOLD;
		else if (streq(line, "setlinewidth")) {
			while (*p) p++; p++; 
			cp->penwidth = atol(p);
		}
		else if (streq(line,"filled")) cp->fill = P_SOLID;
		else if (streq(line,"unfilled")) { /* no-op */ }
		else {
            fprintf(stderr, "svg_set_style: unsupported style %s - ignoring\n",
                line); 
        }
		cp->style_was_set = TRUE;
	}
	/* if (cp->style_was_set) svg_style(cp); */
}

static void svg_textline(point p, textline_t *line)
{
	char	*anchor;
	point	mp;

	if( cstk[SP].pen == P_NONE ) {
		/* its invisible, don't draw */
		return;
	}
        switch(line->just) {
                case 'l':
                        anchor="start";
			break;
		case 'r':
                        anchor="end";
			break;
		default:
		case 'n':
                        anchor="middle";
			break;
	}

	mp = svgpt(p);
	svg_printf("<text text-anchor=\"%s\" ",anchor);
	if (Rot) {
		svg_printf("transform=\"rotate(-90 %d %d)\" ",mp.x,mp.y);
	}
	svg_printf("x=\"%d\" y=\"%d\" ",mp.x,mp.y);
	svg_font(&cstk[SP]);
	svg_printf(">%s</text>\n",svg_string(line->str));
}

static void svg_ellipse(point p, int rx, int ry, int filled)
{
	point	mp;

	if( cstk[SP].pen == P_NONE ) {
		/* its invisible, don't draw */
		return;
	}
	mp.x = p.x;
	mp.y = p.y;
	mp = svgpt(mp);
	svg_printf("<ellipse cx=\"%d\" cy=\"%d\"",
		mp.x,mp.y);
    if (Rot) {int t; t = rx; rx = ry; ry = t;}
	mp.x = Scale * rx;
	mp.y = Scale * ry;
	svg_printf(" rx=\"%d\" ry=\"%d\"",mp.x,mp.y);
	svg_grstyle(&cstk[SP], filled);
	svg_printf("/>\n");
}

static void svg_bezier(point* A, int n, int arrow_at_start, int arrow_at_end)
{
	if( cstk[SP].pen == P_NONE ) {
		/* its invisible, don't draw */
		return;
	}
	svg_printf("<g");
	svg_grstyle(&cstk[SP],0);
	svg_printf("><path d=\"");
	svgbzptarray(A,n);
	svg_printf("\"/></g>\n");
}

static void svg_polygon(point *A, int n, int filled)
{
	int	i;
	point	p;

	if( cstk[SP].pen == P_NONE ) {
		/* its invisible, don't draw */
		return;
	}
	svg_printf("<polygon");
       	svg_grstyle(&cstk[SP],filled);
	svg_printf(" points=\"");
	for (i = 0; i < n; i++) {
		p = svgpt(A[i]);
		svg_printf("%d,%d ",p.x,p.y);
	}
	/* because Adobe SVG is broken */
	p = svgpt(A[0]);
	svg_printf("%d,%d",p.x,p.y);
	svg_printf("\"/>\n");
}

static void svg_polyline(point* A, int n)
{
	int	i;
	point	p;

	if( cstk[SP].pen == P_NONE ) {
		/* its invisible, don't draw */
		return;
	}
	svg_printf("<polyline");
       	svg_grstyle(&cstk[SP],0);
	svg_printf(" points=\"");
	for (i = 0; i < n; i++) {
		p = svgpt(A[i]);
		svg_printf("%d,%d ",p.x,p.y);
	}
	svg_printf("\"/>\n");
}

static void svg_user_shape(char *name, point *A, int n, int filled)
{
	/* int	i; */
	point	p;
	point	sz;
	char	*imagefile;

	if( cstk[SP].pen == P_NONE ) {
		/* its invisible, don't draw */
		return;
	}
	imagefile = agget(Curnode,"shapefile");
	if (imagefile == 0) {
		svg_polygon(A, n, filled);
	       	return;
	}
	p = Curnode->u.coord;
	p.x -= Curnode->u.lw;
	p.y += Curnode->u.ht/2;
	p = svgpt(p);
	sz.x = ROUND(Scale*(Curnode->u.lw+Curnode->u.rw));
	sz.y = ROUND(Scale*Curnode->u.ht);
	svg_printf("<image xlink:href=\"%s\" width=\"%dpx\" height=\"%dpx\" preserveAspectRatio=\"xMidYMid meet\" x=\"%d\" y=\"%d\" \n",
		imagefile,sz.x,sz.y,p.x,p.y);
	
#ifdef NOTDEF
	svg_printf("<clipPath <polygon points=\"");
		for (i = 0; i < n; i++) {
			p = svgpt(A[i]);
			svg_printf("%d,%d ",p.x,p.y);
		}
		/* because Adobe SVG is broken */
		p = svgpt(A[0]);
		svg_printf("%d,%d ",p.x,p.y);
		svg_printf("\"/> />\n");
#endif
	svg_printf(" />\n");
}

codegen_t	SVG_CodeGen = {
	svg_reset,
	svg_begin_job, svg_end_job,
	svg_begin_graph, svg_end_graph,
	svg_begin_page, svg_end_page,
	svg_begin_cluster, svg_end_cluster,
	svg_begin_nodes, svg_end_nodes,
	svg_begin_edges, svg_end_edges,
	svg_begin_node, svg_end_node,
	svg_begin_edge, svg_end_edge,
	svg_begin_context, svg_end_context,
	svg_set_font, svg_textline,
	svg_set_pencolor, svg_set_fillcolor, svg_set_style,
	svg_ellipse, svg_polygon,
	svg_bezier, svg_polyline,
	0 /* svg_arrowhead */, svg_user_shape,
	0 /* svg_comment */, 0 /* svg_textsize */
};
