/* dspfree.c -- This module contains routines that provide FreeModule,         
 *           FreeTask, FreeSegment, etc.. These routines unload objects     
 *           from DSP memory and unhook running tasks.                      
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

#include <stdio.h>
#include <string.h>
#include "dspmgr.h"

#include "dspactiv.h"                  // Include Activate Routines
#include "dspispos.h"                  // CH01 Include Mwave/OS interface
#include "dspfree.h"                   // Include function prototypes
#include "dspglist.h"                  // Include CLL manipulation Routines
#include "dspgpc.h"                    // Include GPC routines
#include "dspitcb.h"                   // Include ITCB routines
#include "dspipc.h"                    // Include IPC routines
#include "dspmem.h"                    // Include DSP memory Routines
#include "dsppcmem.h"                  // Include Memory Routines
#include "dspstruc.h"                  // Include Struc Handling
#include "dspalloc.h"                  // Include CalcFM_CPS prot.

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FreeModule                                              */
/*                                                                          */
/* FUNCTION: Free a running DSP module from DSP memory and deallocate       */
/*           all of it's space on DSP. Also trash the dsp mgr data tree     */
/*           for the module.                                                */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMOD prmod - Ptr to module to free.                              */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC FreeModule(PRMOD prmod,USHORT usFlags)

{

   RC         ulRC;
   PRGPCCL    prgpcclNode;
   PRITCBCL   pritcbclNode;
   PRMOD      prmodPrev;
   PRMOD      prmodTail;
   PRMOD      prmodCur;


   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspfree::FreeModule entry prmod %x usFlags %x\n",(int)prmod,usFlags);
   /*************************************************************************/
   /* Make sure that there is a module to work with.                        */
   /* If so, then change the activation state for this module               */
   /* to INACTIVE.                                                          */
   /*************************************************************************/

   if (prmod == NULL)
      return (DSP_INV_HANDLE);
   if (prmod->MOD_achValid != MODVALIDATE)
      return (DSP_INV_HANDLE);
   if ((ulRC = ChangeModuleState(prmod, (USHORT)(usFlags|(USHORT)
      DSP_ACTIVATE_INACTIVE))) != DSP_NOERROR)
      return (ulRC);

   /*************************************************************************/
   /* If still OK, go through each task and free it.                        */
   /* Functions within FreeTask will manage the linked-list                 */
   /* pointers, so just test for a null list.                               */
   /*************************************************************************/

   while ((prmod->MOD_prtskTail != NULL) && (ulRC == DSP_NOERROR))
      ulRC = FreeTask(prmod->MOD_prtskTail);

   /*************************************************************************/
   /* Remove the GPCCL from the Module. This takes care of the              */
   /* Linked lists attached to the module.                                  */
   /* Reclaim the memory used for the modname.                              */
   /* IF OK, Remove the empty module.                                       */
   /*************************************************************************/

   if ((prmod->MOD_prgpcclTail != NULL) && (ulRC == DSP_NOERROR)) {
      do {
         prgpcclNode = prmod->MOD_prgpcclTail->GPCCL_prgpcclNext;
         ulRC = RemoveGPCCL(prmod, prgpcclNode);
      }  while ((prmod->MOD_prgpcclTail != NULL) && (ulRC == DSP_NOERROR));

   }                                   /* endif                             */

   /*************************************************************************/
   /* Remove the ITCBCL from the Module. This takes care of the             */
   /* Linked lists attached to the module.                                  */
   /* Reclaim the memory used for the modname.                              */
   /* IF OK, Remove the empty module.                                       */
   /*************************************************************************/

   if ((prmod->MOD_pritcbclTail != NULL) && (ulRC == DSP_NOERROR)) {
      do {
         pritcbclNode = prmod->MOD_pritcbclTail->ITCBCL_pritcbclNext;
         ulRC = RemoveITCBCL(prmod, pritcbclNode);
      }  while ((prmod->MOD_pritcbclTail != NULL) && (ulRC == DSP_NOERROR));

   }                                   /* endif                             */

   /*************************************************************************/
   /* Remove the module from the managers Linked list of modules            */
   /* and recoup the Memory allocated for it.                               */
   /* If the load failed, the module is not linked in with other            */
   /* modules so just pass nulls, otherwise find the Previous and           */
   /* Tail.                                                                 */
   /*************************************************************************/

   if (ulRC == DSP_NOERROR) {

      if ((usFlags&DSP_LOAD_MASK) == DSP_LOAD_FAILED) {

         prmodTail = NULL;
         prmodPrev = NULL;
         ulRC = RemoveRModule(&prmodTail, prmod, prmodPrev);

      }
      else {
         prmodPrev = pg->prmgrTail->MGR_prmodTail;
         prmodCur = prmodPrev->MOD_prmodNext;

         while (prmodCur != prmod) {
            prmodPrev = prmodCur;
            prmodCur = prmodCur->MOD_prmodNext;
         }
         ulRC = RemoveRModule(&(pg->prmgrTail->MGR_prmodTail), prmod,
            prmodPrev);
      }
   }
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfree::FreeModule exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FreeTask                                                */
/*                                                                          */
/* FUNCTION: This routine will free the Linked-Lists attached to the        */
/*           task and will recover the memory used by the task.             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRTSK prtsk - Ptr to task to free.                                */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC FreeTask(PRTSK prtsk)

{

   RC         ulRC = DSP_NOERROR;      /* Assume DSP_NOERROR                */
   PRSEG      prseg;
   PRTSK      prtskPrev;
   PRTSK      prtskCur;
   PRFM       prfm;
   PRLDMA     prldma;
   PRITCB     pritcb;
   PRHW       prhw;
   PRGPC      prgpc;
   ULONG      ulOldFM_CPS,ulFM_CPS;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfree:FreeTask entry prtsk %x\n",(int)prtsk);

   if (prtsk == NULL)
      return (DSP_INV_HANDLE);
   if (prtsk->TSK_achValid != TSKVALIDATE)
      return (DSP_INV_HANDLE);

   ulRC = CheckDSPStatus(prtsk->TSK_prdsp);       /* CH01 */
   if (ulRC != DSP_NOERROR)
      return (ulRC);

   /*************************************************************************/
   /* Make sure that there is a task to work with.                          */
   /* Check the activation state of this task.  If its still                */
   /* ACTIVE change to INACTIVE by calling ChangeTaskState with             */
   /* DSP_ACTIVATE_INACTIVE flag set.                                       */
   /*************************************************************************/

   if ((prtsk->TSK_usflgState == TSK_ACTIVE) || (prtsk->TSK_usflgState ==
      TSK_STANDBY))

      if ((ulRC = ChangeTaskState(prtsk, DSP_ACTIVATE_INACTIVE)) !=
         DSP_NOERROR)
         return (ulRC);

   /*************************************************************************/
   /* Now we can remove control blocks hanging off this Task:               */
   /* TempTask Structure.                                                   */
   /* Code and Data Segments.                                               */
   /*************************************************************************/

   if (prtsk->TSK_prtsktmp != NULL)
      if ((ulRC = RemoveRTasktmp(prtsk)) != DSP_NOERROR)
         return (ulRC);

   while ((prtsk->TSK_prsegITail != NULL) && (ulRC == DSP_NOERROR)) {

      prseg = prtsk->TSK_prsegITail;
      ulRC = FreeSegment(prseg);

   }
   while ((prtsk->TSK_prsegDTail != NULL) && (ulRC == DSP_NOERROR)) {

      prseg = prtsk->TSK_prsegDTail;
      ulRC = FreeSegment(prseg);

   }

   /*************************************************************************/
   /* Remove the IPC Control Blocks if there are any.                       */
   /* Remove the IPCTE entry.                                               */
   /* Pass the RemoveR... routine the correct pointers.                     */
   /*************************************************************************/

   if ((prtsk->TSK_pripcte != NULL) && (ulRC == DSP_NOERROR)) {
      ulRC = RemoveIPC(prtsk);
   }

   /*************************************************************************/
   /* Free the HW that the task was driving byt traversing the RDSP Struct  */
   /* Note that the HW should only be there for BIOS Tasks                  */
   /* Traverse the list of HW devices and free them from  the BIOS Task.    */
   /*************************************************************************/

   if ((prtsk->TSK_prhwTail != NULL) && ((prtsk->TSK_usflgType&TSK_BIOSMASK)
      == TSK_BIOS) && (prtsk->TSK_prdsp != NULL) &&
      (ulRC == DSP_NOERROR)) {

      if (prtsk->TSK_prdsp->DSP_prhwTail != NULL) {
         prhw = prtsk->TSK_prdsp->DSP_prhwTail;
         do {
            if (prhw->HW_prtsk == prtsk)
               prhw->HW_prtsk = NULL;
            prhw = prhw->HW_prhwNext;
         }  while (prhw != prtsk->TSK_prdsp->DSP_prhwTail);

         while ((prtsk->TSK_prhwTail != NULL) && (ulRC == DSP_NOERROR)) {
            prhw = prtsk->TSK_prhwTail;
            ulRC = RemoveRHw(&(prtsk->TSK_prhwTail), prhw, prtsk->TSK_prhwTail
               );
         }
      }
   }

   /*************************************************************************/
   /* Remove the LDMA Control Blocks if there are any.                      */
   /* Pass the RemoveR... routine the correct pointers.                     */
   /*************************************************************************/

   while ((prtsk->TSK_prldmaTail != NULL) && (ulRC == DSP_NOERROR)) {

      prldma = (PRLDMA)prtsk->TSK_prldmaTail->LDMA_prldmaNext;

      ulRC = RemoveRLDMA(&(prtsk->TSK_prldmaTail), prldma,
         prtsk->TSK_prldmaTail);

   }

   /*************************************************************************/
   /* Remove the ITCB Control Blocks if there are any.                      */
   /* Pass the RemoveR... routine the correct pointers.                     */
   /* NOTE: need to call FreeMem to free dsp mem for buffer                 */
   /*************************************************************************/

   while ((prtsk->TSK_pritcbTail != NULL) && (ulRC == DSP_NOERROR)) {

      pritcb = prtsk->TSK_pritcbTail->ITCB_pritcbNext;

      if (pritcb->ITCB_usUseCount != 0)
         return (DSP_ITCB_CONNECTED);

      ulRC = RemoveRITCB(&(prtsk->TSK_pritcbTail), pritcb,
         prtsk->TSK_pritcbTail);

   }

   /*************************************************************************/
   /* Remove the GPC Control Blocks if there are any.                       */
   /* Pass the RemoveR... routine the correct pointers.                     */
   /* NOTE: need to call FreeMem to free dsp mem for buffer                 */
   /*************************************************************************/

   while ((prtsk->TSK_prgpcTail != NULL) && (ulRC == DSP_NOERROR)) {

      prgpc = prtsk->TSK_prgpcTail->GPC_prgpcNext;

      if (prgpc->GPC_usUseCount != 0)
         return (DSP_GPC_CONNECTED);

      if ((prgpc->GPC_usflgKind == GPC_OWNER) && (prgpc->GPC_prmemBuffer !=
         NULL))

         if ((ulRC = FreeMem(prtsk->TSK_prdsp,
            prgpc->GPC_prmemBuffer, MEM_DATA)) != DSP_NOERROR)
            return (ulRC);

      ulRC = RemoveRGPC(&(prtsk->TSK_prgpcTail), prgpc, prtsk->TSK_prgpcTail);

   }

   /*************************************************************************/
   /* Recover the CPS for this task and Put back in DSP if there            */
   /* is a DSP present.                                                     */
   /*************************************************************************/

   if (prtsk->TSK_prdsp != NULL)
      prtsk->TSK_prdsp->DSP_ulCPSFree += prtsk->TSK_ulCPS;

   /*************************************************************************/
   /* If there is a frame manager for this task and it's use                */
   /* count is zero, unchain the FMCB from the DSP and Task                 */
   /* and then free the DSP and PC memory associated wit it.                */
   /*************************************************************************/

   if ((prtsk->TSK_prfm != NULL) && (ulRC == DSP_NOERROR)) {
      prfm = prtsk->TSK_prfm;

      if ((prfm->FM_usUseCount -= 1) == 0) {

         /*******************************************************************/
         /*Determine the MIPS freed up by the removal of the frame mgr      */
         /*and adjust DSP_ulCPSFree appropriately.                          */
         /*First determine how many MIPS are associated with exisiting      */
         /*group of frame managers.                                         */
         /*******************************************************************/

         if ((ulRC = CalcFM_CPS(prtsk, &ulOldFM_CPS)) != DSP_NOERROR)
            return  ulRC;

         ulRC = RemoveFrameMgr(prtsk, prfm);/* Remove From ISPOS ONLY       */
         if ((ulRC = FreeMem(prtsk->TSK_prdsp, prfm->FM_prmemFMCB, MEM_DATA))
            != DSP_NOERROR)
            return (ulRC);

         ulRC = RemoveRFrameMgr(prfm, prtsk);/* Remove Structure            */

         /*******************************************************************/
         /*Now determine how many MIPS are associated with the updated      */
         /*group of frame managers.                                         */
         /*******************************************************************/

         if ((ulRC = CalcFM_CPS(prtsk, &ulFM_CPS)) != DSP_NOERROR)
            return  ulRC;

         /*******************************************************************/
         /*Now update the count of Free CPS by the difference between       */
         /*the MIPS required for the old manager group and the MIPS         */
         /*required for the new manger group.                               */
         /*******************************************************************/

         prtsk->TSK_prdsp->DSP_ulCPSFree += ulOldFM_CPS-ulFM_CPS;
      }
   }

   /*************************************************************************/
   /* RemoveRTask takes the task off the linked-list and recovers           */
   /* the structures memory.  So find the right pointers for                */
   /* manipulation; Namely the previous to the current task.                */
   /*************************************************************************/

   prtskPrev = prtsk->TSK_prmod->MOD_prtskTail;
   prtskCur = prtskPrev->TSK_prtskNext;

   while (prtskCur != prtsk) {
      prtskPrev = prtskCur;
      prtskCur = prtskCur->TSK_prtskNext;
   }
   if (ulRC == DSP_NOERROR)
      ulRC = RemoveRTask(&(prtsk->TSK_prmod->MOD_prtskTail), prtsk, prtskPrev)
         ;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfree::FreeTask exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FreeSegment                                             */
/*                                                                          */
/* FUNCTION:  Remove each of the linked lists attached to the control       */
/*            block and then de-link the Segment from the linked list.      */
/*                                                                          */
/*            NOTE: There should be no GPC connections at this time!!!      */
/* INPUT:                                                                   */
/*        PRSEG prseg - Ptr to segment to free.                             */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC FreeSegment(PRSEG prseg)

{

   RC         ulRC = DSP_NOERROR;
   PRSEG      prsegCur;
   PRSEG      prsegPrev;
   PRSEG      pFirstDSeg;
   PRSEG      *pprsegTail;
   USHORT     usFlag;
   PRTSK      prtsk;
   PRLDMA     prldmaNode,pPrevNode,pNextNode;
   USHORT     usCount;
   LONG       lDum1;
   ULONG      ulDum2;
   BOOL       bFound = FALSE;

   /*************************************************************************/
   /* Make sure that there is a segment to work with.                       */
   /*************************************************************************/

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfree::FreeSegment entry prseg %x\n",(int)prseg);

   if (prseg == NULL)
      return (DSP_INV_HANDLE);
   if (prseg->SEG_achValid != SEGVALIDATE)
      return (DSP_INV_HANDLE);

   /*************************************************************************/
   /* Does Caller want to Block the FreeSegment of the DDS                  */
   /*************************************************************************/

   prtsk = prseg->SEG_prtsk; /* CH02 parent task pointer */

   if ((prseg->SEG_usflg&SEG_DDSMASK) == SEG_DDS_INV) {
      prseg->SEG_usflg &= ~SEG_DDS_INV;/* unblock for next caller           */

      /* Is it a Data Segemnt                                               */

      if ((prseg->SEG_usflg&SEG_TYPEMASK) == SEG_DATA) {
         pFirstDSeg = prtsk->TSK_prsegDTail->SEG_prsegNext;

        /* Is Segment the First Data Segment/ Default Data Segment          */

         if (prseg == pFirstDSeg)
            return (DSP_INV_DDS_FREE);
      }
   }

   if ((ulRC=CheckDSPStatus(prtsk->TSK_prdsp)))
      return (ulRC);  /* CH01 Check for card unplugged */

   /*************************************************************************/
   /* CH02 Process any GPCs contained in segment.  Disconnect then remove.  */
   /*************************************************************************/

   if (prseg->SEG_usflg & SEG_GPC) /* Does segment contain GPC? */
   {
      PRGPC prgpcPrev, prgpc, prgpcEnd;
      prgpcPrev=prtsk->TSK_prgpcTail;   /* Previous GPC ptr init */
      prgpcEnd=prgpcPrev;               /* Stop processing at tail */

      /* process all GPCs in task, look for those owned by this segment */
      do
      {
         prgpc = prgpcPrev->GPC_prgpcNext; /* Current GPC */
         if (prgpc->GPC_prseg == prseg)  /* does this seg own GPC? */
         {
            if (prgpc->GPC_usUseCount != 0)  /* if connected then disconnect?*/
               return (DSP_GPC_CONNECTED);   /* easy way out! */

            if ((prgpc->GPC_usflgKind == GPC_OWNER) && (prgpc->GPC_prmemBuffer !=
               NULL))  /* free DSP memory associated with GPC */

               if ((ulRC = FreeMem(prtsk->TSK_prdsp,
                  prgpc->GPC_prmemBuffer, MEM_DATA)) != DSP_NOERROR)
                  return (ulRC);

            if ((ulRC = RemoveRGPC(&(prtsk->TSK_prgpcTail), prgpc, prgpcPrev)))
               return(ulRC);

            if (NULL == prtsk->TSK_prgpcTail) break; /* Quit while loop */
         } /* End if */
         else prgpcPrev=prgpc;  /* advance previous ptr when not found */
      } while (prgpc != prtsk->TSK_prgpcTail); /* End do */

   }  /* End If */

   /*************************************************************************/
   /* Now we can remove control blocks hanging off this Segment.            */
   /* TempSeg  Structure.   Code and Data Segments.                         */
   /*************************************************************************/

   if (prseg->SEG_prsegtmp != NULL)
      if ((ulRC = RemoveRSegmenttmp(prseg)) != DSP_NOERROR)
         return (ulRC);

   /*************************************************************************/
   /* free dsp memory allocated to segment                                  */
   /* FreeMem will discard the memperm struc if appropriate                 */
   /*************************************************************************/

   if ((prseg->SEG_prmem != NULL) && (ulRC == DSP_NOERROR)) {
      if ((prseg->SEG_usflg&SEG_TYPEMASK) == SEG_DATA)
         usFlag = MEM_DATA;
      else
         usFlag = MEM_INST;

      if ((ulRC = FreeMem(prtsk->TSK_prdsp, prseg->SEG_prmem,
         usFlag)) != DSP_NOERROR)
         return (ulRC);
   }

   /*************************************************************************/
   /* Remove the Non-discardable label definitions if any                   */
   /* ?? Use size instead of pointer determine if any???                    */
   /*************************************************************************/

   if ((prseg->SEG_parstblStatic != NULL) && (ulRC == DSP_NOERROR))

      if (FreePerm((PVOID)prseg->SEG_parstblStatic, prseg->SEG_ulStaticSize))
         return (DSP_INTERNAL_ERROR);

   /*************************************************************************/
   /* Remove the Public label definitions if any                            */
   /*************************************************************************/

   if ((prseg->SEG_parpub != NULL) && (ulRC == DSP_NOERROR))

      if (FreePerm((PVOID)prseg->SEG_parpub, prseg->SEG_ulPubsSize))
         return (DSP_INTERNAL_ERROR);

   /*************************************************************************/
   /* Check if any DMA is associated with this Segment by checking the      */
   /* parent task structure.  If there is an associated DMA structure,      */
   /* remove it.                                                            */
   /*************************************************************************/

   if (prtsk->TSK_prldmaTail != NULL)  /* any DMA for parent task           */
      {
      usCount = 0;
      pNextNode = prtsk->TSK_prldmaTail;
      do {
         usCount++;
         pNextNode = pNextNode->LDMA_prldmaNext;
      }  while (pNextNode != prtsk->TSK_prldmaTail);

      /*Thread through dma structure looking for any for this segment       */

      pPrevNode = prtsk->TSK_prldmaTail;/* set addr of prev node for rmv    */
      prldmaNode = pPrevNode->LDMA_prldmaNext;/* node being checked         */
      do                               /* until all dmas in this task       */
                                       /* traversed                         */
         {
         pNextNode = prldmaNode->LDMA_prldmaNext;/* node following current  */
         if (prldmaNode->LDMA_prseg == prseg)/* dma for this segment?       */

            /*remove the dma associated with the segment                    */

            {
            ulRC = RemoveRLDMA(&(prtsk->TSK_prldmaTail), /* addr of head ptr*/
               prldmaNode,             /* ptr to node to delete             */
               pPrevNode);             /* ptr to previous node              */
            if (ulRC != DSP_NOERROR)
               return (ulRC);
            bFound = TRUE;
            prldmaNode = pPrevNode;    /* rldmaNode is gone now             */
         }
         pPrevNode = prldmaNode;       /* reset prev ptr for next loop      */
         prldmaNode = pNextNode;       /* cycle to next dma structure in    */
                                       /* list                              */
         usCount--;
      }  while (usCount != 0);         /* still more to check               */

      /**********************************************************************/

      if (bFound)                      /* if at least one DMA entry freed   */
         {                             /* recover bus capacity...lDum1 &    */
                                       /* ulDum2 are unused                 */

         /* CH12 Ignore resource return codes during FreeSeg                */

         (void)AllocateDMA(&lDum1, &ulDum2);
      }

   }

   /* Check if any IPC is associated with this Segment by checking the      */
   /* parent task structure.  If there is an associated IPC structure,      */
   /* remove it.                                                            */
   /*************************************************************************/
   /*if this tasks IPC is associated with this segment, remove it           */

   if (prtsk->TSK_prseg_IPC == prseg) {

      /*if the IPC has been allocated, remove and free the IPC              */

      if (prtsk->TSK_pripcte != NULL) {
         ulRC = RemoveIPC(prtsk);      /* remove the IPC entry from IPC     */
                                       /* table                             */
      }

      prtsk->TSK_prseg_IPC = NULL;
      prtsk->TSK_ulOffset_IPC = 0;
      if (ulRC != DSP_NOERROR)
         return (ulRC);
   }

   /*************************************************************************/
   /* RemoveRSeg  takes the seg off the linked-list and recovers            */
   /* the structures memory.  So find the right pointers for                */
   /* manipulation; Namely the previous to the current seg.                 */
   /* Also find out if your deleting a data segment or Instruction          */
   /* segment.                                                              */
   /*************************************************************************/

   if (((prseg->SEG_usflg)&(SEG_TYPEMASK)) == SEG_DATA)
      pprsegTail = &(prseg->SEG_prtsk->TSK_prsegDTail);
   else
      pprsegTail = &(prseg->SEG_prtsk->TSK_prsegITail);

   prsegPrev = *pprsegTail;
   prsegCur = prsegPrev->SEG_prsegNext;

   while (prsegCur != prseg) {
      prsegPrev = prsegCur;
      prsegCur = prsegCur->SEG_prsegNext;
   }                                   /* endwhile                          */
   if (ulRC == DSP_NOERROR)
      ulRC = RemoveRSegment(pprsegTail, prseg, prsegPrev);

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfree::FreeSegment exit ulRC %lx\n",ulRC); 
   return (ulRC);
}
