/*  dspfxutl.c --This module contains routines that locate information about    
 *           structures used by the FIXUP routines such as ISPOS labels     
 *           and Segment addresses. See Routine prologs for more detailed   
 *           information for individual functions                           
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <stdio.h>
#include <string.h>
#include "dspmgr.h"
#include "dspfxutl.h"
#include "dspstcmp.h"

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: SegLabelToPointer                                       */
/*                                                                          */
/* FUNCTION: Allocates memory for a RFMCB data structure.                   */
/*                                                                          */
/* INPUT:                                                                   */
/*  pszModName - Zero terminated string containing the module name or NULL  */
/*  pszLabel   - Zero terminated string containing the segment label to     */
/*               search for                                                 */
/*                                                                          */
/* OUTPUT:                                                                  */
/*  ppsegment  - Pointer to start of the segment with the specified label.  */
/*                                                                          */
/* RETURN:                                                                  */
/*                                                                          */
/*    DSP_NOERROR - OK, Segment pointer set                                 */
/*                                                                          */
/*    DSP_NAME_NOT_FOUND - Search could not find segment.                   */
/*                                                                          */
/* DESCRIPTION: Given a zero terminated string of a segment name, and the   */
/*             name (possibly null) of the module that contains the         */
/*             segment, return a pointer to that segment                    */
/*                                                                          */
/* SIDE EFFECTS: NONE                                                       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC SegLabelToPointer(PSZ pszLabel,     /* Zero terminated string holding    */
                                       /* segment label to search for       */
                      PRTSK prtsk,     /* For BIOS Tasks                    */
                      PRSEG *ppsegment /* Ptr to start of segment with the  */
                                       /* specified label.                  */
)

{

   /* Local variables                                                       */

   RC         ulRC;
   PRSEG      prsegBas;
   PRSEG      prsegCur;
   PRTSK      prtskBas;
   PRTSK      prtskCur;


   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspfxutl::SegLabelToPointer entry pszLabel %s prtsk %x\n",pszLabel,(int)prtsk);
   /* Traverse tasks owned by module                                        */

   prtskCur = prtsk;                   /* Ptr to tasks in this module       */

   /* make sure we have tasks                                               */

   if (prtskCur == NULL) {
      return  DSP_NAME_NOT_FOUND;      /* no tasks or seg to find           */
   }

   else {
      prtskBas = prtskCur;             /* save orig to tell when were done  */
   }                                   /* endif                             */
   do {

      if (prtskCur->TSK_prsegITail != NULL) {/* If there are Code Segments  */
         prsegCur = prtskCur->TSK_prsegITail;/* set a ptr to tail segment   */
         prsegBas = prsegCur;          /* save orig to tell when done       */
         do {                          /* Loop thru Code segments looking   */
                                       /* for seg label                     */
         if ((pszLabel != NULL) && (prsegCur->SEG_pszRealSegName != NULL))
               {
               if (mwstrcmp(pszLabel, prsegCur->SEG_pszRealSegName) == 0)/* */
                  {
                  *ppsegment = prsegCur;/* point to seg found               */
                  return  DSP_NOERROR;
               }                       /* endif                             */
            }                          /* endif                             */

            prsegCur = prsegCur->SEG_prsegNext;/* get Ptr to next seg in    */
                                       /* Task                              */
         }  while (prsegBas != prsegCur);/* enddo                           */
      }                                /* End If                            */
      if (prtskCur->TSK_prsegDTail != NULL) {/* If there are Data Segments  */
         prsegCur = prtskCur->TSK_prsegDTail;/* set a ptr to tail segment   */
         prsegBas = prsegCur;          /* save orig to tell when done       */
         do {                          /* Loop thru Data segments looking   */
                                       /* for seg label                     */
         if ((pszLabel != NULL) && (prsegCur->SEG_pszRealSegName != NULL))
               {
               if (mwstrcmp(pszLabel, prsegCur->SEG_pszRealSegName) == 0)
                  {
                  *ppsegment = prsegCur;/* point to seg found               */
                  return  DSP_NOERROR;
               }                       /* endif                             */
            }                          /* endif                             */
            prsegCur = prsegCur->SEG_prsegNext;/* get Ptr to next seg in    */
                                       /* Task                              */
         }  while (prsegBas != prsegCur);/* enddo                           */
      }                                /* End If                            */
      prtskCur = prtskCur->TSK_prtskNext;/* move Ptr to nxt task in module  */
   }  while (prtskCur != prtskBas);    /* enddo                             */
   ulRC = DSP_NAME_NOT_FOUND;
   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspfxutl::SegLabelToPointer error ulRC %lx Segment: %s\n",ulRC, pszLabel); 
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfxutl::SegLabelToPointer exit ulRC %lx\n",ulRC);
   return (ulRC);
}


/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: LabelToOffset                                           */
/*                                                                          */
/* FUNCTION: Finds offset or address of Label in a given segment.           */
/*                                                                          */
/* INPUT:                                                                   */
/*   PRSEG prseg - Ptr to segment containing label being searched for       */
/*   PSZ pszName - Zero terminated string containing the label              */
/*   USHORT *pusAddrFlg - Type of OFFSET to return RELATIVE or ABSOLUTE     */
/*                                                                          */
/* OUTPUT:                                                                  */
/*   PULONG pulOffset - Offset relative to the start of the Segment where   */
/*                      the label is located or Absolute Address            */
/*                                                                          */
/* RETURN:                                                                  */
/*                                                                          */
/*    DSP_NOERROR - OK, label found                                         */
/*                                                                          */
/*    DSP_LABEL_NOT_FOUND - Search could not find label.                    */
/*                                                                          */
/* DESCRIPTION:  Given a pointer to a segment, and a zero terminated string */
/*               containing the name of the segment label to search for,    */
/*               return the offset or address of the label                  */
/*                                                                          */
/* SIDE EFFECTS: NONE                                                       */
/*                                                                          */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC LabelToOffset(PRSEG prseg,          /* Ptr to segment containing label   */
                                       /*   being searched for              */
                  PSZ pszName,         /* Zero terminated string containing */
                                       /*   the label                       */
                  PULONG pulOffset,    /* Offset relative to the start of   */
                                       /*   the Segment where the segment   */
                                       /*   is located.                     */
                  USHORT *pusAddrFlg   /* Type of offset to return relative */
                                       /*   or absolute                     */
)

{

   /* Local variables                                                       */

   PARPUB     prPub;
   USHORT     PubCnt,i;

   /*************************************************************************/
   /* Verify Segment                                                        */
   /*************************************************************************/

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspfxutl::LabelToOffset entry prseg %x pszName %s\n",(int)prseg,pszName);
   if (prseg == NULL)
      return (DSP_INV_HANDLE);  /* CH01 */

   if (prseg->SEG_achValid != SEGVALIDATE)
      return (DSP_INV_HANDLE);  /* CH01 */

   /*************************************************************************/
   /* Step 1 - Access the public definitions array of the segment at prseg  */
   /*************************************************************************/

   prPub = prseg->SEG_parpub;          /* Get ptr to segs Public Label defs */
   PubCnt = prseg->SEG_usPubsCount;    /* Get Count of segs public defs     */

   /*************************************************************************/
   /* Step 2 - Search string table for label and return offset if found     */
   /*************************************************************************/

   if (prPub != NULL) {
      for (i = 0; i < PubCnt; i++, prPub++) {/* search labels in PUB table  */
         if (mwstrcmp(
            pszName, (PSZ)((BYTE *)(prseg->SEG_parstblStatic)+prPub->
            PUB_usLabel+1)) == 0) {

            if ((prPub->PUB_usflg&PUB_EQUMASK) == PUB_EQUATE)

                /* Extend sign to 32 bits for PUBLIC Equates                */

               *pulOffset = (LONG)((SHORT)prPub->PUB_ulOffset);

             /***************************************************************/
             /* If !=PUB_EQUATE && ABSOLUTE address requested, process.     */
             /***************************************************************/

            else
               if (*pusAddrFlg == DSP_ABSOLUTE)
                  {
                  if (NULL != prseg->SEG_prmem)
                     *pulOffset = prPub->PUB_ulOffset+
                        prseg->SEG_prmem->MEM_ulSegAddr;
                  else
                     return  DSP_LABEL_NOT_FOUND;
               }
               else

                  *pulOffset = prPub->PUB_ulOffset;

	       MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspfxutl::LabelToOffset exit ulRC=0\n");
            return  DSP_NOERROR;       /* set found return code             */
         }                             /* endif                             */
      }                                /* endfor                            */

      return (DSP_LABEL_NOT_FOUND);
   }                                   /* endif                             */

   /*************************************************************************/
   /* Segment has no labels to search.                                      */
   /* If Label is the Segment Name, Return Offset/address of Segment        */
   /*************************************************************************/

   if (0 == mwstrcmpnull(prseg->SEG_pszVirtName, pszName)) {   /*CH01*/
      if (*pusAddrFlg == DSP_ABSOLUTE) {
         if (NULL != prseg->SEG_prmem)
            *pulOffset = prseg->SEG_prmem->MEM_ulSegAddr;
         else
            return  DSP_LABEL_NOT_FOUND;
      }

      else
         *pulOffset = 0;

      MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspfxutl::LabelToOffset exit ulRC=0\n");
      return (DSP_NOERROR);
   }

   return (DSP_LABEL_NOT_FOUND);
}



/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  FMLabelToOffset                                        */
/*                                                                          */
/* FUNCTION: Finds offset of Label in a given segment.                      */
/*                                                                          */
/* INPUT:                                                                   */
/*   PSZ pszLabel - Zero terminated string containing the FM label to       */
/*               search for                                                 */
/* OUTPUT:                                                                  */
/*   PULONG pulOffset - FM (Frame Mgr) Offset                               */
/*                                                                          */
/* RETURN:                                                                  */
/*                                                                          */
/*    DSP_NOERROR - OK, Segment pointer set                                 */
/*                                                                          */
/*    DSP_LABEL_NOT_FOUND - Search could not find segment.                  */
/*                                                                          */
/* DESCRIPTION: Given a zero terminated string of a possible FM offset,     */
/*              return a pointer (in words) to the FM offset                */
/*                                                                          */
/* SIDE EFFECTS: NONE                                                       */
/*                                                                          */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC FMLabelToOffset(PSZ pszLabel,PULONG pulOffset)
{

    MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfxutl::FMLabelToOffset entry pszLabel %s\n",pszLabel);
   if (mwstrcmp(pszLabel, "SYSTMP0") == 0)
      {
      *pulOffset = FMCB_SYSTEMP0;
      MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspfxutl::FMLabelToOffset exit ulRC=0\n");
      return  DSP_NOERROR;
   }

   if (mwstrcmp(pszLabel, "SYSTMP1") == 0)
      {
      *pulOffset = FMCB_SYSTEMP1;
      MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspfxutl::FMLabelToOffset exit ulRC=0\n");
      return  DSP_NOERROR;
   }

   if (mwstrcmp(pszLabel, "SYSTMP2") == 0)
      {
      *pulOffset = FMCB_SYSTEMP2;
      MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspfxutl::FMLabelToOffset exit ulRC=0\n");
      return  DSP_NOERROR;
   }

   if (mwstrcmp(pszLabel, "SYSTMP3") == 0)
      {
      *pulOffset = FMCB_SYSTEMP3;
      MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspfxutl::FMLabelToOffset exit ulRC=0\n");
      return  DSP_NOERROR;
   }

   if (mwstrcmp(pszLabel, "SYSTMP4") == 0)
      {
      *pulOffset = FMCB_SYSTEMP4;
      MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspfxutl::FMLabelToOffset exit ulRC=0\n");
      return  DSP_NOERROR;
   }

 /* none of the above, return label not found error                         */
   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspfxutl::FMLabelToOffset exit ulRC=DSP_LABEL_NOT_FOUND\n");
   return  DSP_LABEL_NOT_FOUND;
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: IsposLabelToAddress                                     */
/*                                                                          */
/* FUNCTION: Finds offset of Label in a given segment.                      */
/*                                                                          */
/* INPUT:                                                                   */
/*   PRDSP prdspDSP - Pointer to DSP That owns an ISPOS Module              */
/*   PSZ pszLabel   - Zero terminated string holding ISPOS label            */
/*                    to search for                                         */
/*                                                                          */
/* OUTPUT:                                                                  */
/*  PULONG pulAddress    The address of the ISPOS label                     */
/*                                                                          */
/* RETURN:                                                                  */
/*                                                                          */
/*    DSP_NOERROR - OK, Segment pointer set                                 */
/*                                                                          */
/*    DSP_NAME_NOT_FOUND - Search could not find segment.                   */
/*                                                                          */
/* DESCRIPTION: Given a zero terminated string of an ISPOS label, return    */
/*              the address of that lable in ISPOS                          */
/*                                                                          */
/* SIDE EFFECTS: NONE                                                       */
/*                                                                          */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC IsposLabelToAddress(PRDSP prdspDSP, /* Pointer to DSP That owns an ISPOS */
                                       /* Module                            */
                       PSZ pszLabel,   /* Zero terminated string holding    */
                                       /* ISPOS label to search for         */
                       PULONG pulAddress /* The address of the ISPOS label  */
)

{

   /* Local variables                                                       */

   PRMOD      prmodCur;
   PRSEG      prsegBas;
   PRSEG      prsegCur;
   PRTSK      prtskBas;
   PRTSK      prtskCur;
   PARPUB     prPub;
   USHORT     PubCnt,i;

   /* Locate ISPOS module name from DSP Structure                           */
   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspfxutl::IsposLabelToAddress entry prdspDSP %x pszLabel %s\n",(int)prdspDSP,pszLabel);

   prmodCur = prdspDSP->DSP_prmodOS;
   if (prmodCur==NULL)  /* CH01 If MWAVEOS is not loaded */
      return( DSP_NAME_NOT_FOUND);

   /* Traverse tasks owned by ISPOS module                                  */

   prtskCur = prmodCur->MOD_prtskTail; /* Ptr to tasks in this module       */

   /* make sure we have tasks                                               */

   if (prtskCur == NULL) {
      return  DSP_NAME_NOT_FOUND;      /* no tasks, no seg to find          */
   }

   else {
      prtskBas = prtskCur;             /* save orig to tell when were done  */
   }                                   /* endif                             */
   do {

      if (prtskCur->TSK_prsegITail != NULL) {/* If there are Code Segments  */
         prsegCur = prtskCur->TSK_prsegITail;/* set a ptr to tail segment   */
         prsegBas = prsegCur;          /* save orig to tell when done       */
         do {                          /* Loop thru Code segments looking   */
                                       /* for seg label                     */

       /* Step 1 - Access the public definitions array of the segment at prseg*/

            prPub = prsegCur->SEG_parpub;/* Get pointer to segs Public Label*/
                                       /* defs                              */
            PubCnt = prsegCur->SEG_usPubsCount;/* Get Count of segs public  */
                                       /* defs                              */

          /* Step 2 - Search string table for label and return offset if found*/

            for (i = 0; i < PubCnt; i++, prPub++) {/* search labels in PUB  */
                                       /* table                             */
               if (mwstrcmp(
                  pszLabel, (PSZ)((BYTE *)(prsegCur->SEG_parstblStatic)+prPub
                  ->PUB_usLabel+1)) == 0) {

                  *pulAddress = prPub->PUB_ulOffset;
                  if ((prPub->PUB_usflg&PUB_EQUMASK) != PUB_EQUATE)
                     *pulAddress += prsegCur->SEG_prmem->MEM_ulSegAddr;

		  MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspfxutl::IsposLabelToAddress exit ulRC=0\n");

                  return  DSP_NOERROR;
               }                       /* endif                             */
            }                          /* endfor                            */

            prsegCur = prsegCur->SEG_prsegNext;/* get Ptr to next seg in    */
                                       /* Task                              */
         }  while (prsegBas != prsegCur);/* enddo                           */
      }                                /* End If                            */
      if (prtskCur->TSK_prsegDTail != NULL) {/* If there are Data Segments  */
         prsegCur = prtskCur->TSK_prsegDTail;/* set a ptr to tail segment   */
         prsegBas = prsegCur;          /* save orig to tell when done       */
         do {                          /* Loop thru Data segments looking   */
                                       /* for seg label                     */

       /* Step 1 - Access the public definitions array of the segment at prseg*/

            prPub = prsegCur->SEG_parpub;/* Get pointer to segs Public Label*/
                                       /* defs                              */
            PubCnt = prsegCur->SEG_usPubsCount;/* Get Count of segs public  */
                                       /* defs                              */

          /* Step 2 - Search string table for label and return offset if found*/

            for (i = 0; i < PubCnt; i++, prPub++) {/* search labels in PUB  */
                                       /* table                             */
               if (mwstrcmp(
                  pszLabel, (PSZ)((BYTE *)(prsegCur->SEG_parstblStatic)+prPub
                  ->PUB_usLabel+1)) == 0) {

                  *pulAddress = prPub->PUB_ulOffset;
                  if ((prPub->PUB_usflg&PUB_EQUMASK) != PUB_EQUATE)
                     *pulAddress += prsegCur->SEG_prmem->MEM_ulSegAddr;

		  MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfxutl::IsposLabelToAddress exit ulRC=0 ulAddress %lx\n",*pulAddress);

                  return  DSP_NOERROR;
               }                       /* endif                             */
            }                          /* endfor                            */

            prsegCur = prsegCur->SEG_prsegNext;/* get Ptr to next seg in    */
                                       /* Task                              */
         }  while (prsegBas != prsegCur);/* enddo                           */
      }                                /* End If                            */
      prtskCur = prtskCur->TSK_prtskNext;/* move Ptr to nxt task in module  */
   }  while (prtskCur != prtskBas);    /* enddo                             */

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspfxutl::IsposLabelToAddress exit ulRC=DSP_LABEL_NOT_FOUND\n");

   return  DSP_LABEL_NOT_FOUND;

}                                      /* End IsposLabelToAddress           */




/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: LabelToAddress                                          */
/*                                                                          */
/* FUNCTION: Finds address of a label in a task.                            */
/*                                                                          */
/* INPUT:                                                                   */
/*                                                                          */
/* OUTPUT:                                                                  */
/*   PULONG pulDSPAddr- Address of the label.                               */
/*                                                                          */
/* RETURN:                                                                  */
/*                                                                          */
/*    DSP_NOERROR - OK, label found                                         */
/*                                                                          */
/*    DSP_LABEL_NOT_FOUND - Search could not find label.                    */
/*                                                                          */
/*    DSP_INV_HANDLE      - invalid task handle.                            */
/*                                                                          */
/* DESCRIPTION:                                                             */
/*                                                                          */
/* SIDE EFFECTS: NONE                                                       */
/*                                                                          */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC LabelToAddress(HTASK hTask,PSZ pszName,PULONG pulDSPAddr)
{
   RC         ulRC = DSP_INV_HANDLE;   /* Assume DSP_INV_HANDLE            */
   PRSEG      prsegTemp;
   PRSEG      prsegTempRoot;
   USHORT     usAddrFlg = DSP_ABSOLUTE;

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspfxutl::LabelToAddress hTask %x pszName %s\n",(int)hTask,pszName);

   if (hTask->TSK_achValid == TSKVALIDATE) {
      prsegTemp = hTask->TSK_prsegDTail->SEG_prsegNext; // For data segments
      prsegTempRoot = prsegTemp;
      if ((prsegTempRoot != NULL) && (prsegTemp->SEG_achValid == SEGVALIDATE)){
         do {
            ulRC = LabelToOffset(prsegTemp, pszName, pulDSPAddr, &usAddrFlg);
            if (ulRC == DSP_NOERROR) {
		MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfxutl::LabelToAddress exit ulRC %lx\n",ulRC);

		return (ulRC);
	    }
            prsegTemp = prsegTemp->SEG_prsegNext;
         }  while (prsegTemp != prsegTempRoot);
      }
      prsegTemp = hTask->TSK_prsegITail->SEG_prsegNext; // For Instruction segments
      prsegTempRoot = prsegTemp;
      if ((prsegTempRoot != NULL) && (prsegTemp->SEG_achValid == SEGVALIDATE)){
         do {
            ulRC = LabelToOffset(prsegTemp, pszName, pulDSPAddr, &usAddrFlg);
	    if (ulRC == DSP_NOERROR) {
		MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfxutl::LabelToAddress exit ulRC %lx\n",ulRC);
		return (ulRC);
	    }
            prsegTemp = prsegTemp->SEG_prsegNext;
         }  while (prsegTemp != prsegTempRoot);
      }
      else {
         ulRC = DSP_LABEL_NOT_FOUND;
	 MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspfxutl::LabelToAddress error ulRC %lx Label: %s\n", ulRC,pszName);
      }
   }
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfxutl::LabelToAddress exit ulRC %lx\n",ulRC);
   return (ulRC);
}

