/*
 *   dspstruct.c -- Data structure allocation/deallocation and linkages
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <stdio.h>
#include <string.h>
#include "dspmgr.h"

#include "dspstruc.h"                  // Include function prototypes
#include "dsppcmem.h"                  // Include function prototypes
#include "dspglist.h"                  // Include function prototypes
#include "dspbios.h"                   // contains Run_IPC_Thread

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AllocRModule                                            */
/*                                                                          */
/* FUNCTION: Allocates memory for a RMOD data structure and initializes     */
/*           it to NULL.                                                    */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMOD * pprmodNode  - Ptr to Ptr to the new data structure        */
/*                              allocated.                                  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRMOD * pprmodNode  - Update To point to new data structure.      */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AllocRModule(PRMOD *pprmodNode)

{
   RC         ulRC;                    // error return code
   PRMOD      prmod;                   // Ptr to new data structure

   /* Allocate memory for data structure                                    */

   if ((ulRC = AllocatePerm((ULONG)sizeof(RMOD), (PVOID *)&prmod)) ==
      DSP_NOERROR) {

      /* NULL out fields in structure                                       */
      memset(prmod, INIT_VAL, sizeof(RMOD));

      prmod->MOD_achValid = MODVALIDATE;
      prmod->MOD_usflgState = MOD_INACTIVE;

      /* pass back to caller, the ptr to data structure                     */
      *pprmodNode = prmod;
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRModule                                           */
/*                                                                          */
/* FUNCTION: De-allocates memory for a RMOD data structure and removes it   */
/*           from the managers list of modules.                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMOD * pprmodTail - Ptr to linked-list.                          */
/*        PRMOD   prmodNode  - Pointer to the data structure to delete.     */
/*        PRMOD   prmodPrev  - Pointer to the previous Node in linked-list  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRMOD * pptmodTail - Tail may be updated.                         */
/*        ULONG   ReturnCode - 0 if no error                                */
/*                             !0 if error                                  */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveRModule(PRMOD *pprmodListTail,PRMOD prmodNode,PRMOD prmodPrev)

{
   ULONG      ulSize;

   /*************************************************************************/
   /* Unlink the node from the linked list, recover the                     */
   /* structure memory.                                                     */
   /*************************************************************************/

   CLLDelNode((PVOID *)&(prmodNode->MOD_prmodNext), (PVOID *)
      &(prmodPrev->MOD_prmodNext), (PVOID)prmodPrev, (PVOID *)pprmodListTail);

   prmodNode->MOD_achValid = 0;  /* unmark module structure */

   if (prmodNode->MOD_pszName != NULL) {
      ulSize = (ULONG)strlen(prmodNode->MOD_pszName);
      if (FreePerm((PVOID)prmodNode->MOD_pszName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (FreePerm((PVOID)prmodNode, (ULONG)sizeof(RMOD)))
      return (DSP_INTERNAL_ERROR);

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AllocRMemory                                            */
/*                                                                          */
/* FUNCTION: Allocates memory for a RMEM data structure and initializes it. */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMEM * pprmemNode  - Ptr to Ptr to the new data structure        */
/*                              allocated.                                  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRMEM * pprmemNode  - Update To point to new data structure.      */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AllocRMemory(PRMEM *pprmemNode)

{
   RC         ulRC;                    // error return code
   PRMEM      prmem;                   // Ptr to new data structure

   /* Allocate memory for data structure                                    */
   if ((ulRC = AllocatePerm((ULONG)sizeof(RMEM), (PVOID *)&prmem)) ==
      DSP_NOERROR) {

      /* NULL out fields in structure                                       */
      memset(prmem, INIT_VAL, sizeof(RMEM));

      prmem->MEM_usUseCount = 1;       // At least one at this point

      /* pass back to caller, the ptr to data structure                     */
      *pprmemNode = prmem;
   }
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRManagerTail                                         */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMGR * pprmgrListTail  - Add the new structure allocated to the  */
/*                                  tail of this linked-list.               */
/*        PRMGR * pprmgrNode  - Ptr to Ptr to the new data structure        */
/*                              allocated.                                  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRMGR * pprmgrListTail  - Update Tail pointer                     */
/*        PRMGR * pprmgrNode  - Update To point to new data structure.      */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRManagerTail(PRMGR *pprmgrListTail,PRMGR *pprmgrNode)

{
   RC         ulRC;                    // error return code
   PRMGR      prmgr;                   // Ptr to new data structure

   /* Allocate memory for data structure                                    */
   if ((ulRC = AllocatePerm((ULONG)sizeof(RMGR), (PVOID *)&prmgr)) ==
      DSP_NOERROR) {

      /* NULL out fields in structure                                       */
      memset(prmgr, INIT_VAL, sizeof(RMGR));

      prmgr->MGR_achValid = MGRVALIDATE;  /* Validate structure */

      /* Add data structure to manager linked-list                          */
      CLLInsTail((PVOID)prmgr, (PVOID *)&prmgr->MGR_prmgrNext, (PVOID *)
         &(*pprmgrListTail)->MGR_prmgrNext, (PVOID *)pprmgrListTail);

      /* pass back to caller, the ptr to data structure                     */
      *pprmgrNode = prmgr;
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRBiotTail                                            */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRBIOT * pprbiotListTail  - Add the new structure allocated to the*/
/*                                    tail of this linked-list.             */
/*        PRBIOT * pprbiotNode  - Ptr to Ptr to the new data structure      */
/*                                allocated.                                */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRBIOT * pprbiotListTail  - Update Tail pointer                   */
/*        PRBIOT * pprbiotNode  - Update To point to new data structure.    */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRBiotTail(PRBIOT *pprBiotListTail,PRBIOT *pprBiotNode)

{
   RC         ulRC;                    // error return code
   PRBIOT     prBiot;                  // Ptr to new data structure

   /* Allocate memory for data structure                                    */
   if ((ulRC = AllocatePerm((ULONG)sizeof(RBIOT), (PVOID *)&prBiot)) ==
      DSP_NOERROR) {

      /* NULL out fields in structure                                       */
      memset(prBiot, INIT_VAL, sizeof(RBIOT));

      /* Add data structure to hardware device linked-list                  */
      CLLInsTail((PVOID)prBiot, (PVOID *)&prBiot->BT_prBiotNext, (PVOID *)
         &(*pprBiotListTail)->BT_prBiotNext, (PVOID *)pprBiotListTail);

      /* pass back to caller, the ptr to data structure                     */
      *pprBiotNode = prBiot;
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRBiot                                             */
/*                                                                          */
/* FUNCTION: De-allocates memory for a RBIOT data structure and will        */
/*           unlink it from the linked-list it is attached to.              */
/*           Typically, the BIOT Structures should be removed when the      */
/*           Manager Structure is removed.                                  */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRBIOT * pprBiotListTail  - Ptr to linked-list.                   */
/*        PRBIOT prBiotNode  - Pointer to the data structure to delete.     */
/*        PRBIOT prBiotPrev  - Pointer to the previous Node in linked-list  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRHWDV * pprBiotListTail  - Tail may be updated.                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

//RC      RemoveRBiot (PRBIOT * pprBiotListTail, PRBIOT prBiotNode, PRBIOT prBiotPrev)
//
//{
//   RC     ulRC = DSP_NOERROR;           // error return code
//   ULONG  ulSize;                       // error return code
//
//   /* un-link node from linked-list */
//   CLLDelNode((PVOID *)&prBiotNode->BT_prBiotNext,
//              (PVOID *)&prBiotPrev->BT_prBiotNext,
//              (PVOID)prBiotPrev,
//              (PVOID *)pprBiotListTail);
//
//   /* Get Rid of the Strings in the BIOT structure if there are any. */
//
//   if ((ulRC==DSP_NOERROR)&&(prBiotNode->BT_pszConnector!=NULL)) {
//      ulSize = (ULONG)strlen(prBiotNode->BT_pszConnector);
//      FreePerm((PVOID)prBiotNode->BT_pszConnector, ++ulSize);
//   }
//
//   if ((ulRC==DSP_NOERROR)&&(prBiotNode->BT_pszTaskname!=NULL)) {
//      ulSize = (ULONG)strlen(prBiotNode->BT_pszTaskname);
//      FreePerm((PVOID)prBiotNode->BT_pszTaskname, ++ulSize);
//   }
//
//   if ((ulRC==DSP_NOERROR)&&(prBiotNode->BT_pszFilename!=NULL)) {
//      ulSize = (ULONG)strlen(prBiotNode->BT_pszFilename);
//      FreePerm((PVOID)prBiotNode->BT_pszFilename, ++ulSize);
//   }
//
//   /* free structure memory */
//   if(FreePerm((PVOID)prBiotNode, (ULONG)sizeof(RBIOT)))
//      return(DSP_INTERNAL_ERROR);
//
//   return(DSP_NOERROR);
//}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRDspTail                                             */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRDSP * pprdspListTail  - Add the new structure allocated to the  */
/*                                  tail of this linked-list.               */
/*        PRDSP * pprdspNode  - Ptr to Ptr to the new data structure        */
/*                              allocated.                                  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRDSP * pprdspListTail  - Update Tail pointer                     */
/*        PRDSP * pprdspNode  - Update To point to new data structure.      */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRDspTail(PRDSP *pprdspListTail,PRDSP *pprdspNode)

{
   RC         ulRC;                    // error return code
   PRDSP      prdsp;                   // Ptr to new data structure

   /* Allocate memory for data structure                                    */
   ulRC = AllocatePerm((ULONG)sizeof(RDSP), (PVOID *)&prdsp);
   if (ulRC == DSP_NOERROR) {

      /* NULL out fields in structure                                       */
      memset(prdsp, INIT_VAL, sizeof(RDSP));

      /* Add data structure to DSP linked-list                              */
      CLLInsTail((PVOID)prdsp, (PVOID *)&prdsp->DSP_prdspNext, (PVOID *)
         &(*pprdspListTail)->DSP_prdspNext, (PVOID *)pprdspListTail);

      prdsp->DSP_achValid = DSPVALIDATE; /* Validate structure */

      prdsp->DSP_hDSP = prdsp;     // Non-Global PTR to Its self

      prdsp->DSP_hDSPG = prdsp;    // Global PTR to Its sel

      /* pass back to caller, the ptr to data structure                     */
      *pprdspNode = prdsp;
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRIPCTable                                            */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/*           The IPC Table is composed of entries that correlate tasks      */
/*           to Callback addresses for Interrupts.                          */
/*           There are 16 IPCs per DSP.                                     */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRIPCTE *ppripcteListTail  - Add the new struct allocated to the  */
/*                                     tail of this linked-list.            */
/*        PRIPCTE *ppripcteNode - Ptr to Ptr to the new data structure      */
/*                                allocated.                                */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRIPCTE *ppripcteListTail  - Update Tail pointer                  */
/*        PRIPCTE *ppripcteNode      - Update To point to new data struct   */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/* NOTE:    Since we never get rid of DSPs? We never get rid of this?       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRIPCTableTail(PRIPCTE *ppripcteListTail)
{
   RC         ulRC;                    // error return code
   ULONG      i;                       // Counter
   USHORT     usMask;                  // Mask
   PRIPCTE    pripcte;                 // Ptr to new data structure
   PRIPCTE    pripctenew;              // Ptr to new data structure

   /*************************************************************************/
   /* Allocate memory for data structure only if pointer is NULL            */
   /*************************************************************************/
   ulRC = DSP_NOERROR;
   if (*ppripcteListTail == NULL)      /* get new table pointer             */
      ulRC = AllocatePerm((ULONG)sizeof(RIPCTE)*NUMIPC, (PVOID *)&pripctenew);
   else
      pripctenew = *ppripcteListTail;  /* use existing table pointer        */

   if (ulRC == DSP_NOERROR) {

      /**********************************************************************/
      /* Initialize each entry in the table                                 */
      /**********************************************************************/

      pripcte = pripctenew;            /* pointer to first element in IPC   */
                                       /* table                             */

      /**********************************************************************/
      /* The order in which these elements are stored is significant.       */
      /* The pfnCallBack is set to NULL first to disable the inter-         */
      /* rupt handler from calling the function (reinit only).              */
      /**********************************************************************/

      usMask = 1;
      for (i = 1; i <= NUMIPC; i++) {
         pripcte->IPCTE_usMask = usMask;
         usMask = usMask << 1;
         if ((PRIPCTE)pripcte->IPCTE_pfnCallBack == pripcte) {
            pripcte->IPCTE_pfnCallBack = NULL;
         }
         else {
            pripcte->IPCTE_pfnCallBack = NULL;
         }
         pripcte->IPCTE_usIntCount = 0;
         pripcte->IPCTE_prtsk = NULL;
         pripcte->IPCTE_usType = (USHORT)IPCTE_IDCCALL;
         pripcte->IPCTE_achValid = IPCTEVALIDATE; /* Validate structure */
         pripcte++;                    /* point to next element in IPC table*/
      }                                /* endfor                            */

      /**********************************************************************/
      /* Add to the IPCTableTail in DSP                                     */
      /* Return the New Table.                                              */
      /**********************************************************************/

      *ppripcteListTail = pripctenew;
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRFrameMgrTail                                        */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRFM * pprfmListTail  - Add the new structure allocated to the    */
/*                                tail of this linked-list.                 */
/*        PRFM * pprfmNode  - Ptr to Ptr to the new data structure          */
/*                            allocated.                                    */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRFM * pprfmListTail  - Update Tail pointer                       */
/*        PRFM * pprfmNode  - Update To point to new data structure.        */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRFrameMgrTail(PRFM *pprfmListTail,PRFM *pprfmNode)

{
   RC         ulRC;                    // error return code
   PRFM       prfm;                    // Ptr to new data structure
   PRFM       prfmHead;                // Ptr to HeadACHE

   /* Allocate memory for data structure                                    */
   if ((ulRC = AllocatePerm((ULONG)sizeof(RFM), (PVOID *)&prfm)) ==
      DSP_NOERROR) {

      /* NULL out fields in structure                                       */
      memset(prfm, INIT_VAL, sizeof(RFM));

      /* Add data structure to Frame manager linked-list                    */
      prfmHead = (*pprfmListTail == NULL)?NULL:(*pprfmListTail)->FM_prfmNext;
      CDLLInsTail((PVOID)prfm, (PVOID *)&prfm->FM_prfmNext, (PVOID *)
         &prfm->FM_prfmPrev, (PVOID *)&(*pprfmListTail)->FM_prfmNext, (PVOID
         *)&prfmHead->FM_prfmPrev, (PVOID *)pprfmListTail);

      prfm->FM_achValid = FMVALIDATE;  /* Validate structure */

      prfm->FM_usUseCount = 1;         // At least one at this point
      prfm->FM_usflgType = FM_REALTIME;

      /* pass back to caller, the ptr to data structure                     */
      *pprfmNode = prfm;
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRFrameMgr                                         */
/*                                                                          */
/* FUNCTION: .........De-allocates memory for a data structure and will     */
/*           De-allocate all children data structures from the DSP Manager  */
/*           data tree.                                                     */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRFM  * pprfmListHead  - Ptr to linked-list.                      */
/*        PRFM  prfmNode  - Pointer to the data structure to delete.        */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRFM  * pprfmListHead  - Head may be updated.                     */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveRFrameMgr(PRFM prfm,PRTSK prtsk)

{
   ULONG      ulSize;

   /*************************************************************************/
   /* Unlink the node from the linked list, recover the                     */
   /* structure memory.                                                     */
   /*************************************************************************/

   CDLLDelNode((PVOID *)&prfm->FM_prfmNext, (PVOID *)&prfm->FM_prfmPrev,
      (PVOID *)&prfm->FM_prfmPrev->FM_prfmNext, (PVOID *)
      &prfm->FM_prfmNext->FM_prfmPrev, (PVOID *)
      &prtsk->TSK_prdsp->DSP_prfmTail);

   prfm->FM_achValid = 0;  /* Unmark FM structure */

   prtsk->TSK_prfm = NULL;

   if (prfm->FM_pszName != NULL) {
      ulSize = (ULONG)strlen(prfm->FM_pszName);
      FreePerm((PVOID)prfm->FM_pszName, ++ulSize);
   }

   if (FreePerm((PVOID)prfm, (ULONG)sizeof(RFM)))
      return (DSP_INTERNAL_ERROR);

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRGPCConnListTail                                     */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRGPCCL * pprgpcclListTail  - Add the new structure allocated to  */
/*                                      the tail of this linked-list.       */
/*        PRGPCCL * pprgpcclNode  - Ptr to Ptr to the new data structure    */
/*                                  allocated.                              */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRGPCCL * pprgpcclListTail  - Update Tail pointer                 */
/*        PRGPCCL * pprgpcclNode  - Update To point to new data structure.  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRGPCConnListTail(PRGPCCL *pprgpcclListTail,PRGPCCL *pprgpcclNode)

{
   RC         ulRC;                    // error return code
   PRGPCCL    prgpccl;                 // Ptr to new data structure

   /* Allocate memory for data structure                                    */
   if ((ulRC = AllocateTemp((ULONG)sizeof(RGPCCL), 1, (PVOID *)&prgpccl)) ==
      DSP_NOERROR) {

      /* NULL out fields in structure                                       */
      memset(prgpccl, INIT_VAL, sizeof(RGPCCL));

      /* Add data structure to GPC connection list linked-list              */
      CLLInsTail((PVOID)prgpccl, (PVOID *)&prgpccl->GPCCL_prgpcclNext,
                (PVOID *)&(*pprgpcclListTail)->GPCCL_prgpcclNext,
                (PVOID *)pprgpcclListTail);

      /* pass back to caller, the ptr to data structure                     */
      *pprgpcclNode = prgpccl;
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRGPCConnList                                      */
/*                                                                          */
/* FUNCTION: De-allocates memory for a RGPCCL data structure and will       */
/*           unlink it from the linked-list it is attached to. This routine */
/*           will free memory of ASCIIZ strings that this structure points  */
/*           to.                                                            */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRGPCCL * pprgpcclListTail  - Ptr to linked-list.                 */
/*        PRGPCCL prgpcclNode  - Pointer to the data structure to delete.   */
/*        PRGPCCL prgpcclPrev  - Pointer to the previous Node in linked-list*/
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRGPCCL * pprgpcclListTail  - Tail may be updated.                */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveRGPCConnList(PRGPCCL *pprgpcclListTail,PRGPCCL prgpcclNode,
                      PRGPCCL prgpcclPrev)

{
   ULONG      ulSize;                  // string length

   /* un-link node from linked-list                                         */
   CLLDelNode((PVOID *)&prgpcclNode->GPCCL_prgpcclNext, (PVOID *)
      &prgpcclPrev->GPCCL_prgpcclNext, (PVOID)prgpcclPrev, (PVOID *)
      pprgpcclListTail);

   /*************************************************************************/
   /* Process all of GPC1 then GPC2                                         */
   /* Determine length of string and free                                   */
   /*************************************************************************/

   if (prgpcclNode->GPCCL_pszGpc1FileName != NULL) {
      ulSize = (ULONG)strlen(prgpcclNode->GPCCL_pszGpc1FileName);
      if (FreePerm((PVOID)prgpcclNode->GPCCL_pszGpc1FileName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (prgpcclNode->GPCCL_pszGpc1ModName != NULL) {
      ulSize = (ULONG)strlen(prgpcclNode->GPCCL_pszGpc1ModName);
      if (FreePerm((PVOID)prgpcclNode->GPCCL_pszGpc1ModName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (prgpcclNode->GPCCL_pszGpc1TaskName != NULL) {
      ulSize = (ULONG)strlen(prgpcclNode->GPCCL_pszGpc1TaskName);
      if (FreePerm((PVOID)prgpcclNode->GPCCL_pszGpc1TaskName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (prgpcclNode->GPCCL_pszGpc1SegName != NULL) {
      ulSize = (ULONG)strlen(prgpcclNode->GPCCL_pszGpc1SegName);
      if (FreePerm((PVOID)prgpcclNode->GPCCL_pszGpc1SegName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (prgpcclNode->GPCCL_pszGpc1GPCName != NULL) {
      ulSize = (ULONG)strlen(prgpcclNode->GPCCL_pszGpc1GPCName);
      if (FreePerm((PVOID)prgpcclNode->GPCCL_pszGpc1GPCName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (prgpcclNode->GPCCL_pszGpc2FileName != NULL) {
      ulSize = (ULONG)strlen(prgpcclNode->GPCCL_pszGpc2FileName);
      if (FreePerm((PVOID)prgpcclNode->GPCCL_pszGpc2FileName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (prgpcclNode->GPCCL_pszGpc2ModName != NULL) {
      ulSize = (ULONG)strlen(prgpcclNode->GPCCL_pszGpc2ModName);
      if (FreePerm((PVOID)prgpcclNode->GPCCL_pszGpc2ModName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (prgpcclNode->GPCCL_pszGpc2TaskName != NULL) {
      ulSize = (ULONG)strlen(prgpcclNode->GPCCL_pszGpc2TaskName);
      if (FreePerm((PVOID)prgpcclNode->GPCCL_pszGpc2TaskName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (prgpcclNode->GPCCL_pszGpc2SegName != NULL) {
      ulSize = (ULONG)strlen(prgpcclNode->GPCCL_pszGpc2SegName);
      if (FreePerm((PVOID)prgpcclNode->GPCCL_pszGpc2SegName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (prgpcclNode->GPCCL_pszGpc2GPCName != NULL) {
      ulSize = (ULONG)strlen(prgpcclNode->GPCCL_pszGpc2GPCName);
      if (FreePerm((PVOID)prgpcclNode->GPCCL_pszGpc2GPCName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   /* free structure memory                                                 */
   if (FreeTemp((PVOID)prgpcclNode, (ULONG)sizeof(RGPCCL)))
      return (DSP_INTERNAL_ERROR);

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRITCBConnListTail                                    */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRITCBCL * ppritcbcclListTail  - Add the new structure allocated  */
/*                                         to the tail of this linked-list. */
/*        PRITCBCL * ppritcbclNode  - Ptr to Ptr to the new data structure  */
/*                                    allocated.                            */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRITCBCL *ppritcbcclListTail  - Update Tail pointer               */
/*        PRITCBCL * ppritcbclNode - Update To point to new data structure. */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRITCBConnListTail(PRITCBCL *ppritcbclListTail,PRITCBCL *ppritcbclNode)

{
   RC         ulRC;                    // error return code
   PRITCBCL   pritcbcl;                // Ptr to new data structure

   /* Allocate memory for data structure                                    */

   if ((ulRC = AllocateTemp((ULONG)sizeof(RITCBCL), 1, (PVOID *)&pritcbcl)) ==
      DSP_NOERROR) {

      /* NULL out fields in structure                                       */
      memset(pritcbcl, INIT_VAL, sizeof(RITCBCL));

      /* Add data structure to ITCB connection list linked-list             */
      CLLInsTail((PVOID)pritcbcl, (PVOID *)&pritcbcl->ITCBCL_pritcbclNext,
         (PVOID *)&(*ppritcbclListTail)->ITCBCL_pritcbclNext, (PVOID *)
         ppritcbclListTail);

      /* pass back to caller, the ptr to data structure                     */
      *ppritcbclNode = pritcbcl;
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRITCBConnList                                     */
/*                                                                          */
/* FUNCTION: De-allocates memory for a RITCBCL data structure and will      */
/*           unlink it from the linked-list it is attached to. This routine */
/*           will free memory of ASCIIZ strings that this structure points  */
/*           to.                                                            */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRITCBCL * ppritcbclListTail  - Ptr to linked-list.               */
/*        PRITCBCL pritcbclNode  - Pointer to the data structure to delete. */
/*        PRITCBCL pritcbclPrev  - Pointer to the prev Node in linked-list  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRITCBCL * ppritcbclListTail  - Tail may be updated.              */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveRITCBConnList(PRITCBCL *ppritcbclListTail, PRITCBCL pritcbclNode,
                       PRITCBCL pritcbclPrev)

{
   ULONG      ulSize;                  // string length

   /* un-link node from linked-list                                         */
   CLLDelNode((PVOID *)&pritcbclNode->ITCBCL_pritcbclNext, (PVOID *)
      &pritcbclPrev->ITCBCL_pritcbclNext, (PVOID)pritcbclPrev, (PVOID *)
      ppritcbclListTail);

   /*************************************************************************/
   /* Process all of ITCBP then ITCBS                                       */
   /* Determine length of string and free                                   */
   /*************************************************************************/

   if (pritcbclNode->ITCBCL_pszITCBPFileName != NULL) {
      ulSize = (ULONG)strlen(pritcbclNode->ITCBCL_pszITCBPFileName);
     if (FreePerm((PVOID)pritcbclNode->ITCBCL_pszITCBPFileName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (pritcbclNode->ITCBCL_pszITCBPModName != NULL) {
      ulSize = (ULONG)strlen(pritcbclNode->ITCBCL_pszITCBPModName);
      if (FreePerm((PVOID)pritcbclNode->ITCBCL_pszITCBPModName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (pritcbclNode->ITCBCL_pszITCBPTaskName != NULL) {
      ulSize = (ULONG)strlen(pritcbclNode->ITCBCL_pszITCBPTaskName);
     if (FreePerm((PVOID)pritcbclNode->ITCBCL_pszITCBPTaskName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (pritcbclNode->ITCBCL_pszITCBPSegName != NULL) {
      ulSize = (ULONG)strlen(pritcbclNode->ITCBCL_pszITCBPSegName);
      if (FreePerm((PVOID)pritcbclNode->ITCBCL_pszITCBPSegName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (pritcbclNode->ITCBCL_pszITCBPName != NULL) {
      ulSize = (ULONG)strlen(pritcbclNode->ITCBCL_pszITCBPName);
      if (FreePerm((PVOID)pritcbclNode->ITCBCL_pszITCBPName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (pritcbclNode->ITCBCL_pszITCBSFileName != NULL) {
      ulSize = (ULONG)strlen(pritcbclNode->ITCBCL_pszITCBSFileName);
     if (FreePerm((PVOID)pritcbclNode->ITCBCL_pszITCBSFileName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (pritcbclNode->ITCBCL_pszITCBSModName != NULL) {
      ulSize = (ULONG)strlen(pritcbclNode->ITCBCL_pszITCBSModName);
      if (FreePerm((PVOID)pritcbclNode->ITCBCL_pszITCBSModName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (pritcbclNode->ITCBCL_pszITCBSTaskName != NULL) {
      ulSize = (ULONG)strlen(pritcbclNode->ITCBCL_pszITCBSTaskName);
     if (FreePerm((PVOID)pritcbclNode->ITCBCL_pszITCBSTaskName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (pritcbclNode->ITCBCL_pszITCBSSegName != NULL) {
      ulSize = (ULONG)strlen(pritcbclNode->ITCBCL_pszITCBSSegName);
      if (FreePerm((PVOID)pritcbclNode->ITCBCL_pszITCBSSegName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (pritcbclNode->ITCBCL_pszITCBSName != NULL) {
      ulSize = (ULONG)strlen(pritcbclNode->ITCBCL_pszITCBSName);
      if (FreePerm((PVOID)pritcbclNode->ITCBCL_pszITCBSName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   /* free structure memory                                                 */
   if (FreeTemp((PVOID)pritcbclNode, (ULONG)sizeof(RITCBCL)))
      return (DSP_INTERNAL_ERROR);

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRTaskTail                                            */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRTSK * pprtskListTail  - Add the new structure allocated to the  */
/*                                  tail of this linked-list.               */
/*        PRTSK * pprtskNode  - Ptr to Ptr to the new data structure        */
/*                              allocated.                                  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRTSK * pprtskListTail  - Update Tail pointer                     */
/*        PRTSK * pprtskNode  - Update To point to new data structure.      */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRTaskTail(PRTSK *pprtskListTail,PRTSK *pprtskNode)

{
   RC         ulRC;             // error return code
   PRTSK      prtsk;            // Ptr to new data structure (permanent part)
   PRTSKTMP   prtsktmp;         // Ptr to new data structure (temporary part)

   /* Allocate memory for permanent part of data structure                  */
   if ((ulRC = AllocatePerm((ULONG)sizeof(RTSK), (PVOID *)&prtsk)) ==
      DSP_NOERROR) {

      /* Allocate memory for temporary part of data structure               */
      if ((ulRC = AllocateTemp((ULONG)sizeof(RTSKTMP), 1, (PVOID *)&prtsktmp))
         == DSP_NOERROR) {

         /* NULL out fields in permanent part of data structure             */
         memset(prtsk, INIT_VAL, sizeof(RTSK));

         /* Add permanent part to task linked-list                          */
         CLLInsTail((PVOID)prtsk, (PVOID *)&prtsk->TSK_prtskNext, (PVOID *)
            &(*pprtskListTail)->TSK_prtskNext, (PVOID *)pprtskListTail);

         prtsk->TSK_achValid = TSKVALIDATE;  /* Validate structure */

         prtsk->TSK_usflgType = 0xFFFF;
         prtsk->TSK_usflgState = TSK_INACTIVE;
         prtsk->TSK_prtsktmp = prtsktmp; // point to temp structure

         /* NULL out fields in temporary part of data structure             */
         memset(prtsktmp, INIT_VAL, sizeof(RTSKTMP));

         /* pass back to caller, the ptr to data structure                  */
         *pprtskNode = prtsk;
      }
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRTask                                             */
/*                                                                          */
/* FUNCTION: .........De-allocates memory for a data structure and will     */
/*           De-allocate all children data structures from the DSP Manager  */
/*           data tree.                                                     */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRTSK * pprtskListTail  - Ptr to linked-list.                     */
/*        PRTSK prtskNode  - Pointer to the data structure to delete.       */
/*        PRTSK prtskPrev  - Pointer to the previous Node in linked-list    */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRTSK * pprtskListTail  - Tail may be updated.                    */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveRTask(PRTSK *pprtskListTail,PRTSK prtskNode,PRTSK prtskPrev)

{
   ULONG      ulSize;

   /*************************************************************************/
   /* Unlink the node from the linked list, recover the structure memory    */
   /*************************************************************************/

   CLLDelNode((PVOID *)&(prtskNode->TSK_prtskNext),
              (PVOID *)&(prtskPrev->TSK_prtskNext),
              (PVOID)prtskPrev, (PVOID *)pprtskListTail);

   prtskNode->TSK_achValid = 0;  /* Unmark task structure */

   if (prtskNode->TSK_pszVirtName != NULL) {
      ulSize = (ULONG)strlen(prtskNode->TSK_pszVirtName);
      if (FreePerm((PVOID)prtskNode->TSK_pszVirtName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (FreePerm((PVOID)prtskNode, (ULONG)sizeof(RTSK)))
      return (DSP_INTERNAL_ERROR);

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRTasktmp                                          */
/*                                                                          */
/* FUNCTION: De-allocates memory for temporary task structure and also      */
/*           all strings that it points to.                                 */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRTSK prtskNode  - Pointer to the permanent task structure        */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveRTasktmp(PRTSK prtskNode)

{
   ULONG      ulSize;                  // String length
   PRTSKTMP   prtsktmp;                // ptr to temporary structure

   prtsktmp = prtskNode->TSK_prtsktmp;
   prtskNode->TSK_prtsktmp = NULL;

   /* Determine length of string and free it                                */
   if (prtsktmp->TSK_rslActiveEntry.pszSegment != NULL) {
      ulSize = (ULONG)strlen(prtsktmp->TSK_rslActiveEntry.pszSegment);
      if (FreePerm((PVOID)prtsktmp->TSK_rslActiveEntry.pszSegment, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   /* Determine length of string and free it                                */
   if (prtsktmp->TSK_rslActiveEntry.pszLabel != NULL) {
      ulSize = (ULONG)strlen(prtsktmp->TSK_rslActiveEntry.pszLabel);
     if (FreePerm((PVOID)prtsktmp->TSK_rslActiveEntry.pszLabel, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   /* Determine length of string and free it                                */
   if (prtsktmp->TSK_rslStandbyEntry.pszSegment != NULL) {
      ulSize = (ULONG)strlen(prtsktmp->TSK_rslStandbyEntry.pszSegment);
      if (FreePerm((PVOID)prtsktmp->TSK_rslStandbyEntry.pszSegment, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   /* Determine length of string and free it                                */
   if (prtsktmp->TSK_rslStandbyEntry.pszLabel != NULL) {
      ulSize = (ULONG)strlen(prtsktmp->TSK_rslStandbyEntry.pszLabel);
    if (FreePerm((PVOID)prtsktmp->TSK_rslStandbyEntry.pszLabel, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   /* Determine length of string and free it                                */
   if (prtsktmp->TSK_rslALUEntry.pszSegment != NULL) {
      ulSize = (ULONG)strlen(prtsktmp->TSK_rslALUEntry.pszSegment);
      if (FreePerm((PVOID)prtsktmp->TSK_rslALUEntry.pszSegment, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   /* Determine length of string and free it                                */
   if (prtsktmp->TSK_rslALUEntry.pszLabel != NULL) {
      ulSize = (ULONG)strlen(prtsktmp->TSK_rslALUEntry.pszLabel);
      if (FreePerm((PVOID)prtsktmp->TSK_rslALUEntry.pszLabel, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   /* Determine length of string and free it                                */
   if (prtsktmp->TSK_pszRealName != NULL) {
      ulSize = (ULONG)strlen(prtsktmp->TSK_pszRealName);
      if (FreePerm((PVOID)prtsktmp->TSK_pszRealName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   /* Determine length of string and free it                                */
   if (prtsktmp->TSK_pszFileName != NULL) {
      ulSize = (ULONG)strlen(prtsktmp->TSK_pszFileName);
      if (FreePerm((PVOID)prtsktmp->TSK_pszFileName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (prtsktmp->TSK_pszFMGroupName != NULL) {
      ulSize = (ULONG)strlen(prtsktmp->TSK_pszFMGroupName);
      if (FreePerm((PVOID)prtsktmp->TSK_pszFMGroupName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   /* free structure memory                                                 */
   if (FreeTemp((PVOID)prtsktmp, (ULONG)sizeof(RTSKTMP)))
      return (DSP_INTERNAL_ERROR);

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRSegmentTail                                         */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRSEG * pprsegListTail  - Add the new structure allocated to the  */
/*                                  tail of this linked-list.               */
/*        PRSEG * pprsegNode  - Ptr to Ptr to the new data structure        */
/*                              allocated.                                  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRSEG * pprsegListTail  - Update Tail pointer                     */
/*        PRSEG * pprsegNode  - Update To point to new data structure.      */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRSegmentTail(PRSEG *pprsegListTail,PRSEG *pprsegNode)

{
   RC         ulRC;            // error return code
   PRSEG      prseg;           // Ptr to new data structure (permanent part)
   PRSEGTMP   prsegtmp;        // Ptr to new data structure (temporary part)

   /* Allocate memory for permanent part of data structure                  */
   if ((ulRC = AllocatePerm((ULONG)sizeof(RSEG), (PVOID *)&prseg)) ==
      DSP_NOERROR) {

      /* Allocate memory for temporary part of data structure               */
      if ((ulRC = AllocateTemp((ULONG)sizeof(RSEGTMP), 1, (PVOID *)&prsegtmp))
         == DSP_NOERROR) {

         /* NULL out fields in permanent part of data structure             */
         memset(prseg, INIT_VAL, sizeof(RSEG));

         /* Add permanent part to segment linked-list                       */
         CLLInsTail((PVOID)prseg, (PVOID *)&prseg->SEG_prsegNext, (PVOID *)
            &(*pprsegListTail)->SEG_prsegNext, (PVOID *)pprsegListTail);

         prseg->SEG_achValid = SEGVALIDATE;  /* Validate structure */

         prseg->SEG_usflg = SEG_PURE|SEG_REAL;
         prseg->SEG_prsegtmp = prsegtmp; // point to temp structure

         /* NULL out fields in temporary part of data structure             */
         memset(prsegtmp, INIT_VAL, sizeof(RSEGTMP));

         prsegtmp->SEG_ulAlign = 1;

         /* pass back to caller, the ptr to data structure                  */
         *pprsegNode = prseg;
      }
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRSegment                                          */
/*                                                                          */
/* FUNCTION: .........De-allocates memory for a data structure and will     */
/*           De-allocate all children data structures from the DSP Manager  */
/*           data tree.                                                     */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRSEG * pprsegListTail  - Ptr to linked-list.                     */
/*        PRSEG prsegNode  - Pointer to the data structure to delete.       */
/*        PRSEG prsegPrev  - Pointer to the previous Node in linked-list    */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRSEG * pprsegListTail  - Tail may be updated.                    */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveRSegment(PRSEG *pprsegListTail,PRSEG prsegNode,PRSEG prsegPrev)

{
   ULONG      ulSize;

   /*************************************************************************/
   /* Unlink the node from the linked list, recover the structure memory    */
   /*************************************************************************/

   CLLDelNode((PVOID *)&(prsegNode->SEG_prsegNext),
              (PVOID *)&(prsegPrev->SEG_prsegNext),
              (PVOID)prsegPrev, (PVOID *)pprsegListTail);

   prsegNode->SEG_achValid = 0;  /* Unmark segment structure */

   if (prsegNode->SEG_pszVirtName != NULL) {
      ulSize = (ULONG)strlen(prsegNode->SEG_pszVirtName);
      if (FreePerm((PVOID)prsegNode->SEG_pszVirtName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   /*************************************************************************/
   /* free the memory occupied by the RealSegName                           */
   /*************************************************************************/

   if ((prsegNode->SEG_pszRealSegName != NULL) &&
      (prsegNode->SEG_pszRealSegName != prsegNode->SEG_pszVirtName))
      {
      ulSize = (ULONG)strlen(prsegNode->SEG_pszRealSegName);
      if (FreePerm((PVOID)prsegNode->SEG_pszRealSegName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   if (FreePerm((PVOID)prsegNode, (ULONG)sizeof(RSEG)))
      return (DSP_INTERNAL_ERROR);

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRSegmenttmp                                       */
/*                                                                          */
/* FUNCTION: De-allocates memory for a temporary segment data structure and */
/*           of the ASCIIZ strings it points to. This routine will also     */
/*           de-allocate the binary image linked-list, the discardable      */
/*           string table, and the fixup label table.                       */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRSEG prsegNode  - Pointer to the permenent segment structure.    */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveRSegmenttmp(PRSEG prsegNode)

{
   ULONG      ulSize;                  /* String length                     */
   ULONG      ulRC;                    /* Return Code                       */
   PRSEGTMP   prsegtmp;                /* ptr to temporary structure        */
   PRIMG      primgTail;               /* Pointer to Image linked-list      */

   prsegtmp = prsegNode->SEG_prsegtmp;
   prsegNode->SEG_prsegtmp = NULL;

   /* Determine length of string and free it                                */

   if (prsegtmp->SEG_pszRealName != NULL) {
      ulSize = (ULONG)strlen(prsegtmp->SEG_pszRealName);
      if (FreePerm((PVOID)prsegtmp->SEG_pszRealName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   /* Determine length of string and free it                                */

   if (prsegtmp->SEG_pszFileName != NULL) {
      ulSize = (ULONG)strlen(prsegtmp->SEG_pszFileName);
      if (FreePerm((PVOID)prsegtmp->SEG_pszFileName, ++ulSize))
         return (DSP_INTERNAL_ERROR);
   }

   /* Free Discard table                                                    */

   if ((PVOID)prsegtmp->SEG_parstblDiscard != NULL && FreeTemp((PVOID)
      prsegtmp->SEG_parstblDiscard, prsegtmp->SEG_ulDiscardSize))
      return (DSP_INTERNAL_ERROR);

   /* Free Fixup table                                                      */

   if ((PVOID)prsegtmp->SEG_parfix != NULL && FreeTemp((PVOID)
      prsegtmp->SEG_parfix, prsegtmp->SEG_ulFixupSize))
      return (DSP_INTERNAL_ERROR);

   /* Free image linked-list. It should have been freed by this point,
      let's make sure.                                                      */

   primgTail = prsegtmp->SEG_primgTail;

   while (primgTail != NULL) {
      ulRC = RemoveRImage((PRIMG *)&primgTail,
                          (PRIMG)primgTail->IMG_primgNext,
                          (PRIMG)primgTail);
      if (ulRC != DSP_NOERROR)
         return (DSP_INTERNAL_ERROR);
   }

   /* Free structure memory                                                 */
   if (FreeTemp((PVOID)prsegtmp, (ULONG)sizeof(RSEGTMP)))
      return (DSP_INTERNAL_ERROR);

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRImageTail                                           */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRIMG * pprimgListTail  - Add the new structure allocated to the  */
/*                                  tail of this linked-list.               */
/*        PRIMG * pprimgNode  - Ptr to Ptr to the new data structure        */
/*                              allocated.                                  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRIMG * pprimgListTail  - Update Tail pointer                     */
/*        PRIMG * pprimgNode  - Update To point to new data structure.      */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRImageTail(PRIMG *pprimgListTail,PRIMG *pprimgNode)

{
   RC         ulRC;                    // error return code

   if ((ulRC = AddRImageHead(pprimgListTail, pprimgNode)) == DSP_NOERROR)
      *pprimgListTail = *pprimgNode;
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRImageHead                                           */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the head of a linked-list of similar structures in the DSP     */
/*           Manager data tree. (Used to add the TCB structure to DDS).     */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRIMG * pprimgListTail  - Add the new structure allocated to the  */
/*                                  tail of this linked-list.               */
/*        PRIMG * pprimgNode  - Ptr to Ptr to the new data structure        */
/*                              allocated.                                  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRIMG * pprimgListTail  - Update Tail pointer                     */
/*        PRIMG * pprimgNode  - Update To point to new data structure.      */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRImageHead(PRIMG *pprimgListTail,PRIMG *pprimgNode)

{
   RC         ulRC;                    // error return code
   PRIMG      primg;                   // Ptr to new data structure

   /* Allocate memory for data structure                                    */
   if ((ulRC = AllocateTemp((ULONG)sizeof(RIMG), 1, (PVOID *)&primg)) ==
      DSP_NOERROR) {

      /* NULL out fields in structure                                       */
      memset(primg, INIT_VAL, sizeof(RIMG));

      /* Add data structure to image linked-list                            */
      CLLInsHead((PVOID)primg, (PVOID *)&primg->IMG_primgNext, (PVOID *)
         &(*pprimgListTail)->IMG_primgNext, (PVOID *)pprimgListTail);

      /* pass back to caller, the ptr to data structure                     */
      *pprimgNode = primg;
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRImage                                            */
/*                                                                          */
/* FUNCTION: De-allocates memory for the Image structure as well as the     */
/*           actual image memory also. It also removes this node from the   */
/*           linked-list it is attached to.                                 */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRIMG * pprimgListTail  - Ptr to linked-list.                     */
/*        PRIMG primgNode  - Pointer to the data structure to delete.       */
/*        PRIMG primgPrev  - Pointer to the previous Node in linked-list    */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRIMG * pprimgListTail  - Tail may be updated.                    */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveRImage(PRIMG *pprimgListTail,PRIMG primgNode,PRIMG primgPrev)

{
   /* un-link node from linked-list                                         */
   CLLDelNode((PVOID *)&primgNode->IMG_primgNext, (PVOID *)
      &primgPrev->IMG_primgNext, (PVOID)primgPrev, (PVOID *)pprimgListTail);

   /* free structure memory                                                 */
   if (FreeTemp((PVOID)primgNode, (ULONG)sizeof(RIMG)))
      return (DSP_INTERNAL_ERROR);

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRLDMATail                                            */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRLDMA * pprldmaListTail - Add the new structure allocated to the */
/*                                   tail of this linked-list.              */
/*        PRLDMA * pprldmaNode     - Ptr to Ptr to the new data structure   */
/*                                   allocated.                             */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRDMA * pprldmaListTail  - Update Tail pointer                    */
/*        PRDMA * pprldmaNode      - Update To point to new data structure  */
/*        ULONG  ReturnCode        - 0 if no error                          */
/*                                   !0 if error                            */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRLDMATail(PRLDMA *pprldmaListTail,PRLDMA *pprldmaNode)

{
   RC         ulRC;                    // error return code
   PRLDMA     prldma;                  // Ptr to new data structure

   /*************************************************************************/
   /* Allocate memory for data structure, add it                            */
   /* to the list, and init the structure.                                  */
   /*************************************************************************/

   if ((ulRC = AllocatePerm((ULONG)sizeof(RLDMA), (PVOID *)&prldma)) ==
      DSP_NOERROR) {

      memset(prldma, INIT_VAL, sizeof(RLDMA));/* Null out the structure     */

      CLLInsTail((PVOID)prldma, (PVOID *)&prldma->LDMA_prldmaNext, (PVOID *)
         &(*pprldmaListTail)->LDMA_prldmaNext, (PVOID *)pprldmaListTail);

      prldma->LDMA_achValid = LDMAVALIDATE;  /* Validate structure */

      prldma->LDMA_usflg = LDMA_NOTSHARABLE;

      prldma->LDMA_ulStride = prldma->LDMA_ulHold = prldma->LDMA_usSlotNum =
         prldma->LDMA_ulOffset = 0;

      /* pass back to caller, the ptr to data structure                     */
      *pprldmaNode = prldma;
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRLDMA                                             */
/*                                                                          */
/* FUNCTION: De-allocates memory for the LDMA structure and unlinks it from */
/*           the linked-list it is attached to.                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRLDMA * pprldmaListTail  - Ptr to linked-list.                   */
/*        PRLDMA prldmaNode  - Pointer to the data structure to delete.     */
/*        PRLDMA prldmaPrev  - Pointer to the previous Node in linked-list  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRLDMA * pprldmaListTail  - Tail may be updated.                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      DEVELOPER         DESCR.                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveRLDMA(PRLDMA *pprldmaListTail,PRLDMA prldmaNode,PRLDMA prldmaPrev)

{
   /* un-link node from linked-list                                         */

   CLLDelNode((PVOID *)&prldmaNode->LDMA_prldmaNext,
              (PVOID *)&prldmaPrev->LDMA_prldmaNext,
              (PVOID) prldmaPrev,
              (PVOID *)pprldmaListTail);

   /* free structure memory                                                 */
   if (prldmaNode->LDMA_pszDMAName != NULL) {
      FreePerm(prldmaNode->LDMA_pszDMAName,
               (strlen(prldmaNode->LDMA_pszDMAName)+1));
   }

   return (FreePerm((PVOID)prldmaNode, (ULONG)sizeof(RLDMA)));
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRGPCTail                                             */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRGPC * pprgpcListTail  - Add the new structure allocated to the  */
/*                                  tail of this linked-list.               */
/*        PRGPC * pprgpcNode  - Ptr to Ptr to the new data structure        */
/*                              allocated.                                  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRGPC * pprgpcListTail  - Update Tail pointer                     */
/*        PRGPC * pprgpcNode  - Update To point to new data structure.      */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRGPCTail(PRGPC *pprgpcListTail,PRGPC *pprgpcNode)

{
   RC         ulRC;                    // error return code
   PRGPC      prgpc;                   // Ptr to new data structure

   /* Allocate memory for data structure                                    */

   if ((ulRC = AllocatePerm((ULONG)sizeof(RGPC), (PVOID *)&prgpc)) ==
      DSP_NOERROR) {

      /* NULL out fields in structure                                       */
      memset(prgpc, INIT_VAL, sizeof(RGPC));

      /* Add data structure to GPC linked-list                              */
      CLLInsTail((PVOID)prgpc, (PVOID *)&prgpc->GPC_prgpcNext, (PVOID *)
         &(*pprgpcListTail)->GPC_prgpcNext, (PVOID *)pprgpcListTail);

      prgpc->GPC_achValid = GPCVALIDATE;  /* Validate structure */

      /* pass back to caller, the ptr to data structure                     */
      *pprgpcNode = prgpc;
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRGPC                                              */
/*                                                                          */
/* FUNCTION: .........De-allocates memory for a data structure and will     */
/*           De-allocate all children data structures from the DSP Manager  */
/*           data tree.                                                     */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRGPC * pprgpcListTail  - Ptr to linked-list.                     */
/*        PRGPC prgpcNode  - Pointer to the data structure to delete.       */
/*        PRGPC prgpcPrev  - Pointer to the previous Node in linked-list    */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRGPC * pprgpcListTail  - Tail may be updated.                    */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveRGPC(PRGPC *pprgpcListTail,PRGPC prgpcNode,PRGPC prgpcPrev)

{
   /*************************************************************************/
   /* Unlink the node from the linked list, recover the structure memory    */
   /*************************************************************************/

   CLLDelNode((PVOID *)&(prgpcNode->GPC_prgpcNext),
              (PVOID *)&(prgpcPrev->GPC_prgpcNext),
              (PVOID)prgpcPrev, (PVOID *)pprgpcListTail);

   if (FreePerm((PVOID)prgpcNode, (ULONG)sizeof(RGPC)))
      return (DSP_INTERNAL_ERROR);

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRITCBTail                                            */
/*                                                                          */
/* FUNCTION: .........Allocates memory for a data structure and adds it to  */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRITCB * ppritcbListTail  - Add the new structure allocated to the*/
/*                                  tail of this linked-list.               */
/*        PRITCB * ppritcbNode  - Ptr to Ptr to the new data structure      */
/*                              allocated.                                  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRITCB * ppritcbListTail  - Update Tail pointer                   */
/*        PRITCB * ppritcbNode  - Update To point to new data structure.    */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRITCBTail(PRITCB *ppritcbListTail,PRITCB *ppritcbNode)

{
   RC         ulRC;                    // error return code
   PRITCB     pritcb;                  // Ptr to new data structure

   /* Allocate memory for data structure                                    */
   if ((ulRC = AllocatePerm((ULONG)sizeof(RITCB), (PVOID *)&pritcb)) ==
      DSP_NOERROR) {

      /* NULL out fields in structure                                       */
      memset(pritcb, INIT_VAL, sizeof(RITCB));

      /* Add data structure to ITCB linked-list                             */
      CLLInsTail((PVOID)pritcb, (PVOID *)&pritcb->ITCB_pritcbNext,
                 (PVOID *)&(*ppritcbListTail)->ITCB_pritcbNext,
                 (PVOID *)ppritcbListTail);

      pritcb->ITCB_achValid = ITCBVALIDATE;  /* Validate structure */

      /* pass back to caller, the ptr to data structure                     */
      *ppritcbNode = pritcb;
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRITCB                                             */
/*                                                                          */
/* FUNCTION: .........De-allocates memory for a data structure and will     */
/*           De-allocate all children data structures from the DSP Manager  */
/*           data tree.                                                     */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRITCB * ppritcbListTail  - Ptr to linked-list.                   */
/*        PRITCB pritcbNode  - Pointer to the data structure to delete.     */
/*        PRITCB pritcbPrev  - Pointer to the previous Node in linked-list  */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRITCB * ppritcbListTail  - Tail may be updated.                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveRITCB(PRITCB *ppritcbListTail,PRITCB pritcbNode,PRITCB pritcbPrev)

{
   /*************************************************************************/
   /* Unlink the node from the linked list, recover the structure memory    */
   /*************************************************************************/

   CLLDelNode((PVOID *)&(pritcbNode->ITCB_pritcbNext),
              (PVOID *)&(pritcbPrev->ITCB_pritcbNext),
              (PVOID)pritcbPrev,
              (PVOID *)ppritcbListTail);

    /* ITCB names are now in Segment's parstblStatic structure              */

   if (pritcbNode->ITCB_pritcblab != NULL) {
      if (FreePerm((PVOID)pritcbNode->ITCB_pritcblab,
         (ULONG)(pritcbNode->ITCB_usLabelCount)*(sizeof(RITCBLAB))))
         return (DSP_INTERNAL_ERROR);
   }

   if (FreePerm((PVOID)pritcbNode, (ULONG)sizeof(RITCB)))
      return (DSP_INTERNAL_ERROR);

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddRHwTail                                              */
/*                                                                          */
/* FUNCTION: Allocates memory for a RHW data structure and adds it to       */
/*           the tail of a linked-list of similar structures in the DSP     */
/*           Manager data tree.                                             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRHW * pprhwListTail  - Add the new structure allocated to the    */
/*                                tail of this linked-list.                 */
/*        PRHW * pprhwNode  - Ptr to Ptr to the new data structure          */
/*                            allocated.                                    */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRHW * pprhwListTail  - Update Tail pointer                       */
/*        PRHW * pprhwNode  - Update To point to new data structure.        */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AddRHwTail(PRHW *pprhwListTail,PRHW *pprhwNode)

{
   RC         ulRC;                    // error return code
   PRHW       prhw;                    // Ptr to new data structure

   /* Allocate memory for data structure                                    */

   if ((ulRC = AllocatePerm((ULONG)sizeof(RHW), (PVOID *)&prhw)) ==
      DSP_NOERROR) {

      /* NULL out fields in structure                                       */
      memset(prhw, INIT_VAL, sizeof(RHW));

      /* Add data structure to hardware linked-list                         */
      CLLInsTail((PVOID)prhw, (PVOID *)&prhw->HW_prhwNext, (PVOID *)
         &(*pprhwListTail)->HW_prhwNext, (PVOID *)pprhwListTail);

      /* pass back to caller, the ptr to data structure                     */
      *pprhwNode = prhw;
   }

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: RemoveRHw                                               */
/*                                                                          */
/* FUNCTION: De-allocates memory for a RHW data structure and deletes       */
/*           it from the linked-list it resides in.                         */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRHW  * pprhwListTail  - Ptr to linked-list.                      */
/*        PRHW  prhwNode  - Pointer to the data structure to delete.        */
/*        PRHW  prhwPrev  - Pointer to the previous Node in linked-list     */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRHW  * pprhwListTail  - Tail may be updated.(Linked-list updated)*/
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC RemoveRHw(PRHW *pprhwListTail,PRHW prhwNode,PRHW prhwPrev)

{
   /* un-link node from linked-list                                         */
   CLLDelNode((PVOID *)&prhwNode->HW_prhwNext,
              (PVOID *)&prhwPrev->HW_prhwNext,
              (PVOID)prhwPrev, (PVOID *)pprhwListTail);

   /* free structure memory                                                 */
   return (FreePerm((PVOID)prhwNode, (ULONG)sizeof(RHW)));
}

/****************************************************************************/
/*                                                                          */
/* SUBROUTINE NAME: CheckDSPStatus()                                        */
/*                                                                          */
/* FUNCTION: Checks the validity and status of the DSP structure.           */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRDSP * prds    - Pointer to a RDSP structure                     */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error  DSP_INV_HANDLE,                */
/*                                           DSP_NOT_AVAILABLE              */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/****************************************************************************/

RC CheckDSPStatus(PRDSP prdsp)

{
   if (prdsp) {  /* CH01 Dont check NULL handle */

      if (prdsp->DSP_achValid != DSPVALIDATE)
         return (DSP_INV_HANDLE);

      if (prdsp->DSP_usStatus != 0) {
	MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspstruc::CheckDSPStatus error ulRC %x Card Unplugged?\n",DSP_NOT_AVAILABLE);
         return (DSP_NOT_AVAILABLE);
      }
   }

   return (DSP_NOERROR);
}
