#include <BRepGProp_Face.ixx>
#include <BRep_Tool.hxx>
#include <TopoDS.hxx>
#include <GeomAdaptor_Surface.hxx>
#include <Geom_Surface.hxx>
#include <Geom_BezierSurface.hxx>
#include <Geom_BSplineSurface.hxx>
#include <Geom2d_BezierCurve.hxx>
#include <Geom2d_BSplineCurve.hxx>
#include <math.hxx>
#include <Bnd_Box2d.hxx>
#include <BndLib_Add2dCurve.hxx>
  
//=======================================================================
//function : BRepGProp_Face
//purpose  : 
//=======================================================================

BRepGProp_Face::BRepGProp_Face(){}

BRepGProp_Face::BRepGProp_Face(const TopoDS_Face& F){ 
  Load(F);
}

//=======================================================================
//function : NaturalRestriction
//purpose  : 
//=======================================================================

Standard_Boolean BRepGProp_Face::NaturalRestriction() const{
  return  BRep_Tool::NaturalRestriction(mySurface.Face());
} 

//=======================================================================
//function : UIntegrationOrder
//purpose  : 
//=======================================================================

Standard_Integer BRepGProp_Face::UIntegrationOrder() const {
       
 Standard_Integer Nu;
 switch (mySurface.GetType()) {

 case GeomAbs_Plane :
   Nu =4;
   break;

 case GeomAbs_BezierSurface :
   {
   Nu = (*((Handle(Geom_BezierSurface)*)&((mySurface.Surface()).Surface())))->UDegree()+1;
   Nu = Max(4,Nu);
   }
   break;
 case GeomAbs_BSplineSurface :
   {
   Standard_Integer a = (*((Handle(Geom_BSplineSurface)*)&((mySurface.Surface()).Surface())))->UDegree()+1;
   Standard_Integer b = (*((Handle(Geom_BSplineSurface)*)&((mySurface.Surface()).Surface())))->NbUKnots()-1;
   Nu = Max(4,a*b);
   }
   break;

   default :
     Nu = 9;
   break;
 }
 return Max(8,2*Nu);
}

Standard_Integer BRepGProp_Face::VIntegrationOrder() const {
 Standard_Integer Nv;
 switch (mySurface.GetType()) {

 case GeomAbs_Plane :
   Nv = 4;
   break;

 case GeomAbs_BezierSurface :
   {
   Nv = (*((Handle(Geom_BezierSurface)*)&((mySurface.Surface()).Surface())))->VDegree()+1;
   Nv = Max(4,Nv);
   }
   break;

 case GeomAbs_BSplineSurface :
   {
   Standard_Integer a = (*((Handle(Geom_BSplineSurface)*)&((mySurface.Surface()).Surface())))->VDegree()+1;
   Standard_Integer b = (*((Handle(Geom_BSplineSurface)*)&((mySurface.Surface()).Surface())))->NbVKnots()-1;
   Nv = Max(4,a*b);
   }
   break;

   default :
     Nv = 9;
   break;
 }
 return Max(8,2*Nv);
}

Standard_Integer BRepGProp_Face::IntegrationOrder() const 
{
  Standard_Integer N;

  switch (myCurve.GetType()) {
    
  case GeomAbs_Line :
    N = 2;
    break;

  case GeomAbs_Circle :
  case GeomAbs_Ellipse :
  case GeomAbs_Hyperbola :
    N = 9;
    break;

  case GeomAbs_Parabola :
    N = 9;
    break;

  case GeomAbs_BezierCurve :
    {
      N = (*((Handle(Geom2d_BezierCurve)*)&(myCurve.Curve())))->Degree() + 1;
    }
    break;

  case GeomAbs_BSplineCurve :
    {
    Standard_Integer a = (*((Handle(Geom2d_BSplineCurve)*)&(myCurve.Curve())))->Degree() + 1;
    Standard_Integer b = (*((Handle(Geom2d_BSplineCurve)*)&(myCurve.Curve())))->NbKnots() - 1;
    N = a * b;
    }
    break;

    default :
      N = 9;
    break;
  }

  return Max(4,2*N);
}

void BRepGProp_Face::Bounds(Standard_Real& U1,
			    Standard_Real& U2,
			    Standard_Real& V1,
			    Standard_Real& V2)const 
{
  U1 = mySurface.FirstUParameter();
  U2 = mySurface.LastUParameter();
  V1 = mySurface.FirstVParameter();
  V2 = mySurface.LastVParameter();
}

void BRepGProp_Face::Load(const TopoDS_Edge& E) 
{ 
  Standard_Real a,b;
  Handle(Geom2d_Curve) C = 
    BRep_Tool::CurveOnSurface(E, mySurface.Face(), a,b);
  if (E.Orientation() == TopAbs_REVERSED) { 
    Standard_Real x = a;
    a = C->ReversedParameter(b);
    b = C->ReversedParameter(x);
    C = C->Reversed();
  }
  myCurve.Load(C,a,b);
}

void BRepGProp_Face::Load(const TopoDS_Face& F) 
{ 
  TopoDS_Shape aLocalShape = F.Oriented(TopAbs_FORWARD);
  mySurface.Initialize(TopoDS::Face(aLocalShape));
//  mySurface.Initialize(TopoDS::Face(F.Oriented(TopAbs_FORWARD)));
  mySReverse = (F.Orientation() == TopAbs_REVERSED);
}

gp_Pnt2d BRepGProp_Face::Value2d(const Standard_Real U) const 
{
  return myCurve.Value(U);
}

void BRepGProp_Face::D12d(const Standard_Real U,
			gp_Pnt2d& P,
			gp_Vec2d& V1) const 
{
  myCurve.D1(U,P,V1);
}

void BRepGProp_Face::Normal (const Standard_Real U,
			     const Standard_Real V,
			     gp_Pnt& P,
			     gp_Vec& VNor) const 
{
  gp_Vec D1U,D1V;
  mySurface.D1(U,V,P,D1U,D1V);
  VNor = D1U.Crossed(D1V);
  if (mySReverse) VNor.Reverse();
  
}

Standard_Real BRepGProp_Face::FirstParameter()const 
{
  return myCurve.FirstParameter();
}

Standard_Real BRepGProp_Face::LastParameter()const 
{
  return myCurve.LastParameter();
}

//  APO 17.04.2002 (OCC104)
// This is functions that calculate coeff. to optimize "integration order".
//They had been produced experementally for some hard example.
static Standard_Real AS = -0.15, AL = -0.50, B = 1.0, C = 0.75, D = 0.25;
static inline Standard_Real SCoeff(const Standard_Real Eps){
  return Eps < 0.1? AS*(B+Log10(Eps)) + C: C;
}
static inline Standard_Real LCoeff(const Standard_Real Eps){
  return Eps < 0.1? AL*(B+Log10(Eps)) + D: D;
}
Standard_Integer BRepGProp_Face::SIntOrder(const Standard_Real Eps) const{
  Standard_Integer Nv, Nu;
  switch (mySurface.GetType()) {
  case GeomAbs_Plane:  
    Nu = 1; Nv = 1; break;
  case GeomAbs_Cylinder: 
    Nu = 2; Nv = 1; break;
  case GeomAbs_Cone: 
    Nu = 2; Nv = 1; break;
  case GeomAbs_Sphere: 
    Nu = 2; Nv = 2; break;
  case GeomAbs_Torus:
    Nu = 2; Nv = 2; break;
  case GeomAbs_BezierSurface: 
    Nv = (*((Handle(Geom_BezierSurface)*)&((mySurface.Surface()).Surface())))->VDegree();
    Nu = (*((Handle(Geom_BezierSurface)*)&((mySurface.Surface()).Surface())))->UDegree();
    break;
  case GeomAbs_BSplineSurface: 
    Nv = (*((Handle(Geom_BSplineSurface)*)&((mySurface.Surface()).Surface())))->VDegree();
    Nu = (*((Handle(Geom_BSplineSurface)*)&((mySurface.Surface()).Surface())))->UDegree();
    break;
  default:  
    Nu = 2; Nv = 2;  break;
  }
  return Min(RealToInt(Ceiling(SCoeff(Eps)*Max((Nu+1),(Nv+1)))), math::GaussPointsMax());
}
Standard_Integer BRepGProp_Face::SUIntSubs() const{
  Standard_Integer N;
  switch (mySurface.GetType()) {
  case GeomAbs_Plane:  
    N = 2;  break;
  case GeomAbs_Cylinder: 
    N = 4;  break;
  case GeomAbs_Cone: 
    N = 4;  break;
  case GeomAbs_Sphere: 
    N = 4; break;
  case GeomAbs_Torus:
    N = 4; break;
  case GeomAbs_BezierSurface:  
    N = 2;  break;
  case GeomAbs_BSplineSurface: 
    N = (*((Handle(Geom_BSplineSurface)*)&((mySurface.Surface()).Surface())))->NbUKnots();  break;
  default:  
    N = 2;  break;
  }
  return N - 1;
};
Standard_Integer BRepGProp_Face::SVIntSubs() const{
  Standard_Integer N;
  switch (mySurface.GetType()) {
  case GeomAbs_Plane:  
    N = 2;  break;
  case GeomAbs_Cylinder: 
    N = 2;  break;
  case GeomAbs_Cone: 
    N = 2;  break;
  case GeomAbs_Sphere: 
    N = 3; break;
  case GeomAbs_Torus:
    N = 4; break;
  case GeomAbs_BezierSurface: 
    N = 2;  break;
  case GeomAbs_BSplineSurface: 
    N = (*((Handle(Geom_BSplineSurface)*)&((mySurface.Surface()).Surface())))->NbVKnots();
    break;
  default:  
    N = 2;  break;
  }
  return N - 1;
}
void BRepGProp_Face::UKnots(TColStd_Array1OfReal& Knots) const{
  switch (mySurface.GetType()) {
  case GeomAbs_Plane:
    Knots(1) = mySurface.FirstUParameter();  Knots(2) = mySurface.LastUParameter();  
    break;
  case GeomAbs_Cylinder: 
  case GeomAbs_Cone: 
  case GeomAbs_Sphere: 
  case GeomAbs_Torus:
    Knots(1) = 0.0;  Knots(2) = PI*2.0/3.0;  Knots(3) = PI*4.0/3.0;  Knots(4) = PI*6.0/3.0;
    break;
  case GeomAbs_BSplineSurface: 
    (*((Handle(Geom_BSplineSurface)*)&((mySurface.Surface()).Surface())))->UKnots(Knots);
    break;
  default: 
    Knots(1) = mySurface.FirstUParameter();  Knots(2) = mySurface.LastUParameter();
    break;
  }  
}
void BRepGProp_Face::VKnots(TColStd_Array1OfReal& Knots) const{
  switch (mySurface.GetType()) {
  case GeomAbs_Plane:
  case GeomAbs_Cylinder: 
  case GeomAbs_Cone: 
    Knots(1) = mySurface.FirstUParameter();  Knots(2) = mySurface.LastUParameter();  
    break;
  case GeomAbs_Sphere: 
    Knots(1) = -PI/2.0;  Knots(2) = 0.0;  Knots(3) = +PI/2.0;
    break;
  case GeomAbs_Torus:
    Knots(1) = 0.0;  Knots(2) = PI*2.0/3.0;  Knots(3) = PI*4.0/3.0;  Knots(4) = PI*6.0/3.0;
    break;
  case GeomAbs_BSplineSurface: 
    (*((Handle(Geom_BSplineSurface)*)&((mySurface.Surface()).Surface())))->VKnots(Knots);
    break;
  default: 
    Knots(1) = mySurface.FirstUParameter();  Knots(2) = mySurface.LastUParameter();
    break;
  }  
}
Standard_Integer BRepGProp_Face::LIntOrder(const Standard_Real Eps) const{
  Bnd_Box2d aBox;
  BndLib_Add2dCurve aB;
  aB.Add(myCurve, 1.e-7, aBox);
  Standard_Real aXmin, aXmax, aYmin, aYmax;
  aBox.Get(aXmin, aYmin, aXmax, aYmax);
  Standard_Real aVmin = mySurface.FirstVParameter();
  Standard_Real aVmax = mySurface.LastVParameter();

  Standard_Real anR = Min((aYmax-aYmin)/(aVmax-aVmin), 1.);

//  Standard_Integer anRInt = Max(RealToInt(Ceiling(SVIntSubs()*anR)), 2);
  Standard_Integer anRInt = RealToInt(Ceiling(SVIntSubs()*anR));
  Standard_Integer aLSubs = LIntSubs();


//  Standard_Real NL, NS = Max(SIntOrder(1.0)*anRInt/LIntSubs(), 1);
  Standard_Real NL, NS = Max(SIntOrder(1.)*anRInt/aLSubs, 1);
  switch (myCurve.GetType()) {
  case GeomAbs_Line:  
    NL = 1;  break;
  case GeomAbs_Circle:
    NL = 2 * 3;  break; //correction for the spans of converted curve
  case GeomAbs_Ellipse:
    NL = 2 * 3;  break; //
  case GeomAbs_Parabola:  
    NL = 2 * 3;  break;
  case GeomAbs_Hyperbola: 
    NL = 3 * 3;  break;
  case GeomAbs_BezierCurve: 
    NL = (*((Handle(Geom2d_BezierCurve)*)&(myCurve.Curve())))->Degree();
    break;
  case GeomAbs_BSplineCurve: 
    NL = (*((Handle(Geom2d_BSplineCurve)*)&(myCurve.Curve())))->Degree();
    break;
  default:  
    NL = 3 * 3;  break;
  }

  NL = Max(NL, NS);

  Standard_Integer nn = NL+1;
  if(aLSubs <= 4) nn = RealToInt(Ceiling(LCoeff(Eps)*(NL+1)));

  //return Min(RealToInt(Ceiling(LCoeff(Eps)*(NL+1)*NS)), math::GaussPointsMax());
  return Min(nn, math::GaussPointsMax());
}
Standard_Integer BRepGProp_Face::LIntSubs() const{
  Standard_Integer N;
  switch (myCurve.GetType()) {
  case GeomAbs_Line:  
    N = 2;  break;
  case GeomAbs_Circle:
  case GeomAbs_Ellipse:
    N = 4;  break;
  case GeomAbs_Parabola:
  case GeomAbs_Hyperbola:
    N = 2;  break;
  case GeomAbs_BSplineCurve: 
    N = (*((Handle(Geom2d_BSplineCurve)*)&(myCurve.Curve())))->NbKnots();
    break;
  default:  
    N = 2;  break;
  }
  return N - 1;
}
void BRepGProp_Face::LKnots(TColStd_Array1OfReal& Knots) const{
  switch (myCurve.GetType()) {
  case GeomAbs_Line:  
    Knots(1) = myCurve.FirstParameter();  Knots(2) = myCurve.LastParameter();
    break;
  case GeomAbs_Circle:
  case GeomAbs_Ellipse:
    Knots(1) = 0.0;  Knots(2) = PI*2.0/3.0;  Knots(3) = PI*4.0/3.0;  Knots(2) = PI*6.0/3.0;
    break;
  case GeomAbs_Parabola:
  case GeomAbs_Hyperbola:
    Knots(1) = myCurve.FirstParameter();  Knots(2) = myCurve.LastParameter();
    break;
  case GeomAbs_BSplineCurve:
    (*((Handle(Geom2d_BSplineCurve)*)&(myCurve.Curve())))->Knots(Knots);
    break;
  default: 
    Knots(1) = myCurve.FirstParameter();  Knots(2) = myCurve.LastParameter();
    break;
  }  
}
