/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
// $Id: QTFile.h,v 1.7 1999/06/02 18:15:37 serenyi Exp $
//
// QTFile:
//   The central point of control for a file in the QuickTime File Format.

#ifndef QTFile_H
#define QTFile_H


//
// Includes
#include "OSHeaders.h"
#include "OSFileSource.h"
#include "QTFile_FileControlBlock.h"

//
// External classes
class OSMutex;

class QTAtom_mvhd;
class QTTrack;


//
// QTFile class
class QTFile {

public:
	//
	// Class constants
	enum {
		kDefaultDataBufferUnits		= 1
	};
	
	
	//
	// Class error codes
	enum ErrorCode {
		errNoError					= 0,
		errFileNotFound				= 1,
		errInvalidQuickTimeFile		= 2,
		errInternalError			= 100
	};


	//
	// Class typedefs.
	struct AtomTOCEntry {
		// TOC id (used to compare TOCs)
		UInt32			TOCID;

		// Atom information
		OSType			AtomType, beAtomType; // be = Big Endian

		UInt64			AtomDataPos;
		UInt32			AtomDataLength, AtomHeaderSize;

		// TOC pointers
		AtomTOCEntry	*NextOrdAtom;

		AtomTOCEntry	*PrevAtom, *NextAtom;
		AtomTOCEntry	*Parent, *FirstChild;
	};

	struct TrackListEntry {
		// Track information
		UInt32			TrackID;
		QTTrack			*Track;
		bool			IsHintTrack;

		// List pointers
		TrackListEntry	*NextTrack;
	};


public:
	//
	// Constructors and destructor.
						QTFile(bool Debug = false, bool DeepDebug = false);
	virtual				~QTFile(void);


	//
	// Open a movie file and generate the atom table of contents.
			ErrorCode	Open(const char * MoviePath);
			
			OSMutex*	GetMutex() { return fReadMutex; }

	//
	// Table of Contents functions.
			bool		FindTOCEntry(const char * AtomPath,
									 AtomTOCEntry **TOCEntry,
									 AtomTOCEntry *LastFoundTOCEntry = NULL);
	
	//
	// Track List functions
	inline	UInt32		GetNumTracks(void) { return fNumTracks; }
			bool		NextTrack(QTTrack **Track, QTTrack *LastFoundTrack = NULL);
			bool		FindTrack(UInt32 TrackID, QTTrack **Track);
			bool		IsHintTrack(QTTrack *Track);
	
	//
	// Accessors
	inline	char *		GetMoviePath(void) { return fMoviePath; }
			Float64		GetTimeScale(void);
			Float64		GetDurationInSeconds(void);

	//
	// Read functions.
			bool		Read(UInt64 Offset, char * const Buffer, UInt32 Length, QTFile_FileControlBlock * FCB = NULL);
	

	//
	// Debugging functions.
			void		DumpAtomTOC(void);


protected:
	//
	// Protected member functions.
			bool		GenerateAtomTOC(void);
	
	//
	// Protected member variables.
	bool				fDebug, fDeepDebug;

	UInt32				fNextTOCID;
	OSFileSource			fMovieFD;
	char				*fMoviePath;

	AtomTOCEntry		*fTOC, *fTOCOrdHead, *fTOCOrdTail;
	
	UInt32				fNumTracks;
	TrackListEntry		*fFirstTrack, *fLastTrack;

	QTAtom_mvhd			*fMovieHeaderAtom;
	
	OSMutex				*fReadMutex;
};

#endif // QTFile_H
