/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
// $Id: QTRTPFile.h,v 1.19 1999/06/02 18:15:38 serenyi Exp $
//
// QTRTPFile:
//   An interface to QTFile for TimeShare.

#ifndef QTRTPFile_H
#define QTRTPFile_H


//
// Includes
#include "OSHeaders.h"
#ifndef __MW_
#if __MACOS__
	#include <stat.h>
#else
	#include <sys/stat.h>
#endif
#endif

//
// Constants
#define QTRTPFILE_MAX_PACKET_LENGTH		2048


//
// QTRTPFile class
class OSMutex;

class QTFile;
class QTFile_FileControlBlock;
class QTHintTrack;
class QTHintTrack_HintTrackControlBlock;

class QTRTPFile {

public:
	//
	// Class error codes
	enum ErrorCode {
		errNoError					= 0,
		errFileNotFound				= 1,
		errInvalidQuickTimeFile		= 2,
		errNoHintTracks				= 3,
		errTrackIDNotFound			= 4,
		errInternalError			= 100
	};


	//
	// Class typedefs.
	struct RTPFileCacheEntry {
		//
		// Init mutex (do not use this entry until you have acquired and
		// released this.
		OSMutex		*InitMutex;
		
		//
		// File information
	    char*		fFilename;
		QTFile		*File;
		
		//
		// Reference count for this cache entry
		int			ReferenceCount; 
		
		//
		// List pointers
		RTPFileCacheEntry	*PrevEntry, *NextEntry;
	};
	
	struct RTPTrackListEntry {
		//
		// Track information
		UInt32			TrackID;
		QTHintTrack		*HintTrack;
		QTHintTrack_HintTrackControlBlock	*HTCB;
		bool			IsTrackActive, IsPacketAvailable;
		UInt32			QualityLevel;
		
		//
		// Server information
		void			*Cookie;
		UInt32			SSRC;
		UInt16			FileSequenceNumberRandomOffset, BaseSequenceNumberRandomOffset,
						LastSequenceNumber;
		SInt32			SequenceNumberAdditive;
		UInt32			FileTimestampRandomOffset, BaseTimestampRandomOffset;

		//
		// Sample/Packet information
		UInt32			CurSampleNumber;
		UInt16			NumPacketsInThisSample, CurPacketNumber;

		Float64			CurPacketTime;
		char			CurPacket[QTRTPFILE_MAX_PACKET_LENGTH];
		UInt32			CurPacketLength;

		//
		// List pointers
		RTPTrackListEntry	*NextTrack;
	};


public:
	//
	// Global initialize function; CALL THIS FIRST!
	static void			Initialize(void);
	
	
	//
	// Constructors and destructor.
						QTRTPFile(bool Debug = false, bool DeepDebug = false);
	virtual				~QTRTPFile(void);


	//
	// Initialization functions.
	virtual	ErrorCode	Initialize(const char * FilePath);
	
	//
	// Accessors
			Float64		GetMovieDuration(void);
			UInt64		GetAddedTracksRTPBytes(void);
	virtual	char *		GetSDPFile(int * SDPFileLength);
	
	//
	// Track functions
			ErrorCode	AddTrack(UInt32 TrackID, bool UseRandomOffset = true);
			Float64		GetTrackDuration(UInt32 TrackID);
			UInt32		GetTrackTimeScale(UInt32 TrackID);
			
			void		SetTrackSSRC(UInt32 TrackID, UInt32 SSRC);
			void		SetTrackCookie(UInt32 TrackID, void * Cookie);
			
			//
			// What sort of packets do you want?
			enum
			{
				kAllPackets = 0,
				kNoBFrames = 1,
				kKeyFramesOnly = 2,
				kEveryOtherKeyFrame = 3,
				kEveryFourthKeyFrame = 4
				//this continues until infinity...
			};
			
			void		SetTrackQualityLevel(UInt32 TrackID, UInt32 inNewQualityLevel);

	//
	// Packet functions
	virtual	ErrorCode	Seek(Float64 Time, Float64 MaxBackupTime = 3.0);
	virtual	UInt32		GetSeekTimestamp(UInt32 TrackID);

	virtual	UInt16		GetNextTrackSequenceNumber(UInt32 TrackID);
	virtual	Float64		GetNextPacket(char ** Packet, int * PacketLength, void ** Cookie);


protected:
	//
	// Protected cache functions and variables.
	static	OSMutex				*gFileCacheMutex, *gFileCacheAddMutex;
	static	RTPFileCacheEntry	*gFirstFileCacheEntry;
	
	static	ErrorCode	new_QTFile(const char * FilePath, QTFile ** File, bool Debug = false, bool DeepDebug = false);
	static	void		delete_QTFile(QTFile * File);

	static	void		AddFileToCache(const char *inFilename, QTRTPFile::RTPFileCacheEntry ** NewListEntry);
	static	bool		FindAndRefcountFileCacheEntry(const char *inFilename, QTRTPFile::RTPFileCacheEntry **CacheEntry);

	//
	// Protected member functions.
			bool		FindTrackEntry(UInt32 TrackID, RTPTrackListEntry **TrackEntry);
			bool		PrefetchNextPacket(RTPTrackListEntry * TrackEntry, bool DoSeek = false, Float64 Time = 0.0);

	//
	// Protected member variables.
	bool				fDebug, fDeepDebug;

	QTFile				*fFile;
	QTFile_FileControlBlock	*fFCB;
	
	UInt32				fNumHintTracks;
	RTPTrackListEntry	*fFirstTrack, *fLastTrack;
	
	char				*fSDPFile;
	UInt32				fSDPFileLength;
	
	Float64				fRequestedSeekTime, fSeekTime;

	RTPTrackListEntry	*fLastPacketTrack;
	
	UInt32				fBytesPerSecond;
};

#endif // QTRTPFile
