/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.painter;

import java.awt.*;
import java.awt.geom.GeneralPath;
import java.awt.image.BufferedImage;

import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.utils.SubstanceColorUtilities;
import org.jvnet.substance.utils.SubstanceCoreUtilities;

/**
 * Gradient painter that returns images with flat appearance and wavy color
 * areas. This class is part of officially supported API.
 * 
 * @author Kirill Grouchnikov
 */
public class WaveGradientPainter extends BaseGradientPainter {
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.SubstanceGradientPainter#getDisplayName()
	 */
	public String getDisplayName() {
		return "Wave";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.SubstanceGradientPainter#getContourBackground(int,
	 *      int, java.awt.Shape, boolean, org.jvnet.substance.color.ColorScheme,
	 *      org.jvnet.substance.color.ColorScheme, float, boolean, boolean)
	 */
	public BufferedImage getContourBackground(int width, int height,
			Shape contour, boolean isFocused, ColorScheme colorScheme1,
			ColorScheme colorScheme2, float cyclePos, boolean hasShine,
			boolean useCyclePosAsInterpolation) {

		BufferedImage mixResult = this.getMixContourBackground(width, height,
				contour, isFocused, colorScheme1, colorScheme2, cyclePos,
				hasShine, useCyclePosAsInterpolation);
		if (mixResult != null)
			return mixResult;

		// create rectangular background and later draw it on
		// result image with contour clip.
		BufferedImage rectangular = SubstanceCoreUtilities.getBlankImage(width,
				height);
		Graphics2D rgraphics = (Graphics2D) rectangular.getGraphics();
		rgraphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		ColorScheme interpolationScheme1 = colorScheme1;
		ColorScheme interpolationScheme2 = useCyclePosAsInterpolation ? colorScheme2
				: colorScheme1;

		double cycleCoef = 1.0 - cyclePos / 10.0;

//		Color topBorderColor = SubstanceColorUtilities.getTopBorderColor(
//				interpolationScheme1, interpolationScheme2, cycleCoef);
//		Color midBorderColor = SubstanceColorUtilities.getMidBorderColor(
//				interpolationScheme1, interpolationScheme2, cycleCoef);
//		Color bottomBorderColor = SubstanceColorUtilities.getBottomBorderColor(
//				interpolationScheme1, interpolationScheme2, cycleCoef);

		Color darkFillColor = SubstanceColorUtilities.getTopFillColor(
				interpolationScheme1, interpolationScheme2, cycleCoef,
				useCyclePosAsInterpolation);
		Color midFillColor = SubstanceColorUtilities.getMidFillColor(
				interpolationScheme1, interpolationScheme2, cycleCoef,
				useCyclePosAsInterpolation);
		Color lightFillColor = SubstanceColorUtilities.getBottomFillColor(
				interpolationScheme1, interpolationScheme2, cycleCoef,
				useCyclePosAsInterpolation);

		if (!useCyclePosAsInterpolation) {
			lightFillColor = SubstanceColorUtilities.getInterpolatedColor(
					interpolationScheme1.getUltraLightColor(),
					interpolationScheme1.getExtraLightColor(), cycleCoef);
			midFillColor = SubstanceColorUtilities.getInterpolatedColor(
					interpolationScheme1.getLightColor(), interpolationScheme1
							.getMidColor(), cycleCoef);
			darkFillColor = SubstanceColorUtilities.getInterpolatedColor(
					interpolationScheme1.getMidColor(), interpolationScheme1
							.getMidColor(), cycleCoef);
		} else {
			lightFillColor = SubstanceColorUtilities.getInterpolatedColor(
					interpolationScheme1.getUltraLightColor(),
					interpolationScheme2.getUltraLightColor(), cycleCoef);
			midFillColor = SubstanceColorUtilities.getInterpolatedColor(
					interpolationScheme1.getLightColor(), interpolationScheme2
							.getLightColor(), cycleCoef);
			darkFillColor = SubstanceColorUtilities.getInterpolatedColor(
					interpolationScheme1.getMidColor(), interpolationScheme2
							.getMidColor(), cycleCoef);
		}

		// Fill background
		GradientPaint gradientTop = new GradientPaint(0, 0, lightFillColor,
				width / 4, height / 2, midFillColor);
		GeneralPath clipTop = new GeneralPath();
		clipTop.moveTo(0, 0);
		clipTop.lineTo(width, 0);
		clipTop.curveTo(5 * width / 6, height / 3, 3 * width / 4, height / 2,
				width / 2, height / 2);
		clipTop.curveTo(width / 3, height / 2, width / 4, height, 0,
				7 * height / 8);
		clipTop.lineTo(0, 0);

		rgraphics.setClip(clipTop);
		rgraphics.setPaint(gradientTop);
		rgraphics.fillRect(0, 0, width, height);

		GradientPaint gradientBottom = new GradientPaint(2 * width / 3,
				2 * height / 3, darkFillColor, width, height, midFillColor);

		GeneralPath clipBottom = new GeneralPath();
		clipBottom.moveTo(0, height);
		clipBottom.lineTo(width, height);
		clipBottom.lineTo(width, 0);
		clipBottom.curveTo(5 * width / 6, height / 3, 3 * width / 4,
				height / 2, width / 2, height / 2);
		clipBottom.curveTo(width / 3, height / 2, width / 4, height, 0,
				7 * height / 8);
		clipBottom.lineTo(0, height);

		rgraphics.setClip(clipBottom);
		rgraphics.setPaint(gradientBottom);
		rgraphics.fillRect(0, 0, width, height);

		rgraphics.setClip(null);
		GeneralPath mid = new GeneralPath();
		mid.moveTo(width, 0);
		mid.curveTo(5 * width / 6, height / 3, 3 * width / 4, height / 2,
				width / 2, height / 2);
		mid
				.curveTo(width / 3, height / 2, width / 4, height, 0,
						7 * height / 8);
		rgraphics.draw(mid);

		BufferedImage image = SubstanceCoreUtilities.getBlankImage(width,
				height);
		Graphics2D graphics = (Graphics2D) image.getGraphics();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		graphics.setClip(contour);
		graphics.drawImage(rectangular, 0, 0, null);

		graphics.setClip(null);

//		// Draw border
//		GradientPaint gradientBorderTop = new GradientPaint(0, 0,
//				topBorderColor, 0, height / 2, midBorderColor);
//		graphics.setPaint(gradientBorderTop);
//		graphics.setClip(0, 0, width, height / 2);
//		graphics.draw(contour);
//
//		GradientPaint gradientBorderBottom = new GradientPaint(0, height / 2,
//				midBorderColor, 0, height - 2, bottomBorderColor);
//		graphics.setPaint(gradientBorderBottom);
//		graphics.setClip(0, height / 2, width, 1 + height / 2);
//		graphics.draw(contour);

		return image;
	}
}
