/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.painter.highlight;

import java.awt.*;
import java.awt.geom.GeneralPath;
import java.awt.image.BufferedImage;
import java.util.*;

import org.jvnet.lafwidget.layout.TransitionLayout;
import org.jvnet.substance.SubstanceLookAndFeel;
import org.jvnet.substance.border.SubstanceBorderPainter;
import org.jvnet.substance.button.BaseButtonShaper;
import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.painter.GlassGradientPainter;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.*;
import org.jvnet.substance.utils.SubstanceConstants.Side;

/**
 * Highlight painter that paints a glass gradient. This class is part of
 * officially supported API.
 * 
 * @author Kirill Grouchnikov
 * @since version 4.3
 */
public class GlassHighlightPainter implements SubstanceHighlightPainter {
	/**
	 * The display name for the highlight painters of this class.
	 */
	public static final String DISPLAY_NAME = "Glass";

	/**
	 * Cache for small objects.
	 */
	protected static Map<String, BufferedImage> smallImageCache = new SoftHashMap<String, BufferedImage>();

	/**
	 * Single gradient painter instance.
	 */
	protected GlassGradientPainter painter;

	/**
	 * Creates new classic title painter.
	 */
	public GlassHighlightPainter() {
		this.painter = new GlassGradientPainter();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.SubstanceGradientPainter#getDisplayName()
	 */
	public String getDisplayName() {
		return DISPLAY_NAME;
	}

	public void paintHighlight(Graphics2D graphics, Component comp, int width,
			int height, float borderAlpha, Set<Side> openSides,
			ColorScheme colorScheme1, ColorScheme colorScheme2, float cyclePos) {

		SubstanceBorderPainter borderPainter = SubstanceCoreUtilities
				.getBorderPainter(comp);

		int size = Math.max(width, height);
		if (openSides == null) {
			openSides = new HashSet<Side>();
		}
		if (size < 150) {
			synchronized (this) {
				String openKey = "";
				for (Side oSide : openSides) {
					openKey += oSide.name() + "-";
				}

				String key = width + "*" + height + "*"
						+ SubstanceCoreUtilities.getSchemeId(colorScheme1)
						+ "*"
						+ SubstanceCoreUtilities.getSchemeId(colorScheme2)
						+ "*" + cyclePos + "*" + borderAlpha + "*" + openKey
						+ "*" + borderPainter.getDisplayName();
				BufferedImage result = smallImageCache.get(key);
				if (result == null) {
					result = SubstanceCoreUtilities
							.getBlankImage(width, height);
					this.internalPaint((Graphics2D) result.getGraphics(), comp,
							width, height, colorScheme1, colorScheme2,
							cyclePos, borderPainter, borderAlpha, openSides);
					smallImageCache.put(key, result);
				}
				graphics.drawImage(result, 0, 0, null);
				return;
			}
		}

		this.internalPaint(graphics, comp, width, height, colorScheme1,
				colorScheme2, cyclePos, borderPainter, borderAlpha, openSides);
	}

	protected void internalPaint(Graphics2D graphics, Component comp,
			int width, int height, ColorScheme colorScheme1,
			ColorScheme colorScheme2, float cyclePos,
			SubstanceBorderPainter borderPainter, float borderAlpha,
			Set<SubstanceConstants.Side> openSides) {

		Graphics2D g2d = (Graphics2D) graphics.create();

		g2d.drawImage(this.painter.getContourBackground(width + 6, height + 6,
				new Rectangle(width + 6, height + 6), false, colorScheme1,
				colorScheme2, cyclePos, false, true), -3, -3, null);
		//		
		// SubstanceImageCreator.paintRectangularBackground(g2d, 0, 0, width,
		// height, colorScheme2, 0.0f, false);
		// g2d.setComposite(TransitionLayout.getAlphaComposite(comp,
		// 1.0f - cyclePos / 10.0f, graphics));
		// SubstanceImageCreator.paintRectangularBackground(g2d, 0, 0, width,
		// height, colorScheme1, 0.0f, false);
		// g2d.setComposite(TransitionLayout.getAlphaComposite(comp, 1.0f,
		// graphics));

		if (borderAlpha > 0.0f) {
			int openDelta = 3 + (int) (Math.ceil(3.0 * SubstanceSizeUtils
					.getBorderStrokeWidth(SubstanceSizeUtils
							.getComponentFontSize(comp))));
			int deltaLeft = openSides.contains(Side.LEFT) ? openDelta : 0;
			int deltaRight = openSides.contains(Side.RIGHT) ? openDelta : 0;
			int deltaTop = openSides.contains(Side.TOP) ? openDelta : 0;
			int deltaBottom = openSides.contains(Side.BOTTOM) ? openDelta : 0;

			int borderDelta = (int) Math.floor(SubstanceSizeUtils
					.getBorderStrokeWidth(SubstanceSizeUtils
							.getComponentFontSize(comp)) / 2.0);
			GeneralPath contour = BaseButtonShaper.getBaseOutline(width
					+ deltaLeft + deltaRight, height + deltaTop + deltaBottom,
					0.0f, null, borderDelta);
			// (null, new Insets(
			// borderDelta, borderDelta, borderDelta, borderDelta), width
			// + deltaLeft + deltaRight, height + deltaTop + deltaBottom);

			graphics.translate(-deltaLeft, -deltaTop);

			if (borderAlpha > 0.0f) {
				g2d.setComposite(TransitionLayout.getAlphaComposite(comp,
						borderAlpha, graphics));
				int borderThickness = (int) SubstanceSizeUtils
						.getBorderStrokeWidth(SubstanceSizeUtils
								.getComponentFontSize(comp));
				GeneralPath contourInner = BaseButtonShaper.getBaseOutline(
						width + deltaLeft + deltaRight, height + deltaTop
								+ deltaBottom, 0.0f, null, borderDelta
								+ borderThickness);
				borderPainter.paintBorder(g2d, comp, width + deltaLeft
						+ deltaRight, height + deltaTop + deltaBottom, contour,
						contourInner, colorScheme1, colorScheme2, cyclePos,
						true);
				g2d.dispose();
			}
		}
	}

	/**
	 * Resets image maps (used when setting new theme).
	 * 
	 * @see SubstanceLookAndFeel#setCurrentTheme(String)
	 * @see SubstanceLookAndFeel#setCurrentTheme(SubstanceTheme)
	 */
	public static synchronized void reset() {
		smallImageCache.clear();
	}
}
