#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "interface.h"

#include "treedata.h"

#define thisentry(c) ((sel_entry)nodeobj(c))

typedef struct {
	selection_type type; /* NodeSel,SubtreeSel,BPSel */
	void *item;
	double p;	/* percent from first end */
	} sel_entryd, *sel_entry;

typedef struct {
	/* holds tree data in clipboard */
	int isroot;
	int type;
	double ftscalex;
	double ftscaley;
	double fscale;
	double fdst;
	int fshowinternals;
	int fshowterminals;
	int fshowpct;
	double fpv;
	tcolor fc;
	tfont fpfont;
	double btscalex;
	double btscaley;
	double bscale;
	double bdst;
	int bshowinternals;
	int bshowterminals;
	int bshowpct;
	double bpv;
	tcolor bc;
	tfont bpfont;
} clipdatad, *clipdata;

treenode lastchosen;
void *selection;
void *clipboard;
clipdata clipd;
selection_type tsel; /* NodeSel,SubtreeSel,BPSel,MultiSel */
selection_type tclip; /* SubtreeSel */

extern tree_duplicate();
extern tree_free();

#define select_entry(t) \
switch(t->type)\
{\
	case NodeSel:\
		tree_select_node(t->item);\
		update_full_flags(t->item);\
		break;\
	case BranchPointSel:\
		tree_select_branch_point(t->item, t->p);\
		break;\
	case SubtreeSel:\
		tree_select_subtree(t->item);\
		update_full_flags(t->item);\
		break;\
	case TextSel:\
		ttext_select(t->item);\
		break;\
	case NoSel:\
		break;\
}

#define unselect_entry(t) \
switch(t->type)\
{\
	case NodeSel:\
		tree_unselect_node(t->item);\
		update_full_flags(t->item);\
		break;\
	case BranchPointSel:\
		canvas_add_branch_point(t->item, t->p);\
		tree_unselect_branch(t->item);\
		break;\
	case SubtreeSel:\
		tree_unselect_subtree(t->item);\
		update_full_flags(t->item);\
		break;\
	case TextSel:\
		ttext_unselect(t->item);\
		break;\
	case NoSel:\
		break;\
}

#define nodeon(n) (ndta(n)->selected)
#define nodefull(n) (ndta(n)->full)
#define branchon(n) (bdta(n)->selected)

void update_full_flags(n)
treenode n;
{
	list t;
	int isfull, isfullsofar;


	if(nodeon(n))
	{
		isfullsofar=1;
		ndta(n)->full=0;
	}
	else
	{
		ndta(n)->full=0;
		isfullsofar=0;
		n=parent(n);
	}
	while(n!=NULL)
	{
		if(!nodeon(n))
			break;

		if(isfullsofar)
		{
			/* full so far, so must check for this node */
			isfull=1;
			tforsubtree(n, t)
				if(!nodefull(subtree(t)))
				{
					isfull=0;
					isfullsofar=0;
				}
		}
		else
			/* not full so far, so this one can't be full */
			isfull=0;

		if(nodefull(n)!=isfull)
		{
			/* changed, so set and try above */
			ndta(n)->full=isfull;
			n=parent(n);
		}
		else
			/* unchanged, so no need to look higher */
			break;
	}
}

static sel_entry alloc_and_add(type, item, p)
selection_type type;
void *item;
double p;
{
	sel_entry t;
	list l;

	if(type==BranchPointSel && selection!=NULL)
	{
		lfor(selection, l)
		{
			t=(sel_entry)nodeobj(l);
			if(t->item==item)
				return(t);
		}
	}
	
	t=(sel_entry)malloc(sizeof(sel_entryd));
	t->type=type;
	t->item=item;
	t->p=p;

	/* recreate list */
	if(selection==NULL)
		selection=newlist();

	startlist(selection);
	addnode(selection, t);
	return(t);
}

static void freesel()
{
	void *t;

	if(selection==NULL)
		return;

	startlist(selection);
	while((t=listnext(selection))!=NULL)
		free(t);
	freelist(selection);
}

void incnum(n)
int *n;
{
	if(*n==0)
		*n=1;
	else
		*n=2;
}

static void rebuild_selection(n, multi, branches, nodes, subtrees,
	insidesubtree)
treenode n;
int *multi, *branches, *nodes, *subtrees;
int insidesubtree;
{
	treebranch b;

	if(nodeon(n) && nodefull(n) && !insidesubtree)
	{
		incnum(subtrees);
		incnum(multi);
		insidesubtree=1;
		alloc_and_add(SubtreeSel, n, 0.0);
	}
	else if(nodeon(n) && !nodefull(n))
	{
		incnum(nodes);
		incnum(multi);
		alloc_and_add(NodeSel, n, 0.0);
	}

	startlist(n->branches);
	while((b=listnext(n->branches))!=NULL)
	{
		if(b->up==n)
		{
			if(branchon(b))
			{
				incnum(branches);
				incnum(multi);
				alloc_and_add(BranchPointSel, b, bdta(b)->percentage);
			}
			rebuild_selection(b->down,
				multi, branches, nodes, subtrees, insidesubtree);
		}
	}
}

static int multi, branches, nodes, subtrees;
static int numtrees, clippresent;

static void grey_all_menus_accordingly()
{
	clippresent=tclip!=NoSel;
	grey_menus(numtrees, clippresent, multi, subtrees, nodes, branches);
	grey_extras(numtrees, clippresent, multi, subtrees, nodes, branches);
	grey_tree_info(numtrees, clippresent, multi, subtrees, nodes, branches);
	grey_scale_info(numtrees, clippresent, multi, subtrees, nodes, branches);
	grey_node_info(numtrees, clippresent, multi, subtrees, nodes, branches);
	grey_branch_info(numtrees, clippresent, multi, subtrees, nodes, branches);
}

void set_selection_type()
{
	list t;

	freesel();
	selection=newlist();
	numtrees=number_of_trees();
	subtrees=branches=nodes=0;
	multi=0;
	lfor(globals->trees, t)
		rebuild_selection(treeroot(nodeobj(t)),
			&multi, &branches, &nodes, &subtrees, 0);

	if(multi==0)
	{
		tsel=NoSel;
		freelist(selection);
		selection=NULL;
	}
	else if(multi==2)
		tsel=MultiSel;
	else if(subtrees==1)
		tsel=SubtreeSel;
	else if(nodes==1)
		tsel=NodeSel;
	else if(branches==1)
		tsel=BranchPointSel;
	grey_all_menus_accordingly();
}

selection_type get_selection_type()
{
	return(tsel);
}

void init_selection()
{
	tsel=NoSel;
	tclip=NoSel;
	selection=NULL;
	clipboard=NULL;
	clipd=NULL;
	lastchosen=NULL;
	set_selection_type();
}

static void rebuild_selection_list()
{
	freesel();
	selection=newlist();
}

static void add_entry(type, item, p)
selection_type type;
void *item;
double p;
{
	static list c;
	static sel_entry t;

	/* create entry */
	t=alloc_and_add(type, item, p);
	select_entry(t);

	/* set last chosen */
	switch(type)
	{
		case SubtreeSel:
			lastchosen=item;
			update_tree_info_frame(tree_get_tree(item));
			update_scale_info_frame(tree_get_tree(item));
			update_node_info_frame(item);
			break;
		case BranchPointSel:
			lastchosen=((treebranch)item)->down;
			update_tree_info_frame(tree_get_tree(item));
			update_scale_info_frame(tree_get_tree(item));
			update_branch_info_frame(item);
			break;
		case NodeSel:
			lastchosen=item;
			update_tree_info_frame(tree_get_tree(item));
			update_scale_info_frame(tree_get_tree(item));
			update_node_info_frame(item);
			break;
		case TextSel:
			break;
		default:
			break;
	}

	set_selection_type();

}

void set_last_point(item)
void *item;
{
	nodetypedata nd;
	treetypedata td;
	treenode n=NULL;

	switch(tree_get_type(item))
	{
		case tr_node:
			if(globals->mode==MouseMove)
				n=treeroot(tree_get_tree(item));
			else
				n=item;
			break;
		case tr_branch:
			n=((treebranch)item)->down;
			break;
		default:
			break;
	}

	if(n!=NULL)
	{
		nd=tndta(n, NoTreeType);
		td=ttdta(tree_get_tree(n), NoTreeType);
		globals->px=nodex(nd, td);
		globals->py=nodey(nd, td);
	}
}

int add_to_selection(item, type, add, p)
void *item; /* node, text, branch, tree */
selection_type type;
int add;
double p; /* percent from up node for branch */
{
	sel_entry t;

	/* first, must unselect all others, possibly */
	if(add==0)
	{
		if(tsel!=NoSel)
		{
			/* unselect all objects in selection */
			startlist(selection);
			while((t=listnext(selection))!=NULL)
				unselect_entry(t);

			/* free the list */
			freesel();
		}

		selection=NULL;
		tsel=NoSel;
	}
	
	if(type!=NoSel)
	{
		add_entry(type, item, p);
		set_last_point(item);
	}
	else
		set_selection_type();
	return(1);
}

int add_node_list_to_selection(l, tl, add)
list l; /* list of nodes */
list tl; /* list of trees */
int add;
{
	list t;
	sel_entry te;
	treenode n;

	/* first, must unselect all others, possibly */
	add_to_selection(NULL, NoSel, add, 0.0);

	te=(sel_entry)malloc(sizeof(sel_entryd));
	te->type=NodeSel;


	lfor(l, t)
	{
		te->item=nodeobj(t);
		select_entry(te);
	}
	set_selection_type();

	return(1);
}

int remove_from_selection(item, type)
void *item;
selection_type type;
{
	sel_entryd t;

	t.item=item;
	t.type=type;
	unselect_entry((&t));

	set_selection_type();
	return(1);
}

copy_clip_data_to_tree(t)
tree t;
{
	treetypedata selflower, selboxed;

	selflower=ttdta(t, Flower);
	selboxed=ttdta(t, Boxed);

	set_tree_type(t, clipd->type);
	selflower->tscalex=clipd->ftscalex;
	selflower->tscaley=clipd->ftscaley;
	selflower->scale=clipd->fscale;
	selflower->dst=clipd->fdst;
	selflower->showinternals=clipd->fshowinternals;
	selflower->showterminals=clipd->fshowterminals;
	selflower->showpct=clipd->fshowpct;
	selflower->pv=clipd->fpv;
	selflower->c=clipd->fc;
	selflower->pfont=clipd->fpfont;
	selboxed->tscalex=clipd->btscalex;
	selboxed->tscaley=clipd->btscaley;
	selboxed->scale=clipd->bscale;
	selboxed->dst=clipd->bdst;
	selboxed->showinternals=clipd->bshowinternals;
	selboxed->showterminals=clipd->bshowterminals;
	selboxed->showpct=clipd->bshowpct;
	selboxed->pv=clipd->bpv;
	selboxed->c=clipd->bc;
	selboxed->pfont=clipd->bpfont;
}

static copy_tree_data_to_clip(t)
tree t;
{
	treetypedata selflower, selboxed;

	selflower=ttdta(t, Flower);
	selboxed=ttdta(t, Boxed);
	clipd=(clipdata)malloc(sizeof(clipdatad));

	clipd->type=tdta(t)->type;
	clipd->ftscalex=selflower->tscalex;
	clipd->ftscaley=selflower->tscaley;
	clipd->fscale=selflower->scale;
	clipd->fdst=selflower->dst;
	clipd->fshowinternals=selflower->showinternals;
	clipd->fshowterminals=selflower->showterminals;
	clipd->fshowpct=selflower->showpct;
	clipd->fpv=selflower->pv;
	clipd->fc=selflower->c;
	clipd->fpfont=selflower->pfont;
	clipd->btscalex=selboxed->tscalex;
	clipd->btscaley=selboxed->tscaley;
	clipd->bscale=selboxed->scale;
	clipd->bdst=selboxed->dst;
	clipd->bshowinternals=selboxed->showinternals;
	clipd->bshowterminals=selboxed->showterminals;
	clipd->bshowpct=selboxed->showpct;
	clipd->bpv=selboxed->pv;
	clipd->bc=selboxed->c;
	clipd->bpfont=selboxed->pfont;
}

int copy_to_clipboard()
{
	sel_entry e, n;

	/* first, check for errors */
	switch(tsel)
	{
		case NodeSel:
			if(globals->mode!=MouseText)
			{
				tterr("Clipboard can only hold one full subtree",
					BeepMessage);
				return(0);
			}
			break;
		case BranchPointSel:
			tterr("Clipboard cannot hold a Branch",
				BeepMessage);
			return(0);
			break;
		case SubtreeSel:
			break;
		case TextSel:
			break;
		case MultiSel:
			tterr("Clipboard can only hold one continuous selection",	
				BeepMessage);
			return(0);
		case NoSel:
		default:
			/* can't clip nothing */
			return(0);
			break;
	}
	startlist(selection);
	e=listnext(selection);
	/* ok, so must erase clipboard */
	switch(tclip)
	{
		case NodeSel:
		case TextSel:
			if(globals->mode==MouseText)
				/* just free the string */
				free(clipboard);
			break;
		case BranchPointSel:
			/* can't have branch point on clipboard */
			break;
		case SubtreeSel:
			if(globals->mode==MouseText)
				free(clipboard);
			else
			{
				freesubtree(clipboard);
				free(clipd);
			}
			break;
		case NoSel:
		default:
			break;
	}
	tclip=NoSel;
	switch(e->type)
	{

		case NodeSel:
		case TextSel:
			if(globals->mode==MouseText)
			{
				clipboard=ttext_get_selected_string(tndta(e->item,
					NoTreeType)->label);
				tclip=TextSel;
				tterr("Text copied to clipboard", Message);
			}
			break;
		case BranchPointSel:
			/* can't have branch point on clipboard */
			break;
		case SubtreeSel:
			if(globals->mode==MouseText)
			{
				clipboard=ttext_get_selected_string(tndta(e->item,
					NoTreeType)->label);
				tclip=TextSel;
				tterr("Text copied to clipboard", Message);
			}
			else
			{
				clipboard=(void *)duplicatesubtree(parent(e->item),
					e->item, tree_duplicate);
				tclip=e->type;
				copy_tree_data_to_clip(tree_get_tree(e->item));
				clipd->isroot=(parent(e->item)==NULL);
				tterr("Subtree copied to clipboard", Message);
			}
			break;
		case NoSel:
		default:
			break;
	}
	grey_all_menus_accordingly();
	return(1);
}

int cut_selection()
/* removes the selection from the tree */
{
	sel_entry e;
	treenode newn, oldn;
	treebranch pb;
	tree t;
	int no_errors;

	if(tsel==NoSel)
		return(1);

	startlist(selection);
	e=listnext(selection);

	no_errors=1;

	switch(tsel)
	{
		case NodeSel:
		case TextSel:
			if(globals->mode!=MouseText)
			{
				t=tree_get_tree(e->item);
				unselect_entry(e);
				canvas_add_node(e->item);
				newn=(treenode)merge_branch(e->item);
				if(newn==NULL)
				{
					tterr("Cannot merge branches if node has more than one child",
						BeepMessage);
					no_errors=0;
				}
				else
				{
					e->item=newn;
					invalidate_format(t, Flower);
					invalidate_format(t, Boxed);
					invalidate_spacing(t, Flower);
					invalidate_spacing(t, Boxed);
					invalidate_refit(t, Flower);
					invalidate_refit(t, Boxed);
					modify();
				}
				select_entry(e);
				set_selection_type();
			}
			else
			{
				ttext_delete_string(tndta(e->item, NoTreeType)->label);
				modify();
			}
			break;
		case BranchPointSel:
			/* can't cut branch point */
			tterr("Cannot cut a Branch Point", BeepMessage);
			no_errors=0;
			break;
		case SubtreeSel:
			if(globals->mode!=MouseText)
			{
				t=tree_get_tree(e->item);
				unselect_entry(e);
				newn=parent(e->item);
				t=tree_get_tree(e->item);
				if(treeroot(t)==e->item)
				{
					/* if root node is selected, remove whole tree */
					canvas_add_tree(t);
					removesubtree(newn, e->item);
					freesubtree(e->item);
					tree_remove(t);
					tree_free(t, tr_tree, t_data(t));
					freetree(t);
					free(e);
					freelist(selection);
					selection=NULL;
					tsel=NoSel;
					set_selection_type();
				}
				else
				{
					canvas_add_subtree(e->item);
					pb=getparentandbranch(e->item, &oldn);
					canvas_add_branch(pb);
					removesubtree(newn, e->item);
					freesubtree(e->item);
					/* for selecting the parent after a cut */
					/*e->item=newn;*/
					/*select_entry(e);*/
					/* for unselecting everything after a cut */
					free(e);
					freelist(selection);
					selection=NULL;
					tsel=NoSel;
					set_selection_type();
					invalidate_spacing(t, Flower);
					invalidate_spacing(t, Boxed);
					invalidate_refit(t, Flower);
					invalidate_refit(t, Boxed);
					if(globals->automerge)
					{
						newn=(treenode)merge_branch(newn);
						if(newn!=NULL)
							canvas_add_node(parent(newn));
					}
				}
				modify();
			}
			else
			{
				ttext_delete_string(tndta(e->item, NoTreeType)->label);
				modify();
			}
			break;
		case MultiSel:
			tterr("Cut operates on one continuous selection only",
				BeepMessage);
			no_errors=0;
			break;
		case NoSel:
		default:
			break;
	}

	return(no_errors);
}

int delete_item(n, type)
/* deletes the item from the tree */
treenode n;
selection_type type;
{
	treenode newn, oldn;
	treebranch pb;
	tree t;
	int no_errors;

	no_errors=1;

	if(tree_get_tree(n)==NULL)
		return(1);

	switch(type)
	{
		case NodeSel:
		case TextSel:
			if(globals->mode!=MouseText)
			{
				t=tree_get_tree(n);
				canvas_add_node(n);
				newn=(treenode)merge_branch(n);
				if(newn==NULL)
				{
					tterr("Cannot merge branches if node has more than one child",
						BeepMessage);
					no_errors=0;
				}
				else
				{
					invalidate_format(t, Flower);
					invalidate_format(t, Boxed);
					invalidate_spacing(t, Flower);
					invalidate_spacing(t, Boxed);
					invalidate_refit(t, Flower);
					invalidate_refit(t, Boxed);
					modify();
				}
			}
			else
			{
				ttext_delete_string(tndta(n, NoTreeType)->label);
				modify();
			}
			break;
		case BranchPointSel:
			break;
		case SubtreeSel:
			if(globals->mode!=MouseText)
			{
				t=tree_get_tree(n);
				newn=parent(n);
				t=tree_get_tree(n);
				if(treeroot(t)==n)
				{
					/* if root node is selected, remove whole tree */
					canvas_add_tree(t);
					removesubtree(newn, n);
					freesubtree(n);
					tree_remove(t);
					tree_free(t, tr_tree, t_data(t));
					freetree(t);
					set_selection_type();
				}
				else
				{
					canvas_add_subtree(n);
					pb=getparentandbranch(n, &oldn);
					canvas_add_branch(pb);
					removesubtree(newn, n);
					freesubtree(n);
					set_selection_type();
					invalidate_spacing(t, Flower);
					invalidate_spacing(t, Boxed);
					invalidate_refit(t, Flower);
					invalidate_refit(t, Boxed);
					if(globals->automerge)
					{
						newn=(treenode)merge_branch(newn);
						if(newn!=NULL)
							canvas_add_node(parent(newn));
					}
				}
				modify();
			}
			else
			{
				ttext_delete_string(tndta(n, NoTreeType)->label);
				modify();
			}
			break;
		default:
			break;
	}

	return(no_errors);
}

int delete_selection()
{
	sel_entry e;

	startlist(selection);

	while((e=listnext(selection))!=NULL)
	{
		delete_item(e->item, e->type);
		if(globals->mode!=MouseText)
			free(e);
		if(selection==NULL)
			break;
	}
	if(globals->mode!=MouseText)
	{
		if(selection!=NULL)
			freelist(selection);
		selection=NULL;
		tsel=NoSel;
		set_selection_type();
	}
}

int paste_selection()
{
	sel_entry e;
	treenode newnode;
	tree newtree;
	tree oldtree;

	if(tclip==NoSel)
	{
		tterr("Nothing to paste", Message);
		return(0);
	}

	/* so now tclip must == SubtreeSel || TextSel */

	if(tsel==NoSel)
	{
		if(tclip==TextSel)
		{
			tterr("Must choose a place to paste the text", Notice);
			return(0);
		}
		else if(tclip==SubtreeSel)
		{
			/* make tree out of subtree, and add */
			newnode=duplicatesubtree(NULL,
				clipboard, tree_duplicate);
			newtree=(tree)tree_add_subtree(newnode);
			copy_clip_data_to_tree(newtree);
			reformat_new_tree(newtree);
			add_to_selection(newnode, SubtreeSel, 0, 0.0);
			canvas_add_tree(newtree);
			modify();
		}
		return(1);
	}

	startlist(selection);
	e=listnext(selection);

	switch(tsel)
	{
		case SubtreeSel:
		case NodeSel:
			if(globals->mode==MouseText)
			{
				if(tclip!=TextSel)
				{
					tterr("Can only paste Text into Text", Notice);
					return(0);
				}
				else
					ttext_insert_string(tndta(e->item,
						NoTreeType)->label, clipboard);
			}
			else
			{
				nodetypedata nd;
				treenode p;
				double dist;
				int specified;
				tree t;

				if(tclip==TextSel)
				{
					tterr("Can't paste Text onto Node", Notice);
					return(0);
				}

				t=tree_get_tree(e->item);

				unselect_entry(e);
				p=parent(e->item);
				if(p==NULL)
					p=(treenode)tree_get_tree(e->item);
				newnode=duplicatesubtree(NULL,
					clipboard, tree_duplicate);
				if(clipd->isroot)
					replacesubtree(p, e->item, newnode, 0.0, 0, 1);
				else
					replacesubtree(p, e->item, newnode, 0.0, 0, 0);
				/*nd=tndta((treenode)e->item, Boxed);*/
				/*translate_subtree(newnode, nd->x, nd->y, Boxed);*/
				/*nd=tndta((treenode)e->item, Flower);*/
				/*translate_subtree(newnode, nd->x, nd->y, Flower);*/
				freesubtree(e->item);
				e->item=newnode;
				e->type=SubtreeSel;
				invalidate_spacing(t, Flower);
				invalidate_spacing(t, Boxed);
				invalidate_refit(t, Flower);
				invalidate_refit(t, Boxed);
				invalidate_format(t, Flower);
				invalidate_format(t, Boxed);
				select_entry(e);
				set_selection_type();
				modify();
			}
			break;
		case BranchPointSel:
			if(tclip==TextSel)
			{
				tterr("Can't paste Text onto Branch Point", Notice);
				return(0);
			}
			else
			{
				treenode breaknode;
				treebranch b1;
				treenode n1;
				nodetypedata nd;
				list t;
				treenode c;

				oldtree=tree_get_tree(e->item);

				/* break branch and paste */
				unselect_entry(e);
				breaknode=(treenode)break_branch(e->item, e->p);
				firstsubtree(breaknode, &t);
				c=subtree(t);
				node_set_name(breaknode, strdup(""));
				b1=getparentandbranch(breaknode, &n1);
				newnode=duplicatesubtree(NULL,
					clipboard, tree_duplicate);
				addsubtree(breaknode, newnode, 1.0, 1);
				tndta(breaknode, Boxed)->y=tndta(c, Boxed)->y;
				tndta(breaknode, Flower)->a=tndta(c, Flower)->a;
				/*position_breaknode(breaknode, Flower);*/
				/*position_breaknode(breaknode, Boxed);*/
				/*nd=tndta(breaknode, Boxed);*/
				/*translate_subtree(newnode, nd->x, nd->y, Boxed);*/
				/*nd=tndta(breaknode, Flower);*/
				/*translate_subtree(newnode, nd->x, nd->y, Flower);*/
				set_tree_type(oldtree, tdta(oldtree)->type);
				e->item=newnode;
				e->type=SubtreeSel;
				invalidate_spacing(oldtree, Flower);
				invalidate_spacing(oldtree, Boxed);
				invalidate_refit(oldtree, Flower);
				invalidate_refit(oldtree, Boxed);
				invalidate_format(oldtree, Flower);
				invalidate_format(oldtree, Boxed);
				select_entry(e);
				set_selection_type();
				modify();
			}
			break;
		case MultiSel:
			tterr("Must choose only one place to paste", BeepMessage);
			break;
		case NoSel:
		default:
			break;
	}
	return(1);
}

int paste_selection_subtree()
{
	sel_entry e;
	treenode newnode;
	tree newtree;
	tree oldtree;

	if(tclip==NoSel)
		return(paste_selection());

	if(tsel==NoSel)
		return(paste_selection());

	startlist(selection);
	e=listnext(selection);

	switch(tsel)
	{
		case SubtreeSel:
		case NodeSel:
			if(tclip==TextSel)
				return(paste_selection());
			else
			{
				nodetypedata nd;
				treenode p;
				double dist;
				int specified;
				tree t;

				t=tree_get_tree(e->item);

				unselect_entry(e);
				newnode=duplicatesubtree(NULL,
					clipboard, tree_duplicate);
				addsubtree(e->item, newnode, 1.0, 1);
				/*nd=tndta((treenode)e->item, Boxed);*/
				/*translate_subtree(newnode, nd->x, nd->y, Boxed);*/
				/*nd=tndta((treenode)e->item, Flower);*/
				/*translate_subtree(newnode, nd->x, nd->y, Flower);*/
				e->item=newnode;
				e->type=SubtreeSel;
				invalidate_spacing(t, Flower);
				invalidate_spacing(t, Boxed);
				invalidate_refit(t, Flower);
				invalidate_refit(t, Boxed);
				invalidate_format(t, Flower);
				invalidate_format(t, Boxed);
				select_entry(e);
				set_selection_type();
				modify();
			}
			break;
		case BranchPointSel:
		case TextSel:
		case MultiSel:
		case NoSel:
			return(paste_selection());
			break;
		default:
			break;
	}
	return(1);
}

treenode first_selected_node()
{
	return(lastchosen);
}

int get_one_selection(type, i, p)
selection_type type;
void **i;
double *p;
/* verifies that selection contains one item of the given type, returns */
/* returns number of items:  0=none, 1=1, 2=more, -1=wrong type */
{
	sel_entry t;

	if(tsel==NoSel)
		return(0);
	else if(tsel==MultiSel)
		return(2);

	startlist(selection);
	t=listnext(selection);

	if(t->type==type)
	{
		*i=t->item;
		*p=t->p;
		return(1);
	}
	else
		return(0);
}

list used;
typedef struct {
	void *i;
	selection_type t;
	} usedentryd, *usedentry;

static void additem(i, type)
void *i;
selection_type type;
{
	usedentry e;
	list t;

	lfor(used, t)
		if(((usedentry)nodeobj(t))->i==i)
			break;
	if(t==NULL)
	{
		e=(usedentry)malloc(sizeof(usedentryd));

		e->t=type;
		e->i=i;
		addnode(used, e);
	}
}

static void add_subtree_nodes(n)
treenode n;
{
	list t;

	additem(n, NodeSel);
	tforsubtree(n, t)
		add_subtree_nodes(subtree(t));
}

static void add_subtree_branches(n)
treenode n;
{
	list t;

	tforsubtree(n, t)
	{
		additem(nodeobj(t), BranchPointSel);
		add_subtree_branches(subtree(t));
	}
}

static void free_used()
{
	list t;

	lfor(used, t)
		free(nodeobj(t));
	freelist(used);
}

int do_on_selection(sc, c, p1, p2, d1, d2, i1, i2, i3, i4)
command sc; /* type of selections */
command c;
void *p1, *p2;
double d1, d2;
int i1, i2, i3, i4;
{
	list t;
	int tmp, result;
	void *i;
	int usenodes,usebps,usesubs,usetrees;
	treenode tn;
	tree tt;
	treebranch tb;
	selection_type ttype;

	if(tsel==NoSel)
	{
		tterr("Nothing Selected", BeepMessage);
		return(0);
	}

	usenodes=usebps=usesubs=usetrees=0;
	switch(sc)
	{
		case EveryNode:
			usenodes=1;
			break;
		case EveryNodeAndSubtree:
			usenodes=1;
			usesubs=1;
			break;
		case EveryBranch:
			usebps=1;
			break;
		case EveryTree:
			usetrees=1;
			break;
		default:
			break;
	}

	used=newlist();
	lfor(selection, t)
	{
		switch(thisentry(t)->type)
		{
			case NodeSel:
				tn=thisentry(t)->item;
				if(usenodes)
					additem(tn, NodeSel);
				else if(usetrees)
					additem(tree_get_tree(tn), TreeSel);
				break;
			case SubtreeSel:
				tn=thisentry(t)->item;
				if(usesubs)
					additem(tn, SubtreeSel);
				else if(usetrees)
					additem(tree_get_tree(tn), TreeSel);
				else if(usenodes)
					add_subtree_nodes(tn);
				else if(usebps)
					add_subtree_branches(tn);
				break;
			case BranchPointSel:
				tb=thisentry(t)->item;
				if(usebps)
					additem(tb, BranchPointSel);
				else if(usetrees)
					additem(tree_get_tree(tb), TreeSel);
				break;
			default:
				break;
		}
	}

	result=1;
	lfor(used, t)
	{
		tmp=do_command(c, ((usedentry)nodeobj(t))->i, p2, d1, d2,
			i1, i2, i3, i4, ((usedentry)nodeobj(t))->t);
		result=result&&tmp;
	}
	free_used();
	return(result);
}

static draw_entry(t, c)
sel_entry t;
tcanvas c;
{
	switch(t->type)
	{
		case NodeSel:
			draw_node_selection(t->item, c);
			break;
		case SubtreeSel:
			draw_subtree_selection(t->item, c);
			break;
		case BranchPointSel:
			draw_branch_point_selection(t->item, t->p, c);
			break;
		case TextSel:
			break;
		case NoSel:
		default:
			break;
	}
}

draw_selection(c)
tcanvas c;
{
	list t;

	if(tsel==NoSel || selection==NULL)
		return;

	lfor(selection, t)
		draw_entry(nodeobj(t), c);
}

write_selected_names(f)
FILE *f;
{
	list t;
	sel_entry c;

	if(tsel==NoSel || selection==NULL)
		return;

	lfor(selection, t)
	{
		c=nodeobj(t);
		if(c->type==NodeSel)
			write_node_name(f, c->item);
		else if(c->type==SubtreeSel)
			write_subtree_names(f, c->item);
	}
}

int remove_selections_on_tree(t)
tree t;
{
	list n;
	sel_entry c;

	/* first, remove all selections on this tree */
	lfor(selection, n)
	{
		c=nodeobj(n);

		if((c->type==SubtreeSel ||
				c->type==NodeSel ||
				c->type==BranchPointSel)
			&& tree_get_tree(c->item)==t)
		{
			if((c->type==SubtreeSel || c->type==NodeSel) &&
				lastchosen==c->item)
				lastchosen=NULL;

			n=rmnode(n);
			free(c);
		}
	}
	set_selection_type();
	return(1);
}

static void get_selection_commands(type, cm, selected, shifton, ctrlon)
selection_type *type;
int selected;
command *cm;
int shifton, ctrlon;
{
	switch(*type)
	{
		case NodeSel:
		case SubtreeSel:
			if(shifton)
			{
				if(selected)
					*cm=RemoveFromSelection;
				else
					*cm=AddToSelection;
			}
			else
				*cm=Select;
			break;
		case NodeSubtreeSel:
			if(shifton)
			{
				if(selected)
					*cm=RemoveFromSelection;
				else
					*cm=AddToSelection;
			}
			else
				*cm=Select;
			if(ctrlon)
				*type=NodeSel;
			else
				*type=SubtreeSel;
			break;
		case BranchPointSel:
			if(shifton)
			{
				if(selected)
					*cm=RemoveFromSelection;
				else
					*cm=AddToSelection;
			}
			else
				*cm=Select;
			break;
		case NoSel:
			if(shifton)
				*cm=AddToSelection;
			else
				*cm=Select;
			break;
		default:
			break;
	}
}

static void select_by_values(type, item, p, selected, shifton, ctrlon)
selection_type type;
void *item;
double p;
int selected, shifton, ctrlon;
{
	command cm;

	get_selection_commands(&type, &cm, selected, shifton, ctrlon);

	if(type!=BranchPointSel)
		p=0.0;

	do_command(cm, item, NULL, p, 0.0, type, 0, 0, 0, 0);
}

void select_by_state(type, item, p, e)
selection_type type;
void *item;
double p;
tevent e;
{
	switch(type)
	{
		case NodeSel:
		case SubtreeSel:
		case NodeSubtreeSel:
			select_by_values(type, item, p, ndta((treenode)item)->selected,
				tevent_modifier_down(e, tm_shift)||
					(tevent_button(e)==tb_adjust),
				tevent_modifier_down(e, tm_ctrl));
			break;
		case BranchPointSel:
			select_by_values(type, item, p, bdta((treebranch)item)->selected,
				tevent_modifier_down(e, tm_shift)||
					(tevent_button(e)==tb_adjust),
				0);
			break;
		case NoSel:
			select_by_values(type, item, p, 0,
				tevent_modifier_down(e, tm_shift)||
					(tevent_button(e)==tb_adjust),
				0);
			break;
		default:
			return;
	}
}
