#!/usr/bin/env python

"""
Filesystem utilities.

Copyright (C) 2014, 2015, 2017 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import errno
from imiptools.config import settings
from os.path import abspath, commonprefix, exists, join, split
from os import chmod, getpid, makedirs, mkdir, rename, rmdir
from time import sleep, time

DEFAULT_PERMISSIONS = settings["DEFAULT_PERMISSIONS"]
DEFAULT_DIR_PERMISSIONS = settings["DEFAULT_DIR_PERMISSIONS"]

def check_dir(base, filename):

    "Return whether 'base' contains 'filename'."

    return commonprefix([base, abspath(filename)]) == base

def remaining_parts(base, filename):

    "Return the remaining parts from 'base' provided by 'filename'."

    if not check_dir(base, filename):
        return None

    filename = abspath(filename)

    parts = []
    while True:
        filename, part = split(filename)
        if check_dir(base, filename):
            parts.insert(0, part)
        else:
            break

    return parts

def fix_permissions(filename, is_dir=False):

    """
    Fix permissions for 'filename', with 'is_dir' indicating whether the object
    should be a directory or not.
    """

    try:
        chmod(filename, is_dir and DEFAULT_DIR_PERMISSIONS or DEFAULT_PERMISSIONS)
    except OSError:
        pass

def make_path(base, parts):

    """
    Make the path within 'base' having components defined by the given 'parts'.
    Note that this function does not check the parts for suitability. To do so,
    use the FileBase methods instead.
    """

    for part in parts:
        pathname = join(base, part)
        if not exists(pathname):
            mkdir(pathname)
            fix_permissions(pathname, True)
        base = pathname

class FileBase:

    "Basic filesystem operations."

    lock_name = "__lock__"

    def __init__(self, store_dir):
        self.store_dir = abspath(store_dir)
        if not exists(self.store_dir):
            makedirs(self.store_dir)
            fix_permissions(self.store_dir, True)
        self.lock_depth = 0

    def get_file_object(self, base, *parts):

        """
        Within the given 'base' location, return a path corresponding to the
        given 'parts'.
        """

        # Handle "empty" components.

        pathname = join(base, *parts)
        return check_dir(base, pathname) and pathname or None

    def get_object_in_store(self, *parts):

        """
        Return the name of any valid object stored within a hierarchy specified
        by the given 'parts'.
        """

        parent = expected = self.store_dir

        # Handle "empty" components.

        parts = [p for p in parts if p]

        for part in parts:
            filename = self.get_file_object(expected, part)
            if not filename:
                return None
            parent = expected
            expected = filename

        if not exists(parent):
            make_path(self.store_dir, parts[:-1])

        return filename

    def move_object(self, source, target):

        "Move 'source' to 'target'."

        if not self.ensure_parent(target):
            return False
        rename(source, target)

    def ensure_parent(self, target):

        "Ensure that the parent of 'target' exists."

        parts = remaining_parts(self.store_dir, target)
        if not parts or not self.get_file_object(self.store_dir, *parts[:-1]):
            return False

        make_path(self.store_dir, parts[:-1])
        return True

    # Locking methods.
    # This uses the directory creation method exploited by MoinMoin.util.lock.
    # However, a simple single lock type mechanism is employed here.

    def make_lock_dir(self, *parts):

        "Make the lock directory defined by the given 'parts'."

        parts = parts and list(parts) or []
        parts.append(self.lock_name)
        d = self.get_object_in_store(*parts)
        if not d: raise OSError(errno.ENOENT, "Could not get lock in store: %r in %r" % (parts, self.store_dir))
        mkdir(d)
        parts.append(str(getpid()))
        d = self.get_object_in_store(*parts)
        if not d: raise OSError(errno.ENOENT, "Could not get lock in store: %r in %r" % (parts, self.store_dir))
        mkdir(d)

    def remove_lock_dir(self, *parts):

        "Remove the lock directory defined by the given 'parts'."

        parts = parts and list(parts) or []

        parts.append(self.lock_name)
        parts.append(str(getpid()))
        rmdir(self.get_object_in_store(*parts))
        parts.pop()
        rmdir(self.get_object_in_store(*parts))

    def owning_lock_dir(self, *parts):

        "Return whether this process owns the lock directory."

        parts = parts and list(parts) or []
        parts.append(self.lock_name)
        parts.append(str(getpid()))
        return exists(self.get_object_in_store(*parts))

    def acquire_lock(self, timeout=None, *parts):

        """
        Acquire an exclusive lock on the directory or a path within it described
        by 'parts'.
        """

        start = now = time()

        while not timeout or now - start < timeout:
            try:
                self.make_lock_dir(*parts)
                break
            except OSError, exc:
                if exc.errno != errno.EEXIST:
                    raise
                elif self.owning_lock_dir(*parts):
                    self.lock_depth += 1
                    break
            sleep(1)
            now = time()

    def release_lock(self, *parts):

        """
        Release an acquired lock on the directory or a path within it described
        by 'parts'.
        """

        try:
            if self.lock_depth != 0:
                self.lock_depth -= 1
            else:
                self.remove_lock_dir(*parts)
        except OSError, exc:
            if exc.errno not in (errno.ENOENT, errno.ENOTEMPTY):
                raise

# vim: tabstop=4 expandtab shiftwidth=4
