/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.openapi.roots;

import com.intellij.openapi.application.ReadAction;
import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiElement;
import org.jetbrains.annotations.NotNull;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * @author nik
 */
public abstract class GeneratedSourcesFilter {
  public static final ExtensionPointName<GeneratedSourcesFilter> EP_NAME = ExtensionPointName.create("com.intellij.generatedSourcesFilter");

  public static boolean isGeneratedSourceByAnyFilter(@NotNull VirtualFile file, @NotNull Project project) {
    return ReadAction.compute(() -> !project.isDisposed() && file.isValid() &&
                                    Arrays.stream(EP_NAME.getExtensions()).anyMatch(filter -> filter.isGeneratedSource(file, project)));
  }

  public abstract boolean isGeneratedSource(@NotNull VirtualFile file, @NotNull Project project);

  /**
   * Returns all elements that have been processed by a code generator to derive the given element.
   *
   * @param element the generated element
   * @return a list of original elements. An empty result indicates that the element is not considered to be generated by the filter.
   */
  @NotNull
  public List<? extends PsiElement> getOriginalElements(@NotNull PsiElement element) {
    return Collections.emptyList();
  }
}
